/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ImageResponseFilter.h"
#include "AbstractImageInspector.h"
#include "SplittableBuffer.h"
#include "GlobalState.h"
#include "Conf.h"
#include "HttpStatusLine.h"
#include "HttpResponseMetadata.h"
#include "HttpHeadersCollection.h"
#include "HeuristicScore.h"
#include "ImmediateResponse.h"
#include "ReplacementImage.h"
#include "DownloadProgress.h"
#include "ErrorDescriptor.h"
#include "ErrorFactory.h"
#include "ErrorCodes.h"
#include "IntrusivePtr.h"
#include "RequestStatus.h"
#include "RequestTag.h"
#include "BString.h"
#include <stddef.h>

using namespace std;

ImageResponseFilter::ImageResponseFilter(
	ResponseFilterChain& chain, bool ignore_size,
 	std::auto_ptr<AbstractImageInspector> inspector)
:	ResponseFilterBase(chain),
	m_state(INSPECTING),
	m_ignoreSize(ignore_size),
	m_ptrInspector(inspector)
{
}

ImageResponseFilter::~ImageResponseFilter()
{
}

void
ImageResponseFilter::processMetadata(
	RequestStatus&, std::auto_ptr<HttpResponseMetadata> metadata)
{
	m_ptrMetadata = metadata;
}

void
ImageResponseFilter::processBodyData(
	RequestStatus& status, SplittableBuffer& data, bool eof)
{
	switch (m_state) {
		case FORWARDING: {
			outputBodyData(status, data, eof);
			break;
		}
		case SUBSTITUTING: {
			data.clear();
			break;
		}
		case INSPECTING: {
			m_ptrInspector->consumeDataChunk(data, eof);
			switch (m_ptrInspector->getStatus()) {
				case AbstractImageInspector::COMPLETE: {
					if (isAd()) {
						m_state = SUBSTITUTING;
 						outputAdSubstitution(status);
						considerRequestCancel(status);
						break;
					}
					// fall through
				}
				case AbstractImageInspector::FAILED: {
					m_state = FORWARDING;
					outputMetadata(status, m_ptrMetadata);
					outputBodyData(
						status,
						m_ptrInspector->accumulatedData(),
						eof
					);
					break;
				}
				default: break;
			}
			break;
		}
	}
}

bool
ImageResponseFilter::isAd()
{
	return m_ptrInspector->isAnimated() || (
		!m_ignoreSize &&
		HeuristicScore::isCommonAdSize(
			m_ptrInspector->getWidth(),
			m_ptrInspector->getHeight()
		)
	);
}

void
ImageResponseFilter::outputAdSubstitution(RequestStatus& status)
{
	auto_ptr<ImmediateResponse> response(
		ReplacementImage::createHttpResponse(
			m_ptrInspector->getWidth(),
			m_ptrInspector->getHeight(),
			HttpStatusLine::SC_OK
		)
	);
	outputResponse(status, *response);
}

void
ImageResponseFilter::considerRequestCancel(RequestStatus& status)
{
	DownloadProgress const& dp = getFilterChain().getDownloadProgress();
	size_t threshold = GlobalState::ReadAccessor()->config().getSaveTrafficThreshold();
	if (dp.total() - dp.received() > threshold*1024) { // or if dp.total() is undefined
		status.cancel(
			ErrorFactory::errDummyError(
				ErrorCodes::REQUEST_CANCEL_TO_SAVE_TRAFFIC,
				"request cancelled to save traffic"
			)
		);
	}
}
