/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ServerReachabilityDB.h"
#include "ServerReachability.h"
#include "InetAddr.h"
#include <ace/config-lite.h>
#include <ace/Synch.h>
#include <ace/Singleton.h>
#include <boost/multi_index_container.hpp>
#include <boost/multi_index/member.hpp>
#include <boost/multi_index/ordered_index.hpp>
#include <boost/multi_index/sequenced_index.hpp>
#include <utility>
#include <cassert>

using namespace std;
using namespace boost::multi_index;

struct ServerReachabilityDB::Entry
{
	Entry(InetAddr const& addr, Record const& record)
	: addr(addr), record(record) {}
	
	InetAddr addr;
	Record record;
};


class ServerReachabilityDB::Impl : public ServerReachabilityDB
{
public:
	Impl();
	
	virtual ~Impl();
	
	virtual void put(InetAddr const& addr, Record const& rec);
	
	virtual std::vector<Record> get(std::vector<InetAddr> const& addrs) const;
private:
	class AddrTag {};
	class SeqTag {};
	
	typedef ACE_Thread_Mutex Mutex;
	typedef multi_index_container<
		Entry,
		indexed_by<
			ordered_unique<
				tag<AddrTag>,
				member<Entry, InetAddr, &Entry::addr>
			>,
			sequenced<
				tag<SeqTag>
			>
		>
	> Container;
	typedef Container::index<AddrTag>::type AddrIdx;
	typedef Container::index<SeqTag>::type SeqIdx;
	
	Record getRecordFor(InetAddr const& addr) const;
	
	mutable Mutex m_mutex;
	Container m_container;
};


ServerReachabilityDB*
ServerReachabilityDB::instance()
{
	return ACE_Singleton<Impl, ACE_Recursive_Thread_Mutex>::instance();
}


ServerReachabilityDB::Impl::Impl()
{
}

ServerReachabilityDB::Impl::~Impl()
{
}

void
ServerReachabilityDB::Impl::put(InetAddr const& addr, Record const& rec)
{
	ACE_GUARD_RETURN(Mutex, guard, m_mutex, );
	
	AddrIdx& addr_idx = m_container.get<AddrTag>();
	SeqIdx& seq_idx = m_container.get<SeqTag>();
	
	Entry entry(addr, rec);
	
	SeqIdx::iterator seq_pos;
	pair<AddrIdx::iterator, AddrIdx::iterator> addr_range(
		addr_idx.equal_range(entry.addr)
	);
	if (addr_range.first != addr_range.second) {
		// replacing an entry with the same address
		addr_idx.replace(addr_range.first, entry);
		seq_pos = m_container.project<SeqTag>(addr_range.first);
	} else if (m_container.size() >= CAPACITY) {
		// replacing the first entry in sequence (the oldest one)
		seq_pos = seq_idx.begin();
		seq_idx.replace(seq_pos, entry);
	} else {
		// inserting a new entry
		addr_idx.insert(addr_range.first, entry);
		return; // entry inserted at the end of seq_idx, no need to relocate it
	}
	
	seq_idx.relocate(seq_idx.end(), seq_pos);
}

std::vector<ServerReachabilityDB::Record>
ServerReachabilityDB::Impl::get(std::vector<InetAddr> const& addrs) const
{
	vector<Record> res;
	ACE_GUARD_RETURN(Mutex, guard, m_mutex, res);
	
	vector<InetAddr>::const_iterator it = addrs.begin();
	vector<InetAddr>::const_iterator const end = addrs.end();
	for (; it != end; ++it) {
		res.push_back(getRecordFor(*it));
	}
	
	return res;
}

ServerReachabilityDB::Record
ServerReachabilityDB::Impl::getRecordFor(InetAddr const& addr) const
{
	AddrIdx const& addr_idx = m_container.get<AddrTag>();
	AddrIdx::const_iterator it = addr_idx.find(addr);
	if (it != addr_idx.end()) {
		return it->record;
	} else {
		return Record();
	}
}
