/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Socks5AsyncConnector.h"
#include "SocksConnectorListener.h"
#include "SocksError.h"

using namespace std;

Socks5AsyncConnector::Socks5AsyncConnector()
:	m_state(ST_INACTIVE),
	m_refCount(0),
	m_targetAddr(std::string(), -1),
	m_pReactor(0)
{
}

Socks5AsyncConnector::~Socks5AsyncConnector()
{
	abort();
}

void
Socks5AsyncConnector::initiate(
	Listener& listener, Reactor& reactor,
	AutoClosingSAP<ACE_SOCK_Stream>& proxy_conn,
	SymbolicInetAddr const& target_addr,
	std::string const& username, std::string const& password,
	TimeDelta const* timeout)
{
	abort();
	
	m_state = ST_AUTHENTICATING;
	m_proxyConn = proxy_conn;
	m_targetAddr = target_addr;
	m_pReactor = &reactor;
	m_observerLink.setObserver(&listener);
	
	m_timeoutTimerId = m_pReactor->registerTimer(
		IntrusivePtr<EventHandlerBase>(this), timeout
	);
	
	m_authenticator.startAuthentication(
		*this, *m_pReactor, m_proxyConn.get_handle(),
		username, password
	);
	// It may call abort() through a callback, so nothing must be below.
}

void
Socks5AsyncConnector::abort()
{
	if (m_state != ST_INACTIVE) {
		m_observerLink.setObserver(0);
		unregisterTimer();
		m_authenticator.abort();
		m_requester.abort();
		m_proxyConn.close();
		(void)SymbolicInetAddr(std::string(), -1).swap(m_targetAddr);
		m_state = ST_INACTIVE;
	}
	assert(m_refCount == 0); // otherwise we are leaking reactor registrations
}

void
Socks5AsyncConnector::unregisterTimer()
{
	if (m_timeoutTimerId) {
		assert(m_pReactor);
		m_pReactor->unregisterTimer(m_timeoutTimerId);
		m_timeoutTimerId = ReactorTimerId();
	}
}

void
Socks5AsyncConnector::ref()
{
	++m_refCount;
}

void
Socks5AsyncConnector::unref()
{
	if (--m_refCount == 0) {
		m_pReactor = 0;
		m_timeoutTimerId = ReactorTimerId();
	}
}

void
Socks5AsyncConnector::handleTimeout(ReactorTimerId const&)
{
	handleConnectFailure(SocksError(SocksError::SOCKS_SERVER_TIMEOUT));
}

void
Socks5AsyncConnector::onAuthSuccess()
{
	m_state = ST_REQUESTING_CONNECTION;
	
	m_requester.requestConnection(
		*this, *m_pReactor,
		m_proxyConn.get_handle(), m_targetAddr
	);
	// It may call abort() through a callback, so nothing must be below.
}

void
Socks5AsyncConnector::onAuthFailure(SocksError const& err)
{
	handleConnectFailure(err);
}

void
Socks5AsyncConnector::onRequestSuccess()
{
	AutoClosingSAP<ACE_SOCK_Stream> strm(m_proxyConn);
	Listener* listener = m_observerLink.getObserver();
	abort(); // this will detach the listener
	if (listener) {
		listener->onConnThroughSocksDone(strm);
	}
}

void
Socks5AsyncConnector::onRequestFailure(SocksError const& err)
{
	handleConnectFailure(err);
}

void
Socks5AsyncConnector::handleConnectFailure(SocksError const& err)
{
	Listener* listener = m_observerLink.getObserver();
	abort(); // this will detach the listener
	if (listener) {
		listener->onConnThroughSocksFailed(err);
	}
}

