/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "FileIO.h"
#include "LimitedSpaceManager.h"
#include <ace/config-lite.h>
#include <ace/OS_NS_unistd.h>

namespace HttpCache
{

FileIO::FileIO(FileHandle& file_handle,
	IntrusivePtr<LimitedSpaceObject> const& limited_object)
:	m_curPos(0),
	m_ptrLimitedObject(limited_object),
	m_handle(file_handle)
{
}

FileIO::~FileIO()
{
}

ssize_t
FileIO::read(void* const buf, size_t const len)
{
	return ACE_OS::read(m_handle.get(), buf, len);
}

size_t
FileIO::read_n(void* buf, size_t const len, ssize_t* last_result)
{
	size_t read = 0;
	ssize_t const res = ACE_OS::read_n(m_handle.get(), buf, len, &read);
	if (last_result) {
		*last_result = res;
	}
	return read;
}

ssize_t
FileIO::write(void const* const buf, size_t const len)
{
	GrowRequest grow_req(m_ptrLimitedObject, m_curPos + len);
	if (!grow_req.requestGranted()) {
		errno = ENOSPC;
		return -1;
	}
	
	ssize_t const written = ACE_OS::write(m_handle.get(), buf, len);
	if (written > 0) {
		m_curPos += written;
		grow_req.setNewSize(m_curPos);
	}
	
	return written;
}

size_t
FileIO::write_n(void const* const buf, size_t const len)
{
	GrowRequest grow_req(m_ptrLimitedObject, m_curPos + len);
	if (!grow_req.requestGranted()) {
		errno = ENOSPC;
		return -1;
	}
	
	size_t written = 0;
	ACE_OS::write_n(m_handle.get(), buf, len, &written);
	if (written > 0) {
		m_curPos += written;
		grow_req.setNewSize(m_curPos);
	}
	return written;
}

off_t
FileIO::seek(off_t const offset, int const whence)
{
	off_t const new_pos = ACE_OS::lseek(m_handle.get(), offset, whence);
	if (new_pos != -1) {
		m_curPos = new_pos;
	}
	return new_pos;
}

off_t
FileIO::tell()
{
	return m_curPos;
}

int
FileIO::truncate(off_t const len)
{
	ResizeRequest resize_req(m_ptrLimitedObject, len);
	int const res = ACE_OS::ftruncate(m_handle.get(), len);
	if (res != -1) {
		resize_req.confirm();
	}
	return res;
}

} // namespace HttpCache
