/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "AnimatedTrayIcon.h"
#include <algorithm>
#include <cassert>

using namespace std;

namespace wxGUI
{

AnimatedTrayIcon::AnimatedTrayIcon()
:	m_timer(*this)
{
}

AnimatedTrayIcon::~AnimatedTrayIcon()
{
}

void
AnimatedTrayIcon::setIcon(
	wxIcon const& icon, wxString const& tooltip, int flags)
{
	m_nextAnimation.clear();
	m_nextAnimation.frames.assign(1, Frame(icon, 0));
	m_nextAnimation.tooltip = tooltip;
	m_nextAnimation.flags = flags;
	if (!(flags & FINISH_CURRENT_ITERATION)
	    || m_curAnimation.isFinalFrame()) {
		switchToNextAnimation();
	}
}

void
AnimatedTrayIcon::removeIcon()
{
	m_timer.Stop();
	RemoveIcon();
	m_curAnimation.clear();
	m_nextAnimation.clear();
}

void
AnimatedTrayIcon::onTimer()
{
	if (m_curAnimation.isFinalFrame()) {
		return;
	}
	
	if (++m_curAnimation.curFrame < m_curAnimation.frames.size()) {
		switchToFrame(m_curAnimation.curFrame);
	} else if (!m_nextAnimation.frames.empty()) {
		switchToNextAnimation();
	} else if (m_curAnimation.flags & LOOP) {
		assert(!m_curAnimation.frames.empty());
		switchToFrame(0);
	}
}

void
AnimatedTrayIcon::switchToFrame(int frame)
{
	m_curAnimation.curFrame = frame;
	Frame& frm = m_curAnimation.frames[frame];
	SetIcon(frm.icon, m_curAnimation.tooltip);
	if (m_curAnimation.isFinalFrame()) {
		m_timer.Stop();
	} else {
		m_timer.Start(frm.delay_ms, true);
	}
}

void
AnimatedTrayIcon::switchToNextAnimation()
{
	m_timer.Stop();
	m_nextAnimation.swap(m_curAnimation);
	m_nextAnimation.clear();
	if (m_curAnimation.frames.empty()) {
		RemoveIcon();
	} else {
		switchToFrame(0);
	}
}


/*==================== AnimatedTrayIcon::Animation ======================*/

bool
AnimatedTrayIcon::Animation::isFinalFrame() const
{
	if (frames.size() < 2) {
		return true;
	}
	if (flags & LOOP) {
		return false;
	}
	if (curFrame < frames.size() - 1) {
		return false;
	}
	return true;
}

void
AnimatedTrayIcon::Animation::clear()
{
	frames.clear();
	curFrame = 0;
	tooltip.Clear();
	flags = 0;
}

void
AnimatedTrayIcon::Animation::swap(Animation& other)
{
	frames.swap(other.frames);
	std::swap(curFrame, other.curFrame);
	tooltip.swap(other.tooltip);
	std::swap(flags, other.flags);
}


/*===================== AnimatedTrayIcon::Timer ========================*/

AnimatedTrayIcon::Timer::Timer(AnimatedTrayIcon& owner)
:	m_rOwner(owner)
{
}

AnimatedTrayIcon::Timer::~Timer()
{
}

void
AnimatedTrayIcon::Timer::Notify()
{
	m_rOwner.onTimer();
}

} // namespace wxGUI
