/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef WXGUI_CONTENTFILTERGROUP_H_
#define WXGUI_CONTENTFILTERGROUP_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ContentFilterList.h"
#include "FilterFileStructure.h"
#include "FilterGroupTag.h"
#include <wx/string.h>
#include <wx/filename.h>

class RegexFilterDescriptor;
class FilterTag;

namespace wxGUI
{

class ContentFilterGroup
{
public:
	enum FilterTagPolicy { PRESERVE_FILTER_TAGS, NEW_FILTER_TAGS };
	
	explicit ContentFilterGroup(wxString const& fname);
	
	/**
	 * \brief Creates a shallow copy of another group.
	 *
	 * Group and filter tags will be preserved.
	 */
	ContentFilterGroup(ContentFilterGroup const& other);
	
	/**
	 * \brief Creates a deep copy of another group.
	 *
	 * The new copy will have its own group tag.
	 * Filter tags may optionally be preserved or recreated.
	 */
	ContentFilterGroup(
		wxString const& fname,
		ContentFilterGroup const& prototype,
		FilterTagPolicy filter_tag_policy);
	
	virtual ~ContentFilterGroup();
	
	ContentFilterGroup& operator=(ContentFilterGroup const& rhs);
	
	void swap(ContentFilterGroup& other);
	
	bool load();
	
	bool save() const;
	
	bool saveFilterFileOnly() const;
	
	bool saveEnabledFileOnly() const;
	
	bool deleteFromDisk() const;
	
	bool renameOnDisk(wxString const& new_fname) const;
	
	FilterGroupTag const& getTag() const { return m_tag; }
	
	wxString const& getFileName() const { return m_fileName; }
	
	ContentFilterList const& getFilters() const { return m_filters; }
	
	/**
	 * \brief Find a filter by tag.
	 */
	RegexFilterDescriptor const* findFilter(FilterTag const& id) const;
	
	/**
	 * \brief Append a filter.
	 * \return true on success, false if a filter with the same tag
	 *         already exists.
	 */
	bool appendFilter(RegexFilterDescriptor const& filter);
	
	/**
	 * \brief Insert a filter before another one.
	 * \return true on success, false if the filter indicated by \p before
	 *         wasn't found.
	 */
	bool insertFilter(RegexFilterDescriptor const& filter, FilterTag const& before);
	
	/**
	 * \brief Update a filter having the same tag.
	 * \return true on success, false if the filter with the same tag
	 *         wasn't found.
	 */
	bool updateFilter(RegexFilterDescriptor const& filter);
	
	/**
	 * \brief Rename a filter.
	 * \return true on success, false if the filter indicated by \p id
	 *         wasn't found.
	 */
	bool renameFilter(FilterTag const& id, std::string const& new_name);
	
	/**
	 * \brief Delete a filter.
	 * \return true on success, false if the filter indicated by \p id
	 *         wasn't found.
	 */
	bool deleteFilter(FilterTag const& id);
	
	/**
	 * \brief Swap positions of two filters.
	 * \return true on success, false if any of the two filters wasn't
	 *         found.
	 */
	bool swapFilters(FilterTag const& id1, FilterTag const& id2);
	
	/**
	 * \brief Enable or disable a filter.
	 * \return true on success, false if the filter indicated by \p id
	 *         wasn't found.
	 */
	bool filterSetEnabled(FilterTag const& id, bool enabled);
	
	/**
	 * \brief Enable or disable all filters.
	 */
	void allFiltersSetEnabled(bool enabled);
private:
	wxFileName getFilterFileDir() const;
	
	wxFileName getEnabledFileDir() const;
	
	wxFileName getFilterFilePath(wxString const& filter_fname) const;
	
	wxFileName getFilterFilePath() const;
	
	wxFileName getEnabledFilePath(wxString const& filter_fname) const;
	
	wxFileName getEnabledFilePath() const;
	
	void loadEnabledFilterList(ContentFilterList& filters);
	
	wxString m_fileName;
	FilterGroupTag m_tag;
	FilterFileStructure m_fileStructure;
	ContentFilterList m_filters;
};


inline void swap(ContentFilterGroup& o1, ContentFilterGroup& o2)
{
	o1.swap(o2);
}

} // namespace wxGUI

#endif
