/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ImageHeuristicScore.h"
#include "FlashHeuristicScore.h"
#include "BString.h"
#include "URI.h"
#include <boost/test/auto_unit_test.hpp>

static void processImageUrls(
	ImageHeuristicScore& score,
	URI const& page, URI const& link, URI const& img);

static void processFlashUrls(
	FlashHeuristicScore& score, URI const& page, URI const& flash);

static void checkAtLeast(
	int id, HeuristicScore const& score, HeuristicScore::Status status);

static void checkAtMost(
	int id, HeuristicScore const& score, HeuristicScore::Status status);

BOOST_AUTO_UNIT_TEST(test1)
{
	ImageHeuristicScore score;
	URI const page(BString("http://nnm.ru/"));
	URI const img(BString("http://www.it-territory.ru/images/3/600x120_3.jpg"));
	URI const link(BString("http://partners.territory.ru/count.php?site_id=854&event_id=1&user_id=-1&url=http://dwar.ru/index.php%3F1%26site_id=854"));
	score.processSize(600, 120);
	processImageUrls(score, page, link, img);
	score.processLinkTarget(BString("_blank"));
	score.processNoscript(false);
	score.processAssociatedMap(false);
	score.processScriptLevel(0, false);
	checkAtLeast(1, score, score.AD);
}

BOOST_AUTO_UNIT_TEST(test2)
{
	ImageHeuristicScore score;
	URI const page(BString("http://www.alcohol-soft.com/"));
	URI const img(BString("http://images.alcohol-soft.com/www/download.gif"));
	URI const link(BString("http://www.free-downloads.net/programs/Alcohol_120_"));
	score.processSize(-1, -1);
	processImageUrls(score, page, link, img);
	score.processLinkTarget(BString(""));
	score.processNoscript(false);
	score.processAssociatedMap(false);
	score.processScriptLevel(0, false);
	checkAtMost(2, score, score.PROBABLY_NOT_AD);
}

BOOST_AUTO_UNIT_TEST(test3)
{
	FlashHeuristicScore score;
	URI const page(BString("http://911blogger.com/"));
	URI const flash(BString("http://www.youtube.com/v/6d8MIENVtKw"));
	score.processSize(425, 350);
	processFlashUrls(score, page, flash);
	score.processOptions(false, true);
	score.processNoscript(false);
	score.processScriptLevel(0, false);
	checkAtMost(3, score, score.PROBABLY_AD);
}

BOOST_AUTO_UNIT_TEST(test4)
{
	FlashHeuristicScore score;
	URI const page(BString("http://911blogger.com/"));
	URI const flash(BString("http://www.webcameron.org.uk/videos/vidplayer.swf?filePath=../content/video/2090.flv"));
	score.processSize(320, 280);
	processFlashUrls(score, page, flash);
	score.processOptions(false, false);
	score.processNoscript(false);
	score.processScriptLevel(0, false);
	checkAtMost(4, score, score.PROBABLY_AD);
}

static void processImageUrls(
	ImageHeuristicScore& score,
	URI const& page, URI const& link, URI const& img)
{
	URI const link_abs(page, link);
	URI const img_abs(page, img);
	score.processUrls(page, img_abs, img, link_abs, link);
}

static void processFlashUrls(
	FlashHeuristicScore& score, URI const& page, URI const& flash)
{
	URI const flash_abs(page, flash);
	score.processUrl(flash_abs, flash, page);
}

static void checkAtLeast(
	int const id, HeuristicScore const& score,
	HeuristicScore::Status const status)
{
	char const* str = 0;
	if (score.getStatus() >= status) {
		str = "OK";
	} else {
		str = "TOO LITTLE";
	}
	BOOST_MESSAGE('[' << id << "] " << str << ": " << score.getNumericScore());	
}

static void checkAtMost(
	int const id, HeuristicScore const& score,
	HeuristicScore::Status const status)
{
	char const* str = 0;
	if (score.getStatus() <= status) {
		str = "OK";
	} else {
		str = "TOO MUCH";
	}
	BOOST_MESSAGE('[' << id << "] " << str << ": " << score.getNumericScore());	
}
