/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "exception.h"
#include <gtk/gtk.h>
#include <config.h>
#include "utilities.h"
#include "directories.h"
#include "gwrappers.h"
#include "stylesheetutils.h"
#include "shell.h"
#include "projectutils.h"
#include "scripting.h"
#include "import.h"
#include "categorize.h"
#include "progresswindow.h"
#include "books.h"
#include "generalconfig.h"
#include "synchronize.h"
#include "notes_utils.h"
#include "export_utils.h"
#include "projectconfig.h"
#include "checks.h"
#include "check_validate_usfm.h"
#include "check_count_words.h"


void scripting_usage (GOptionEntry entries[], bool subscript);
ustring scripting_get_from_gchar (gchar * argument);
void scripting_handle_error (GError * error);
void scripting_ensure_project (const ustring& project);
void initialize_gui (bool gui, int argc, char *argv[]);
int export_stylesheet (int argc, char *argv[]);
int upgrade_data (int argc, char *argv[]);
int import_book (int argc, char *argv[]);
int export_book (int argc, char *argv[]);
int shutdown_actions (int argc, char *argv[]);
int export_project_usfm (int argc, char *argv[]);
int export_project_sword (int argc, char *argv[]);
int general_configuration (int argc, char *argv[]);
int project_configuration (int argc, char *argv[]);
int create_project (int argc, char *argv[]);
int delete_project (int argc, char *argv[]);
int validate_markers (int argc, char *argv[]);
int count_words (int argc, char *argv[]);


int scripting (int argc, char *argv[])
{
  // Parse options.
  bool version = false;
  bool upgradedata = false;
  bool importbook = false;
  bool exportbook = false;
  bool exportprojectusfm = false;
  bool exportprojectsword = false;
  bool exportstylesheet = false;
  bool shutdownactions = false;
  bool generalconfiguration = false;
  bool projectconfiguration = false;
  bool createproject = false;
  bool deleteproject = false;
  bool validatemarkers = false;
  bool countwords = false;
  GOptionEntry entries[] = 
  {
    { "version", gchar (NULL), 0, G_OPTION_ARG_NONE, &version, "Version number", NULL },
    { "upgrade-data", gchar (NULL), 0, G_OPTION_ARG_NONE, &upgradedata, "Upgrade all relevant data", NULL },
    { "import-book", gchar (NULL), 0, G_OPTION_ARG_NONE, &importbook, "Import a book", NULL },
    { "export-book", gchar (NULL), 0, G_OPTION_ARG_NONE, &exportbook, "Export a book", NULL },
    { "export-project-usfm", gchar (NULL), 0, G_OPTION_ARG_NONE, &exportprojectusfm, "Export a project to USFM", NULL },
    { "export-project-sword", gchar (NULL), 0, G_OPTION_ARG_NONE, &exportprojectsword, "Export a project to SWORD", NULL },
    { "export-stylesheet", gchar (NULL), 0, G_OPTION_ARG_NONE, &exportstylesheet, "Export a stylesheet", NULL },
    { "shutdown-actions", gchar (NULL), 0, G_OPTION_ARG_NONE, &shutdownactions, "Called on Bibledit shutdown", NULL },
    { "general-configuration", gchar (NULL), 0, G_OPTION_ARG_NONE, &generalconfiguration, "Access Bibledit's general configuration", NULL },
    { "project-configuration", gchar (NULL), 0, G_OPTION_ARG_NONE, &projectconfiguration, "Access Bibledit's project configuration", NULL },
    { "create-project", gchar (NULL), 0, G_OPTION_ARG_NONE, &createproject, "Create a project", NULL },
    { "delete-project", gchar (NULL), 0, G_OPTION_ARG_NONE, &deleteproject, "Delete a project", NULL },
    { "validate-markers", gchar (NULL), 0, G_OPTION_ARG_NONE, &validatemarkers, "Validate USFM markers", NULL },
    { "count-words", gchar (NULL), 0, G_OPTION_ARG_NONE, &countwords, "Count words", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_set_ignore_unknown_options (context, true);
  g_option_context_set_help_enabled (context, false);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  if (error) {
    gw_error (error->message);
    g_error_free (error);
  }

  // Take action depending on options.
  if (version) {
    cout << VERSION << endl;
    return 0;
  } else if (upgradedata) {
    return upgrade_data (argc, argv);
  } else if (importbook) {
    return import_book (argc, argv);
  } else if (exportbook) {
    export_book (argc, argv);
  } else if (exportprojectusfm) {
    export_project_usfm (argc, argv);
  } else if (exportprojectsword) {
    export_project_sword (argc, argv);
  } else if (exportstylesheet) {
    export_stylesheet (argc, argv);
  } else if (shutdownactions) {
    shutdown_actions (argc, argv);
  } else if (generalconfiguration) {
    general_configuration (argc, argv);
  } else if (projectconfiguration) {
    project_configuration (argc, argv);
  } else if (createproject) {
    create_project (argc, argv);
  } else if (deleteproject) {
    delete_project (argc, argv);
  } else if (validatemarkers) {
    validate_markers (argc, argv);
  } else if (countwords) {
    count_words (argc, argv);
  } else {
    scripting_usage (entries, false);
  }
  
  // Ready.
  return 0;
}


void scripting_usage (GOptionEntry entries[], bool subscript)
{
  int pointer = 0;
  cout << "Possible options:" << endl << endl;
  while (entries[pointer].long_name) {
    cout << "--" << entries[pointer].long_name;
    int length = strlen (entries[pointer].long_name);
    for (unsigned int i = length; i < 25; i++) cout << " ";
    cout << entries[pointer].description << endl;
    pointer++;
  }
  cout << endl;
  if (subscript) {
    ustring s = "Bibledit needs more information";
    gw_error (s.c_str());
  } else {
    cout << "Add --help to any of the above to get specific help, e.g. --" << entries[3].long_name << " --help" << endl << endl;
  }
}


ustring scripting_get_from_gchar (gchar * argument)
{
  ustring value;
  if (argument) { 
    value = argument; 
    g_free (argument);
  }
  return value;
}


void scripting_handle_error (GError * error)
{
  if (error) {
    ustring s = error->message;
    g_error_free (error); 
    gw_error (s.c_str()); 
  }
}


void scripting_ensure_project (const ustring& project)
{
  if (!project_exists (project)) {
    cerr << "Project " << project << " does not exist" << endl;
    exit (EXIT_FAILURE);
  }
}


void initialize_gui (bool gui, int argc, char *argv[])
{
  if (gui)
    gtk_init (&argc, &argv);
}


int export_stylesheet (int argc, char *argv[])
// Exports a stylesheet from Bibledit's database to a zip file with individual style files.
{
  // Parse options.
  ustring stylesheet;
  ustring file;
  gchar * g_stylesheet = NULL;
  gchar * g_file = NULL;
  GOptionEntry entries[] = 
  {
    { "stylesheet", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_stylesheet, "Name of the stylesheet to be exported", NULL },
    { "file", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_file, "Name of the file to write to", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  stylesheet = scripting_get_from_gchar (g_stylesheet);
  file = scripting_get_from_gchar (g_file);

  // Usage.
  if (stylesheet.empty() || file.empty()) scripting_usage (entries, true);

  // Check stylesheet.
  if (!stylesheet_exists (stylesheet)) {
    cerr << "Stylesheet " << stylesheet << " not found." << endl;
    return 1;
  }

  // Make and clear a temporal directory to work in.
  ustring tempdir = gw_build_filename (directories_get_temp(), "stylesheet-export");
  remove_directory (tempdir);
  create_directory (tempdir);
  
  // Get all styles, and export them to xml files in that temporal directory.
  vector<Style> styles;
  stylesheet_get_styles (stylesheet, styles);
  for (unsigned int i = 0; i < styles.size(); i++) {
    ustring filename = gw_build_filename (tempdir, styles[i].marker + STYLE_SUFFIX);
    styles[i].export_it (filename);
  }
  
  // Zip all individual styles. 
  ustring zipfile = "zip.zip";
  ustring command;
  command = "cd ";
  command.append (shell_quote_space (tempdir));
  command.append ("; zip");
  command.append (shell_quote_space (zipfile));
  command.append ("*");
  command.append (STYLE_SUFFIX);
  system (command.c_str());  

  // Get output filename.
  file.append (".zip");
  
  // Copy zipfile to outputfile.
  command = "cp";
  command.append (shell_quote_space (gw_build_filename (tempdir, zipfile)));
  command.append (shell_quote_space (file));
  cout << "Saving to " << file << endl;
  system (command.c_str());
  
  return 0;
}


int upgrade_data (int argc, char *argv[])
// Upgrades older data to the currently used formats.
{
  // Parse options.
  bool gui = false;
  GOptionEntry entries[] = 
  {
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Show graphical progressbar", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (gui, argc, argv);

  // Upgrade data.
  stylesheets_upgrade ();
  projects_initial_check (gui);
  notes_database_verify (gui);
  notes_categories_check ();
  upgrade_configuration ();
  vector <ustring> projects = projects_get_all ();
  for (unsigned int i = 0; i < projects.size(); i++) {
    upgrade_project_configuration (projects[i]);
  }
  
  return 0;
}


int import_book (int argc, char *argv[])
// Imports a USFM file into Bibledit's database, with good error checking.
{
  // Parse options.
  ustring project;
  ustring file;
  ustring book;
  ustring encoding;  
  gchar * g_project = NULL;
  gchar * g_file = NULL;
  gchar * g_book = NULL;
  gchar * g_encoding = NULL;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project to import the book into", NULL },
    { "file", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_file, "Name of the USFM file which should be imported", NULL },
    { "book", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_book, "Optional: the name of the book, e.g. Matthew", NULL },
    { "encoding", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_encoding, "Optional: the encoding of the file, e.g. iso-8859-15", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  project = scripting_get_from_gchar (g_project);
  file = scripting_get_from_gchar (g_file);
  book = scripting_get_from_gchar (g_book);
  encoding = scripting_get_from_gchar (g_encoding);

  // Usage.
  if (project.empty() || file.empty()) scripting_usage (entries, true);

  // Check project.
  scripting_ensure_project (project);
    
  // Read the book.
  ImportBookRead ibr (file, encoding);
  ibr.usfm ();
  if (ibr.lines.size() == 0) {
    cerr << "Could not read file " << file << "." << endl;
    return 1;
  }
  
  // Handle bookname.
  if (book.empty()) {
    book = ibr.bookname;
  }
  if (book.empty()) {
    cerr << "Could not determine the bookname." << endl;
    return 1;
  }
  
  // Each line gets associated chapter and verse information.
  CategorizeChapterVerse ccv (ibr.lines);

  // Store it in the databases.
  project_store_book (project, book, ccv);

  // Success.  
  return 0;
}


int export_book (int argc, char *argv[])
// Exports a USFM file from Bibledit's database, with good error checking.
{
  // Parse options.
  ustring project;
  ustring book;
  ustring file;
  gchar * g_project = NULL;
  gchar * g_book = NULL;
  gchar * g_file = NULL;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project to be exported", NULL },
    { "book", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_book, "Name of the book, e.g. Matthew", NULL },
    { "file", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_file, "Name of the file to which should be exported", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  project = scripting_get_from_gchar (g_project);
  book = scripting_get_from_gchar (g_book);
  file = scripting_get_from_gchar (g_file);

  // Usage.
  if (project.empty() || file.empty() || book.empty()) 
    scripting_usage (entries, true);
  
  // Project is there?
  scripting_ensure_project (project);

  // Book is there?
  if (!project_book_exists (project, book)) {
    cerr << "Book " << book << " does not exist." << endl;
    return 1;
  }
  
  // Get and write book.
  vector <ustring> lines = project_retrieve_book (project, book);
  write_lines (file, lines);

  // Success.  
  return 0;
}


int shutdown_actions (int argc, char *argv[])
// Takes certain actions when Bibledit shuts down.
{
  // Parse options.
  GOptionEntry entries[] = 
  {
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);

  {
    // Open a configuration to read parameters from.
    GeneralConfiguration configuration (0);
    
    // Synchronize all projects on shutdown?
    if (configuration.synchronize_on_shutdown ()) {
      SynchronizeProjects sp (0);      
    }

    unsigned int startuptime = configuration.startup_time ();
    // Set about to vacuum the sqlite databases.
    // Vacuuming a database is done only when it got changed. Saves time.
    // Project related databases.
    vector <ustring> projects = projects_get_all ();
    for (unsigned int i = 0; i < projects.size(); i++) {
      project_vacuum (projects[i], startuptime);
      project_configuration_vacuum (projects[i], startuptime);
    }
    // Stylesheets.
    vector <ustring> stylesheets;
    stylesheet_get_ones_available (stylesheets);
    for (unsigned int i = 0; i < stylesheets.size(); i++) {
      stylesheet_vacuum (stylesheets[i], startuptime);
    }
    // Notes.
    notes_vacuum (startuptime);
  }

  // After configuration is no longer accessed, vacuum it.
  configuration_vacuum ();
  
  // Success.  
  return 0;
}


int export_project_usfm (int argc, char *argv[])
// Exports a project to USFM files, one per book
// Does good error checking.
{
  // Parse options.
  ustring project;
  ustring directory;
  bool gui = false;
  gchar * g_project = NULL;
  gchar * g_directory = NULL;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project to be exported", NULL },
    { "directory", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_directory, "Where to store the output", NULL },
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Show graphical progressbar", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (gui, argc, argv);
  project = scripting_get_from_gchar (g_project);
  directory = scripting_get_from_gchar (g_directory);

  // Usage.
  if (project.empty() || directory.empty()) scripting_usage (entries, true);

  // Check project.
  scripting_ensure_project (project);

  // Output directory.
  create_directory (directory);

  // Get books.
  vector <ustring> books = project_get_books (project);

  // Progress information.
  ProgressWindow * progresswindow = NULL;
  if (gui) {
    progresswindow = new ProgressWindow ("Exporting project", true);
    progresswindow->set_iterate (0, 1, books.size());
  }

  // Export all books to usfm.
  for (unsigned int i = 0; i < books.size(); i++) {
    // Progress info.
    cout << books[i] << endl;
    if (gui) {
      progresswindow->iterate ();
      progresswindow->set_text (books[i]);
      if (progresswindow->cancel)
        return 1;
    }
    vector <ustring> lines = project_retrieve_book (project, books[i]);
    ustring filename = gw_build_filename (directory, books[i] + ".usfm");
    write_lines (filename, lines);
  }
  
  // Destroy window.
  if (progresswindow)
    delete progresswindow;

  // Success.  
  return 0;
}


int export_project_sword (int argc, char *argv[])
// Exports a project to a file fit for compiling by the Sword compiler.
// Does good error checking.
{
  // Parse options.
  ustring project;
  bool gui = false;
  ustring directory;
  gchar * g_project = NULL;
  gchar * g_directory = NULL;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project to be exported", NULL },
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Show graphical progressbar", NULL },
    { "directory", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_directory, "Optional: where to store the module", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (gui, argc, argv);
  project = scripting_get_from_gchar (g_project);
  directory = scripting_get_from_gchar (g_directory);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);

  // Check project.
  scripting_ensure_project (project);

  // Take the action.
  export_to_sword_script (project, directory, gui);
  
  // Success.  
  return 0;
}


int general_configuration (int argc, char *argv[])
// Reads or writes to the general configuration.
{
  // Parse options.
  bool read;
  bool write;
  bool reset;
  ustring key;
  ustring value;
  gchar * g_key = NULL;
  gchar * g_value = NULL;
  GOptionEntry entries[] = 
  {
    { "read", gchar (NULL), 0, G_OPTION_ARG_NONE, &read, "Read a value from the configuration", NULL },
    { "write", gchar (NULL), 0, G_OPTION_ARG_NONE, &write, "Write a value to the configuration", NULL },
    { "reset", gchar (NULL), 0, G_OPTION_ARG_NONE, &reset, "Reset the configuration to default values", NULL },
    { "key", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_key, "When reading/writing: The key to select the value", NULL },
    { "value", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_value, "When writing: The value to be stored", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  key = scripting_get_from_gchar (g_key);
  value = scripting_get_from_gchar (g_value);

  // Usage.
  if (read && key.empty()) scripting_usage (entries, true);
  if (write && (key.empty() || value.empty())) scripting_usage (entries, true);
  if (!read && !write && !reset) scripting_usage (entries, true);

  if (read) {
    GeneralConfiguration configuration (0);
    cout << configuration.value_get (key, "") << endl;
  } else if (write) {
    GeneralConfiguration configuration (0);
    configuration.value_set (key, value);
  } else if (reset) {
    unlink (configuration_filename ().c_str());
  }
  
  // Ready.
  return 0;
}


int project_configuration (int argc, char *argv[])
// Reads or writes to the project configuration.
{
  // Parse options.
  bool read;
  bool write;
  bool reset;
  ustring project;
  ustring key;
  ustring value;
  gchar * g_project = NULL;
  gchar * g_key = NULL;
  gchar * g_value = NULL;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Which project's configuration to access", NULL },
    { "read", gchar (NULL), 0, G_OPTION_ARG_NONE, &read, "Read a value from the configuration", NULL },
    { "write", gchar (NULL), 0, G_OPTION_ARG_NONE, &write, "Write a value to the configuration", NULL },
    { "reset", gchar (NULL), 0, G_OPTION_ARG_NONE, &reset, "Reset the configuration to default values", NULL },
    { "key", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_key, "When reading/writing: The key to select the value", NULL },
    { "value", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_value, "When writing: The value to be stored", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  project = scripting_get_from_gchar (g_project);
  key = scripting_get_from_gchar (g_key);
  value = scripting_get_from_gchar (g_value);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);
  if (read && key.empty()) scripting_usage (entries, true);
  if (write && (key.empty() || value.empty())) scripting_usage (entries, true);
  if (!read && !write && !reset) scripting_usage (entries, true);

  // Project is there?
  scripting_ensure_project (project);

  if (read) {
    ProjectConfiguration projectconfig (project);
    cout << projectconfig.value_get (key, "") << endl;
  } else if (write) {
    ProjectConfiguration projectconfig (project);
    projectconfig.value_set (key, value);
  } else if (reset) {
    unlink (project_configuration_filename (project).c_str());
  }
  
  // Ready.
  return 0;
}


int create_project (int argc, char *argv[])
{
  // Parse options.
  ustring project;
  gchar * g_project = NULL;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project to create", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  project = scripting_get_from_gchar (g_project);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);
    
  // Execute task.
  if (project_exists (project)) {
    cerr << "Project already exists" << endl;
    return 1;
  }
  project_create (project);
  return 0;  
}


int delete_project (int argc, char *argv[])
{
  // Parse options.
  ustring project;
  gchar * g_project = NULL;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project to delete", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  project = scripting_get_from_gchar (g_project);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);

  // Execute task.
  project_delete (project);
  return 0;  
}


int validate_markers (int argc, char *argv[])
{
  // Parse options.
  ustring project;
  bool gui = false;
  bool checksheet = false;
  gchar * g_project = NULL;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project", NULL },
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Optional: Show graphical progressbar", NULL },
    { "check-stylesheet", gchar (NULL), 0, G_OPTION_ARG_NONE, &checksheet, "Optional: Are the markers in the stylesheet?", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (gui, argc, argv);
  project = scripting_get_from_gchar (g_project);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);

  // Project is there?
  scripting_ensure_project (project);

  // Execute task.
  vector <ustring> books;
  CheckValidateUsfm check (project, books, gui, checksheet);
  checks_output_references_comments (check.references, check.comments);

  return 0;  
}


int count_words (int argc, char *argv[])
{
  // Parse options.
  ustring project;
  gchar * g_project = NULL;
  bool sortword = false;
  bool sortcount = false;
  int excludecount = 0;
  ustring extracharacters;
  gchar * g_extracharacters = NULL;
  bool gui = false;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project", NULL },
    { "sort-word", gchar (NULL), 0, G_OPTION_ARG_NONE, &sortword, "Optional: Sort on word", NULL },
    { "sort-count", gchar (NULL), 0, G_OPTION_ARG_NONE, &sortcount, "Optional: Sort on count", NULL },
    { "exclude-count", gchar (NULL), 0, G_OPTION_ARG_INT, &excludecount, "Optional: Exclude words with a count of n or higher", NULL },
    { "extra-characters", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_extracharacters, "Optional: Extra characters that are part of a word", NULL },
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Optional: Show graphical progressbar", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (true, argc, argv);
  project = scripting_get_from_gchar (g_project);
  extracharacters = scripting_get_from_gchar (g_extracharacters);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);

  // Project is there?
  scripting_ensure_project (project);

  // Sort on only one thing.
  if (sortword && sortcount) {
    cout << "Cant't sort on more than one key" << endl;
    return 1;
  }
  
  // Execute task.
  vector <ustring> books;
  CheckCountWords check (project, books, extracharacters, sortword, sortcount, excludecount, gui);
  checks_output_two_columns (check.words, check.counts);

  return 0;  
}
