/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "scripture_reordered.h"
#include "projectutils.h"
#include "projectconfig.h"
#include "books.h"


ScriptureReordered::ScriptureReordered (const ustring& project)
// This object encapsulates a reordered Scripture.
{
  myproject = project;
  reordering_read ();
  vector<ustring> scripture_books = project_get_books (project);
  for (unsigned int i = 0; i < reordered_books.size(); i++) {
    for (unsigned int i2 = 0; i2 < scripture_books.size(); i2++) {
      if (reordered_books[i] == scripture_books[i2]) {
        if (reordered_includes[i]) {
          books.push_back (reordered_books[i]);
          portions.push_back (reordered_portions[i]);
        }
      }
    }
  }  
}


void ScriptureReordered::reordering_read ()
/*
Reads reordering information from disk.

Updates the reordering information based on the actual books in the object.
This means that, even if books, includes and portions are not in the settings on disk,
they get updated by the actual books in this Scripture. And any excess books,
if they are no more in the Scripture, are removed.
Extra books are added at their proper locations.
*/
{
  // Load scripture.
  vector <ustring> scripture_books = project_get_books (myproject);
  
  // Read the reordering information.
  ProjectConfiguration projectconfig (myproject);
  reordered_books = projectconfig.reordered_books ();
  reordered_includes = projectconfig.reordered_includes ();
  reordered_portions = projectconfig.reordered_portions ();
  
  // First pass: remove books no longer in the object.
  {
    set<ustring> bookset (scripture_books.begin(), scripture_books.end());
    vector<ustring> mybooks;
    vector<bool> myincludes;
    vector<ustring> myportions;
    for (unsigned int i = 0; i < reordered_books.size(); i++) {
      if (bookset.find (reordered_books[i]) != bookset.end()) {
        mybooks.push_back (reordered_books[i]);
        myincludes.push_back (reordered_includes[i]);
        myportions.push_back (reordered_portions[i]);
      }
    }
    reordered_books.assign (mybooks.begin(), mybooks.end());
    reordered_includes.assign (myincludes.begin(), myincludes.end());
    reordered_portions.assign (myportions.begin(), myportions.end());
  }

  // Second pass: add books missing in the reordering information.
  set<ustring> reorder_set (reordered_books.begin(), reordered_books.end());
  for (unsigned int i = 0; i < scripture_books.size(); i++) {
    if (reorder_set.find (scripture_books[i]) == reorder_set.end()) {
      // We've found a book which needs to be inserted in the list in the proper place.
      // Measure the "distance" between a book to insert and any of the books 
      // already in the project. Distance can be positive, if the book follows,
      // or negative, if the book goes before.
      vector<ustring> books2;
      vector<int> distances;
      int my_offset = books_english_to_id (scripture_books[i]);
      for (unsigned int i2 = 0; i2 < reordered_books.size(); i2++) {
        int your_offset = books_english_to_id (reordered_books[i2]);
        books2.push_back (reordered_books[i2]);
        distances.push_back (my_offset - your_offset);
      }
      // Find the smallest absolute distance, and the corresponding book.
      int smallest_distance = 10000;
      unsigned int smallest_offset = 0;
      for (unsigned int i2 = 0; i2 < distances.size(); i2++) {
        if (abs (distances[i2]) < smallest_distance) {
          smallest_distance = abs (distances[i2]);
          smallest_offset = i2;
        }          
      }
      // If offset is positive, insert the book after this one, if negative, insert before.
      vector<ustring>::iterator book_iter = reordered_books.begin();
      vector<bool>::iterator includes_iter = reordered_includes.begin();
      vector<ustring>::iterator portions_iter = reordered_portions.begin();
      // Check whether there were any books at all.
      if (!reordered_books.empty()) {
        for (unsigned int i2 = 0; i2 < smallest_offset; i2++) {
          book_iter++;
          includes_iter++;
          portions_iter++;
        }
        if (distances[smallest_offset] > 0) {
          book_iter++;
          includes_iter++;
          portions_iter++;
        }
      }
      reordered_books.insert (book_iter, scripture_books[i]);
      reordered_includes.insert (includes_iter, true);
      reordered_portions.insert (portions_iter, CHAPTER_VERSE_SELECTION_ALL);
    }
  }
}


bool ScriptureReordered::reordering_in_order ()
// Indicates whether the reordered books are in the standard order.
{
  int previousoffset = 0;
  for (unsigned int i = 0; i < reordered_books.size(); i++) {
    int currentoffset = books_english_to_id (reordered_books[i]);
    if (currentoffset < previousoffset)
      return false;
    previousoffset = currentoffset;    
  }
  return true;
}


bool ScriptureReordered::reordering_portions_all ()
// Indicates whether all books have their portions set to "all"  and are included.
{
  for (unsigned int i = 0; i < reordered_includes.size(); i++)
    if (!reordered_includes[i])
      return false;
  for (unsigned int i = 0; i < reordered_portions.size(); i++)
    if (reordered_portions[i] != CHAPTER_VERSE_SELECTION_ALL)
      return false;
  return true;
}
