/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "utilities.h"
#include <libgen.h>
#include <glib.h>
#include <config.h>
#include "gui_navigation.h"
#include "combobox.h"
#include "projectconfig.h"
#include "books.h"
#include "projectutils.h"
#include "gtkwrappers.h"
#include "bible.h"


GuiNavigation::GuiNavigation (int dummy):
reference (0),
track (0)
{
  // Initialize variables.
  settingcombos = false;
  spinbutton_book_previous_value = 0;
  spinbutton_chapter_previous_value = 0;
  spinbutton_verse_previous_value = 0;
  delayer_event_id = 0;
  later_event_id = 0;
  track_event_id = 0;
}


GuiNavigation::~GuiNavigation ()
{
}


void GuiNavigation::build (GtkWidget * toolbar, GtkWidget * menu_next_reference, GtkWidget * menu_previous_reference)
{
  // Save pointers to menu items.
  menu_back = menu_previous_reference;
  menu_forward = menu_next_reference;
  
  // Signalling buttons, but not visible.
  GtkToolItem * toolitem_immediate = gtk_tool_item_new ();
  gtk_toolbar_insert (GTK_TOOLBAR (toolbar), GTK_TOOL_ITEM (toolitem_immediate), -1);
  reference_signal_immediate = gtk_button_new ();
  gtk_container_add (GTK_CONTAINER (toolitem_immediate), reference_signal_immediate);
  GtkToolItem * toolitem_delayed = gtk_tool_item_new ();
  gtk_toolbar_insert (GTK_TOOLBAR (toolbar), GTK_TOOL_ITEM (toolitem_delayed), -1);
  reference_signal_delayed = gtk_button_new ();
  gtk_container_add (GTK_CONTAINER (toolitem_delayed), reference_signal_delayed);
  GtkToolItem * toolitem_later = gtk_tool_item_new ();
  gtk_toolbar_insert (GTK_TOOLBAR (toolbar), GTK_TOOL_ITEM (toolitem_later), -1);
  reference_signal_late = gtk_button_new ();
  gtk_container_add (GTK_CONTAINER (toolitem_later), reference_signal_late);

  // Gui proper.
  GtkToolItem * toolitem1 = gtk_tool_item_new ();
  gtk_widget_show (GTK_WIDGET (toolitem1));
  gtk_toolbar_insert (GTK_TOOLBAR (toolbar), GTK_TOOL_ITEM (toolitem1), -1);

  button_back = gtk_button_new ();
  gtk_widget_show (button_back);
  gtk_container_add (GTK_CONTAINER (toolitem1), button_back);

  image1 = gtk_image_new_from_stock ("gtk-go-back", GTK_ICON_SIZE_BUTTON);
  gtk_widget_show (image1);
  gtk_container_add (GTK_CONTAINER (button_back), image1);

  GtkToolItem * toolitem2 = gtk_tool_item_new ();
  gtk_widget_show (GTK_WIDGET (toolitem2));
  gtk_toolbar_insert (GTK_TOOLBAR (toolbar), GTK_TOOL_ITEM (toolitem2), -1);

  button_forward = gtk_button_new ();
  gtk_widget_show (button_forward);
  gtk_container_add (GTK_CONTAINER (toolitem2), button_forward);

  image2 = gtk_image_new_from_stock ("gtk-go-forward", GTK_ICON_SIZE_BUTTON);
  gtk_widget_show (image2);
  gtk_container_add (GTK_CONTAINER (button_forward), image2);

  GtkToolItem * toolitem3 = gtk_tool_item_new ();
  gtk_widget_show (GTK_WIDGET (toolitem3));
  gtk_toolbar_insert (GTK_TOOLBAR (toolbar), GTK_TOOL_ITEM (toolitem3), -1);

  combo_book = gtk_combo_box_new_text ();
  gtk_widget_show (combo_book);
  gtk_container_add (GTK_CONTAINER (toolitem3), combo_book);

  GtkToolItem * toolitem4 = gtk_tool_item_new ();
  gtk_widget_show (GTK_WIDGET (toolitem4));
  gtk_toolbar_insert (GTK_TOOLBAR (toolbar), GTK_TOOL_ITEM (toolitem4), -1);

  spinbutton_book_adj = gtk_adjustment_new (0, -1e+06, 1e+06, 1, 10, 10);
  spinbutton_book = gtk_spin_button_new (GTK_ADJUSTMENT (spinbutton_book_adj), 1, 0);
  gtk_widget_show (spinbutton_book);
  gtk_container_add (GTK_CONTAINER (toolitem4), spinbutton_book);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spinbutton_book), TRUE);
  gtk_editable_set_editable (GTK_EDITABLE (spinbutton_book), false);  
  gtk_entry_set_visibility (GTK_ENTRY (spinbutton_book), false);
  gtk_entry_set_invisible_char (GTK_ENTRY (spinbutton_book), 0);
  gtk_entry_set_width_chars (GTK_ENTRY (spinbutton_book), 0);
  GTK_WIDGET_UNSET_FLAGS (spinbutton_book, GTK_CAN_FOCUS);
  
  GtkToolItem * toolitem6 = gtk_tool_item_new ();
  gtk_widget_show (GTK_WIDGET (toolitem6));
  gtk_toolbar_insert (GTK_TOOLBAR (toolbar), GTK_TOOL_ITEM (toolitem6), -1);

  combo_chapter = gtk_combo_box_new_text ();
  gtk_widget_show (combo_chapter);
  gtk_container_add (GTK_CONTAINER (toolitem6), combo_chapter);

  GtkToolItem * toolitem5 = gtk_tool_item_new ();
  gtk_widget_show (GTK_WIDGET (toolitem5));
  gtk_toolbar_insert (GTK_TOOLBAR (toolbar), GTK_TOOL_ITEM (toolitem5), -1);

  spinbutton_chapter_adj = gtk_adjustment_new (0, -1e+06, 1e+06, 1, 10, 10);
  spinbutton_chapter = gtk_spin_button_new (GTK_ADJUSTMENT (spinbutton_chapter_adj), 1, 0);
  gtk_widget_show (spinbutton_chapter);
  gtk_container_add (GTK_CONTAINER (toolitem5), spinbutton_chapter);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spinbutton_chapter), TRUE);
  gtk_editable_set_editable (GTK_EDITABLE (spinbutton_chapter), false);  
  gtk_entry_set_visibility (GTK_ENTRY (spinbutton_chapter), false);
  gtk_entry_set_invisible_char (GTK_ENTRY (spinbutton_chapter), 0);
  gtk_entry_set_width_chars (GTK_ENTRY (spinbutton_chapter), 0);
  GTK_WIDGET_UNSET_FLAGS (spinbutton_chapter, GTK_CAN_FOCUS);

  GtkToolItem * toolitem8 = gtk_tool_item_new ();
  gtk_widget_show (GTK_WIDGET (toolitem8));
  gtk_toolbar_insert (GTK_TOOLBAR (toolbar), GTK_TOOL_ITEM (toolitem8), -1);

  combo_verse = gtk_combo_box_new_text ();
  gtk_widget_show (combo_verse);
  gtk_container_add (GTK_CONTAINER (toolitem8), combo_verse);

  GtkToolItem * toolitem7 = gtk_tool_item_new ();
  gtk_widget_show (GTK_WIDGET (toolitem7));
  gtk_toolbar_insert (GTK_TOOLBAR (toolbar), GTK_TOOL_ITEM (toolitem7), -1);

  spinbutton_verse_adj = gtk_adjustment_new (0, -1e+06, 1e+06, 1, 10, 10);
  spinbutton_verse = gtk_spin_button_new (GTK_ADJUSTMENT (spinbutton_verse_adj), 1, 0);
  gtk_widget_show (spinbutton_verse);
  gtk_container_add (GTK_CONTAINER (toolitem7), spinbutton_verse);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spinbutton_verse), TRUE);
  gtk_editable_set_editable (GTK_EDITABLE (spinbutton_verse), false);  
  gtk_entry_set_visibility (GTK_ENTRY (spinbutton_verse), false);
  gtk_entry_set_invisible_char (GTK_ENTRY (spinbutton_verse), 0);
  gtk_entry_set_width_chars (GTK_ENTRY (spinbutton_verse), 0);
  GTK_WIDGET_UNSET_FLAGS (spinbutton_verse, GTK_CAN_FOCUS);

  // Resize the spinbuttons.
  gint defaultheight;
  GtkRequisition sizerequisition;
  gtk_widget_size_request (combo_verse, &sizerequisition);
  defaultheight = (int) (sizerequisition.height * 0.8);
  gtk_widget_set_size_request (spinbutton_book, int (defaultheight * 0.7), -1);
  gtk_widget_set_size_request (spinbutton_chapter, int (defaultheight * 0.7), -1);
  gtk_widget_set_size_request (spinbutton_verse, int (defaultheight * 0.7), -1);
  
  g_signal_connect ((gpointer) menu_forward, "activate", G_CALLBACK (on_menu_forward_activate), gpointer(this));
  g_signal_connect ((gpointer) menu_back, "activate", G_CALLBACK(on_menu_back_activate), gpointer(this));
  g_signal_connect ((gpointer) button_back, "clicked", G_CALLBACK (on_button_back_clicked), gpointer(this));
  g_signal_connect ((gpointer) button_forward, "clicked", G_CALLBACK (on_button_forward_clicked), gpointer(this));
  g_signal_connect ((gpointer) combo_book, "changed", G_CALLBACK (on_combo_book_changed), gpointer(this));
  g_signal_connect ((gpointer) spinbutton_book, "value_changed", G_CALLBACK (on_spinbutton_book_value_changed), gpointer(this));
  g_signal_connect ((gpointer) combo_chapter, "changed", G_CALLBACK (on_combo_chapter_changed), gpointer(this));
  g_signal_connect ((gpointer) spinbutton_chapter, "value_changed", G_CALLBACK (on_spinbutton_chapter_value_changed), gpointer(this));
  g_signal_connect ((gpointer) combo_verse, "changed", G_CALLBACK (on_combo_verse_changed), gpointer(this));
  g_signal_connect ((gpointer) spinbutton_verse, "value_changed", G_CALLBACK (on_spinbutton_verse_value_changed), gpointer(this));
}


void GuiNavigation::sensitive (bool sensitive)
{
  // Tracker.
  if (!sensitive) track.clear();
  tracker_sensitivity ();
  // Set the sensitivity of the reference controls.
  gtk_widget_set_sensitive (combo_book, sensitive);
  gtk_widget_set_sensitive (spinbutton_book, sensitive);
  gtk_widget_set_sensitive (combo_chapter, sensitive);
  gtk_widget_set_sensitive (spinbutton_chapter, sensitive);
  gtk_widget_set_sensitive (combo_verse, sensitive);
  gtk_widget_set_sensitive (spinbutton_verse, sensitive);
  if (!sensitive) {
    // We are programatically going to change the comboboxes, 
    // and therefore don't want a signal during that operation.
    settingcombos = true;
    project.clear();
    gtk_combo_box_set_active (GTK_COMBO_BOX (combo_book), -1);
    gtk_combo_box_set_active (GTK_COMBO_BOX (combo_chapter), -1);
    gtk_combo_box_set_active (GTK_COMBO_BOX (combo_verse), -1);
    settingcombos = false;
  }
}


void GuiNavigation::set_project (const ustring& value)
// Sets the project of the object, and loads the books.
{
  // If the project is the same as the one already loaded, bail out.
  if (value == project) return;
    
  // Save project, language.
  project = value;
  ProjectConfiguration projectconfig (project);
  language = projectconfig.language ();
  
  // Load books.
  load_books ();

  // Clear the references tracker.
  track.clear ();
}


void GuiNavigation::clamp (Reference& reference)
// This clamps the reference, that is, it brings it within the limits of 
// the project.
{
  // If the reference exists, fine, bail out.
  if (reference_exists (reference, false)) return;
  
  // Clamp the book.
  if (!project_book_exists (project, reference.book)) {
    vector <unsigned int> books = project_get_books (project);
    if (books.empty()) { 
      reference.book = 0;
    } else {
      reference.book = CLAMP (reference.book, books[0], books[books.size() -1]);
    }
  }
  
  // Clamp the chapter.
  vector <unsigned int> chapters = project_get_chapters (project, reference.book);
  set <unsigned int> chapterset (chapters.begin(), chapters.end());
  if (chapterset.find (reference.chapter) == chapterset.end()) {
    reference.chapter = 0;
    if (!chapters.empty()) reference.chapter = chapters[0];
  }
  
  // Clamp the verse.
  vector <ustring> verses = project_get_verses (project, reference.book, reference.chapter);
  set <ustring> verseset (verses.begin(), verses.end());
  if (verseset.find (reference.verse) == verseset.end()) {
    reference.verse = "0";
    if (!verses.empty()) reference.verse = verses[0];
  }
}


void GuiNavigation::display (const Reference& ref)
// This tries to have the reference displayed, if it is in the project.
// If the requested reference is not available in the project, it does nothing.
{
  // Project configuration.
  ProjectConfiguration projectconfig (project);
  language = projectconfig.language();
  
  // Find out if there is a change in book, chapter, verse.
  unsigned int currentbook = books_name_to_id (language, combobox_get_active_string (combo_book));
  bool newbook = (ref.book != currentbook);
  unsigned int currentchapter = convert_to_int (combobox_get_active_string (combo_chapter));
  bool newchapter = (ref.chapter != currentchapter);
  ustring currentverse = combobox_get_active_string (combo_verse);
  bool newverse = (ref.verse != currentverse);

  // If a new book, then there is also a new chapter, and so on.
  if (newbook) newchapter = true;
  if (newchapter) newverse = true;
    
  // If there is no change in the reference, do nothing.
  if (!newverse) return;

  // There's a change, see if our project has that reference. 
  // If not, polish the verse up, if still not, bail out.
  Reference ref2 = ref;
  if (!reference_exists (ref2, false)) {
    polish_up_verse (ref2);
    if (!reference_exists (ref2, true)) {
      return;
    }
  }
  
  // Handle new book.
  if (newbook) {
    set_book (ref.book);
    load_chapters (ref.book);
  }

  // Handle new chapter.
  if (newchapter) {
    set_chapter (ref.chapter);
    load_verses (ref.book, ref.chapter);
  }

  // Handle new verse.
  if (newverse) {
    set_verse (ref.verse);
    // Give signal.
    reference = ref;
    signal ();
  }
}


void GuiNavigation::nextbook ()
{
  vector <ustring> strings = combobox_get_strings (combo_book);
  if (strings.size () == 0)
    return;
  ustring ubook = combobox_get_active_string (combo_book);
  unsigned int index = 0;
  for (unsigned int i = 0; i < strings.size (); i++) {
    if (ubook == strings[i])
      index = i;
  }
  if (index == (strings.size () - 1)) {
    return;
  }
  reference.book = books_name_to_id (language, strings[++index]);
  reference.chapter = 1;
  // Find the first verse.
  vector<ustring> verses = project_get_verses (project, reference.book, reference.chapter);
  if (verses.size () > 1)
    // Get the first verse of the chapter which is not "0".
    if (verses[0] == "0")
      reference.verse = verses[1];
    else
      reference.verse = verses[0];
  else
    reference.verse = "0";
  clamp (reference);
  set_book (reference.book);
  load_chapters (reference.book);
  set_chapter (reference.chapter);
  load_verses (reference.book, reference.chapter);
  set_verse (reference.verse);
  signal ();
}


void GuiNavigation::previousbook ()
{
  vector <ustring> strings = combobox_get_strings (combo_book);
  if (strings.size () == 0)
    return;
  ustring ubook = combobox_get_active_string (combo_book);
  unsigned int index = 0;
  for (unsigned int i = 0; i < strings.size (); i++) {
    if (ubook == strings[i])
      index = i;
  }
  if (index == 0) {
    return;
  }
  reference.book = books_name_to_id (language, strings[--index]);
  reference.chapter = 1;
  // Find proper first verse.
  vector<ustring> verses = project_get_verses (project, reference.book, reference.chapter);
  if (verses.size () > 1)
    // Get the first verse of the chapter which is not "0".
    if (verses[0] == "0")
      reference.verse = verses[1];
    else
      reference.verse = verses[0];
  else
    reference.verse = "0";
  clamp (reference);
  set_book (reference.book);
  load_chapters (reference.book);
  set_chapter (reference.chapter);
  load_verses (reference.book, reference.chapter);
  set_verse (reference.verse);
  signal ();
}


void GuiNavigation::nextchapter ()
{
  unsigned int chapter = convert_to_int (combobox_get_active_string (combo_chapter));
  vector <ustring> strings = combobox_get_strings (combo_chapter);
  if (strings.size () == 0)
    return;
  unsigned int index = 0;
  for (unsigned int i = 0; i < strings.size (); i++) {
    if (chapter == convert_to_int (strings[i]))
      index = i;
  }
  if (index == (strings.size () - 1)) {
    crossboundarieschapter (true);
    return;
  }
  chapter = convert_to_int (strings[++index]);
  // Find proper first verse.
  ustring verse;
  vector<ustring> verses = project_get_verses (project, reference.book, chapter);
  if (verses.size () > 1)
    verse = verses[1];
  else
    verse = "0";
  clamp (reference);
  load_verses (reference.book, chapter);
  set_verse (verse);
  set_chapter (chapter);
  reference.verse = verse;
  reference.chapter = chapter;
  signal ();
}


void GuiNavigation::previouschapter ()
{
  unsigned int chapter = convert_to_int (combobox_get_active_string (combo_chapter));
  vector <ustring> strings = combobox_get_strings (combo_chapter);
  if (strings.size () == 0)
    return;
  unsigned int index = 0;
  for (unsigned int i = 0; i < strings.size (); i++) {
    if (chapter == convert_to_int (strings[i]))
      index = i;
  }
  if (index == 0) {
    crossboundarieschapter (false);
    return;
  }
  chapter = convert_to_int (strings[--index]);
  // Find proper first verse.
  ustring verse;
  vector<ustring> verses = project_get_verses (project, reference.book, chapter);
  if (verses.size () > 1)
    verse = verses[1];
  else
    verse = "0";
  clamp (reference);
  load_verses (reference.book, chapter);
  set_verse (verse);
  set_chapter (chapter);
  reference.verse = verse;
  reference.chapter = chapter;
  signal ();
}


void GuiNavigation::nextverse ()
{
  ustring verse = combobox_get_active_string (combo_verse);
  vector <ustring> strings = combobox_get_strings (combo_verse);
  if (strings.size () == 0)
    return;
  unsigned int index = 0;
  for (unsigned int i = 0; i < strings.size (); i++) {
    if (verse == strings[i])
      index = i;
  }
  if (index == (strings.size () - 1)) {
    crossboundariesverse (true);
    return;
  }
  verse = strings[++index];
  set_verse (verse);
  reference.verse = verse;
  signal ();
}


void GuiNavigation::previousverse ()
{
  ustring verse = combobox_get_active_string (combo_verse);
  vector <ustring> strings = combobox_get_strings (combo_verse);
  if (strings.size () == 0)
    return;
  unsigned int index = 0;
  for (unsigned int i = 0; i < strings.size (); i++) {
    if (verse == strings[i])
      index = i;
  }
  if (index == 0) {
    crossboundariesverse (false);
    return;
  }
  verse = strings[--index];
  set_verse (verse);
  reference.verse = verse;
  signal ();
}


void GuiNavigation::on_menu_back_activate (GtkMenuItem *menuitem, gpointer user_data)
{
  ((GuiNavigation *) user_data)->on_back ();
}


void GuiNavigation::on_menu_forward_activate (GtkMenuItem *menuitem, gpointer user_data)
{
  ((GuiNavigation *) user_data)->on_forward ();
}


void GuiNavigation::on_button_back_clicked (GtkButton *button, gpointer user_data)
{
  ((GuiNavigation *) user_data)->on_back ();
}


void GuiNavigation::on_button_forward_clicked (GtkButton *button, gpointer user_data)
{
  ((GuiNavigation *) user_data)->on_forward ();
}


void GuiNavigation::on_combo_book_changed (GtkComboBox *combobox, gpointer user_data)
{
  ((GuiNavigation *) user_data)->on_combo_book ();
}


void GuiNavigation::on_combo_chapter_changed (GtkComboBox *combobox, gpointer user_data)
{
  ((GuiNavigation *) user_data)->on_combo_chapter ();
}


void GuiNavigation::on_combo_verse_changed (GtkComboBox *combobox, gpointer user_data)
{
  ((GuiNavigation *) user_data)->on_combo_verse ();
}


void GuiNavigation::on_spinbutton_book_value_changed (GtkSpinButton *spinbutton, gpointer user_data)
{
  ((GuiNavigation *) user_data)->on_spinbutton_book ();
}


void GuiNavigation::on_spinbutton_chapter_value_changed (GtkSpinButton *spinbutton, gpointer user_data)
{
  ((GuiNavigation *) user_data)->on_spinbutton_chapter ();
}


void GuiNavigation::on_spinbutton_verse_value_changed (GtkSpinButton *spinbutton, gpointer user_data)
{
  ((GuiNavigation *) user_data)->on_spinbutton_verse ();
}


void GuiNavigation::on_back ()
{
  track.get_previous_reference (reference);
  signal (false);
}


void GuiNavigation::on_forward ()
{
  track.get_next_reference (reference);
  signal (false);
}


void GuiNavigation::on_combo_book ()
{
  if (settingcombos) return;
  reference.book = books_name_to_id (language, combobox_get_active_string (combo_book));
  reference.chapter = 1;
  reference.verse = "1";
  clamp (reference);
  load_chapters (reference.book);
  set_chapter (reference.chapter);
  load_verses (reference.book, reference.chapter);
  set_verse (reference.verse);
  signal (); 
}


void GuiNavigation::on_combo_chapter ()
{
  if (settingcombos) return;
  reference.chapter = convert_to_int (combobox_get_active_string (combo_chapter));
  reference.verse = "1";
  clamp (reference);
  load_verses (reference.book, reference.chapter);
  set_verse (reference.verse);
  signal ();
}


void GuiNavigation::on_combo_verse ()
{
  if (settingcombos) return;
  reference.verse = combobox_get_active_string (combo_verse);
  signal ();
}


void GuiNavigation::on_spinbutton_book ()
{
  if (settingcombos) return;
  int value = int (gtk_adjustment_get_value (GTK_ADJUSTMENT (spinbutton_book_adj)));
  bool next = (value < spinbutton_book_previous_value);
  unsigned int amount = abs (value - spinbutton_book_previous_value);
  for (unsigned int i = 0; i < amount; i++) {
    if (next) nextbook();
    else previousbook();
  }
  spinbutton_book_previous_value = value;
}


void GuiNavigation::on_spinbutton_chapter ()
{
  if (settingcombos) return;
  int value = int (gtk_adjustment_get_value (GTK_ADJUSTMENT (spinbutton_chapter_adj)));
  bool next = (value < spinbutton_chapter_previous_value);
  unsigned int amount = abs (value - spinbutton_chapter_previous_value);
  for (unsigned int i = 0; i < amount; i++) {
    if (next) nextchapter();
    else previouschapter();
  }
  spinbutton_chapter_previous_value = value;
}


void GuiNavigation::on_spinbutton_verse ()
{
  if (settingcombos) return;
  int value = int (gtk_adjustment_get_value (GTK_ADJUSTMENT (spinbutton_verse_adj)));
  bool next = (value < spinbutton_verse_previous_value);
  unsigned int amount = abs (value - spinbutton_verse_previous_value);
  for (unsigned int i = 0; i < amount; i++) {
    if (next) nextverse();
    else previousverse();
  }
  spinbutton_verse_previous_value = value;
}


bool GuiNavigation::reference_exists (Reference& reference, bool gui) 
// Returns true if the reference exists.
// If the references does not exist, it gives a message.
{
  bool exists = project_book_exists (project, reference.book);
  if (exists) {
    vector <unsigned int> chapters = project_get_chapters (project, reference.book);
    set <unsigned int> chapterset (chapters.begin(), chapters.end());
    if (chapterset.find (reference.chapter) == chapterset.end()) {
      exists = false;
    }
  }
  if (exists) {
    vector <ustring> verses = project_get_verses (project, reference.book, reference.chapter);
    set <ustring> verseset (verses.begin(), verses.end());
    if (verseset.find (reference.verse) == verseset.end()) {
      exists = false;
    }
  }
  if (!exists)
    if (gui)
      gtkw_dialog_warning (NULL, "Project " + project + " does not have " + reference.human_readable (language));
  return exists;  
}


void GuiNavigation::polish_up_verse (Reference& reference)
// This function ensures that the verse we try to go to is within the range
// of currently being displayed verses.
{
  // See whether the verse is among the ones displayed.
  vector<ustring> verses = project_get_verses (project, reference.book, reference.chapter);
  set<ustring> verses_set (verses.begin(), verses.end());
  if (verses_set.find (reference.verse) != verses_set.end())
    return;
  // The verse was not found. See whether it can be found within the current verses.
  // E.g. is a combined verse (10-12) is displayed, and we go to verse 11, then
  // this 11 is within the range 10-12, hence we should go to 12-12.
  // It supports sequences in the form of v 1,2,3.
  // It supports ranges in the form of v 1b-3 and v 2-4a and v 2b-5a.
  int int_verse = convert_to_int (number_in_string (reference.verse));
  int_verse = int_verse * 2;
  if (reference.verse.find ("B") != string::npos)
    int_verse++;
  for (unsigned int i = 0; i < verses.size(); i++) {
    // Get a container full of encoded verses.
    vector<int> encoded_verses = verses_encode (verses[i]);
    set<int> expanded_verses (encoded_verses.begin(), encoded_verses.end());
    // See whether our verse is in it.
    if (expanded_verses.find (int_verse) != expanded_verses.end()) {
      reference.verse = verses[i];
      return;
    }
  }
  // Last attempt: take first verse in chapter.
  reference.verse = verses[0];  
}


void GuiNavigation::load_books ()
{
  settingcombos = true;
  vector <unsigned int> books = project_get_books (project);
  vector <ustring> localizedbooks;
  for (unsigned int i = 0; i < books.size(); i++) {
    ustring localizedbook = books_id_to_name (language, books[i]);
    localizedbooks.push_back (localizedbook); 
  }
  combobox_set_strings (combo_book, localizedbooks);
  settingcombos = false;
}


void GuiNavigation::set_book (unsigned int book)
{
  settingcombos = true;
  ustring localizedbook = books_id_to_name (language, book);
  combobox_set_string (combo_book, localizedbook);
  settingcombos = false;
}


void GuiNavigation::load_chapters (unsigned int book)
{
  settingcombos = true;
  vector <unsigned int> chapters = project_get_chapters (project, book);
  combobox_set_strings (combo_chapter, chapters);
  settingcombos = false;
}


void GuiNavigation::set_chapter (unsigned int chapter)
{
  settingcombos = true;
  combobox_set_string (combo_chapter, chapter);
  settingcombos = false;
}


void GuiNavigation::load_verses (unsigned int book, unsigned int chapter)
{
  settingcombos = true;
  vector <ustring> verses = project_get_verses (project, book, chapter);
  combobox_set_strings (combo_verse, verses);
  settingcombos = false;
}


void GuiNavigation::set_verse (const ustring& verse)
{
  settingcombos = true;
  combobox_set_string (combo_verse, verse);
  settingcombos = false;
}


void GuiNavigation::signal (bool track)
{
  // Emit the immediate signal.
  gtk_button_clicked (GTK_BUTTON (reference_signal_immediate));
  // Postpone any active delayed signal.
  if (delayer_event_id) {
    GSource *source = g_main_context_find_source_by_id (NULL, delayer_event_id);
    if (source) g_source_destroy (source);
  }
  // Start the time out for the delayed signal.
  delayer_event_id = g_timeout_add_full (G_PRIORITY_DEFAULT, 100, GSourceFunc (signal_delayer), gpointer(this), NULL);
  // Same story for the later signal.
  if (later_event_id) {
    GSource *source = g_main_context_find_source_by_id (NULL, later_event_id);
    if (source) g_source_destroy (source);
  }
  later_event_id = g_timeout_add_full (G_PRIORITY_DEFAULT, 200, GSourceFunc (signal_later), gpointer(this), NULL);
  // Same thing again for the tracker signal.
  // This signal has a delay of some seconds, so that, 
  // if a reference is displaying for some seconds, it will be tracked. 
  // References that display for a shorter time will not be tracked.
  if (track_event_id) {
    GSource *source = g_main_context_find_source_by_id (NULL, track_event_id);
    if (source) g_source_destroy (source);
  }
  if (track) {
    track_event_id = g_timeout_add_full (G_PRIORITY_DEFAULT, 2000, GSourceFunc (signal_track), gpointer(this), NULL);
  }
  // Sensitivity of tracker controls.
  tracker_sensitivity ();
}


bool GuiNavigation::signal_delayer (gpointer user_data)
{
  ((GuiNavigation *) user_data)->signal_delayed ();
  return false;
}


void GuiNavigation::signal_delayed ()
{
  gtk_button_clicked (GTK_BUTTON (reference_signal_delayed));
}


bool GuiNavigation::signal_later (gpointer user_data)
{
  ((GuiNavigation *) user_data)->signal_late ();
  return false;
}


void GuiNavigation::signal_late ()
{
  gtk_button_clicked (GTK_BUTTON (reference_signal_late));
}


bool GuiNavigation::signal_track (gpointer user_data)
{
  ((GuiNavigation *) user_data)->signal_tracking ();
  return false;
}


void GuiNavigation::signal_tracking ()
{
  track.store (reference);
}


void GuiNavigation::crossboundariesverse (bool forward)
// Handles the situation where a requested change in a verse needs to cross
// the boundaries of a book or a chapter.
{
  // Index of the currently opened book.
  int bookindex = -1;
  vector<unsigned int> allbooks = project_get_books (project);
  for (unsigned int i = 0; i < allbooks.size (); i++) {
    if (reference.book == allbooks[i])
      bookindex = i;
  }
  // If the requested book does not exist, bail out.
  if (bookindex < 0) {
    return;
  }
  // Get the previous book, and the next book.
  int previousbookindex = bookindex - 1;
  previousbookindex = CLAMP (previousbookindex, 0, bookindex);
  unsigned int nextbookindex = bookindex + 1;
  nextbookindex = CLAMP (nextbookindex, 0, allbooks.size() - 1);
  // Get a list of all references in these on the most three books.
  vector<unsigned int> books;
  vector<unsigned int> chapters;
  vector<ustring> verses;
  for (unsigned int i = previousbookindex; i <= nextbookindex; i++) {
    // Get the book metrics.
    vector<unsigned int> bookchapters = project_get_chapters (project, allbooks[i]);
    for (unsigned int i2 = 0; i2 < bookchapters.size(); i2++) {
      vector<ustring> chapterverses = project_get_verses (project, allbooks[i], bookchapters[i2]);
      for (unsigned int i3 = 0; i3 < chapterverses.size(); i3++) {
        books.push_back (allbooks[i]);
        chapters.push_back (bookchapters[i2]);
        verses.push_back (chapterverses[i3]);
      }
    }
  }
  // Find the current reference in the list.
  ustring verse = reference.verse;
  unsigned int referenceindex = 0;
  for (unsigned int i = 0; i < books.size(); i++) {
    if (verse == verses[i]) {
      if (reference.chapter == chapters[i]) {
        if (reference.book == books[i]) {
          referenceindex = i;
          break;
        }
      }
    }
  }
  // Get the next or previous value.
  if (forward) {
    if (referenceindex == (chapters.size() - 1))
      return;
    referenceindex++;
  } else {
    if (referenceindex == 0)
      return;
    referenceindex--;
  }
  // Ok, give us the new values.
  reference.book = books[referenceindex];
  reference.chapter = chapters[referenceindex];
  reference.verse = verses[referenceindex];
  // Set the values in the comboboxes, and signal a change.
  set_book (reference.book);
  load_chapters (reference.book);
  set_chapter (reference.chapter);
  load_verses (reference.book, reference.chapter);
  set_verse (reference.verse);
  signal ();  
}


void GuiNavigation::crossboundarieschapter (bool forward)
{
  // Index of the currently opened book.
  int bookindex = -1;
  vector<unsigned int> allbooks = project_get_books (project);
  for (unsigned int i = 0; i < allbooks.size (); i++) {
    if (reference.book == allbooks[i])
      bookindex = i;
  }
  // If the requested book does not exist, bail out.
  if (bookindex < 0) {
    return;
  }
  // Get the previous book, and the next book.
  int previousbookindex = bookindex - 1;
  previousbookindex = CLAMP (previousbookindex, 0, bookindex);
  unsigned int nextbookindex = bookindex + 1;
  nextbookindex = CLAMP (nextbookindex, 0, allbooks.size() - 1);
  // Get a list of all references in these on the most three books.
  vector<unsigned int> books;
  vector<unsigned int> chapters;
  vector<ustring> first_verses;
  for (unsigned int i = previousbookindex; i <= nextbookindex; i++) {
    // Get the book metrics.
    vector<unsigned int> bookchapters = project_get_chapters (project, allbooks[i]);
    for (unsigned int i2 = 0; i2 < bookchapters.size(); i2++) {
      books.push_back (allbooks[i]);
      chapters.push_back (bookchapters[i2]);
      vector<ustring> chapterverses = project_get_verses (project, allbooks[i], bookchapters[i2]);
      // We take the first verse of each chapter, if available, else we take v 0.
      if (chapterverses.size() > 1)
        first_verses.push_back (chapterverses[1]);
      else 
        first_verses.push_back (chapterverses[0]);
    }
  }
  // Find the current reference in the list.
  unsigned int referenceindex = 0;
  for (unsigned int i = 0; i < books.size(); i++) {
    if (reference.chapter == chapters[i]) {
      if (reference.book == books[i]) {
        referenceindex = i;
        break;
      }
    }
  }
  // Get the next or previous value.
  if (forward) {
    if (referenceindex == (chapters.size() - 1))
      return;
    referenceindex++;
  } else {
    if (referenceindex == 0)
      return;
    referenceindex--;
  }
  // Ok, give us the new values.
  reference.book = books[referenceindex];
  reference.chapter = chapters[referenceindex];
  if (forward) {
    reference.verse = first_verses[referenceindex];
  } else {
    reference.verse = first_verses[referenceindex];
  }
  // Set the values in the comboboxes, and signal a change.
  set_book (reference.book);
  load_chapters (reference.book);
  set_chapter (reference.chapter);
  load_verses (reference.book, reference.chapter);
  set_verse (reference.verse);
  signal ();  
}


void GuiNavigation::tracker_sensitivity ()
{
  // Menu items.
  gtk_widget_set_sensitive (menu_back, track.previous_reference_available ());
  gtk_widget_set_sensitive (menu_forward, track.next_reference_available ());
  // Buttons.
  gtk_widget_set_sensitive (button_back, track.previous_reference_available ());
  gtk_widget_set_sensitive (button_forward, track.next_reference_available ());
}
