/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <stdlib.h>

#include "bkl-entry.h"
#include "bkl-item-audio.h"
#include "bkl-utils.h"

enum {
    PROP_0,
    PROP_TITLE,
    PROP_ALBUM,
    PROP_ARTISTS,
    PROP_PERFORMERS,
    PROP_CONDUCTOR,
    PROP_COMPOSER,
    PROP_GENRE,
    PROP_COMMENT,
    PROP_COPYRIGHT,
    PROP_LICENSE,
    PROP_ORGANISATION,
    PROP_ISRC,
    PROP_CONTACT,
    PROP_YEAR,
    PROP_TRACK,
    PROP_MAXTRACKS,
    PROP_DISC,
    PROP_MAXDISCS,
    PROP_DURATION,
    PROP_ARTIST_IMAGE,
};

struct _BklItemAudioPrivate {
    char *title;
    char *album;

    GPtrArray *artists;
    GPtrArray *performers;
    char *conductor;
    char *composer;

    char *genre;
    char *comment;
    char *copyright;
    char *license;
    char *organisation;
    char *isrc;
    char *contact;
    char *year;

    char *artist_image;

    guint track;
    guint maxtracks;
    guint disc;
    guint maxdiscs;
    guint duration;
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), BKL_TYPE_ITEM_AUDIO, BklItemAudioPrivate))
G_DEFINE_TYPE (BklItemAudio, bkl_item_audio, BKL_TYPE_ITEM_EXTENDED);

static void
free_string_array (GPtrArray *array)
{
    int i;

    for (i = 0; i < array->len; i++) {
        g_free (array->pdata[i]);
    }

    g_ptr_array_free (array, TRUE);
}

static void
bkl_item_audio_finalize (GObject *object)
{
    BklItemAudio *self = (BklItemAudio *) object;
    BklItemAudioPrivate *priv = self->priv;

    g_free (priv->title);
    g_free (priv->album);

    if (priv->artists) {
        free_string_array (priv->artists);
        priv->artists = NULL;
    }

    if (priv->performers) {
        free_string_array (priv->performers);
        priv->performers = NULL;
    }

    g_free (priv->conductor);
    g_free (priv->composer);
    g_free (priv->genre);
    g_free (priv->comment);
    g_free (priv->copyright);
    g_free (priv->license);
    g_free (priv->organisation);
    g_free (priv->isrc);
    g_free (priv->contact);
    g_free (priv->year);
    g_free (priv->artist_image);

    g_signal_handlers_destroy (object);
    G_OBJECT_CLASS (bkl_item_audio_parent_class)->finalize (object);
}

static void
bkl_item_audio_dispose (GObject *object)
{
    G_OBJECT_CLASS (bkl_item_audio_parent_class)->dispose (object);
}

static void
bkl_item_audio_set_property (GObject      *object,
                             guint         prop_id,
                             const GValue *value,
                             GParamSpec   *pspec)
{
    switch (prop_id) {
    case PROP_TITLE:
        break;

    case PROP_ALBUM:
        break;

    case PROP_ARTISTS:
        break;

    case PROP_PERFORMERS:
        break;

    case PROP_CONDUCTOR:
        break;

    case PROP_COMPOSER:
        break;

    case PROP_GENRE:
        break;

    case PROP_COMMENT:
        break;

    case PROP_COPYRIGHT:
        break;

    case PROP_LICENSE:
        break;

    case PROP_ORGANISATION:
        break;

    case PROP_ISRC:
        break;

    case PROP_CONTACT:
        break;

    case PROP_YEAR:
        break;

    case PROP_TRACK:
        break;

    case PROP_MAXTRACKS:
        break;

    case PROP_DISC:
        break;

    case PROP_MAXDISCS:
        break;

    case PROP_DURATION:
        break;

    case PROP_ARTIST_IMAGE:
        break;

    default:
        break;
    }
}

static void
bkl_item_audio_get_property (GObject    *object,
                             guint       prop_id,
                             GValue     *value,
                             GParamSpec *pspec)
{
    switch (prop_id) {
    case PROP_TITLE:
        break;

    case PROP_ALBUM:
        break;

    case PROP_ARTISTS:
        break;

    case PROP_PERFORMERS:
        break;

    case PROP_CONDUCTOR:
        break;

    case PROP_COMPOSER:
        break;

    case PROP_GENRE:
        break;

    case PROP_COMMENT:
        break;

    case PROP_COPYRIGHT:
        break;

    case PROP_LICENSE:
        break;

    case PROP_ORGANISATION:
        break;

    case PROP_ISRC:
        break;

    case PROP_CONTACT:
        break;

    case PROP_YEAR:
        break;

    case PROP_TRACK:
        break;

    case PROP_MAXTRACKS:
        break;

    case PROP_DISC:
        break;

    case PROP_MAXDISCS:
        break;

    case PROP_DURATION:
        break;

    case PROP_ARTIST_IMAGE:
        break;

    default:
        break;
    }
}

static void
bkl_item_audio_class_init (BklItemAudioClass *klass)
{
    GObjectClass *o_class = (GObjectClass *)klass;

    o_class->dispose = bkl_item_audio_dispose;
    o_class->finalize = bkl_item_audio_finalize;
    o_class->set_property = bkl_item_audio_set_property;
    o_class->get_property = bkl_item_audio_get_property;

    g_type_class_add_private (klass, sizeof (BklItemAudioPrivate));

    g_object_class_install_property (o_class, PROP_TITLE,
                                     g_param_spec_string ("title", "", "", "",
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_ALBUM,
                                     g_param_spec_string ("album", "", "", "",
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_ARTISTS,
                                     g_param_spec_pointer ("artists", "", "",
                                                           G_PARAM_READWRITE |
                                                           G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_PERFORMERS,
                                     g_param_spec_pointer ("performers", "", "",
                                                           G_PARAM_READWRITE |
                                                           G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_CONDUCTOR,
                                     g_param_spec_string ("conductor", "", "", "",
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_COMPOSER,
                                     g_param_spec_string ("composer", "", "", "",
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_GENRE,
                                     g_param_spec_string ("genre", "", "", "",
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_COMMENT,
                                     g_param_spec_string ("comment", "", "", "",
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_COPYRIGHT,
                                     g_param_spec_string ("copyright", "", "", "",
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_LICENSE,
                                     g_param_spec_string ("license", "", "", "",
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_ORGANISATION,
                                     g_param_spec_string ("organisation", "", "", "",
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_ISRC,
                                     g_param_spec_string ("isrc", "", "", "",
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_CONTACT,
                                     g_param_spec_string ("contact", "", "", "",
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_YEAR,
                                     g_param_spec_string ("year", "", "", "",
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_ARTIST_IMAGE,
                                     g_param_spec_string ("artist-image", "", "", "",
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_TRACK,
                                     g_param_spec_uint ("track", "", "",
                                                        0, G_MAXUINT, 0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (o_class, PROP_MAXTRACKS,
                                     g_param_spec_uint ("maxtracks", "", "",
                                                        0, G_MAXUINT, 0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_DISC,
                                     g_param_spec_uint ("disc", "", "",
                                                        0, G_MAXUINT, 0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_MAXDISCS,
                                     g_param_spec_uint ("maxdiscs", "", "",
                                                        0, G_MAXUINT, 0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_DURATION,
                                     g_param_spec_uint ("duration", "", "",
                                                        0, G_MAXUINT, 0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_STRINGS));
}

static void
bkl_item_audio_init (BklItemAudio *self)
{
    self->priv = GET_PRIVATE (self);

    bkl_item_set_item_type ((BklItem *) self, BKL_ITEM_TYPE_AUDIO);
}

BklItemAudio *
bkl_item_audio_new (void)
{
    BklItemAudio *audio;

    audio = g_object_new (BKL_TYPE_ITEM_AUDIO, NULL);
    return audio;
}

#define IS_EMPTY_STRING(s) ((s) == NULL || *(s) == '\0')
BklItemAudio *
bkl_item_audio_new_from_field (KozoField *field)
{
    BklItemAudio *audio;
    const char *data;
    int number;

    audio = g_object_new (BKL_TYPE_ITEM_AUDIO, NULL);

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_TITLE);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_audio_set_title (audio, data);
    }

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_ALBUM);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_audio_set_album (audio, data);
    }

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_ARTISTS);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_audio_set_artists (audio, _bkl_string_to_string_array (data));
    }

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_PERFORMERS);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_audio_set_performers (audio, _bkl_string_to_string_array (data));
    }

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_CONDUCTOR);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_audio_set_conductor (audio, data);
    }

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_COMPOSER);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_audio_set_composer (audio, data);
    }

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_GENRE);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_audio_set_genre (audio, data);
    }

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_COMMENT);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_audio_set_comment (audio, data);
    }

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_COPYRIGHT);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_audio_set_copyright (audio, data);
    }

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_LICENSE);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_audio_set_license (audio, data);
    }

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_ORGANISATION);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_audio_set_organisation (audio, data);
    }

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_ISRC);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_audio_set_isrc (audio, data);
    }

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_CONTACT);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_audio_set_contact (audio, data);
    }

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_YEAR);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_audio_set_year (audio, data);
    }

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_TRACK);
    if (!IS_EMPTY_STRING (data)) {
        number = atoi (data);
        bkl_item_audio_set_track (audio, number);
    }

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_MAXTRACKS);
    if (!IS_EMPTY_STRING (data)) {
        number = atoi (data);
        bkl_item_audio_set_maxtracks (audio, number);
    }

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_DISC);
    if (!IS_EMPTY_STRING (data)) {
        number = atoi (data);
        bkl_item_audio_set_disc (audio, number);
    }

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_MAXDISCS);
    if (!IS_EMPTY_STRING (data)) {
        number = atoi (data);
        bkl_item_audio_set_maxdiscs (audio, number);
    }

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_DURATION);
    if (!IS_EMPTY_STRING (data)) {
        number = atoi (data);
        bkl_item_audio_set_duration (audio, number);
    }

    data = kozo_field_get_value_string (field, BKL_AUDIO_FIELD_ARTIST_IMAGE);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_audio_set_artist_image (audio, data);
    }

    return audio;
}

GSList *
bkl_item_audio_get_fields (BklItemAudio *item)
{
    BklItemAudioPrivate *priv = item->priv;
    GSList *fields = NULL;

    fields = g_slist_prepend (fields, g_strdup (priv->artist_image));
    fields = g_slist_prepend (fields, g_strdup_printf ("%u", priv->duration));
    fields = g_slist_prepend (fields, g_strdup_printf ("%u", priv->maxdiscs));
    fields = g_slist_prepend (fields, g_strdup_printf ("%u", priv->disc));
    fields = g_slist_prepend (fields, g_strdup_printf ("%u", priv->maxtracks));
    fields = g_slist_prepend (fields, g_strdup_printf ("%u", priv->track));
    fields = g_slist_prepend (fields, g_strdup (priv->year));
    fields = g_slist_prepend (fields, g_strdup (priv->contact));
    fields = g_slist_prepend (fields, g_strdup (priv->isrc));
    fields = g_slist_prepend (fields, g_strdup (priv->organisation));
    fields = g_slist_prepend (fields, g_strdup (priv->license));
    fields = g_slist_prepend (fields, g_strdup (priv->copyright));
    fields = g_slist_prepend (fields, g_strdup (priv->comment));
    fields = g_slist_prepend (fields, g_strdup (priv->genre));
    fields = g_slist_prepend (fields, g_strdup (priv->composer));
    fields = g_slist_prepend (fields, g_strdup (priv->conductor));
    fields = g_slist_prepend (fields,
                              _bkl_string_array_to_string (priv->performers));
    fields = g_slist_prepend (fields,
                              _bkl_string_array_to_string (priv->artists));
    fields = g_slist_prepend (fields, g_strdup (priv->album));
    fields = g_slist_prepend (fields, g_strdup (priv->title));

    return fields;
}

void
bkl_item_audio_set_title (BklItemAudio *audio,
                          const char   *title)
{
    g_free (audio->priv->title);
    audio->priv->title = g_strdup (title);
}

const char *
bkl_item_audio_get_title (BklItemAudio *audio)
{
    return audio->priv->title;
}

void
bkl_item_audio_set_album (BklItemAudio *audio,
                          const char   *album)
{
    g_free (audio->priv->album);
    audio->priv->album = g_strdup (album);
}

const char *
bkl_item_audio_get_album (BklItemAudio *audio)
{
    return audio->priv->album;
}

void
bkl_item_audio_set_artists (BklItemAudio *audio,
                            GPtrArray    *artists)
{
    if (audio->priv->artists) {
        free_string_array (audio->priv->artists);
    }
    audio->priv->artists = artists;
}

GPtrArray *
bkl_item_audio_get_artists (BklItemAudio *audio)
{
    return audio->priv->artists;
}

void
bkl_item_audio_set_performers (BklItemAudio *audio,
                               GPtrArray    *performers)
{
    if (audio->priv->performers) {
        free_string_array (audio->priv->performers);
    }
    audio->priv->performers = performers;
}

GPtrArray *
bkl_item_audio_get_performers (BklItemAudio *audio)
{
    return audio->priv->performers;
}

void
bkl_item_audio_set_conductor (BklItemAudio *audio,
                              const char   *conductor)
{
    g_free (audio->priv->conductor);
    audio->priv->conductor = g_strdup (conductor);
}

const char *
bkl_item_audio_get_conductor (BklItemAudio *audio)
{
    return audio->priv->conductor;
}

void
bkl_item_audio_set_composer (BklItemAudio *audio,
                             const char   *composer)
{
    g_free (audio->priv->composer);
    audio->priv->composer = g_strdup (composer);
}

const char *
bkl_item_audio_get_composer (BklItemAudio *audio)
{
    return audio->priv->composer;
}

void
bkl_item_audio_set_genre (BklItemAudio *audio,
                          const char   *genre)
{
    g_free (audio->priv->genre);
    audio->priv->genre = g_strdup (genre);
}

const char *
bkl_item_audio_get_genre (BklItemAudio *audio)
{
    return audio->priv->genre;
}

void
bkl_item_audio_set_comment (BklItemAudio *audio,
                            const char   *comment)
{
    g_free (audio->priv->comment);
    audio->priv->comment = g_strdup (comment);
}

const char *
bkl_item_audio_get_comment (BklItemAudio *audio)
{
    return audio->priv->comment;
}

void
bkl_item_audio_set_copyright (BklItemAudio *audio,
                              const char   *copyright)
{
    g_free (audio->priv->copyright);
    audio->priv->copyright = g_strdup (copyright);
}

const char *
bkl_item_audio_get_copyright (BklItemAudio *audio)
{
    return audio->priv->copyright;
}

void
bkl_item_audio_set_license (BklItemAudio *audio,
                            const char   *license)
{
    g_free (audio->priv->license);
    audio->priv->license = g_strdup (license);
}

const char *
bkl_item_audio_get_license (BklItemAudio *audio)
{
    return audio->priv->license;
}

void
bkl_item_audio_set_organisation (BklItemAudio *audio,
                                 const char   *organisation)
{
    g_free (audio->priv->organisation);
    audio->priv->organisation = g_strdup (organisation);
}

const char *
bkl_item_audio_get_organisation (BklItemAudio *audio)
{
    return audio->priv->organisation;
}

void
bkl_item_audio_set_isrc (BklItemAudio *audio,
                         const char   *isrc)
{
    g_free (audio->priv->isrc);
    audio->priv->isrc = g_strdup (isrc);
}

const char *
bkl_item_audio_get_isrc (BklItemAudio *audio)
{
    return audio->priv->isrc;
}

void
bkl_item_audio_set_contact (BklItemAudio *audio,
                            const char   *contact)
{
    g_free (audio->priv->contact);
    audio->priv->contact = g_strdup (contact);
}

const char *
bkl_item_audio_get_contact (BklItemAudio *audio)
{
    return audio->priv->contact;
}

void
bkl_item_audio_set_year (BklItemAudio *audio,
                         const char   *year)
{
    g_free (audio->priv->year);
    audio->priv->year = g_strdup (year);
}

const char *
bkl_item_audio_get_year (BklItemAudio *audio)
{
    return audio->priv->year;
}

void
bkl_item_audio_set_track (BklItemAudio *audio,
                          guint         track)
{
    audio->priv->track = track;
}

guint
bkl_item_audio_get_track (BklItemAudio *audio)
{
    return audio->priv->track;
}

void
bkl_item_audio_set_maxtracks (BklItemAudio *audio,
                              guint         maxtracks)
{
    audio->priv->maxtracks = maxtracks;
}

guint
bkl_item_audio_get_max_tracks (BklItemAudio *audio)
{
    return audio->priv->maxtracks;
}

void
bkl_item_audio_set_disc (BklItemAudio *audio,
                         guint         disc)
{
    audio->priv->disc = disc;
}

guint
bkl_item_audio_get_disc (BklItemAudio *audio)
{
    return audio->priv->disc;
}

guint
bkl_item_audio_get_maxdiscs (BklItemAudio *audio)
{
    return audio->priv->maxdiscs;
}

void
bkl_item_audio_set_maxdiscs (BklItemAudio *audio,
                             guint         maxdiscs)
{
    audio->priv->maxdiscs = maxdiscs;
}

void
bkl_item_audio_set_duration (BklItemAudio *audio,
                          guint         duration)
{
    audio->priv->duration = duration;
}

guint
bkl_item_audio_get_duration (BklItemAudio *audio)
{
    return audio->priv->duration;
}

void
bkl_item_audio_set_artist_image (BklItemAudio *audio,
                         const char   *artist_image)
{
    g_free (audio->priv->artist_image);
    audio->priv->artist_image = g_strdup (artist_image);
}

const char *
bkl_item_audio_get_artist_image (BklItemAudio *audio)
{
    return audio->priv->artist_image;
}

char *
bkl_item_audio_to_string (BklItemAudio *audio)
{
    BklItemAudioPrivate *priv = audio->priv;
    GString *contents;
    char *str;
    int i;

    contents = g_string_new ("");
    if (priv->title) {
        g_string_append (contents, priv->title);
        g_string_append_c (contents, ' ');
    }

    if (priv->album) {
        g_string_append (contents, priv->album);
        g_string_append_c (contents, ' ');
    }

    if (priv->artists) {
        for (i = 0; i < priv->artists->len; i++) {
            g_string_append (contents, priv->artists->pdata[i]);
            g_string_append_c (contents, ' ');
        }
    }

    if (priv->performers) {
        for (i = 0; i < priv->performers->len; i++) {
            g_string_append (contents, priv->performers->pdata[i]);
            g_string_append_c (contents, ' ');
        }
    }

    if (priv->conductor) {
        g_string_append (contents, priv->conductor);
        g_string_append_c (contents, ' ');
    }

    if (priv->composer) {
        g_string_append (contents, priv->composer);
        g_string_append_c (contents, ' ');
    }

    if (priv->genre) {
        g_string_append (contents, priv->genre);
        g_string_append_c (contents, ' ');
    }

    if (priv->comment) {
        g_string_append (contents, priv->comment);
        g_string_append_c (contents, ' ');
    }

    if (priv->year) {
        g_string_append (contents, priv->year);
    }

    str = contents->str;
    g_string_free (contents, FALSE);

    return str;
}
