/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "kozo-id-cache.h"
#include "kozo-private.h"

struct _KozoIDCache {
        GHashTable *hash;
        guint n_ids;

        GList *id_list;
        GStringChunk *string_chunk;
};

/* Tweakable paramaters */

/* String chunk block size */
#define STRING_CHUNK_BLOCK_SIZE 256

#define LIST_SEPARATOR 1

static int
add_id (KozoIDCache *cache,
        const char  *desc)
{
        int id;

        id = cache->n_ids;
        cache->n_ids++;

        cache->id_list = g_list_append (cache->id_list, (char *) desc);
        g_hash_table_insert (cache->hash, (char *) desc,
                             GINT_TO_POINTER (id));

        return id;
}

KozoIDCache *
kozo_id_cache_new (const char *fields,
                   int         len)
{
        KozoIDCache *cache;
        char *end, *start;
        int i;

        cache = g_new (KozoIDCache, 1);
        cache->hash = g_hash_table_new (g_str_hash, g_str_equal);
        cache->n_ids = 0;
        cache->id_list = NULL;
        cache->string_chunk = g_string_chunk_new (STRING_CHUNK_BLOCK_SIZE);

        if (fields == NULL) {
                return cache;
        }

        start = (char *) fields;
        end = start;
        i = 0;

        while (i < len && *start != '\0' && *end != '\0') {
                if (*end == 1) {
                        char *desc = g_string_chunk_insert_len (cache->string_chunk,
                                                                start,
                                                                end - start);
                        add_id (cache, desc);
                        start = end + 1;
                }

                end++;
                i++;
        }
        return cache;
}

void
kozo_id_cache_free (KozoIDCache *cache)
{
        kozo_id_cache_flush (cache);

        g_hash_table_destroy (cache->hash);
        g_string_chunk_free (cache->string_chunk);

        g_free (cache);
}

int
kozo_id_cache_get (KozoIDCache *cache,
                   const char  *desc)
{
        gpointer retp;

        if (g_hash_table_lookup_extended (cache->hash, desc, NULL, &retp)) {
                return GPOINTER_TO_INT (retp);
        }

        return -1;
}

int
kozo_id_cache_add (KozoIDCache *cache,
                   const char  *desc,
                   GError     **error)
{
        return add_id (cache, g_string_chunk_insert (cache->string_chunk,
                                                     desc));
}

char *
kozo_id_cache_get_string (KozoIDCache *cache)
{
        GString *built;
        GList *l;
        char *str;

        if (cache->id_list == NULL) {
                return NULL;
        }

        built = g_string_new ("");
        for (l = cache->id_list; l; l = l->next) {
                g_string_append (built, l->data);
                g_string_append_c (built, LIST_SEPARATOR);
        }

        str = built->str;
        g_string_free (built, FALSE);
        return str;
}

void
kozo_id_cache_flush (KozoIDCache *cache)
{
}

guint
kozo_id_cache_n_ids (KozoIDCache *cache)
{
        return cache->n_ids;
}
