/** 
 * @namespace   biewlib
 * @file        biewlib/sysdep/ia32/win32/fileio.c
 * @brief       This file contains implementation of file i/o routines for Win32s.
 * @version     -
 * @remark      this source file is part of Binary vIEW project (BIEW).
 *              The Binary vIEW (BIEW) is copyright (C) 1995 Nick Kurshev.
 *              All rights reserved. This software is redistributable under the
 *              licence given in the file "Licence.en" ("Licence.ru" in russian
 *              translation) distributed in the BIEW archive.
 * @note        Requires POSIX compatible development system
 *
 * @author      Nick Kurshev
 * @since       1999
 * @note        Development, fixes and improvements
 * @warning     WinNT may "die", when performs long file query in network.
**/
#include <windows.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include "biewlib/biewlib.h"

static char f_buff[FILENAME_MAX+1];

static char * __FASTCALL__ fnUnix2Dos(const char *fn)
{
  char *ptr;
  strncpy(f_buff,fn,sizeof(f_buff)-1);
  ptr = f_buff;
  while((ptr = strchr(ptr,'/')) != NULL) *ptr = '\\';
  return f_buff;
}

int __FASTCALL__ __OsCreate(const char *name)
{
  char *dosname;
  HANDLE handle;
  dosname = fnUnix2Dos(name);
  handle = CreateFile(dosname,
                      GENERIC_WRITE | GENERIC_READ,
                      0,
                      NULL,
                      CREATE_NEW,
                      FILE_ATTRIBUTE_NORMAL,
                      NULL);
  if(handle == INVALID_HANDLE_VALUE)
  {
    errno = GetLastError();
    handle = (HANDLE)-1;
  }
  return (int)handle;
}

int __FASTCALL__ __OsDelete(const char *name)
{
  char *dosname;
  dosname = fnUnix2Dos(name);
  if(!DeleteFile(dosname))
  {
    errno = GetLastError();
    return -1;
  }
  else return 0;
}

int __FASTCALL__ __OsRename(const char *oldname,const char *newname)
{
  char *newdosname;
  char olddosname[FILENAME_MAX+1];
  newdosname = fnUnix2Dos(oldname);
  strncpy(olddosname,newdosname,sizeof(olddosname)-1);
  newdosname = fnUnix2Dos(newname);
  if(!MoveFile(olddosname,newdosname))
  {
    errno = GetLastError();
    return -1;
  }
  else return 0;
}

void __FASTCALL__ __OsClose(int handle)
{
  CloseHandle((HANDLE)handle);
}

int __FASTCALL__ __OsDupHandle(int handle)
{
  HANDLE hret;
  if(DuplicateHandle(GetCurrentProcess(),
                     (HANDLE)handle,
                     GetCurrentProcess(),
                     &hret,
                     FILE_MAP_ALL_ACCESS,
                     TRUE,
                     DUPLICATE_SAME_ACCESS) == 0)
  {
    errno = GetLastError();
    hret = (HANDLE)-1;
  }
  return (int)hret;
}

int  __FASTCALL__ __OsOpen(const char *fname,int mode)
{
  char *dosname;
  HANDLE handle;
  int access,smode;
  access = smode = 0;
  access = GENERIC_READ;
  dosname = fnUnix2Dos(fname);
  if(mode & FO_WRITEONLY) access |= GENERIC_WRITE;
  if(mode & FO_READWRITE) access |= GENERIC_WRITE | GENERIC_READ;
  if(mode & SO_DENYWRITE) smode |= FILE_SHARE_READ;
  if(mode & SO_DENYREAD) smode |= FILE_SHARE_WRITE;
  if(mode & SO_DENYNONE) smode |= FILE_SHARE_READ | FILE_SHARE_WRITE;
  handle = CreateFile(dosname,
                      access,
                      smode,
                      NULL,
                      OPEN_EXISTING,
                      FILE_ATTRIBUTE_NORMAL,
                      NULL);
  if(handle == INVALID_HANDLE_VALUE)
  {
    errno = GetLastError();
    handle = (HANDLE)-1;
  }
  return (int)handle;
}

long __FASTCALL__ __OsSeek(int handle,long offset,int origin)
{
  return SetFilePointer((HANDLE)handle,offset,NULL,origin);
}

int __FASTCALL__ __OsTruncFile( int handle, unsigned long newsize)
{
  int ret = 0;
  __OsSeek(handle,newsize,SEEKF_START);
  if(SetEndOfFile((HANDLE)handle) == 0)
  {
    errno = GetLastError();
    ret = 1;
  }
  return ret;
}

int __FASTCALL__  __OsRead(int handle, void *buff, unsigned size)
{
  DWORD ret = size;
  if(ReadFile((HANDLE)handle,buff,size,&ret,NULL) == 0)
  {
    errno = GetLastError();
    ret = 0;
  }
  return ret;
}

int __FASTCALL__  __OsWrite(int handle,const void *buff, unsigned size)
{
  DWORD ret = size;
  if(WriteFile((HANDLE)handle,buff,size,&ret,NULL) == 0)
  {
    errno = GetLastError();
    ret = 0;
  }
  return ret;
}

int __FASTCALL__ __OsChSize(int handle, long size)
{
  return __OsTruncFile(handle,size);
}

long __FASTCALL__ __FileLength(int handle)
{
  DWORD ret;
  if((ret = GetFileSize((HANDLE)handle,NULL)) == 0xFFFFFFFFUL)
  {
    errno = GetLastError();
    ret = 0;
  }
  return ret;
}

long __FASTCALL__ __OsTell(int handle) { return __OsSeek(handle,0L,SEEKF_CUR); }

tBool __FASTCALL__ __IsFileExists(const char *name)
{
   int handle = __OsOpen(name,FO_READONLY | SO_DENYNONE);
   if(handle != -1) __OsClose(handle);
   return handle != -1;
}
