/** 
 * @namespace   biew
 * @file        search.c
 * @brief       This file contains implementation of file search interface.
 * @version     -
 * @remark      this source file is part of Binary vIEW project (BIEW).
 *              The Binary vIEW (BIEW) is copyright (C) 1995 Nick Kurshev.
 *              All rights reserved. This software is redistributable under the
 *              licence given in the file "Licence.en" ("Licence.ru" in russian
 *              translation) distributed in the BIEW archive.
 * @note        Requires POSIX compatible development system
 *
 * @author      Nick Kurshev
 * @since       1995
 * @note        Development, fixes and improvements
**/
#include <string.h>
#include <limits.h>
#include <stdlib.h>
#define  _CT_FTM
#include <ctype.h>

#include "colorset.h"
#include "bmfile.h"
#include "tstrings.h"
#include "biewhelp.h"
#include "biewutil.h"
#include "bconsole.h"
#include "reg_form.h"
#include "biewlib/kbd_code.h"
#include "biewlib/pmalloc.h"
#include "biewlib/bbio.h"

#define MAX_SEARCH_SIZE 21

static TWindow *prcntswnd;
extern void ReReadFile( int );

unsigned char search_buff[MAX_SEARCH_SIZE] = "";
unsigned char search_len = 0;
tBool search_Case = False,search_Word = False,search_Backward = False;
static tBool __found;

static unsigned long __NEAR__ __FASTCALL__ __lfind(unsigned long start)
{
  unsigned long flen;
  unsigned long endscan;
  unsigned long tsize,cpos,findptr = UINT_MAX,retval;
  unsigned proc,pproc,pmult;
  unsigned char __search_len;
  int direct;
  tBool cond;
  char *fbuff;
  char cbuff[MAX_SEARCH_SIZE];
  char *nbuff;
  unsigned char ch,ch1;
  unsigned bio_opt,iptr,symb_size;
  int cache[UCHAR_MAX+1];
  symb_size = activeMode->get_symbol_size();
  if(!(fbuff = PMalloc(MAX_SEARCH_SIZE*symb_size))) return 0;
  if(!(nbuff = PMalloc(activeMode->get_symbol_size())))
  {
    PFree(fbuff);
    return 0;
  }
  /**
   * Cache initialization for adapted Boyer-Moore search algorithm
  */
  memset(cache,0,sizeof(cache));
  for(iptr = 0;iptr < search_len;iptr++)
  {
    cache[ search_Case ? search_buff[iptr] : toupper(search_buff[iptr]) ] = iptr+1;
  }
  __found = False;
  flen = BMGetFLength();
  endscan = search_Backward ? 0 : flen;
  direct  = search_Backward ? -1 : 1;
  tsize = labs(endscan-start);
  pmult = 100;
  if(tsize > ULONG_MAX/100) { tsize /= 100; pmult = 1; }
  cond = False;
  pproc = proc = 0;
  start = search_Backward ? start : start+(search_len-1)*symb_size;
  bio_opt = bioGetOptimization(BMbioHandle());
  bioSetOptimization(BMbioHandle(),
                    (bio_opt & (~BIO_OPT_DIRMASK)) |
                    (search_Backward ? BIO_OPT_RBACKSCAN : BIO_OPT_RFORWARD));
  retval = 0;
  start = (start/symb_size)*symb_size; /** align on symbol boundary */
  memcpy(cbuff,search_buff,search_len);
  if(!search_Case) memupr((void *)cbuff,search_len);
  for(cpos = start;start != endscan;start+=search_len*direct*symb_size,cpos=start)
  {
   if(direct == 1 && start > flen - (search_len*symb_size)) break;
   if(direct == -1 && start < (search_len*symb_size)) break;
   proc = (unsigned)((cpos*pmult)/tsize);
   if(proc != pproc)
   {
     if(!ShowPercentInWnd(prcntswnd,pproc=proc))  break;
   }
   BMReadBufferEx(nbuff,symb_size,start,BM_SEEK_SET);
   if((activeMode->flags & __MF_TEXT) == __MF_TEXT) activeMode->convert_cp(nbuff,symb_size);
   ch = nbuff[0];
   if(!search_Case) ch = toupper(ch);
   if(cache[ch])
   {
    if(search_len > 1)
    {
      findptr = start-(cache[ch]-1)*symb_size;
      BMReadBufferEx((void *)fbuff,search_len*symb_size,findptr,BM_SEEK_SET);
      if((activeMode->flags & __MF_TEXT) == __MF_TEXT)
                __search_len = activeMode->convert_cp((char *)fbuff,search_len*symb_size);
      else      __search_len = search_len;
      if(!search_Case) memupr((void *)fbuff,__search_len);
      if(memcmp(fbuff,cbuff,__search_len) == 0) cond = True;
      else
      {
        start -= (search_len-1)*direct*symb_size;
        continue;
      }
    }
    else { findptr = start; cond = True; }
    if(search_Word && cond)
    {
       if(start)
       {
         BMReadBufferEx(nbuff,symb_size,findptr - symb_size,BM_SEEK_SET);
         if((activeMode->flags & __MF_TEXT) == __MF_TEXT) activeMode->convert_cp(nbuff,symb_size);
         ch = nbuff[0];
       }
       else      ch = ' ';
       if(start + search_len < flen)
       {
         BMReadBufferEx(nbuff,symb_size,findptr + (search_len*symb_size),BM_SEEK_SET);
         if((activeMode->flags & __MF_TEXT) == __MF_TEXT) activeMode->convert_cp(nbuff,symb_size);
         ch1 = nbuff[0];
       }
       else      ch1 = ' ';
       if(!(isseparate(ch) && isseparate(ch1))) cond = False;
    }
   }
   if(cond) { __found = True; retval = findptr; break; }
  }
  bioSetOptimization(BMbioHandle(),bio_opt);
  PFree(fbuff);
  PFree(nbuff);
  return retval;
}

int __FASTCALL__ ExpandHex(char * dest,const unsigned char * src,int size,char hard)
{
  int i,k;
    dest[0] = '\0';
    k = 0;
    for(i = 0;i < size;i++)
    {
        char * cptr;
        cptr = Get2Digit(src[i]);
        strcat(dest,cptr); k += 2;
        if(hard == 1) { strcat(dest,!((i + 1)%4) ? ( !((i+1)%16) ? " " : "-" ) : " "); k++; }
        else
          if(hard == 2) { strcat(dest,!((i + 1)%4) ? "  " : " ");  k += !((i + 1)%4) ? 2 : 1; }
          else { strcat(dest," "); k++; }
    }
  return k;
}

static void __NEAR__ __FASTCALL__ SearchPaint(TWindow *wdlg,char ** ebuff,unsigned char searchlen,tBool search_CaseSens,tBool WordOnly,tBool search_Backwarding)
{
 TWindow *using = twUsedWin();
 int i;
 twUseWin(wdlg);
 twDirectWrite(10,3,ebuff[0],searchlen);
 twGotoXY(10+searchlen,3);
 for(i = searchlen;i < 20;i++) twPutChar(TWC_DEF_FILLER);
 twDirectWrite(10,5,ebuff[1],searchlen*3);
 twGotoXY(10+searchlen*3,5);
 for(i = searchlen*3;i < 60;i++) twPutChar('');
 twSetColorAttr(dialog_cset.group.active);
 twGotoXY(40,2); twPutChar(GetBool(search_CaseSens));
 twGotoXY(40,3); twPutChar(GetBool(WordOnly));
 twGotoXY(45,4); twPutS(search_Backwarding ? BACKWARD : FORWARD);
 twSetColorAttr(dialog_cset.main);
 twUseWin(using);
}

tBool __FASTCALL__ SearchDialog( char * searchbuff, unsigned char *searchlen,tBool *search_CaseSens,tBool *WordOnly,tBool *search_Backwarding)
{
  TWindow *hwnd,* ewnd[2];
  tAbsCoord x1,y1,x2,y2;
  tRelCoord X1,Y1,X2,Y2;
  unsigned x[2] = { 0, 0 };
  int rret;
  tBool update,ret;
  char attr[2] = { __ESS_FILLER_7BIT | __ESS_WANTRETURN | __ESS_ENABLEINSERT | __ESS_NON_C_STR,
                   __ESS_WANTRETURN | __ESS_ASHEX | __ESS_NON_C_STR };
  char ebuff1[MAX_SEARCH_SIZE],ebuff2[MAX_SEARCH_SIZE*3];
  char *ebuff[2],*legal[2];
  unsigned mlen[2],flags;
  int ch,i;
  unsigned char active,oactive;
  hwnd = CrtDlgWndnls(FIND_STR,70,5);
  twSetFooterAttr(hwnd,FIND_PRMT,TW_TMODE_CENTER,dialog_cset.footer);
  twGetWinPos(hwnd,&x1,&y1,&x2,&y2);
  X1 = x1;
  Y1 = y1;
  X2 = x2;
  Y2 = y2;
  X1 += 10;
  Y1 += 3;
  X2 = X1 + 19;
  Y2 = Y1;
  ewnd[0] = CreateEditor(X1,Y1,X2,Y2,TWS_CURSORABLE);
  Y1 += 2;
  Y2 += 2;
  X2 = X1 + 59;
  ewnd[1] = CreateEditor(X1,Y1,X2,Y2,TWS_CURSORABLE | TWS_NLSOEM);
  twUseWin(hwnd);
  twGotoXY(2,1);  twPutS(TYPE_STR);
  twGotoXY(2,3);  twPutS("ASCII :");
  twGotoXY(4,5);  twPutS("HEX :");
  twSetColorAttr(dialog_cset.group.active);
  for(i = 0;i < 3;i++) { twGotoXY(38,i + 2); twPutS(msgFindOpt[i]); }
  twSetColorAttr(dialog_cset.main);
  legal[0] = NULL;
  legal[1] = &legalchars[2];
  ebuff1[0] = ebuff2[0] = '\0';
  ebuff[0] = ebuff1;
  ebuff[1] = ebuff2;
  if(searchlen)
  {
    memcpy(ebuff[0],searchbuff,*searchlen);
    ExpandHex(ebuff[1],(unsigned char *)searchbuff,*searchlen,0);
  }
  active = 0;
  oactive = active ? 0 : 1;
  rret = 2;
  ret = True;
  SearchPaint(hwnd,ebuff,*searchlen,*search_CaseSens,*WordOnly,*search_Backwarding);
  update = True;
  while(1)
  {
    if(oactive != active)
    {
      twHideWin(ewnd[oactive]);
      twShowWin(ewnd[active]);
      oactive = active;
    }
    mlen[0] = 20;
    mlen[1] = 60;
    twUseWin(ewnd[active]);
    flags = attr[active];
    if(!update) flags |= __ESS_NOREDRAW;
    ch = eeditstring(ebuff[active],legal[active],&mlen[active],
                     active ? (*searchlen)*3 : *searchlen,
                     &x[active],flags,NULL);
    update = True;
    switch(ch)
    {
       case KE_UPARROW  : active = 0; continue;
       case KE_DOWNARROW: active = 1; continue;
       case KE_TAB      : active = active ? 0 : 1; continue;
       case KE_ENTER    : if(searchlen) { rret = 1; ret = True; } else { rret = 0; ret = False; } break;
       case KE_F(10)    :
       case KE_ESCAPE   : rret = 0; ret = False; break;
       case KE_F(2)     : *search_CaseSens = (*search_CaseSens) ? False : True;
                          update = False;
                          break;
       case KE_F(3)     : (*WordOnly) = (*WordOnly) ? False : True;
                          update = False;
                          break;
       case KE_F(4)     : (*search_Backwarding) = (*search_Backwarding) ? False : True;
                          update = False;
                          break;
       case KE_F(1)     : hlpDisplay(7);
                          update = False;
                          break;
       case KE_LEFTARROW:
       case KE_RIGHTARROW:
                          update = False;
                          break;
       default : break;
    }
    if(rret != 2) break;
    twUseWin(hwnd);
    if(!active) { *searchlen = mlen[0]; memcpy(searchbuff,ebuff[0],mlen[0]); }
    else  { *searchlen = mlen[1] / 3; CompressHex((unsigned char *)searchbuff,ebuff[1],*searchlen,True); }
    if(searchlen) memcpy(ebuff[0],searchbuff,*searchlen);
    else     ebuff[0][0] = '\0';
    mlen[0] = *searchlen;
    ExpandHex(ebuff[1],(unsigned char *)searchbuff,*searchlen,0);
    mlen[1] = (*searchlen)*3;
    for(i = 0;i < 2;i++) if(x[i] > mlen[i]) x[i] = mlen[i];
    SearchPaint(hwnd,ebuff,*searchlen,*search_CaseSens,*WordOnly,*search_Backwarding);
  }
  CloseWnd(ewnd[0]);
  CloseWnd(ewnd[1]);
  CloseWnd(hwnd);
  return ret;
}

extern TWindow * ErrorWnd;

unsigned long __FASTCALL__ Search( tBool mod )
{
  unsigned long found;
  unsigned long fmem,lmem;
  tBool ret;
  fmem = BMGetCurrFilePos();
  ret = mod ? True : SearchDialog((char *)search_buff,&search_len,&search_Case,&search_Word,&search_Backward);
  if(ret && search_len)
  {
    prcntswnd = PercentWnd(PLEASE_WAIT,SEARCHING);
    lmem = fmem;
    if(FoundTextSt != FoundTextEnd)
    {
      unsigned cp_symb_size;
      cp_symb_size = activeMode->get_symbol_size();
      if(search_Backward && lmem) lmem-=cp_symb_size;
      else if(lmem < BMGetFLength()) lmem+=cp_symb_size;
    }
    found = __lfind(lmem);
    CloseWnd(prcntswnd);
    if(__found)  { FoundTextSt = found; FoundTextEnd = found + search_len*activeMode->get_symbol_size(); return found; }
    else  ErrMessageBox(STR_NOT_FOUND,SEARCH_MSG);
  }
  BMSeek(fmem,BM_SEEK_SET);
  return fmem;
}
