;;;; compiler.scm - The CHICKEN Scheme compiler (front-end)
;
;
; "This is insane. What we clearly want to do is not exactly clear, and is rooted in NCOMPLR."
;
;
;------------------------------------------------------------------------------------------------------------------------
; Copyright (c) 2000-2002, Felix L. Winkelmann
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following
; conditions are met:
;
;   Redistributions of source code must retain the above copyright notice, this list of conditions and the following
;     disclaimer. 
;   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
;     disclaimer in the documentation and/or other materials provided with the distribution. 
;   Neither the name of the author nor the names of its contributors may be used to endorse or promote
;     products derived from this software without specific prior written permission. 
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
; AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
; CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
; POSSIBILITY OF SUCH DAMAGE.
;
; Send bugs, suggestions and ideas to: 
;
; felix@call-with-current-continuation.org
;
; Felix L. Winkelmann
; Steinweg 1A
; 37130 Gleichen, OT Weissenborn
; Germany
;
;
; Supported syntax:
;
; - Declaration specifiers:
;
; (unit <unitname>)
; (uses {<unitname>})
; ([not] standard-bindings {<name>})
; ([not] usual-integrations {<name>})
; ([not] extended-bindings (<name>})
; ([number-type] <type>)
; (fixnum-arithmetic)
; (unsafe)
; ([not] safe)
; (notinline)
; ([not] inline)
; ([not] interrupts-enabled)
; (no-bound-checks)
; (no-argc-checks)
; (no-procedure-checks)
; (block-global {<name>})
; (interrupts-disabled)
; (disable-interrupts)
; (always-bound {<name>})
; (foreign-declare {<string>})
; (block)
; (separate)
; (no-fancy-ports)
; (no-winding-callcc)
;
;   <type> = fixnum | flonum | generic
;
; - Source language:
;
; <variable>
; <constant>
; (##core#include (quote <string>))
; (##core#declare {(quote <spec>)})
; (##core#immutable <exp>)
; (##core#qualified (quote <exp>))
; (quote <exp>)
; (if <exp> <exp> [<exp>])
; (let ({(<variable> <exp>)}) <body>)
; (let-macro (<macrodef> ...) <body>)
; (let-id-macro ((<symbol> <exp>) ...) <body>)
; (lambda <variable> <body>)
; (lambda ({<variable>}+ [. <variable>]) <body>)
; (set! <variable> <exp>)
; (##core#named-lambda <name> <llist> <body>)
; (##core#loop-lambda <llist> <body>)
; (##core#undefined)
; (##core#primitive <name>)
; (##core#inline <op> {<exp>})
; (##core#inline_allocate (<op> <words>) {<exp>})
; (##core#inline_ref (<name> <type>))
; (##core#inline_update (<name> <type>) <exp>)
; (##core#compiletimetoo <exp>)
; (##core#compiletimeonly <exp>)
; (##core#elaborationtimetoo <exp>)
; (##core#elaborationtimeonly <exp>)
; (##core#define-module (quote <name>) (quote <clauses>))
; (##core#define-foreign-variable (quote <symbol>) (quote <type>) [(quote <string>)])
; (##core#define-foreign-parameter (quote <symbol>) (quote <type>) [(quote <string>)])
; (##core#define-foreign-type (quote <symbol>) (quote <type>) [<proc1> [<proc2>]])
; (##core#foreign-lambda (quote <type>) (quote <string>) {(quote <type>)})
; (##core#foreign-lambda* (quote <type>) (quote ({(<type> <var>)})) {(quote <string>)})
; (##core#foreign-callback-lambda (quote <type>) (quote <string>) {(quote <type>)})
; (##core#foreign-callback-lambda* (quote <type>) (quote ({(<type> <var>)})) {(quote <string>)})
; (##core#define-inline (quote <name>) <exp>)
; (##core#define-constant (quote <name>) <exp>)
; (##core#foreign-callback-wrapper (quote <name>) (quote <qualifiers>) (quote <type>) (quote {<type>}) <exp>)
; (##core#define-external-variable (quote <name>) (quote <type>))
; (##core#enable-unqualified-quoted-symbols)
; (##core#disable-unqualified-quoted-symbols)
; (<exp> {<exp>})
;
; - Core language:
;
; [##core#variable {<variable>}]
; [if {} <exp> <exp> <exp>)]
; [quote {<exp>}]
; [let {<variable>} <exp-v> <exp>]
; [##core#lambda {<id> <mode> (<variable>... [. <variable>]) <size>} <exp>]
; [set! {<variable>} <exp>]
; [##core#undefined {}]
; [##core#primitive {<name>}]
; [##core#inline {<op>} <exp>...]
; [##core#inline_allocate {<op <words>} <exp>...]
; [##core#inline_ref {<name> <type>}]
; [##core#inline_update {<name> <type>} <exp>]
; [##core#call {<safe-flag> [<debug-symbol>]} <exp-f> <exp>...]
; [##core#callunit {<unitname>} <exp>...]
; [##core#switch {<count>} <exp> <const1> <body1> ... <defaultbody>]
; [##core#cond <exp> <exp> <exp>]
; [##core#recurse {<tail-flag>} <exp1> ...]
; [##core#return <exp>]
; [##core#direct_call {<safe-flag> <debug-symbol> <call-id> <words>} <exp-f> <exp>...]
; [##core#direct_lambda {<id> <mode> (<variable>... [. <variable>]) <size>} <exp>]
;
; - Closure converted/prepared language:
;
; [if {} <exp> <exp> <exp>]
; [quote {<exp>}]
; [##core#bind {<count>} <exp-v>... <exp>]
; [##core#undefined {}]
; [##core#inline {<op>} <exp>...]
; [##core#inline_allocate {<op <words>} <exp>...]
; [##core#inline_ref {<name> <type>}]
; [##core#inline_update {<name> <type>} <exp>]
; [##core#closure {<count>} <exp>...]
; [##core#box {} <exp>]
; [##core#unbox {} <exp>]
; [##core#ref {<index>} <exp>]
; [##core#update {<index>} <exp> <exp>]
; [##core#updatebox {} <exp> <exp>]
; [##core#update_i {<index>} <exp> <exp>]
; [##core#updatebox_i {} <exp> <exp>]
; [##core#call {<safe-flag> [<debug-symbol> [<call-id> <customizable-flag>]]} <exp-f> <exp>...]
; [##core#callunit {<unitname>} <exp>...]
; [##core#local {<index>}]
; [##core#setlocal {<index>} <exp>]
; [##core#global {<literal> <safe-flag> <block-mode>}]
; [##core#setglobal {<literal> <block-mode>} <exp>]
; [##core#setglobal_i {<literal> <block-mode>} <exp>]
; [##core#literal {<literal>}]
; [##core#immediate {<type> [<immediate>]}]     - type: bool/fix/nil/char
; [##core#proc {<name>}]
; [##core#recurse {<tail-flag> <call-id>} <exp1> ...]
; [##core#return <exp>]
; [##core#direct_call {<safe-flag> <debug-symbol> <call-id> <words>} <exp-f> <exp>...]
;
;
; Analysis database entries:
;
; <variable>:
;
;   captured -> <boolean>                    If true: variable is used outside it's home-scope
;   global -> <boolean>                      If true: variable does not occur in any lambda-list
;   call-sites -> ((<lambda-id> <node>) ...) Known call-nodes of a named procedure
;   home -> <lambda-id>                      Procedure which introduces this variable
;   unknown -> <boolean>                     If true: variable can not have a known value
;   assigned -> <boolean>                    If true: variable is assigned somewhere
;   undefined -> <boolean>                   If true: variable is unknown yet but can be known later
;   value -> <node>                          Variable has a known value
;   references -> (<node> ...)               Nodes that are accesses of this variable (##core#variable nodes)
;   side-effecting -> <boolean>              If true: variable names side-effecting standard-binding
;   foldable -> <boolean>                    If true: variable names foldable standard-binding
;   boxed -> <boolean>                       If true: variable has to be boxed after closure-conversion
;   contractable -> <boolean>                If true: variable names contractable procedure
;   collapsable -> <boolean>                 If true: variable refers to collapsable constant
;   removable -> <boolean>                   If true: variable is not used
;   replacable -> <variable>                 Variable can be replaced by another variable
;   replacing -> <boolean>                   If true: variable can replace another variable (don't remove)
;   standard-binding -> <boolean>            If true: variable names a standard binding
;   extended-binding -> <boolean>            If true: variable names an extended binding
;   unused -> <boolean>                      If true: variable is a formal parameter that is never used
;   rest-parameter -> #f | 'vector | 'list   If true: variable holds rest-argument list mode
;   o-r/access-count -> <n>                  Contains number of references as arguments of optimizable rest operators
;   constant -> <boolean>                    If true: variable has fixed value
; 
; <lambda-id>:
;
;   contains -> (<lambda-id> ...)            Procedures contained in this lambda
;   contained-in -> <lambda-id>              Procedure containing this lambda
;   has-unused-parameters -> <boolean>       If true: procedure has unused formal parameters
;   use-expr -> (<lambda-id> ...)            Marks non-direct use-sites of common subexpression
;   outermost -> (<lambda-id> ...)           Lambdas which contain common subexpressions for which this is outermost
;   closure-size -> <integer>                Number of free variables stored in a closure
;   customizable -> <boolean>                If true: all call sites are known, procedure does not escape
;   simple -> <boolean>                      If true: procedure only calls its continuation
;   explicit-rest -> <boolean>               If true: procedure is called with consed rest list
;   captured-variables -> (<var> ...)        List of closed over variables


(declare
 (unit compiler)
 (foreign-declare "
#ifndef C_DEFAULT_TARGET_STACK_SIZE
# include \"c_defaults.h\"
#endif

#ifndef C_INSTALL_HOME
# define C_INSTALL_HOME NULL
#endif

#ifndef C_DEFAULT_TARGET_HEAP_SIZE
# define C_DEFAULT_TARGET_HEAP_SIZE 0
#endif

#ifndef C_DEFAULT_TARGET_STACK_SIZE
# define C_DEFAULT_TARGET_STACK_SIZE 0
#endif") )


#{compiler
  build-information compiler-arguments process-command-line
  default-analysis-database-size default-standard-bindings default-extended-bindings side-effecting-standard-bindings
  non-foldable-standard-bindings foldable-standard-bindings non-foldable-extended-bindings foldable-extended-bindings
  standard-bindings-that-never-return-false side-effect-free-standard-bindings-that-never-return-false
  installation-home decompose-lambda-list external-to-pointer
  foreign-type-table-size file-io-only nonwinding-call/cc
  unit-name insert-timer-checks used-units inlining external-variables
  debug-info-index debug-info-vector-name profile-info-vector-name
  foreign-declarations emit-trace-info block-compilation analysis-database-size line-number-database-size
  always-bound-to-procedure block-globals make-block-variable-literal block-variable-literal? block-variable-literal-name
  target-heap-size target-stack-size try-harder default-installation-home 
  default-default-target-heap-size default-default-target-stack-size verbose-mode original-program-size
  current-program-size line-number-database-2 foreign-lambda-stubs immutable-constants foreign-variables
  rest-parameters-promoted-to-vector inline-table inline-table-used constant-table constants-used mutable-constants
  dependency-list broken-constant-nodes inline-substitutions-enabled loop-lambda-names expand-profile-lambda
  profile-lambda-list profile-lambda-index emit-profile expand-profile-lambda
  direct-call-ids foreign-type-table first-analysis expand-debug-lambda expand-debug-assignment expand-debug-call
  initialize-compiler canonicalize-expression expand-foreign-lambda update-line-number-database! scan-toplevel-assignments
  perform-cps-conversion analyze-expression simplifications perform-high-level-optimizations perform-pre-optimization!
  reorganize-recursive-bindings substitution-table simplify-named-call find-inlining-candidates perform-inlining!
  perform-closure-conversion prepare-for-code-generation compiler-source-file create-foreign-stub expand-foreign-lambda*
  transform-direct-lambdas! expand-foreign-callback-lambda* debug-lambda-list debug-variable-list
  debugging-chicken warnings-enabled bomb check-signature posq stringify symbolify flonum? build-lambda-list
  string->c-identifier c-ify-string words check-and-open-input-file close-checked-input-file fold-inner constant?
  collapsable-literal? immediate? canonicalize-begin-body extract-mutable-constants string->expr get get-all
  put! collect! count! get-line get-line-2 find-lambda-container display-analysis-database varnode qnode 
  build-node-graph build-expression-tree fold-boolean inline-lambda-bindings match-node expression-has-side-effects?
  simple-lambda-node? compute-database-statistics print-program-statistics output gen gen-list 
  pprint-expressions-to-file foreign-type-check estimate-foreign-result-size scan-used-variables scan-free-variables
  topological-sort print-version print-usage initialize-analysis-database
  expand-foreign-callback-lambda default-optimization-passes default-optimization-passes-when-trying-harder
  product copyright compiler-features units-used-by-default words-per-flonum
  parameter-limit default-output-filename eq-inline-operator optimizable-rest-argument-operators
  membership-test-operators membership-unfold-limit valid-compiler-options valid-compiler-options-with-argument
  current-source-file source-file-changed make-random-name final-foreign-type
  generate-code make-variable-list make-argument-list generate-foreign-stubs foreign-type-declaration
  foreign-argument-conversion foreign-result-conversion foreign-type-convert-argument foreign-type-convert-result}

(eval-when (compile eval)
  (match-error-control #:fail) )

(include "tweaks")
(include "parameters")


;;; Parameters:

;; These are not used beyond this unit:
(define-constant foreign-type-table-size 301)
(define-constant default-inline-limit 100)
(define-constant default-analysis-database-size 3001)
(define-constant default-line-number-database-size 997)
(define-constant default-inline-passes 3)
(define-constant inline-table-size 301)
(define-constant constant-table-size 301)

(eval-when (eval)
  (define installation-home #f)
  (define default-target-heap-size #f)
  (define default-target-stack-size #f) )

(eval-when (load)
  (define-foreign-variable installation-home c-string "C_INSTALL_HOME")
  (define-foreign-variable default-target-heap-size int "C_DEFAULT_TARGET_HEAP_SIZE")
  (define-foreign-variable default-target-stack-size int "C_DEFAULT_TARGET_STACK_SIZE") )

(define user-options-pass (make-parameter #f))
(define user-read-pass (make-parameter #f))
(define user-preprocessor-pass (make-parameter #f))
(define user-pass (make-parameter #f))


;;; Global variables containing compilation parameters:

(define unit-name #f)
(define number-type 'generic)
(define standard-bindings '())
(define extended-bindings '())
(define insert-timer-checks #t)
(define used-units '())
(define unsafe #f)
(define inlining #f)
(define always-bound '())
(define always-bound-to-procedure '())
(define foreign-declarations '())
(define emit-trace-info #f)
(define block-compilation #f)
(define analysis-database-size default-analysis-database-size)
(define line-number-database-size default-line-number-database-size)
(define target-heap-size #f)
(define target-stack-size #f)
(define inline-passes default-inline-passes)		; maximum number of inlining passes
(define inline-limit default-inline-limit)		; percentage of original program size
(define try-harder #f)
(define optimize-leaf-routines #f)
(define file-io-only #f)
(define nonwinding-call/cc #f)
(define emit-debug-info #f)
(define debug-calls #f)
(define debug-loops #f)
(define emit-profile #f)
(define no-bound-checks #f)
(define no-argc-checks #f)
(define no-procedure-checks #f)
(define block-globals '())


;;; These are here so that the backend can access them:

(define default-installation-home installation-home)
(define default-default-target-heap-size default-target-heap-size)
(define default-default-target-stack-size default-target-stack-size)


;;; Other global variables:

(define verbose-mode #f)
(define original-program-size #f)
(define current-program-size 0)
(define line-number-database-2 #f)
(define immutable-constants '())
(define expand-only #f)
(define rest-parameters-promoted-to-vector '())
(define inline-table #f)
(define inline-table-used #f)
(define constant-table #f)
(define constants-used #f)
(define mutable-constants '())
(define write-dependencies #f)
(define dependency-output #f)
(define dependency-list '())
(define broken-constant-nodes '())
(define inline-substitutions-enabled #f)
(define direct-call-ids '())
(define first-analysis #t)
(define foreign-type-table #f)
(define foreign-variables '())
(define foreign-lambda-stubs '())
(define foreign-callback-stubs '())
(define external-variables '())
(define debug-lambda-list '())
(define debug-variable-list '())
(define loop-lambda-names '())
(define debug-info-index 0)
(define debug-info-vector-name #f)
(define profile-lambda-list '())
(define profile-lambda-index 0)
(define profile-info-vector-name #f)
(define external-to-pointer '())


;;; Initialize globals:

(randomize)

(define (initialize-compiler)
  (set! unit-name #f)
  (set! number-type 'generic)
  (set! standard-bindings '())
  (set! extended-bindings '())
  (set! insert-timer-checks #t)
  (set! used-units '())
  (set! unsafe #f)
  (set! inlining #f)
  (set! always-bound '())
  (set! foreign-declarations '())
  (set! emit-trace-info #f)
  (set! block-compilation #f)
  (set! target-heap-size #f)
  (set! target-stack-size #f)
  (set! inline-passes default-inline-passes)
  (set! inline-limit default-inline-limit)
  (set! try-harder #f)
  (set! verbose-mode #f)
  (set! original-program-size #f)
  (set! current-program-size 0)
  (if line-number-database-2
      (vector-fill! line-number-database-2 '())
      (set! line-number-database-2 (make-vector line-number-database-size '())) )
  (set! foreign-lambda-stubs '())
  (set! immutable-constants '())
  (set! foreign-variables '())
  (set! expand-only #f)
  (set! rest-parameters-promoted-to-vector '())
  (if inline-table
      (vector-fill! inline-table '())
      (set! inline-table (make-vector inline-table-size '())) )
  (set! inline-table-used #f)
  (if constant-table
      (vector-fill! constant-table '())
      (set! constant-table (make-vector constant-table-size '())) )
  (set! constants-used #f)
  (set! mutable-constants '())
  (set! write-dependencies #f)
  (set! dependency-output #f)
  (set! dependency-list '()) 
  (set! inline-substitutions-enabled #f)
  (set! direct-call-ids '())
  (set! broken-constant-nodes '()) 
  (set! first-analysis #t)
  (set! optimize-leaf-routines #f)
  (set! file-io-only #f)
  (set! nonwinding-call/cc #f)
  (set! foreign-callback-stubs '())
  (set! emit-debug-info #f)
  (set! debug-lambda-list '())
  (set! debug-variable-list '())
  (set! loop-lambda-names '())
  (set! debug-calls #f)
  (set! debug-info-index 0)
  (set! debug-info-vector-name (make-random-name 'debug-info))
  (set! emit-profile #f)
  (set! profile-lambda-list '())
  (set! profile-lambda-index 0)
  (set! profile-info-vector-name (make-random-name 'profile-info))
  (set! always-bound-to-procedure '())
  (set! external-variables '())
  (set! no-bound-checks #f)
  (set! no-argc-checks #f)
  (set! no-procedure-checks #f)
  (set! block-globals '())
  (set! external-to-pointer '())
  (if foreign-type-table
      (vector-fill! foreign-type-table '())
      (set! foreign-type-table (make-vector foreign-type-table-size '())) ) )


;;; Expand macros and canonicalize expressions:

(define (canonicalize-expression exp)

  (define (resolve v ae)
    (cond [(assq v ae) => cdr]
	  [else v] ) )

  (define (walk x ae me dest)
    (cond ((symbol? x)
	   (cond [(assq x ae) 
		  => (lambda (n)
		       (walk (##sys#macroexpand-hook (cdr n) me) ae me dest) ) ] ; with highlevel macros this is a noop.
		 [(and constants-used (##sys#hash-table-ref constant-table x)) 
		  => (lambda (val) (walk (car val) ae me dest)) ]
		 [(and inline-table-used (##sys#hash-table-ref inline-table x))
		  => (lambda (val) (walk val ae me dest)) ]
		 [(assq x foreign-variables)
		  => (lambda (fv) 
		       (let* ([t (second fv)]
			      [ft (final-foreign-type t)] 
			      [body `(##core#inline_ref (,(third fv) ,t))] )
			 (foreign-type-convert-result
			  (if (eq? 'c-string ft)
			      `(##sys#peek-c-string ,body '0)
			      body)
			  t) ) ) ]
		 [else x] ) )
	  ((and (not-pair? x) (constant? x)) `(quote ,x))
	  ((not-pair? x) (quit "syntax error - illegal atomic form `~s'" x))
	  ((symbol? (car x))
	   (let* ([head (car x)]
		  [rest (cdr x)]
		  [ln (get-line x)]
		  [name (resolve head ae)] )
	     (unless (proper-list? x)
	       (if ln
		   (quit "syntax error in line ~s - malformed expression `~s'" ln x)
		   (quit "syntax error - malformed expression `~s'" x) ) )
	     (set! ##sys#syntax-error-culprit x)
	     (let* ([x2 (cons name rest)]
		    [xexpanded (##sys#secondary-macroexpand (##sys#macroexpand-1-hook x2 me))] )
	       (cond [(not (eq? x2 xexpanded))
		      (when ln (update-line-number-database! xexpanded ln))
		      (walk xexpanded ae me dest) ]
		     [(and inline-table-used (##sys#hash-table-ref inline-table name))
		      => (lambda (val) (walk (cons val (cdr x)) ae me dest)) ]
		     [else
		      (case name

			((if)
			 (##sys#check-syntax 'if x '(if _ _ . #(_)))
			 `(if ,(walk (cadr x) ae me #f)
			      ,(walk (caddr x) ae me #f)
			      ,(if (null? (cdddr x)) 
				   '(##core#undefined)
				   (walk (cadddr x) ae me #f) ) ) )

			((quote)
			 (##sys#check-syntax 'quote x '(quote _))
			 (if ##sys#unqualified-quoted-symbols
			     `(quote ,(##sys#unqualify-quoted-symbols (cadr x)))
			     x) )

			((##core#qualified)
			 (fluid-let ([##sys#unqualified-quoted-symbols #f])
			   (walk (cadr x) ae me dest) ) )

			((##core#immutable)
			 (##sys#check-syntax '##core#immutable x '(##core#immutable (quote _)))
			 (let ((c (cadadr x)))
			   (cond (expand-only `(##core#immutable ',c))
				 ((assoc c immutable-constants) => cdr)
				 (else
				  (let ([var (make-random-name 'c)])
				    (set! immutable-constants (alist-cons c var immutable-constants))
				    (set! always-bound (cons var always-bound))
				    var) ) ) ) )

			((##core#undefined ##core#callunit ##core#primitive) x)

			((##core#enable-unqualified-quoted-symbols)
			 (set! ##sys#unqualified-quoted-symbols #t)
			 (walk '(##core#undefined) ae me dest) )

			((##core#disable-unqualified-quoted-symbols)
			 (set! ##sys#unqualified-quoted-symbols #f)
			 (walk '(##core#undefined) ae me dest) )

			((let)
			 (##sys#check-syntax 'let x '(let #((variable _) 0) . #(_ 1)))
			 (let* ([bindings (cadr x)]
				[vars (unzip1 bindings)]
				[aliases (map gensym vars)] )
			   `(let ,(map (lambda (alias b)
					 (list alias (walk (cadr b) ae me (car b))) )
				       aliases bindings)
			      ,(walk (##sys#canonicalize-body (cddr x))
				     (append (map cons vars aliases) ae)
				     me dest) ) ) )

			((lambda ##core#internal-lambda)
			 (##sys#check-syntax 'lambda x '(_ lambda-list . #(_ 1)))
			 (let ([llist (cadr x)]
			       [obody (cddr x)] )
			   (decompose-lambda-list
			    llist
			    (lambda (vars argc rest)
			      (let* ([aliases (map gensym vars)]
				     [body 
				      (walk 
				       (##sys#canonicalize-body (cddr x))
				       (append (map cons vars aliases) ae)
				       me #f) ]
				     [llist2 
				      (build-lambda-list
				       aliases argc
				       (and rest (list-ref aliases (posq rest vars))) ) ] )
				(cond [(and emit-debug-info (eq? 'lambda name))
				       (expand-debug-lambda dest llist2 llist argc rest body obody) ]
				      [(and dest emit-profile (eq? 'lambda name))
				       (expand-profile-lambda dest llist2 body) ]
				      [else `(lambda ,llist2 ,body)] ) ) ) ) ) )

			((##core#named-lambda)
			 (##sys#check-syntax '##core#named-lambda x '(##core#named-lambda symbol lambda-list . #(_ 1)))
			 (walk `(lambda ,@(cddr x)) ae me (cadr x)) )

			((##core#loop-lambda)
			 (when (and dest emit-debug-info debug-calls (not debug-loops))
			   (set! loop-lambda-names (cons dest loop-lambda-names)) )
			 (let* ([vars (cadr x)]
				[obody (cddr x)]
				[aliases (map gensym vars)]
				[body 
				 (walk 
				  (##sys#canonicalize-body obody)
				  (append (map cons vars aliases) ae)
				  me #f) ] )
			   (if (and emit-debug-info debug-loops)
			       (expand-debug-lambda dest aliases vars (length vars) #f body obody)
			       `(lambda ,aliases ,body) ) ) )

			((set!) 
			 (##sys#check-syntax 'set! x '(set! variable _))
			 (let* ([var0 (cadr x)]
				[var (resolve var0 ae)]
				[ln (get-line x)]
				[val (walk (caddr x) ae me var0)] )
			   (when (eq? var var0)
			     (when (macro? var)
			       (warning "assigned global variable `~S' is a macro ~A"
					var
					(if ln (sprintf "in line ~S" ln) "") ) )
			     (when emit-debug-info
			       (set! val (expand-debug-assignment var val x)) ) )
			   (cond [(assq var foreign-variables)
				  => (lambda (fv)
				       (let ([type (second fv)]
					     [tmp (gensym)] )
					 `(let ([,tmp ,(foreign-type-convert-argument val type)])
					    (##core#inline_update 
					     (,(third fv) ,type)
					     ,(foreign-type-check tmp type) ) ) ) ) ]
				 [else `(set! ,var ,val)] ) ) )

			((##core#inline)
			 (##sys#check-syntax '##core#inline x '(##core#inline _ . #(_ 0)))
			 `(##core#inline ,(cadr x) ,@(mapwalk (cddr x) ae me)))

			((##core#inline_allocate)
			 (##sys#check-syntax '##core#inline_allocate x '(##core#inline_allocate (_ number) . #(_ 0)))
			 `(##core#inline_allocate ,(cadr x) ,@(mapwalk (cddr x) ae me)))

			((##core#inline_ref)
			 (##sys#check-syntax '##core#inline_ref x '(##core#inline_ref (string _)))
			 x)
	      
			((##core#inline_update)
			 (##sys#check-syntax '##core#inline_update x '(##core#inline_update (string _) _))
			 `(##core#inline_update ,(cadr x) ,(walk (caddr x) ae me #f)) )

			((##core#compiletimetoo ##core#elaborationtimetoo)
			 (let ((exp (cadr x)))
			   (eval exp)
			   (walk exp ae me dest) ) )

			((##core#compiletimeonly ##core#elaborationtimeonly)
			 (eval (cadr x))
			 '(##core#undefined) )

			((begin) 
			 (##sys#check-syntax 'begin x '(begin . #(_ 0)))
			 (if (pair? (cdr x))
			     (canonicalize-begin-body
			      (let fold ([xs (cdr x)])
				(let ([x (car xs)]
				      [r (cdr xs)] )
				  (if (null? r)
				      (list (walk x ae me dest))
				      (cons (walk x ae me #f) (fold r)) ) ) ) )
			     '(##core#undefined) ) )

                        ((##core#foreign-lambda)
			 (##sys#check-syntax '##core#foreign-lambda x '(##core#foreign-lambda _ _ . #(_ 0)))
                         (walk (expand-foreign-lambda x) ae me dest) )

                        ((##core#foreign-callback-lambda)
			 (##sys#check-syntax '##core#foreign-callback-lambda x '(##core#foreign-callback-lambda _ _ . #(_ 0)))
                         (walk (expand-foreign-callback-lambda x) ae me dest) )

                        ((##core#foreign-lambda*)
			 (##sys#check-syntax '##core#foreign-lambda* x '(##core#foreign-lambda* _ _ . #(_ 1)))
                         (walk (expand-foreign-lambda* x) ae me dest) )

                        ((##core#foreign-callback-lambda*)
			 (##sys#check-syntax '##core#foreign-callback-lambda* x '(##core#foreign-callback-lambda* _ _ . #(_ 1)))
                         (walk (expand-foreign-callback-lambda* x) ae me dest) )

                        ((##core#define-foreign-variable)
			 (##sys#check-syntax '##core#define-foreign-variable x '(##core#define-foreign-variable _ _ . #(_)))
                         (let* ([var (cadr (second x))]
                                [type (cadr (third x))]
                                [name (if (pair? (cdddr x))
                                          (cadr (fourth x))
                                          (symbol->string var) ) ] )
                           (set! foreign-variables
			     (cons (list var type (if (string? name) name (symbol->string name)))
				   foreign-variables))
                           '(##core#undefined) ) )

                        ((##core#define-foreign-parameter)
			 (##sys#check-syntax '##core#define-foreign-parameter x '(##core#define-foreign-parameter _ _ . #(_)))
                         (let* ([var (cadr (second x))]
                                [type (cadr (third x))]
                                [name (if (pair? (cdddr x))
                                          (cadr (fourth x))
                                          (symbol->string var) ) ] 
				[vname (gensym)] )
			   (walk 
			    `(begin
			       (##core#define-foreign-variable ',vname ',type ',name)
			       (set! ,var 
				 (lambda arg
				   (if (pair? arg)
				       (set! ,vname (##sys#slot arg 0))
				       ,vname) ) ) ) 
			    ae me dest) ) )

                        ((##core#define-foreign-type)
			 (##sys#check-syntax '##core#define-foreign-type x '(##core#define-foreign-type _ _ . #(_ 0 2)))
                         (let ([name (cadr (second x))]
                               [type (cadr (third x))] 
			       [conv (cdddr x)] )
			   (cond [(pair? conv)
				  (let ([arg (make-random-name)]
					[ret (make-random-name)] )
				    (##sys#hash-table-set! foreign-type-table name (vector type arg ret))
				    (set! always-bound (cons* arg ret always-bound))
				    (walk
				     `(begin
					(set! ,arg ,(first conv))
					(set! ,ret ,(if (pair? (cdr conv)) (second conv) '##sys#values)) ) 
				     ae me dest) ) ]
				 [else
				  (##sys#hash-table-set! foreign-type-table name type)
				  '(##core#undefined) ] ) ) )

			((##core#define-external-variable)
			 (##sys#check-syntax '##core#define-external-variable x '(##core#define-external-variable _ _))
			 (let* ([sym (cadr (second x))]
				[name (symbol->string sym)]
				[type (cadr (third x))] 
				[rname (make-random-name)] )
			   (set! external-variables (alist-cons name type external-variables))
			   (set! foreign-variables
			     (cons (list rname 'c-pointer (string-append "&" name))
				   foreign-variables) )
			   (set! external-to-pointer (alist-cons sym rname external-to-pointer))
			   '(##core#undefined) ) )

                        ((##core#define-inline)
			 (##sys#check-syntax '##core#define-inline x '(##core#define-inline _ _))
                         (let* ([name (cadr (second x))]
                                [val (third x)] )
                           (receive (val2 mlist) (extract-mutable-constants (walk val ae me name))
                             (##sys#hash-table-set! inline-table name val2)
                             (set! always-bound (append (unzip1 mlist) always-bound))
                             (set! inline-table-used #t)
                             (walk
                              `(begin ,@(map (lambda (m) `(set! ,(car m) ',(cdr m))) mlist))
                              ae me #f) ) ) )

                        ((##core#define-constant)
			 (##sys#check-syntax '##core#define-constant x '(##core#define-constant _ _))
                         (let* ([name (cadr (second x))]
                                [val (third x)] )
                           (set! constants-used #t)
                           (cond [(collapsable-literal? val)
                                  (##sys#hash-table-set! constant-table name (list val))
                                  '(##core#undefined) ]
                                 [else
                                  (let ([var (make-random-name)])
                                    (##sys#hash-table-set! constant-table name (list var))
                                    (set! mutable-constants (alist-cons var val mutable-constants))
                                    (set! always-bound (cons var always-bound))
                                    (walk `(set! ,var ,val) ae me #f) ) ] ) ) )

			((##core#include)
			 (##sys#check-syntax '##core#include x '(_ (quote string)))
			 (let* ([ln (get-line x)]
				[file (##sys#resolve-include-filename (cadadr x))]
				[in (check-and-open-input-file file ln)] 
				[old current-source-file] )
			   (when verbose-mode (printf "including ~s~%~!" file))
			   (set! current-source-file file)
			   (set! source-file-changed #t)
			   (set! dependency-list (cons file dependency-list))
			   (walk 
			    `(begin 
			       ,@(let ([orlc ##sys#read-line-counter])
				   (set! ##sys#read-line-counter 1)
				   (do ((forms '() (cons (##sys#compiler-toplevel-macroexpand-hook x) forms))
					(x (read in) (read in)) )
				       ((eof-object? x) 
					(set! ##sys#read-line-counter orlc)
					(set! current-source-file old)
					(set! source-file-changed #t)
					(reverse forms) )
				     (when ln (update-line-number-database! x ln)) ) ) )
			    ae me #f) ) )

			((##core#declare)
			 (walk `(begin ,@(map process-declaration (map second (cdr x)))) '() me #f) )
	     
			((##core#define-module)
			 (walk
			  (##sys#expand-module-definition (cadadr x) (cadr (caddr x)) quit)
			  ae me dest) )

			((##core#foreign-callback-wrapper)
			 (let-values ([(args lam) (split-at (cdr x) 4)])
			   (let* ([lam (car lam)]
				  [rtype (cadr (third args))]
				  [atypes (cadr (fourth args))]
				  [vars (second lam)] )
			     `(##core#foreign-callback-wrapper
			       ,@(mapwalk args ae me)
			       ,(walk `(##core#internal-lambda 
					,vars
					(let ,(let loop ([vars vars] [types atypes])
						(if (null? vars)
						    '()
						    (let ([var (car vars)])
						      (cons (list var (foreign-type-convert-result var (car types)))
							    (loop (cdr vars) (cdr types)) ) ) ) )
					  ,(foreign-type-convert-argument `(let () ,@(cddr lam)) rtype) ) )
				      ae me #f) ) ) ) )

			(else
			 (let ([handle-call
				(lambda ()
				  (let* ([x2 (mapwalk x ae me)]
					 [head2 (car x2)]
					 [old (##sys#hash-table-ref line-number-database-2 head2)] )
				    (when ln
				      (##sys#hash-table-set!
				       line-number-database-2
				       head2
				       (cons name (alist-cons x2 ln (if old (cdr old) '()))) ) )
				    (if (and emit-debug-info
					     debug-calls
					     (not (memq head loop-lambda-names)) )
					(expand-debug-call head x2)
					x2) ) ) ] )

			   (cond [##sys#strict-mode (handle-call)]

				 [(eq? name 'let-macro)
				  (##sys#check-syntax 'let-macro x '(let-macro #(list 0) . #(_ 1)))
				  (let* ([me2 (##sys#expand-local-macrodefs (cadr x))]
					 [names (map car me2)]
					 [aliases (map gensym names)] )
				    (walk
				     (##sys#canonicalize-body (cddr x))
				     (append (map cons names aliases) ae)
				     (append (map (lambda (alias mdef) (cons alias (cdr mdef))) aliases me2) me)
				     dest) ) ]

				 [(eq? name 'let-id-macro)
				  (##sys#check-syntax 'let-id-macro x '(let-id-macro #((symbol _) 0) . #(_ 1)))
				  (let* ([mdefs (cadr x)]
					 [names (map car mdefs)]
					 [aliases (map gensym names)] )
				    (walk
				     (##sys#canonicalize-body (cddr x))
				     (append (map cons names aliases) ae)
				     (append (map (lambda (alias mdef)
						    (cons alias (lambda (form) (cadr mdef))) )
						  aliases mdefs) me)
				     dest) ) ]
				 
				 [(eq? 'external-pointer name)
				  (##sys#check-syntax 'external-pointer x '(external-pointer symbol))
				  (let* ([sym (cadr x)]
					 [a (assq sym external-to-pointer)] )
				    (if a
					(walk (cdr a) ae me #f)
					(let ([ln (get-line x)])
					  (if ln
					      (quit "syntax error in line ~S - undefined external variable ~S" ln sym)
					      (quit "syntax error - undefined external variable ~S" sym) ) ) ) ) ]
				 
				 [else (handle-call)] ) ) ) ) ] ) ) ) )

	  ((not (proper-list? x))
	   (quit "syntax error - malformed expression `~s'" x) )

	  ((and (pair? (car x)) (eq? 'lambda (caar x)))
	   (let ((var (gensym 't)))
	     (walk
	      `(let ((,var ,(car x)))
		 (,var ,@(cdr x)) )
	      ae me dest) ) )
	  
	  ((and emit-debug-info debug-calls)
	   (expand-debug-call anonymous-object-identifier (mapwalk x ae me)) )

	  (else (mapwalk x ae me)) ) )
  
  (define (mapwalk xs ae me)
    (map (lambda (x) (walk x ae me #f)) xs) )

  (define (check-decl spec minlen . maxlen)
    (let ([n (length (cdr spec))])
      (if (or (< n minlen) (> n (:optional maxlen 99999)))
	  (quit "syntax error in declaration: `~s'" spec) ) ) )

  (define (process-declaration spec)
    (case (car spec)
      ((uses)
       (let ((us (cdr spec)))
	 (apply register-feature! us)
	 (if (pair? us)
	     (let ((units (map (lambda (u) (string->c-identifier (stringify u))) us)))
	       (set! used-units (append used-units units)) ) ) ) )
      ((unit)
       (check-decl spec 1 1)
       (let ([u (cadr spec)])
	 (when unit-name
	   (warning "unit `~A' was already given a name (new name is ignored)" unit-name) )
	 (set! unit-name (string->c-identifier (stringify u))) ) )
      ((standard-bindings)
       (if (null? (cdr spec))
	   (set! standard-bindings default-standard-bindings)
	   (set! standard-bindings (append (cdr spec) standard-bindings)) ) )
      ((extended-bindings)
       (if (null? (cdr spec))
	   (set! extended-bindings default-extended-bindings)
	   (set! extended-bindings (append (cdr spec) extended-bindings)) ) )
      ((usual-integrations)      
       (cond [(null? (cdr spec))
	      (set! standard-bindings default-standard-bindings)
	      (set! extended-bindings default-extended-bindings) ]
	     [else
	      (let ([syms (cdr spec)])
		(set! standard-bindings (lset-intersection eq? syms default-standard-bindings))
		(set! extended-bindings (lset-intersection eq? syms default-extended-bindings)) ) ] ) )
      ((number-type)
       (check-decl spec 1 1)
       (set! number-type (cadr spec)))
      ((fixnum fixnum-arithmetic) (set! number-type 'fixnum))
      ((flonum) (set! number-type 'flonum))
      ((generic) (set! number-type 'generic))
      ((unsafe)
       (set! unsafe #t)
       (##match#set-error-control #:unspecified) )
      ((safe) (set! unsafe #f))
      ((no-bound-checks) (set! no-bound-checks #t))
      ((no-argc-checks) (set! no-argc-checks #t))
      ((no-procedure-checks) (set! no-procedure-checks #t))
      ((interrupts-enabled) (set! insert-timer-checks #t))
      ((disable-interrupts interrupts-disabled) (set! insert-timer-checks #f))
      ((always-bound) (set! always-bound (append (cdr spec) always-bound)))
      ((bound-to-procedure)
       (set! always-bound-to-procedure (append (cdr spec) always-bound-to-procedure))
       (set! always-bound (append (cdr spec) always-bound)) )
      ((foreign-declare)
       (let ([fds (cdr spec)])
	 (if (every string? fds)
	     (set! foreign-declarations (append foreign-declarations fds))
	     (quit "syntax error in declaration: `~s'" spec) ) ) )
      ((block) (set! block-compilation #t))
      ((separate) (set! block-compilation #f))
      ((notinline) (set! inlining #f))
      ((inline) (set! inlining #t))
      ((inline-limit)
       (check-decl spec 1 1)
       (let ([n (cadr spec)])
	 (if (number? n)
	     (set! inline-limit n)
	     (quit "invalid argument to `inline-limit' declaration") ) ) )
      ((not)
       (check-decl spec 1)
       (case (second spec)
	 [(standard-bindings)
	  (if (null? (cddr spec))
	      (set! standard-bindings '())
	      (set! standard-bindings (lset-difference eq? default-standard-bindings (cddr spec))) ) ]
	 [(extended-bindings)
	  (if (null? (cddr spec))
	      (set! extended-bindings '())
	      (set! extended-bindings (lset-difference eq? default-extended-bindings (cddr spec))) ) ]
	 [(usual-integrations)      
	  (cond [(null? (cddr spec))
		 (set! standard-bindings '())
		 (set! extended-bindings '()) ]
		[else
		 (let ([syms (cddr spec)])
		   (set! standard-bindings (lset-difference eq? default-standard-bindings syms))
		   (set! extended-bindings (lset-difference eq? default-extended-bindings syms)) ) ] ) ]
	 [else
	  (check-decl spec 1 1)
	  (case (cadr spec)
	    [(interrupts-enabled) (set! insert-timer-checks #f)]
	    [(safe) 
	     (set! unsafe #t)
	     (##match#set-error-control #:unspecified) ]
	    [(inline) (set! inlining #f)] 
	    [else (warning "illegal declaration specifier `~s'" spec)] ) ] ) )
      ((no-fancy-ports) (set! file-io-only #t))
      ((no-winding-callcc) (set! nonwinding-call/cc #t))
      ((block-global) (set! block-globals (append (cdr spec) block-globals)))
      (else (warning "illegal declaration specifier `~s'" spec)) )
    '(##core#undefined) )

  (walk (##sys#compiler-toplevel-macroexpand-hook exp) '() '() #f) )


;;; Expand "foreign-lambda"/"foreign-callback-lambda" forms and add item to stub-list:

(define-record foreign-stub
  id					; symbol
  return-type				; type-specifier
  name					; string or #f
  argument-types			; (type-specifier...)
  argument-names			; #f or (symbol ...)
  body					; #f or string
  callback)				; boolean

(define (create-foreign-stub rtype sname argtypes argnames body callback)
  (let* ([params (list-tabulate (length argtypes) (lambda (x) (gensym 'a)))]
	 [f-id (gensym 'stub)]
	 [bufvar (gensym)] 
	 [rsize (estimate-foreign-result-size rtype)] )
    (set! foreign-lambda-stubs 
      (cons (make-foreign-stub f-id rtype sname argtypes argnames body callback)
	    foreign-lambda-stubs) )
    (let ([rsize (if callback (+ rsize 24) rsize)] ; 24 -> has to hold cons on 64-bit platform!
	  [head (if callback
		    `((##core#primitive ,f-id))
		    `(##core#inline ,f-id) ) ]
	  [rest (map (lambda (p t) (foreign-type-check (foreign-type-convert-argument p t) t)) params argtypes)] )
      `(lambda ,params
	 ;; Do minor GC (if callback) to make room on stack:
	 ,@(if callback '((##sys#gc #f)) '())
	 ,(cond [(zero? rsize) 
		 (foreign-type-convert-result (append head (cons '(##core#undefined) rest)) rtype) ]
		[(eq? 'c-string (final-foreign-type rtype))
		 `(let ([,bufvar (##sys#allocate-vector ,rsize #t #f #t)])
		    ,(foreign-type-convert-result
		      `(##sys#peek-c-string ,(append head (cons bufvar rest)) 0)
		      rtype) ) ]
		[else
		 ;; Allocate buffer to leave space for results:
		 `(let ([,bufvar (##sys#allocate-vector ,rsize #t #f #t)])
		    ,(foreign-type-convert-result
		      (append head (cons bufvar rest))
		      rtype) ) ] ) ) ) ) )

(define (expand-foreign-lambda exp)
  (let* ([name (cadr (third exp))]
	 [sname (cond ((symbol? name) (symbol->string name))
		      ((string? name) name)
		      (else (quit "name `~s' of foreign procedure has wrong type" name)) ) ]
	 [rtype (cadr (second exp))]
	 [argtypes (map second (cdddr exp))] )
    (create-foreign-stub rtype sname argtypes #f #f #f) ) )

(define (expand-foreign-callback-lambda exp)
  (let* ([name (cadr (third exp))]
	 [sname (cond ((symbol? name) (symbol->string name))
		      ((string? name) name)
		      (else (quit "name `~s' of foreign procedure has wrong type" name)) ) ]
	 [rtype (cadr (second exp))]
	 [argtypes (map second (cdddr exp))] )
    (create-foreign-stub rtype sname argtypes #f #f #t) ) )

(define (expand-foreign-lambda* exp)
  (let* ([rtype (cadr (second exp))]
	 [args (cadr (third exp))]
	 [body (apply string-append (map cadr (cdddr exp)))]
 	 [argtypes (map car args)]
	 [argnames (map cadr args)] )
    (create-foreign-stub rtype #f argtypes argnames body #f) ) )

(define (expand-foreign-callback-lambda* exp)
  (let* ([rtype (cadr (second exp))]
	 [args (cadr (third exp))]
	 [body (apply string-append (map cadr (cdddr exp)))]
 	 [argtypes (map car args)]
	 [argnames (map cadr args)] )
    (create-foreign-stub rtype #f argtypes argnames body #t) ) )


;;; Traverse expression and update line-number db with all contained calls:

(define (update-line-number-database! exp ln)

  (define (mapupdate xs)
    (let loop ((xs xs))
      (if (pair? xs)
	  (begin
	    (walk (car xs))
	    (loop (cdr xs)) ) ) ) )

  (define (walk x)
    (cond ((not-pair? x))
	  ((symbol? (car x))
	   (let* ((name (car x))
		  (old (or (##sys#hash-table-ref ##sys#line-number-database name) '())) )
	     (if (not (assq x old))
		 (##sys#hash-table-set! ##sys#line-number-database name (alist-cons x ln old)) )
	     (mapupdate (cdr x)) ) )
	  (else (mapupdate x)) ) )

  (walk exp) )


;;; Scan toplevel expressions for assignments:

(define (scan-toplevel-assignments node)
  (let ([safe '()]
	[unsafe '()] )

    (define (mark v)
      (if (not (memq v unsafe)) (set! safe (cons v safe))) )

    (debugging 'p "scanning toplevel assignments...")
    (call-with-current-continuation
     (lambda (return)

       (define (scan-each ns e)
	 (for-each (lambda (n) (scan n e)) ns) )

       (define (scan n e)
	 (let ([params (node-parameters n)]
	       [subs (node-subexpressions n)] )
	   (case (node-class n)

	     [(##core#variable)
	      (let ([var (first params)])
		(if (and (not (memq var e)) (not (memq var safe)))
		    (set! unsafe (cons var unsafe)) ) ) ]

	     [(if ##core#cond ##core#switch)
	      (scan (first subs) e)
	      (return #f) ]

	     [(let)
	      (scan (first subs) e)
	      (scan (second subs) (append params e)) ]

	     [(lambda ##core#callunit) #f]

	     [(##core#call) (return #f)]

	     [(set!)
	      (let ([var (first params)])
		(if (not (memq var e)) (mark var))
		(scan (first subs) e) ) ]

	     [else (scan-each subs e)] ) ) )

       (scan node '()) ) )
    (debugging 'o "safe globals" safe)
    (set! always-bound (append safe always-bound)) ) )


;;; Convert canonicalized node-graph into continuation-passing-style:

(define (perform-cps-conversion node)

  (define (cps-lambda id llist subs k)
    (let ([t1 (gensym 'k)])
      (k (make-node
	  '##core#lambda (list id #t (cons t1 llist) 0)
	  (list (walk (car subs)
		      (lambda (r) 
			(make-node '##core#call '(#t) (list (varnode t1) r)) ) ) ) ) ) ) )
  
  (define (walk n k)
    (let ((subs (node-subexpressions n))
	  (params (node-parameters n)) 
	  (class (node-class n)) )
      (case (node-class n)
	((##core#variable quote ##core#undefined ##core#primitive) (k n))
	((if) (let* ((t1 (gensym 'k))
		     (t2 (gensym 'r))
		     (k1 (lambda (r) (make-node '##core#call '(#t) (list (varnode t1) r)))) )
		(make-node 'let
			   (list t1)
			   (list (make-node '##core#lambda (list (gensym 'f) #f (list t2) 0) 
					    (list (k (varnode t2))) )
				 (walk (car subs)
				       (lambda (v)
					 (make-node 'if '()
						    (list v
							  (walk (cadr subs) k1)
							  (walk (caddr subs) k1) ) ) ) ) ) ) ) )
	((let) (let loop ((vars params) (vals subs))
		 (if (null? vars)
		     (walk (car vals) k)
		     (walk (car vals)
			   (lambda (r) 
			     (make-node 'let
					(list (car vars))
					(list r (loop (cdr vars) (cdr vals))) ) ) ) ) ) )
	((lambda) (cps-lambda (gensym 'f) (first params) subs k))
	((set!) (let ((t1 (gensym 't)))
		  (walk (car subs)
			(lambda (r)
			  (make-node 'let (list t1)
				     (list (make-node 'set! (list (first params)) (list r))
					   (k (varnode t1)) ) ) ) ) ) )
	((##core#foreign-callback-wrapper)
	 (let ([id (gensym 'f)]
	       [lam (first subs)] )
	   (set! foreign-callback-stubs
	     (cons (apply make-foreign-callback-stub id params) foreign-callback-stubs) )
	   (cps-lambda id (first (node-parameters lam)) (node-subexpressions lam) k) ) )
	((##core#inline ##core#inline_allocate ##core#inline_ref ##core#inline_update)
	 (walk-inline-call class params subs k) )
	((##core#call) (walk-call (car subs) (cdr subs) params k))
	((##core#callunit) (walk-call-unit (first params) k))
	(else (bomb "bad node (cps)")) ) ) )
  
  (define (walk-call fn args params k)
    (let ((t0 (gensym 'k))
          (t3 (gensym 'r)) )
      (make-node
       'let (list t0)
       (list (make-node '##core#lambda (list (gensym 'f) #f (list t3) 0) 
			(list (k (varnode t3))) )
	     (walk-arguments
	      args
	      (lambda (vars)
		(walk fn
		      (lambda (r) 
			(make-node '##core#call params (cons* r (varnode t0) vars) ) ) ) ) ) ) ) ) )
  
  (define (walk-call-unit unitname k)
    (let ((t0 (gensym 'k))
	  (t3 (gensym 'r)) )
      (make-node
       'let (list t0)
       (list (make-node '##core#lambda (list (gensym 'f) #f (list t3) 0) 
			(list (k (varnode t3))) )
	     (make-node '##core#callunit (list unitname)
			(list (varnode t0)) ) ) ) ) )

  (define (walk-inline-call class op args k)
    (walk-arguments
     args
     (lambda (vars)
       (k (make-node class op vars)) ) ) )
  
  (define (walk-arguments args wk)
    (let loop ((args args) (vars '()))
      (cond ((null? args) (wk (reverse vars)))
            ((atomic? (car args))
             (loop (cdr args) (cons (car args) vars)) )
            (else
             (let ((t1 (gensym 'a)))
               (walk (car args)
                     (lambda (r)
		       (make-node 'let (list t1)
				  (list r
					(loop (cdr args) 
					      (cons (varnode t1) vars) ) ) ) ) ) ) ) ) ) )
  
  (define (atomic? n)
    (let ((class (node-class n)))
      (or (memq class '(quote ##core#variable ##core#undefined))
	  (and (memq class '(##core#inline ##core#inline_allocate ##core#inline_ref ##core#inline_update))
	       (every atomic? (node-subexpressions n)) ) ) ) )
  
  (walk node values) )


;;; Foreign callback stub type:

(define-record foreign-callback-stub
  id					; symbol
  name					; string
  qualifiers				; string
  return-type				; type-specifier
  argument-types)			; (type-specifier ...)


;;; Perform source-code analysis:

(define (analyze-expression node)
  (let ([db (make-vector analysis-database-size '())]
	[explicitly-consed '()] )

    (define (grow n)
      (set! current-program-size (+ current-program-size n)) )

    (define (walk n env localenv here call)
      (let ((subs (node-subexpressions n))
	    (params (node-parameters n)) 
	    (class (node-class n)) )
	(grow 1)
	(case class
	  ((quote ##core#undefined ##core#primitive ##core#proc) #f)

	  ((##core#variable)
	   (let ((var (first params)))
	     (ref var n)
	     (unless (memq var localenv)
	       (grow 1)
	       (cond ((memq var env) (put! db var 'captured #t))
		     ((not (get db var 'global)) (put! db var 'global #t) ) ) ) ) )
	  
	  ((##core#callunit ##core#recurse)
	   (grow 1)
	   (walkeach subs env localenv here #f) )

	  ((##core#call)
	   (grow 1)
	   (let ([fun (car subs)])
	     (if (eq? '##core#variable (node-class fun))
		 (let ([name (first (node-parameters fun))])
		   (collect! db name 'call-sites (cons here n))
		   ;; If call to standard-binding & optimizable rest-arg operator: decrease access count:
		   (if (and (get db name 'standard-binding)
			    (memq name optimizable-rest-argument-operators) )
		       (for-each
			(lambda (arg)
			  (and-let* ([(eq? '##core#variable (node-class arg))]
				     [var (first (node-parameters arg))] )
			    (when (get db var 'rest-parameter) (count! db var 'o-r/access-count)) ) )
			(cdr subs) ) ) ) )
	     (walk (first subs) env localenv here #t)
	     (walkeach (cdr subs) env localenv here #f) ) )

	  ((let)
	   (let ([env2 (append params localenv env)]
		 [var (first params)]
		 [val (first subs)] )
	     (put! db var 'home here)
	     (assign var val env2 here)
	     (walk val env localenv here #f)
	     (walk (second subs) env (append params localenv) here #f) ) )

	  ((##core#lambda ##core#direct_lambda)
	   (grow 1)
	   (decompose-lambda-list
	    (third params)
	    (lambda (vars argc rest)
	      (let ([id (first params)]
		    [size0 current-program-size] )
		(when here
		  (collect! db here 'contains id)
		  (put! db id 'contained-in here) )
		(for-each 
		 (lambda (var)
		   (put! db var 'home here)
		   (put! db var 'unknown #t) )
		 vars)
		(when rest
		  (put! db rest 'rest-parameter
			(if (memq rest rest-parameters-promoted-to-vector)
			    'vector
			    'list) ) )
		; (when (simple-lambda-node? n) (put! db id 'simple #t))
		(walk (car subs) (append localenv env) vars id #f)
		(set-car! (cdddr (node-parameters n)) (- current-program-size size0)) ) ) ) )
	  
	  ((set!) 
	   (let ((var (first params)))
	     (when first-analysis 
	       (cond [(get db var 'standard-binding)
		      (warning "redefinition of standard binding `~S'" var) ]
		     [(get db var 'extended-binding)
		      (warning "redefinition of extended binding `~S'" var) ] ) )
	     (when (and (not (memq var localenv)) 
			(not (memq var env))
			(not (get db var 'global)))
	       (grow 1)
	       (when (and block-compilation first-analysis)
		 (set! block-globals (lset-adjoin eq? block-globals var)) )
	       (put! db var 'global #t) )
	     (assign var (car subs) (append localenv env) here)
	     (put! db var 'assigned #t)
	     (walk (car subs) env localenv here #f) ) )

	  (else (walkeach subs env localenv here #f)) ) ) )

    (define (walkeach xs env lenv here call) 
      (for-each (lambda (x) (walk x env lenv here call)) xs) )

    (define (assign var val env here)
      (cond ((eq? '##core#undefined (node-class val))
	     (put! db var 'undefined #t) )
	    ((and (eq? '##core#variable (node-class val))
		  (eq? var (first (node-parameters val))) ) )
	    ((or block-compilation (memq var env) (get db var 'constant))
	     (let ((props (get-all db var 'unknown 'value))
		   (home (get db var 'home)) )
	       (unless (assq 'unknown props)
		 (if (assq 'value props)
		     (put! db var 'unknown #t)
		     (if (or (not home) (eq? here home))
			 (put! db var 'value val)
			 (put! db var 'unknown #t) ) ) ) ) )
	    (else (put! db var 'unknown #t)) ) )
    
    (define (ref var node)
      (collect! db var 'references node) )

    (define (quick-put! plist prop val)
      (set-cdr! plist (alist-cons prop val (cdr plist))) )

    ;; Return true if <id> directly or indirectly contains any of <other-ids>:
    (define (contains? id other-ids)
      (or (memq id other-ids)
	  (let ((clist (get db id 'contains)))
	    (and clist
		 (any (lambda (id2) (contains? id2 other-ids)) clist) ) ) ) )

    ;; Initialize database:
    (initialize-analysis-database db)

    ;; Walk toplevel expression-node:
    (debugging 'p "analysis traversal phase...")
    (set! current-program-size 0)
    (walk node '() '() #f #f) 

    ;; Complete gathered database information:
    (debugging 'p "analysis gathering phase...")
    (##sys#hash-table-for-each
     (lambda (sym plist)
       (let ([unknown #f]
	     [value #f]
	     [references '()]
	     [captured #f]
	     [call-sites '()]
	     [assigned #f]
	     [undefined #f]
	     [global #f]
	     [o-r/access-count 0]
	     [rest-parameter #f] 
	     [nreferences 0]
	     [ncall-sites 0] )

	 (for-each
	  (lambda (prop)
	    (case (car prop)
	      [(unknown) (set! unknown #t)]
	      [(references) 
	       (set! references (cdr prop))
	       (set! nreferences (length references)) ]
	      [(captured) (set! captured #t)]
	      [(call-sites)
	       (set! call-sites (cdr prop))
	       (set! ncall-sites (length call-sites)) ]
	      [(assigned) (set! assigned #t)]
	      [(undefined) (set! undefined #t)]
	      [(global) (set! global #t)]
	      [(value) (set! value (cdr prop))]
	      [(o-r/access-count) (set! o-r/access-count (cdr prop))]
	      [(rest-parameter) (set! rest-parameter #t)] ) )
	  plist)

	 (set! value (and (not unknown) value))

	 ;; If this is the first analysis and the variable is global and has no references and we are
	 ;;  in block mode, then issue warning:
	 (when (and first-analysis block-compilation global)
	   (when (null? references)
	     (warning "global variable `~S' is never used" sym) ) )

 	 ;; Make 'boxed, if 'assigned & 'captured:
	 (when (and assigned captured)
	   (quick-put! plist 'boxed #t) )

	 ;; Make 'contractable, if it has a procedure as known value, has only one use and one call-site and
	 ;;  if the lambda has no free non-global variables or is an internal lambda:
	 (when value
	   (let ((valparams (node-parameters value)))
	     (when (and (eq? '##core#lambda (node-class value))
			(= 1 nreferences)
			(= 1 ncall-sites)
			(or (not (second valparams))
			    (every (lambda (v) (get db v 'global)) (scan-free-variables value)) ) )
	       (quick-put! plist 'contractable #t) ) ) )

	 ;; Make 'collapsable, if it has a known constant value which is either collapsable or is only
	 ;;  referenced once and if no assignments are made:
	 (when (and value
		    ;; (not (assq 'assigned plist)) - If it has a known value, it's assigned just once!
		    (eq? 'quote (node-class value)) )
	   (let ((val (first (node-parameters value))))
	     (when (or (collapsable-literal? val)
		       (= 1 nreferences) )
	       (quick-put! plist 'collapsable #t) ) ) )
		
	 ;; If it has a known value that is a procedure, and if the number of call-sites is equal to the
	 ;;  number of references (does not escape), then make all formal parameters 'unused which are
	 ;;  never referenced or assigned (if no rest parameter exist):
	 ;;  - also marks the procedure as 'has-unused-parameters.
	 ;;  - if the procedure is internal (a continuation) do NOT mark unused parameters.
	 ;;  - also: if procedure has rest-parameter and no unused params, mark f-id as 'explicit-rest.
	 (when value
	   (let ([has #f])
	     (when (and (eq? '##core#lambda (node-class value))
			(= nreferences ncall-sites) )
	       (let ([lparams (node-parameters value)])
		 (when (and (second lparams) try-harder)
		   (decompose-lambda-list
		    (third lparams)
		    (lambda (vars argc rest)
		      (unless rest
			(for-each
			 (lambda (var)
			   (cond [(and (not (get db var 'references))
				       (not (get db var 'assigned)) )
				  (put! db var 'unused #t)
				  (set! has #t)
				  #t]
				 [else #f] ) )
			 vars) )
		      (cond [has (put! db (first lparams) 'has-unused-parameters #t)]
			    [(and try-harder rest)
			     (set! explicitly-consed (cons rest explicitly-consed))
			     (put! db (first lparams) 'explicit-rest #t) ] ) ) ) ) ) ) ) )

	 ;;  Make 'removable, if it has no references and is not assigned to, and if it has either a value that
	 ;;    does not cause any side-effects or if it is 'undefined:
	 (when (and (not assigned)
		    (null? references)
		    (or (and value
			     (not (expression-has-side-effects? value db)) )
			undefined) )
	   (quick-put! plist 'removable #t) )

	 ;; Make 'replacable, if it has a variable as known value and if either that variable has
	 ;;  a known value itself, or if it is not captured and referenced only once, the target and
	 ;;  the source are never assigned and the source is non-global or we are in block-mode:
	 ;;  - The target-variable is not allowed to be global.
	 ;;  - The variable that can be substituted for the current one is marked as 'replacing.
	 ;;    This is done to prohibit beta-contraction of the replacing variable (It wouldn't be there, if
	 ;;    it was contracted).
	 (when (and value (not global))
	   (when (eq? '##core#variable (node-class value))
	     (let* ([name (first (node-parameters value))]
		    [nrefs (get db name 'references)] )
	       (when (or (and (not (get db name 'unknown)) (get db name 'value))
			 (and (not (get db name 'captured))
			      nrefs
			      (= 1 (length nrefs))
			      (not assigned)
			      (not (get db name 'assigned)) 
			      (or block-compilation (not (get db name 'global))) ) )
		 (quick-put! plist 'replacable name) 
		 (put! db name 'replacing #t) ) ) ) )

	 ;; Make 'replacable, if it has a known value of the form: '(lambda (<xvar>) (<kvar> <xvar>))' and
	 ;;  is an internally created procedure: (See above for 'replacing)
	 (when (and value (eq? '##core#lambda (node-class value)))
	   (let ([params (node-parameters value)])
	     (when (not (second params))
	       (let ([llist (third params)]
		     [body (first (node-subexpressions value))] )
		 (when (and (pair? llist) 
			    (null? (cdr llist))
			    (eq? '##core#call (node-class body)) )
		   (let ([subs (node-subexpressions body)])
		     (when (= 2 (length subs))
		       (let ([v1 (first subs)]
			     [v2 (second subs)] )
			 (when (and (eq? '##core#variable (node-class v1))
				    (eq? '##core#variable (node-class v2))
				    (eq? (first llist) (first (node-parameters v2))) )
			   (let ([kvar (first (node-parameters v1))])
			     (quick-put! plist 'replacable kvar)
			     (put! db kvar 'replacing #t) ) ) ) ) ) ) ) ) ) )

	 ;; If a rest-argument, convert 'rest-parameter property to 'vector, if the variable is never
	 ;;  assigned, and the number of references is identical to the number of accesses in optimizable
	 ;;  rest-argument operators:
	 ;; - Add variable to "rest-parameters-promoted-to-vector", because subsequent optimization will
	 ;;   change variables context (operators applied to it).
	 (when (and try-harder
		    rest-parameter
		    (not assigned)
		    (= nreferences o-r/access-count) )
	   (set! rest-parameters-promoted-to-vector (lset-adjoin eq? rest-parameters-promoted-to-vector sym))
	   (put! db sym 'rest-parameter 'vector) ) ) )

     db)

    ;; Remove explicitly consed rest parameters from promoted ones:
    (set! rest-parameters-promoted-to-vector
      (lset-difference eq? rest-parameters-promoted-to-vector explicitly-consed) )

    ;; Set original program-size, if this is the first analysis-pass:
    (unless original-program-size
      (set! original-program-size current-program-size) )
    db) )


;;; Do some optimizations:
;
; - optimize tail recursion by replacing trivial continuations.
; - perform beta-contraction (inline procedures called only once).
; - remove empty 'let' nodes.
; - evaluate constant expressions.
; - substitute variables bound to constants with the value.
; - remove variable-bindings which are never used (and which are not bound to side-effecting expressions).
; - perform simple copy-propagation.
; - remove assignments to unused variables if the assigned value is free of side-effects and the variable is
;   not global.
; - remove unused formal parameters from functions and change all call-sites accordingly.
; - rewrite calls to standard bindings into more efficient forms.
; - rewrite calls to known non-escaping procedures with rest parameter to cons up rest-list at call-site,
;   also: change procedure's lambda-list.

(define simplifications (make-vector 301 '()))

(define (perform-high-level-optimizations node db)
  (let ([removed-lets 0]
	[removed-ifs 0]
	[replaced-vars 0]
	[simplified-classes '()]
	[dirty #f] )

    (define (test sym item) (get db sym item))
    (define (constant-node? n) (eq? 'quote (node-class n)))
    (define (node-value n) (first (node-parameters n)))
    (define (touch) (set! dirty #t))

    (define (simplify n)
      (and (memq 'apply-simplifications compiler-features)
	   (or (and-let* ([entry (##sys#hash-table-ref simplifications (node-class n))])
		 (any (lambda (s)
			(and-let* ([vars (second s)]
				   [env (match-node n (first s) vars)] 
				   [n2 (apply (third s) db
					      (map (lambda (v) (cdr (assq v env))) vars) ) ] )
			  (let* ([name (caar s)]
				 [counter (assq name simplified-classes)] )
			    (if counter
				(set-cdr! counter (add1 (cdr counter)))
				(set! simplified-classes (alist-cons name 1 simplified-classes)) )
			    (touch)
			    (simplify n2) ) ) )
		      entry) )
	       n) ) )

    (define (walk n)
      (if (memq n broken-constant-nodes)
	  n
	  (simplify
	   (let* ((odirty dirty)
		  (n1 (walk1 n))
		  (subs (node-subexpressions n1)) )
	     (case (node-class n1)

	       ((if)			; (This can be done by the simplificator...)
		(cond ((constant-node? (car subs))
		       (set! removed-ifs (+ removed-ifs 1))
		       (touch)
		       (walk (if (node-value (car subs))
				 (cadr subs)
				 (caddr subs) ) ) )
		      (else n1) ) )

	       ((##core#call)
		(if (eq? '##core#variable (node-class (car subs)))
		    (let ((var (first (node-parameters (car subs)))))
		      (if (and try-harder
			       (or (test var 'standard-binding)
				   (test var 'extended-binding) )
			       (test var 'foldable)
			       (every constant-node? (cddr subs)) )
			  (let ((form (cons var (map (lambda (arg) `(quote ,(node-value arg)))
						     (cddr subs) ) ) ) )
			    (call-with-current-continuation
			     (lambda (c)
			       (parameterize ([error-handler
					       (lambda args
						 (unless odirty (set! dirty #f))
						 (set! broken-constant-nodes (lset-adjoin eq? broken-constant-nodes n1))
						 (c n1) ) ] )
				 (debugging 'o "folding constant expression" form)
				 (let ((x (eval form)))
				   (touch)
				   (make-node ; Build call to continuation with new result...
				    '##core#call
				    '(#t)
				    (list (cadr subs) (qnode x)) ) ) ) ) ) )
			  n1) )
		    n1) )

	       (else n1) ) ) ) ) )

    (define (walk1 n)
      (let ((subs (node-subexpressions n))
	    (params (node-parameters n)) 
	    (class (node-class n)) )
	(case class

	  ((##core#variable)
	   (let replace ((var (first params)))
	     (cond ((test var 'replacable) => replace)
		   ((test var 'collapsable)
		    (touch)
		    (debugging 'o "substituted constant variable" var)
		    (qnode (car (node-parameters (test var 'value)))) )
		   (else
		    (if (not (eq? var (first params)))
			(begin
			  (touch)
			  (set! replaced-vars (+ replaced-vars 1)) ) )
		    (varnode var) ) ) ) )

	  ((let)
	   (let ([var (first params)])
	     (cond [(or (test var 'replacable)
			(test var 'removable)
			(and (test var 'contractable) (not (test var 'replacing))) )
		    (touch)
		    (set! removed-lets (add1 removed-lets))
		    (walk (second subs)) ]
		   [else (make-node 'let params (map walk subs))] ) ) )

	  ((##core#lambda)
	   (let ([llist (third params)])
	     (cond [(test (first params) 'has-unused-parameters)
		    (decompose-lambda-list
		     llist
		     (lambda (vars argc rest)
		       (receive (unused used) (partition (lambda (v) (test v 'unused)) vars)
			 (touch)
			 (debugging 'o "removed unused formal parameters" unused)
			 (make-node
			  '##core#lambda
			  (list (first params) (second params)
				(cond [(and rest (test (first params) 'explicit-rest))
				       (debugging 'o "merged explicitly consed rest parameter" rest)
				       (build-lambda-list used (add1 argc) #f) ]
				      [else (build-lambda-list used argc rest)] )
				(fourth params) )
			  (list (walk (first subs))) ) ) ) ) ]
		   [(and try-harder (test (first params) 'explicit-rest))
		    (decompose-lambda-list
		     llist
		     (lambda (vars argc rest)
		       (touch)
		       (debugging 'o "merged explicitly consed rest parameter" rest)
		       (make-node
			'##core#lambda
			(list (first params)
			      (second params)
			      (build-lambda-list vars (add1 argc) #f)
			      (fourth params) )
			(list (walk (first subs))) ) ) ) ]
		   [else (walk-generic n class params subs)] ) ) )

	  ((##core#call)
	   (let* ([fun (car subs)]
		  [funclass (node-class fun)] )
	     (case funclass
	       [(##core#variable)
		;; Call to named procedure:
		(let* ([var (first (node-parameters fun))]
		       [lval (and (not (test var 'unknown)) (test var 'value))]
		       [args (cdr subs)] )
		  (cond [(test var 'contractable)
			 (let* ((lparams (node-parameters lval))
				(llist (third lparams)) )
			   (check-signature args llist)
			   (debugging 'o "contracted procedure" var)
			   (touch)
			   (walk (inline-lambda-bindings llist args (first (node-subexpressions lval)))) ) ]
			[(and lval (eq? '##core#lambda (node-class lval)))
			 (decompose-lambda-list
			  (third (node-parameters lval))
			  (lambda (vars argc rest)
			    (cond [(test (first (node-parameters lval)) 'has-unused-parameters)
				   (if (< (length args) argc) ; Expression was already optimized (should this happen?)
				       (walk-generic n class params subs)
				       (let loop ((vars vars) (argc argc) (args args) (used '()))
					 (cond [(or (null? vars) (zero? argc))
						(touch)
						(make-node
						 '##core#call
						 (cdr params)
						 (map walk (cons fun (append-reverse used args))) ) ]
					       [(test (car vars) 'unused)
						(touch)
						(debugging 'o "removed unused parameter to known procedure" (car vars) var)
						(if (expression-has-side-effects? (car args) db)
						    (make-node
						     'let
						     (list (gensym 't))
						     (list (walk (car args))
							   (loop (cdr vars) (sub1 argc) (cdr args) used) ) )
						    (loop (cdr vars) (sub1 argc) (cdr args) used) ) ]
					       [else (loop (cdr vars)
							   (sub1 argc)
							   (cdr args)
							   (cons (car args) used) ) ] ) ) ) ]
				  [(test (first (node-parameters lval)) 'explicit-rest)
				   (let ([n (length (third (node-parameters lval)))])
				     (debugging 'o "consed rest parameter at call site" var n)
				     (let-values ([(args rargs) (split-at args n)])
				       (make-node
					'##core#call
					params
					(map walk
					     (cons fun
						   (append 
						    args
						    (list
						     (make-node
						      '##core#inline_allocate 
						      (list "C_a_i_list" (* 3 (length rargs)))
						      rargs) ) ) ) ) ) ) ) ]
				  [else (walk-generic n class params subs)] ) ) ) ]
			[else (walk-generic n class params subs)] ) ) ]
	       [(##core#lambda)
		(if (first params)
		    (walk-generic n class params subs)
		    (make-node '##core#call (cons #t (cdr params)) (map walk subs)) ) ]
	       [else (walk-generic n class params subs)] ) ) )

	  ((set!)
	   (let ((var (first params)))
	     (cond ((test var 'contractable)
		    (touch)
		    (make-node '##core#undefined '() '()) )
		   ((and (or (not (test var 'global)) block-compilation)
			 (not (test var 'references)) 
			 (not (expression-has-side-effects? (first subs) db)) )
		    (touch)
		    (debugging 'o "removed side-effect free assignment to unused variable" var)
		    (make-node '##core#undefined '() '()) )
		   (else 
		    (make-node 'set! params (list (walk (car subs)))) ) ) ) )

	  (else (walk-generic n class params subs)) ) ) )
    
    (define (walk-generic n class params subs)
      (let ((subs2 (map walk subs)))
	(if (every eq? subs subs2)
	    n
	    (make-node class params subs2) ) ) )

    (if (and try-harder (perform-pre-optimization! node db))
	(values node #t)
	(begin
	  (debugging 'p "traversal phase...")
	  (let ((node2 (walk node)))
	    (when (pair? simplified-classes) (debugging 'o "simplifications" simplified-classes))
	    (when (> replaced-vars 0) (debugging 'o "replaced variables" replaced-vars))
	    (when (> removed-lets 0) (debugging 'o "removed binding forms" removed-lets))
	    (when (> removed-ifs 0) (debugging 'o "removed conditional forms" removed-ifs))
	    (values node2 dirty) ) ) ) ) )


;;; Pre-optimization phase:
;
; - Transform expressions of the form '(if (not <x>) <y> <z>)' into '(if <x> <z> <y>)'.
; - Transform expressions of the form '(if (<x> <y> ...) <z> <q>)' into '<z>' if <x> names a
;   standard-binding that is never #f and if it's arguments are free of side-effects.

(define (perform-pre-optimization! node db)
  (let ((dirty #f)
	(removed-nots 0) )

    (define (touch) (set! dirty #t) #t)
    (define (test sym prop) (get db sym prop))

    (debugging 'p "pre-optimization phase...")

    ;; Handle '(if (not ...) ...)':
    (if (test 'not 'standard-binding)
	(for-each 
	 (lambda (site)
	   (let* ((n (cdr site))
		  (subs (node-subexpressions n))
		  (kont (first (node-parameters (second subs))))
		  (lnode (and (not (test kont 'unknown)) (test kont 'value)))
		  (krefs (test kont 'references)) )
	     ;; Call-site has one argument and a known continuation (which is a ##core#lambda)
	     ;;  that has only one use:
	     (if (and lnode krefs (= 1 (length krefs)) (= 3 (length subs))
		      (eq? '##core#lambda (node-class lnode)) )
		 (let* ((llist (third (node-parameters lnode)))
			(body (first (node-subexpressions lnode))) 
			(bodysubs (node-subexpressions body)) )
		   ;; Continuation has one parameter?
		   (if (and (proper-list? llist) (null? (cdr llist)))
		       (let* ((var (car llist))
			      (refs (test var 'references)) )
			 ;; Parameter is only used once?
			 (if (and refs (= 1 (length refs)) (eq? 'if (node-class body)))
			     ;; Continuation contains an 'if' node?
			     (let ((iftest (first (node-subexpressions body))))
			       ;; Parameter is used only once and is the test-argument?
			       (if (and (eq? '##core#variable (node-class iftest))
					(eq? var (first (node-parameters iftest))) )
				   ;; Modify call-site to call continuation directly and swap branches
				   ;;  in the conditional:
				   (begin
				     (set! removed-nots (+ removed-nots 1))
				     (node-parameters-set! n '(#t))
				     (node-subexpressions-set! n (cdr subs))
				     (node-subexpressions-set! 
				      body
				      (cons (car bodysubs) (reverse (cdr bodysubs))) )
				     (touch) ) ) ) ) ) ) ) ) ) )
	 (or (test 'not 'call-sites) '()) ) )
    
    ;; Handle '(if (<func> <a> ...) ...)', where <func> never returns false:
    (for-each
     (lambda (varname)
       (if (test varname 'standard-binding)
	   (for-each
	    (lambda (site)
	      (let* ((n (cdr site))
		     (subs (node-subexpressions n))
		     (kont (first (node-parameters (second subs)))) 
		     (krefs (test kont 'references)) 
		     (lnode (and (not (test kont 'unknown)) (test kont 'value))) )
		;; Call-site has side-effect-free arguments and a known continuation that has only one use?
		(if (and lnode
			 (eq? '##core#lambda (node-class lnode))
			 krefs (= 1 (length krefs))
			 (not (any (lambda (sn) (expression-has-side-effects? sn db)) (cddr subs))) )
		    (let* ((llist (third (node-parameters lnode)))
			   (body (first (node-subexpressions lnode))) )
		      ;; Continuation has one parameter and contains an 'if' node?
		      (if (and (proper-list? llist)
			       (null? (cdr llist))
			       (eq? 'if (node-class body)) )
			  (let* ((var (car llist))
				 (refs (test var 'references)) 
				 (iftest (first (node-subexpressions body))) )
			    ;; Parameter is used only once and is the test-argument?
			    (if (and refs (= 1 (length refs))
				     (eq? '##core#variable (node-class iftest))
				     (eq? var (first (node-parameters iftest))) )
				(let ((bodysubs (node-subexpressions body)))
				  ;; Modify call-site to call continuation directly and swap branches
				  ;;  in the conditional:
				  (debugging 'o "removed call in test-context" varname)
				  (node-parameters-set! n '(#t))
				  (node-subexpressions-set! n (list (second subs) (qnode #t)))
				  (touch) ) ) ) ) ) ) ) )
	    (or (test varname 'call-sites) '()) ) ) )
     side-effect-free-standard-bindings-that-never-return-false)

    (if (> removed-nots 0) (debugging 'o "Removed `not' forms" removed-nots))
    dirty) )


;;; Simplifications:

(define (register-simplifications class . ss)
  (##sys#hash-table-set! simplifications class ss) )


(register-simplifications
 '##core#call
 ;; (<named-call> ...) -> (<primitive-call/inline> ...)
 `((##core#call d (##core#variable (a)) b . c)
   (a b c d)
   ,(lambda (db a b c d)
      (let loop ((entries (or (##sys#hash-table-ref substitution-table a) '())))
	(cond ((null? entries) #f)
	      ((simplify-named-call db d a b (caar entries) (cdar entries) c))
	      (else (loop (cdr entries))) ) ) ) ) )


(register-simplifications
 'let

 ;; (let ((<var1> (##core#inline <eq-inline-operator> <var0> <const1>)))
 ;;   (if <var1> <body1>
 ;;       (let ((<var2> (##core#inline <eq-inline-operator> <var0> <const2>)))
 ;;         (if <var2> <body2>
 ;;             <etc.>
 ;; -> (##core#switch (2) <var0> <const1> <body1> <const2> <body2> <etc.>)
 ;; - <var1> and <var2> have to be referenced once only.
 `((let (var1) (##core#inline (op) (##core#variable (var0)) (quote (const1)))
	(if d1 (##core#variable (var1))
	    body1
	    (let (var2) (##core#inline (op) (##core#variable (var0)) (quote (const2)))
		 (if d2 (##core#variable (var2))
		     body2
		     rest) ) ) )
   (var0 var1 var2 op const1 const2 body1 body2 d1 d2 rest)
   ,(lambda (db var0 var1 var2 op const1 const2 body1 body2 d1 d2 rest)
      (and (equal? op eq-inline-operator)
	   (memq 'target-has-switch compiler-features)
	   (immediate? const1)
	   (immediate? const2)
	   (= 1 (length (get db var1 'references)))
	   (= 1 (length (get db var2 'references)))
	   (make-node
	    '##core#switch
	    '(2)
	    (list (varnode var0)
		  (qnode const1)
		  body1
		  (qnode const2)
		  body2
		  rest) ) ) ) )

 ;; (let ((<var> (##core#inline <eq-inline-operator> <var0> <const>)))
 ;;   (if <var>
 ;;       <body>
 ;;       (##core#switch <n> <var0> <const1> <body1> ... <rest>) ) )
 ;; -> (##core#switch <n+1> <var0> <const> <body> <const1> <body1> ... <rest>)
 ;; - <var> has to be referenced once only.
 `((let (var) (##core#inline (op) (##core#variable (var0)) (quote (const)))
	(if d (##core#variable (var))
	    body
	    (##core#switch (n) (##core#variable (var0)) . clauses) ) )
   (var op var0 const d body n clauses)
   ,(lambda (db var op var0 const d body n clauses)
      (and (equal? op eq-inline-operator)
	   (memq 'target-has-switch compiler-features)
	   (immediate? const)
	   (= 1 (length (get db var 'references)))
	   (make-node
	    '##core#switch
	    (list (add1 n))
	    (cons* (varnode var0)
		   (qnode const)
		   body
		   clauses) ) ) ) )
	      
 ;; (let ((<var1> (##core#undefined)))
 ;;   (let ((<var2> (##core#undefined)))
 ;;     ...
 ;;     (let ((<tmp1> (set! <var1> <x1>))
 ;;       (let ((<tmp2> (set! <var2> <x2>)))
 ;;         ...
 ;;         <body>) ... )
 ;; -> <a simpler sequence of let's>
 ;; - <tmpI> may not be used.
 `((let (var1) (##core#undefined ())
	more)
   (var1 more)
   ,(lambda (db var1 more)
      (let loop1 ([vars (list var1)] 
		  [body more] )
	(let ([c (node-class body)]
	      [params (node-parameters body)] 
	      [subs (node-subexpressions body)] )
	  (and try-harder
	       (eq? c 'let)
	       (null? (cdr params))
	       (let* ([val (first subs)]
		      [valparams (node-parameters val)]
		      [valsubs (node-subexpressions val)] )
		 (case (node-class val)
		   [(##core#undefined) (loop1 (cons (first params) vars) (second subs))]
		   [(set!)
		    (let ([allvars (reverse vars)])
		      (and (pair? allvars)
			   (eq? (first valparams) (first allvars))
			   (let loop2 ([vals (list (first valsubs))]
				       [vars (cdr allvars)] 
				       [body (second subs)] )
			     (let ([c (node-class body)]
				   [params (node-parameters body)]
				   [subs (node-subexpressions body)] )
			       (cond [(and (eq? c 'let)
					   (null? (cdr params))
					   (not (get db (first params) 'references))
					   (pair? vars)
					   (eq? 'set! (node-class (first subs)))
					   (eq? (car vars) (first (node-parameters (first subs)))) )
				      (loop2 (cons (first (node-subexpressions (first subs))) vals)
					     (cdr vars)
					     (second subs) ) ]
				     [(null? vars)
				      (receive (n progress) 
					  (reorganize-recursive-bindings allvars (reverse vals) body) 
					(and progress n) ) ]
				     [else #f] ) ) ) ) ) ]
		   [else #f] ) ) ) ) ) ) )

 ;; (let ((<var> (##core#inline <op> ...)))
 ;;   (if <var> <x> <y>) )
 ;; -> (if (##core#inline <op> ...) <x> <y>)
 ;; - <op> may not be the eq-inline operator (so rewriting to "##core#switch" works).
 ;; - <var> has to be referenced only once.
 `((let (var) (##core#inline (op) . args)
	(if d (##core#variable (var))
	    x
	    y) ) 
   (var op args d x y)
   ,(lambda (db var op args d x y)
      (and (not (equal? op eq-inline-operator))
	   (= 1 (length (get db var 'references)))
	   (make-node
	    'if d
	    (list (make-node '##core#inline (list op) args)
		  x y) ) ) ) ) )


(register-simplifications
 'if

 ;; (if <x>
 ;;     (<var> <y>)
 ;;     (<var> <z>) )
 ;; -> (<var> (##core#cond <x> <y> <z>))
 ;; - inline-substitutions have to be enabled (so IF optimizations have already taken place).
 `((if d1 x
       (##core#call d2 (##core#variable (var)) y)
       (##core#call d3 (##core#variable (var)) z) )
   (d1 d2 d3 x y z var)
   ,(lambda (db d1 d2 d3 x y z var)
      (and inline-substitutions-enabled
	   (make-node
	    '##core#call d2
	    (list (varnode var)
		  (make-node '##core#cond '() (list x y z)) ) ) ) ) )

 ;; (if (##core#inline <memXXX> <x> '(<c1> ...)) ...)
 ;; -> (let ((<var> <x>))
 ;;      (if (##core#cond (##core#inline XXX? <var> '<c1>) #t ...) ...)
 ;; - there is a limit on the number of items in the list of constants.
 `((if d1 (##core#inline (op) x (quote (clist)))
       y
       z)
   (d1 op x clist y z)
   ,(lambda (db d1 op x clist y z)
      (and-let* ([opa (assoc op membership-test-operators)]
		 [(proper-list? clist)]
		 [(< (length clist) membership-unfold-limit)] )
	(let ([var (gensym)]
	      [eop (list (cdr opa))] )
	  (make-node
	   'let (list var)
	   (list 
	    x
	    (make-node
	     'if d1
	     (list
	      (fold-right
	       (lambda (c rest)
		 (make-node
		  '##core#cond '()
		  (list 
		   (make-node '##core#inline eop (list (varnode var) (qnode c)))
		   (qnode #t)
		   rest) ) )
	       (qnode #f)
	       clist)
	      y
	      z) ) ) ) ) ) ) ) )


;;; Perform dependency-analysis and transform letrec's into simpler constructs (if possible):

(define (reorganize-recursive-bindings vars vals body)
  (let ([graph '()]
	[valmap (map cons vars vals)] )

    (define (find-path var1 var2)
      (let find ([var var1] [traversed '()])
	(and (not (memq var traversed))
	     (let ([arcs (cdr (assq var graph))])
	       (or (memq var2 arcs)
		   (let ([t2 (cons var traversed)])
		     (any (lambda (v) (find v t2)) arcs) ) ) ) ) ) )

    ;; Build dependency graph:
    (for-each
     (lambda (var val) (set! graph (alist-cons var (scan-used-variables val vars) graph)))
     vars vals)

    ;; Compute recursive groups:
    (let ([groups '()]
	  [done '()] )
      (for-each
       (lambda (var)
	 (when (not (memq var done))
	   (let ([g (filter
		     (lambda (v) (and (not (eq? v var)) (find-path var v) (find-path v var)))
		     vars) ] )
	     (set! groups (alist-cons (gensym) (cons var g) groups))
	     (set! done (append (list var) g done)) ) ) )
       vars)

      ;; Coalesce groups into a new graph:
      (let ([cgraph '()])
	(for-each
	 (lambda (g)
	   (let ([id (car g)]
		 [deps
		  (append-map
		   (lambda (var) (filter (lambda (v) (find-path var v)) vars)) 
		   (cdr g) ) ] )
	     (set! cgraph
	       (alist-cons 
		id
		(filter-map
		 (lambda (g2) (and (not (eq? g2 g)) (lset<= eq? (cdr g2) deps) (car g2))) 
		 groups)
		cgraph) ) ) )
	 groups) 

	;; Topologically sort secondary dependency graph:
	(let ([sgraph (topological-sort cgraph eq?)]
	      [optimized '()] )

	  ;; Construct new bindings:
	  (let ([n2
		 (fold
		  (lambda (gn body)
		    (let* ([svars (cdr (assq gn groups))]
			   [svar (car svars)] )
		      (cond [(and (null? (cdr svars))
				  (not (memq svar (cdr (assq svar graph)))) )
			     (set! optimized (cons svar optimized))
			     (make-node 'let svars (list (cdr (assq svar valmap)) body)) ]
			    [else
			     (fold-right
			      (lambda (var rest)
				(make-node
				 'let (list var)
				 (list (make-node '##core#undefined '() '()) rest) ) )
			      (fold-right
			       (lambda (var rest)
				 (make-node
				  'let (list (gensym))
				  (list (make-node 'set! (list var) (list (cdr (assq var valmap))))
					rest) ) )
			       body
			       svars)
			      svars) ] ) ) )
		  body
		  sgraph) ] )
	    (cond [(pair? optimized)
		   (debugging 'o "eliminated assignments" optimized)
		   (values n2 #t) ]
		  [else (values n2 #f)] ) ) ) ) ) ) )


;;;; Rewrite named calls to more primitive forms:

(define substitution-table (make-vector 301 '()))

(define (rewrite name . class-and-args)
  (let ((old (or (##sys#hash-table-ref substitution-table name) '())))
    (##sys#hash-table-set! substitution-table name (append old (list class-and-args))) ) )

(define (simplify-named-call db params name cont class classargs callargs)

  (define (test sym prop) (get db sym prop))

  (case class

    ;; (eq?/eqv?/equal? <var> <var>) -> (quote #t)
    ;; (eq?/eqv?/equal? ...) -> (##core#inline <iop> ...)
    ((1) ; classargs = (<argc> <iop>)
     (and (test name 'standard-binding)
	  (or (and (= (length callargs) (first classargs))
		   (let ((arg1 (first callargs))
			 (arg2 (second callargs)) )
		     (and (eq? '##core#variable (node-class arg1))
			  (eq? '##core#variable (node-class arg2))
			  (equal? (node-parameters arg1) (node-parameters arg2))
			  (make-node '##core#call '(#t) (list cont (qnode #t))) ) ) )
	      (and inline-substitutions-enabled
		   (make-node
		    '##core#call '(#t) 
		    (list cont (make-node '##core#inline (list (second classargs)) callargs)) ) ) ) ) )

    ;; (<op> ...) -> (##core#inline <iop> ...)
    ;; (<op> <rest-vector>) -> (##core#inline <iopv> <rest-vector>)
    ((2) ; classargs = (<argc> <iop> <safe> <iopv>)
     (and inline-substitutions-enabled
	  (= (length callargs) (first classargs))
	  (or (test name 'extended-binding) (test name 'standard-binding))
	  (or (third classargs) unsafe)
	  (let ([arg1 (first callargs)]
		[iopv (fourth classargs)] )
	    (make-node
	     '##core#call '(#t)
	     (list 
	      cont
	      (cond [(and iopv
			  (eq? '##core#variable (node-class arg1))
			  (eq? 'vector (get db (first (node-parameters arg1)) 'rest-parameter)) )
		     (make-node '##core#inline (list iopv) callargs) ]
		    [else (make-node '##core#inline (list (second classargs)) callargs)] ) ) ) ) ) )

    ;; (<op>) -> <var>
    ((3) ; classargs = (<var>)
     (and inline-substitutions-enabled
	  (null? callargs)
	  (test name 'standard-binding)
	  (make-node '##core#call '(#t) (list cont (varnode (first classargs)))) ) )

    ;; (<op> a b) -> (<primitiveop> a (quote <i>) b)
    ((4) ; classargs = (<primitiveop> <i>)
     (and inline-substitutions-enabled
	  unsafe
	  (= 2 (length callargs))
	  (test name 'standard-binding)
	  (make-node '##core#call (list #f (first classargs))
		     (list (varnode (first classargs))
			   cont
			   (first callargs)
			   (qnode (second classargs))
			   (second callargs) ) ) ) )

    ;; (<op> a) -> (##core#inline <iop> a (quote <x>))
    ((5) ; classargs = (<iop> <x> <numtype>)
     ;; - <numtype> may be #f
     (and inline-substitutions-enabled
	  (or (test name 'extended-binding)
	      (test name 'standard-binding) )
	  (= 1 (length callargs))
	  (let ((ntype (third classargs)))
	    (or (not ntype) (eq? ntype number-type)) )
	  (make-node '##core#call '(#t)
		     (list cont
			   (make-node '##core#inline (list (first classargs))
				      (list (first callargs)
					    (qnode (second classargs)) ) ) ) ) ) )

    ;; (<op> a) -> (##core#inline <iop1> (##core#inline <iop2> a))
    ((6) ; classargs = (<iop1> <iop2> <safe>)
      (and (or (third classargs) unsafe)
	   inline-substitutions-enabled
	   (= 1 (length callargs))
	   (test name 'standard-binding)
	   (make-node '##core#call '(#t)
		      (list cont
			    (make-node '##core#inline (list (first classargs))
				       (list (make-node '##core#inline (list (second classargs))
							callargs) ) ) ) ) ) )

    ;; (<op> ...) -> (##core#inline <iop> ... (quote <x>))
    ((7) ; classargs = (<argc> <iop> <x> <safe>)
     (and (or (fourth classargs) unsafe)
	  inline-substitutions-enabled
	  (= (length callargs) (first classargs))
	  (or (test name 'standard-binding) (test name 'extended-binding))
	  (make-node '##core#call '(#t)
		     (list cont
			   (make-node '##core#inline (list (second classargs))
				      (append callargs
					      (list (qnode (third classargs))) ) ) ) ) ) )

    ;; (<op> ...) -> <<call procedure <proc> with <classargs>, <cont> and <callargs> >>
    ((8) ; classargs = (<proc> ...)
     (and inline-substitutions-enabled
	  (or (test name 'standard-binding)
	      (test name 'extended-binding) )
	  ((first classargs) db classargs cont callargs) ) )

    ;; (<op> <x1> ...) -> (##core#inline "C_and" (##core#inline <iop> <x1> <x2>) ...)
    ;; (<op> [<x>]) -> (quote #t)
    ((9) ; classargs = (<iop-fixnum> <iop-flonum> <fixnum-safe> <flonum-safe>)
     (and inline-substitutions-enabled
	  (test name 'standard-binding)
	  (if (< (length callargs) 2)
	      (make-node '##core#call '(#t) (list cont (qnode #t)))
	      (and (or (and unsafe (not (eq? number-type 'generic)))
		       (and (eq? number-type 'fixnum) (third classargs))
		       (and (eq? number-type 'flonum) (fourth classargs)) )
		   (let* ([names (map (lambda (z) (gensym)) callargs)]
			  [vars (map varnode names)] )
		     (fold-right
		      (lambda (x n y) (make-node 'let (list n) (list x y)))
		      (make-node
		       '##core#call '(#t)
		       (list 
			cont
			(let ([op (list
				   (if (eq? number-type 'fixnum)
				       (first classargs)
				       (second classargs) ) ) ] )
			  (fold-boolean
			   (lambda (x y) (make-node '##core#inline op (list x y))) 
			   vars) ) ) )
		      callargs names) ) ) ) ) )

    ;; (<op> a [b]) -> (<primitiveop> a (quote <i>) b)
    ((10) ; classargs = (<primitiveop> <i> <bvar> <safe>)
     (and inline-substitutions-enabled
	  (or (fourth classargs) unsafe)
	  (test name 'standard-binding)
	  (let ((n (length callargs)))
	    (and (< 0 n 3)
		 (make-node '##core#call (list #f (first classargs))
			    (list (varnode (first classargs))
				  cont
				  (first callargs)
				  (qnode (second classargs))
				  (if (null? (cdr callargs))
				      (varnode (third classargs))
				      (second callargs) ) ) ) ) ) ) )

    ;; (<op> ...) -> (<primitiveop> ...)
    ((11) ; classargs = (<argc> <primitiveop> <safe>)
     (and inline-substitutions-enabled
	  (or (third classargs) unsafe)
	  (test name 'standard-binding)
	  (= (length callargs) (first classargs))
	  (make-node '##core#call (list #t (second classargs))
		     (cons* (varnode (second classargs))
			    cont
			    callargs) ) ) )

    ;; (<op> a) -> a
    ;; (<op> ...) -> (<primitiveop> ...)
    ((12) ; classargs = (<primitiveop> <safe> <maxargc>)
     (and inline-substitutions-enabled
	  (test name 'standard-binding)
	  (or (third classargs) unsafe)
	  (let ((n (length (callargs))))
	    (and (<= n (fourth classargs))
		 (case n
		   ((1) (make-node '##core#call '(#t) (cons cont callargs)))
		   (else (make-node '##core#call (list #t (second classargs))
				    (cons* (varnode (second classargs))
					   cont callargs) ) ) ) ) ) ) )

    ;; (<op> ...) -> ((##core#proc <primitiveop>) ...)
    ((13) ; classargs = (<primitiveop> <safe>)
     (and inline-substitutions-enabled
	  (or (test name 'extended-binding) (test name 'standard-binding))
	  (or (second classargs) unsafe)
	  (let ((pname (first classargs)))
	    (make-node '##core#call (if (pair? params) (cons #t (cdr params)) params)
		       (cons* (make-node '##core#proc (list pname) '())
			      cont callargs) ) ) ) )

    ;; (<op> <x> ...) -> (##core#inline <iop-safe>/<iop-unsafe> <x> ...)
    ((14) ; classargs = (<numtype> <argc> <iop-safe> <iop-unsafe>)
     (and inline-substitutions-enabled
	  (= (second classargs) (length callargs))
	  (or (test name 'extended-binding)
	      (test name 'standard-binding) )
	  (eq? number-type (first classargs))
	  (or (fourth classargs) unsafe)
	  (make-node
	   '##core#call '(#t)
	   (list cont
		 (make-node
		  '##core#inline
		  (list (if unsafe (fourth classargs) (third classargs)))
		  callargs) ) ) ) )

    ;; (<op> <x>) -> (<primitiveop> <x>)   - if numtype1
    ;;             | <x>                   - if numtype2
    ((15) ; classargs = (<numtype1> <numtype2> <primitiveop> <safe>)
     (and inline-substitutions-enabled
	  (= 1 (length callargs))
	  (or unsafe (fourth classargs))
	  (or (test name 'extended-binding)
	      (test name 'standard-binding) )
	  (cond ((eq? number-type (first classargs))
		 (make-node '##core#call (list #t (third classargs))
			    (cons* (varnode (third classargs)) cont callargs) ) )
		((eq? number-type (second classargs))
		 (make-node '##core#call '(#t) (cons cont callargs)) )
		(else #f) ) ) )

    ;; (<alloc-op> ...) -> (##core#inline_allocate (<aiop> <words>) ...)
    ((16) ; classargs = (<argc> <aiop> <safe> <words>)
     ;; - <argc> may be #f, saying that any number of arguments is allowed,
     ;; - <words> may be a list of one element (the number of words), meaning that
     ;;   the words are to be multiplied with the number of arguments.
     ;; - <words> may also be #t, meaning that the number of words is the same as the
     ;;   number of arguments plus 1.
     (let ([argc (first classargs)]
	   [rargc (length callargs)]
	   [w (fourth classargs)] )
       (and inline-substitutions-enabled
	    (or (not argc) (= rargc argc))
	    (or (test name 'extended-binding) (test name 'standard-binding))
	    (or (third classargs) unsafe)
	    (make-node
	     '##core#call '(#t)
	     (list cont 
		   (make-node
		    '##core#inline_allocate
		    (list (second classargs) 
			  (cond [(eq? #t w) (add1 rargc)]
				[(pair? w) (* rargc (car w))]
				[else w] ) )
		    callargs) ) ) ) ) )

    ;; (<op> ...) -> (##core#inline <iop>/<unsafe-iop> ...)
    ((17) ; classargs = (<argc> <iop-safe> [<iop-unsafe>])
     (and inline-substitutions-enabled
	  (= (length callargs) (first classargs))
	  (or (test name 'extended-binding) (test name 'standard-binding))
	  (make-node
	   '##core#call '(#t)
	   (list cont
		 (make-node '##core#inline
			    (list (if (and unsafe (pair? (cddr classargs)))
				      (third classargs)
				      (second classargs) ) )
			    callargs)) ) ) )

    ;; (<op>) -> (quote <null>)
    ((18) ; classargs = (<null>)
     (and inline-substitutions-enabled
	  (null? callargs)
	  (or (test name 'extended-binding) (test name 'standard-binding))
	  (make-node '##core#call '(#t) (list cont (qnode (first classargs))) ) ) )

    ;; (<op> (quote <const>) [<x>]) -> (##core#inline <iop> (quote <const>) <x> | ##sys#standard-output)
    ;;  - if file-io-only and const satisfies <pred>.
    ;;  - if port is given unsafe has to be #t.
    ((19) ; classargs = (<pred> <iop>)
     (and inline-substitutions-enabled
	  file-io-only
	  (test name 'standard-binding)
	  (and-let* ([port (case (length callargs)
			     [(1) (varnode '##sys#standard-output)]
			     [(2) (and unsafe (second callargs))]
			     [else #f] ) ] )
	    (let ([arg (first callargs)])
	      (and (eq? 'quote (node-class arg))
		   ((first classargs) (first (node-parameters arg)))
		   (make-node '##core#call '(#t)
			      (list cont (make-node '##core#inline (list (second classargs)) 
						    (list port arg) ) ) ) ) ) ) ) )

    ;; (<op> [<x>]) -> (##core#inline <iop> <x> | ##sys#standard-input)
    ;;  - if file-io-only.
    ;;  - if port is given unsafe has to be #t.
    ((20) ; classargs = (<iop> <out?>)
     (and inline-substitutions-enabled
	  file-io-only
	  (or (test name 'standard-binding) (test name 'extended-binding))
	  (and-let* ([port (case (length callargs)
			     [(0) (varnode (if (second classargs) '##sys#standard-output '##sys#standard-input))]
			     [(1) (and unsafe (first callargs))]
			     [else #f] ) ] )
	    (make-node '##core#call '(#t)
		       (list cont (make-node '##core#inline (list (first classargs)) 
					     (list port) ) ) ) ) ) )

    (else (bomb "bad type (optimize)")) ) )


;;; Find procedures that can potentially be inlined:
;
; - Returns a list of inline-situations: ((<lambda-node> <size> <call-node>) ...)
; - Procedures that can potentially be inlined are not self-recursive, have no free 
;   variables and grow the containing procedure not by more than <inline-limit>.

(define (find-inlining-candidates node db)
  (let ([candidates '()])

    (define (test sym prop) (get db sym prop))
    (define (compute-limit n) (+ n (/ (* n inline-limit) 100)))

    (define (walk n here sizeh)
      (let ([subs (node-subexpressions n)])
	(case (node-class n)
	  [(##core#lambda) 
	   (let ([params (node-parameters n)])
	     (walk (first subs) (first params) (fourth params)) ) ]
	  [(##core#call)
	   (let ([fn (first subs)])
	     (when (eq? '##core#variable (node-class fn))
	       ;; Call to named procedure:
	       (let* ([var (first (node-parameters fn))]
		      [lval (and (not (test var 'unknown)) (test var 'value))]
		      [args (cdr subs)] )
		 (when (and lval (eq? '##core#lambda (node-class lval)))
		   ;; Known procedure:
		   (let* ([lparams (node-parameters lval)]
			  [id (first lparams)]
			  [size (fourth lparams)]
			  [llist (third lparams)] )
		     (when (and (not (find-lambda-container here id db)) ; not recursive?
				(<= (+ sizeh size) (compute-limit sizeh)) ; not exceeding limit?
				;; lambda has no free, non-global variables:
				(every (lambda (v) (test v 'global)) (scan-free-variables lval)) )
		       (check-signature args llist)
		       (set! candidates (cons (list lval size n) candidates)) ) ) ) ) )
	     (walkeach subs here sizeh) ) ]
	  [else (walkeach subs here sizeh)] ) ) )

    (define (walkeach ns here sizeh)
      (for-each (lambda (n) (walk n here sizeh)) ns) )

    (debugging 'p "inlining analysis phase...")
    (walk node #f 0)
    (let* ([limit (compute-limit original-program-size)]
	   [count current-program-size]
	   [cs
	    (take-while
	     (lambda (c)
	       (set! count (+ count (second c)))
	       (< count limit) )
	     (sort candidates (lambda (c1 c2) (< (second c1) (second c2)))) ) ] )
      (debugging
       'i "inlining: osize/csize/limit/candidates"
       original-program-size
       current-program-size
       limit
       (delay (map (lambda (c) (cons (first (node-parameters (first c))) (second c))) cs)) )
      cs) ) )


;;; Insert inlined procedures into call-sites:
;
; - identify mutable constants in inlined lambda-bodies and convert literal-reference
;   to variable reference.
; - after inlining a a-list of newly introduced variables and their values is returned.

(define (perform-inlining! candidates node db)
  (let ([mvars '()])

    (define (test sym prop) (get db sym prop))

    (define (walk n)
      (let ([subs (node-subexpressions n)])
	(case (node-class n)
	  [(##core#call)
	   ;; Call to a named proc and in candidates-list?
	   (let ([fn (first subs)])
	     (if (eq? '##core#variable (node-class fn))
		 (let* ([var (first (node-parameters fn))]
			[lval (and (not (test var 'unknown)) (test var 'value))]
			[args (cdr subs)] )
		   (if (and lval (eq? '##core#lambda (node-class lval)))
		       (let ([ic (find (lambda (c) (eq? n (third c))) candidates)])
			 (if ic
			     (insert! var lval n)
			     (walkeach subs) ) )
		       (walkeach subs) ) )
		 (walkeach subs) ) ) ]
	  [else (walkeach subs)] ) ) )

    (define (walkeach ns) (for-each walk ns))

    (define (insert! var lnode cnode)
      (let* ([lnode2 (alpha lnode '())]
	     [llist (third (node-parameters lnode2))]
	     [args (cdr (node-subexpressions cnode))]
	     [body (node-subexpressions lnode2)] )
	(debugging 'i "inlining" var (fourth (node-parameters lnode2)))
	(debugging 'o "inlining" var)
	(let ([inode (inline-lambda-bindings llist args (first body))])
	  (node-class-set! cnode 'let)
	  (node-parameters-set! cnode (node-parameters inode))
	  (node-subexpressions-set! cnode (node-subexpressions inode)) ) ) )

    ;; Perform alpha conversion on complete node tree and gather mutable constants:
    (define (alpha n e)
      (let ([params (node-parameters n)]
	    [class (node-class n)]
	    [subs (node-subexpressions n)] )
	(case class
	  [(quote)
	   (if (not (collapsable-literal? (first params)))
	       (let ([var (gensym)])
		 (set! mvars (alist-cons var (first params) mvars))
		 (node-class-set! n '##core#variable)
		 (node-parameters-set! n (list var))
		 (varnode var) )
	       n) ]
	  [(##core#lambda)
	   (decompose-lambda-list
	    (third params)
	    (lambda (vars argc rest)
	      (let ([aliases (map gensym vars)])
		(make-node
		 '##core#lambda 
		 (list (gensym 'f)
		       (second params)
		       (append 
			(take aliases argc) 
			(if rest (last aliases) '()) )
		       (fourth params) )
		 (mapalpha subs (append (map cons vars aliases) e)) ) ) ) ) ]
	  [(##core#variable) (varnode (lookup (first params) e))]
	  [(let)
	   (let* ([var (first params)]
		  [alias (gensym var)] )
	     (make-node 'let (list alias) (mapalpha subs (append (list (cons var alias)) e))) ) ]
	  [(set!)
	   (make-node 'set! (list (lookup (first params) e)) (mapalpha subs e)) ]
	  [else (make-node class params (mapalpha subs e))] ) ) )

    (define (mapalpha ns e)
      (map (lambda (n) (alpha n e)) ns) )

    (define (lookup v e)
      (cond [(assq v e) => cdr]
	    [else v] ) )

    (debugging 'p "inlining mutation phase...")
    (walk node) 
    mvars) )


;;; Optimize direct leaf routines:

(define (transform-direct-lambdas! node db)
  (let ([dirty #f]
	[inner-ks '()] 
	[hoistable '()] 
	[allocated 0] )

    ;; Process node tree and walk lambdas that meet the following constraints:
    ;;  - Only external lambdas (no CPS redexes),
    ;;  - All calls are either to the direct continuation or (tail-) recursive calls.
    ;;  - No allocation, no rest parameter.
    ;;  - The lambda has a known container variable and all it's call-sites are known.

    (define (walk d n dn)
      (let ([params (node-parameters n)]
	    [subs (node-subexpressions n)] )
	(case (node-class n)
	  [(##core#lambda)
	   (let ([llist (third params)])
	     (if (and d
		      (second params)
		      (not (get db d 'unknown))
		      (proper-list? llist)
		      (and-let* ([val (get db d 'value)]
				 [refs (get db d 'references)]
				 [sites (get db d 'call-sites)] )
			(and (eq? n val)
			     (= (length refs) (length sites)) 
			     (scan (first subs) (first llist) d dn (cons d llist)) ) ) )
		 (transform n d inner-ks hoistable dn allocated) 
		 (walk #f (first subs) #f) ) ) ]
	  [(set!) (walk (first params) (first subs) #f)]
	  [(let)
	   (walk (first params) (first subs) n)
	   (walk #f (second subs) #f) ]
	  [else (for-each (lambda (x) (walk #f x #f)) subs)] ) ) )

    (define (scan n kvar fnvar destn env)
      (let ([closures '()]
	    [recursive #f] )
	(define (rec n v vn e)
	  (let ([params (node-parameters n)]
		[subs (node-subexpressions n)] )
	    (case (node-class n)
	      [(##core#variable)
	       (let ([v (first params)])
		 (or (not (get db v 'boxed))
		     (not (memq v env))
		     (and (not recursive)
			  (begin
			    (set! allocated (+ allocated 2))
			    #t) ) ) ) ]
	      [(##core#lambda)
	       (and v
		    (decompose-lambda-list
		     (third params)
		     (lambda (vars argc rest)
		       (set! closures (cons v closures))
		       (rec (first subs) #f #f (append vars e)) ) ) ) ]
	      [(##core#inline_allocate)
	       (and (not recursive)
		    (begin
		      (set! allocated (+ allocated (second params)))
		      (every (lambda (x) (rec x #f #f e)) subs) ) ) ]
	      [(##core#direct_lambda)
	       (and vn destn
		    (null? (scan-used-variables (first subs) e)) 
		    (begin
		      (set! hoistable (alist-cons v vn hoistable))
		      #t) ) ]
	      [(##core#inline_ref)
	       (and (let ([n (estimate-foreign-result-size (second params))])
		      (or (zero? n)
			  (and (not recursive)
			       (begin
				 (set! allocated (+ allocated n))
				 #t) ) ) )
		    (every (lambda (x) (rec x #f #f e)) subs) ) ]
	      [(##core#call)
	       (let ([fn (first subs)])
		 (and (eq? '##core#variable (node-class fn))
		      (let ([v (first (node-parameters fn))])
			(cond [(eq? v fnvar)
			       (and (zero? allocated)
				    (let ([k (second subs)])
				      (when (eq? '##core#variable (node-class k))
					(set! inner-ks (cons (first (node-parameters k)) inner-ks)) )
				      (set! recursive #t)
				      #t) ) ]
			      [else (eq? v kvar)] ) )
		      (every (lambda (x) (rec x #f #f e)) (cdr subs)) ) ) ]
	      [(##core#direct_call)
	       (let ([n (fourth params)])
		 (or (zero? n)
		     (and (not recursive)
			  (begin
			    (set! allocated (+ allocated n))
			    (every (lambda (x) (rec x #f #f e)) subs) ) ) ) ) ]
	      [(set!) (rec (first subs) (first params) #f e)]
	      [(let)
	       (and (rec (first subs) (first params) n e)
		    (rec (second subs) #f #f (append params e)) ) ]
	      [else (every (lambda (x) (rec x #f #f e)) subs)] ) ) )
	(set! inner-ks '())
	(set! hoistable '())
	(set! allocated 0)
	(and (rec n #f #f env)
	     (lset= eq? closures (delete kvar inner-ks eq?)) ) ) )

    (define (transform n fnvar ks hoistable destn allocated)
      (if (pair? hoistable)
	  (debugging 'o "direct leaf routine with hoistable closures/allocation" fnvar (delay (unzip1 hoistable)) allocated)
	  (debugging 'o "direct leaf routine/allocation" fnvar allocated) )
      (set! dirty #t)
      (let ([params (node-parameters n)]
	    [klambdas '()] 
	    [sites (get db fnvar 'call-sites)] 
	    [ksites '()] )
	(match params
	  [(id _ (kvar vars ...) _)
	   ;; Remove continuation argument:
	   (set-car! (cddr params) vars)
	   ;; Make "##core#direct_lambda":
	   (node-class-set! n '##core#direct_lambda)
	   ;; Transform recursive calls and remove unused continuations:

	   (let rec ([n (first (node-subexpressions n))])
	     (let ([params (node-parameters n)]
		   [subs (node-subexpressions n)] )
	       (case (node-class n)
		 [(##core#call)
		  (let* ([fn (first subs)]
			 [arg0 (second subs)]
			 [fnp (node-parameters fn)] 
			 [arg0p (node-parameters arg0)] )
		    (when (eq? '##core#variable (node-class fn))
		      (cond [(eq? fnvar (first fnp))
			     (set! ksites (alist-cons #f n ksites))
			     (cond [(eq? kvar (first arg0p))
				    (node-class-set! n '##core#recurse)
				    (node-parameters-set! n (list #t id))
				    (node-subexpressions-set! n (cddr subs)) ]
				   [(assq (first arg0p) klambdas)
				    => (lambda (a)
					 (let* ([klam (cdr a)]
						[kbody (first (node-subexpressions klam))] )
					   (node-class-set! n 'let)
					   (node-parameters-set! n (take (third (node-parameters klam)) 1))
					   (node-subexpressions-set!
					    n
					    (list (make-node '##core#recurse (list #f id) (cddr subs)) kbody) )
					   (rec kbody) ) ) ]
				   [else (bomb "missing kvar" arg0p)] ) ]
			    [(eq? kvar (first fnp))
			     (node-class-set! n '##core#return)
			     (node-parameters-set! n '())
			     (node-subexpressions-set! n (cdr subs)) ]
			    [else (bomb "bad call (leaf)")] ) ) ) ]
		 [(let)
		  (let ([var (first params)]
			[val (first subs)] )
		    (cond [(memq var ks)
			   (set! klambdas (alist-cons var val klambdas))
			   (copy-node! (second subs) n)
			   (rec n) ]
			  [else (for-each rec subs)] ) ) ]

		 [else (for-each rec subs)] ) ) )

	   ;; Transform call-sites:
	   (for-each
	    (lambda (site)
	      (let* ([n (cdr site)]
		     [nsubs (node-subexpressions n)] )
		(node-subexpressions-set!
		 n
		 (list (second nsubs)
		       (make-node
			'##core#direct_call
			(list #t #f id allocated)
			(cons (car nsubs) (cddr nsubs)) ) ) ) ) )
	    (lset-difference (lambda (s1 s2) (eq? (cdr s1) (cdr s2))) sites ksites) )

	   ;; Hoist direct lambdas out of container:
	   (when (and destn (pair? hoistable))
	     (let ([destn0 (make-node #f #f #f)])
	       (copy-node! destn destn0) ; get copy of container binding
	       (let ([hoisted
		      (fold-right	; build cascade of bindings for each hoistable direct lambda...
		       (lambda (h rest)
			 (make-node
			  'let (list (car h))
			  (let ([dlam (first (node-subexpressions (cdr h)))])
			    (list (make-node (node-class dlam) (node-parameters dlam) (node-subexpressions dlam))
				  rest) ) ) )
		       destn0
		       hoistable) ] )
		 (copy-node! hoisted destn) ; mutate container binding to hold hoistable bindings
		 (for-each 
		  (lambda (h)		; change old direct lambdas bindings to dummy ones...
		    (let ([vn (cdr h)])
		      (node-parameters-set! vn (list (gensym)))
		      (set-car! (node-subexpressions vn) (make-node '##core#undefined '() '())) ) )
		  hoistable) ) ) ) ]
	  [_ (bomb "invalid parameter list" params)] ) ) )

    (debugging 'p "direct leaf routine optimization pass...")
    (walk #f node #f)
    dirty) )


;;; Convert closures to explicit data structures (effectively flattens function-binding structure):

(define (perform-closure-conversion node db)
  (let ([direct-calls 0]
	[customizable '()] )

    (define (test sym item) (get db sym item))
  
    (define (register-customizable! var id)
      (set! customizable (lset-adjoin eq? customizable var)) 
      (put! db id 'customizable #t) )

    (define (register-direct-call! id)
      (set! direct-calls (add1 direct-calls))
      (set! direct-call-ids (lset-adjoin eq? direct-call-ids id)) )

    ;; Gather free-variable information:
    ;;   - register direct calls.
    ;;   - update (by mutation) call information in "##core#call" nodes.
    (define (gather n here env)
      (let ((subs (node-subexpressions n))
	    (params (node-parameters n)) )
	(case (node-class n)

	  ((quote ##core#variable ##core#undefined ##core#proc ##core#primitive) #f)

	  ((let)
	   (receive (vals body) (split-at subs (length params))
	     (for-each (lambda (n) (gather n here env)) vals)
	     (gather (first body) here (append params env)) ) )

	  ((##core#call)
	   (let* ([fn (first subs)]
		  [mode (first params)]
		  [name (and (pair? (cdr params)) (second params))]
		  [varfn (eq? '##core#variable (node-class fn))] )
	     (node-parameters-set!
	      n
	      (cons mode
		    (if (or name varfn)
			(cons name
			      (if varfn
				  (let* ([varname (first (node-parameters fn))]
					 [val (and (not (test varname 'unknown)) (test varname 'value))] )
				    (if (and val (eq? '##core#lambda (node-class val)))
					(let* ([params (node-parameters val)]
					       [id (first params)]
					       [refs (test varname 'references)]
					       [sites (test varname 'call-sites)] 
					       [custom
						(and refs sites
						     (= (length refs) (length sites)) 
						     (proper-list? (third params)) ) ] )
					  (register-direct-call! id)
					  (when custom (register-customizable! varname id)) 
					  (list id custom) )
					'() ) )
				  '() ) )
			'() ) ) )
	     (for-each (lambda (n) (gather n here env)) subs) ) )

	  ((##core#lambda ##core#direct_lambda)
	   (decompose-lambda-list
	    (third params)
	    (lambda (vars argc rest)
	      (let* ([id (if here (first params) 'toplevel)]
		     [capturedvars (captured-variables (car subs) env)]
		     [csize (length capturedvars)] )
		(put! db id 'closure-size csize)
		(put! db id 'captured-variables capturedvars)
		(gather (car subs) id (append vars env)) ) ) ) )
	
	  (else (for-each (lambda (n) (gather n here env)) subs)) ) ) )

    ;; Create explicit closures:
    (define (transform n here closure)
      (let ((subs (node-subexpressions n))
	    (params (node-parameters n)) 
	    (class (node-class n)) )
	(case class

	  ((quote ##core#undefined ##core#proc) n)

	  ((##core#variable)
	   (let* ((var (first params))
		  (val (ref-var n here closure)) )
	     (if (test var 'boxed)
		 (make-node '##core#unbox '() (list val))
		 val) ) )

	  ((if ##core#call ##core#inline ##core#inline_allocate ##core#callunit ##core#inline_ref ##core#inline_update 
	       ##core#switch ##core#cond ##core#direct_call ##core#recurse ##core#return)
	   (make-node (node-class n) params (maptransform subs here closure)) )

	  ((let)
	   (let* ([var (first params)]
		  [boxedvar (test var 'boxed)]
		  [boxedalias (gensym var)] )
	     (if boxedvar
		 (make-node 
		  'let (list boxedalias)
		  (list (transform (first subs) here closure)
			(make-node
			 'let (list var)
			 (list (make-node '##core#box '() (list (varnode boxedalias)))
			       (transform (second subs) here closure) ) ) ) )
		 (make-node
		  'let params
		  (maptransform subs here closure) ) ) ) )

	  ((##core#lambda ##core#direct_lambda)
	   (let ([llist (third params)])
	     (decompose-lambda-list
	      llist
	      (lambda (vars argc rest)
		(let* ([boxedvars (filter (lambda (v) (test v 'boxed)) vars)]
		       [boxedaliases (map cons boxedvars (map gensym boxedvars))]
		       [cvar (gensym 'c)]
		       [id (if here (first params) 'toplevel)]
		       [capturedvars (or (test id 'captured-variables) '())]
		       [csize (or (test id 'closure-size) 0)] )
		  ;; If rest-parameter is boxed: mark it as 'boxed-rest
		  ;;  (if we don't do this than preparation will think the (boxed) alias
		  ;;  of the rest-parameter is never used)
		  (and-let* ([rest]
			     [(test rest 'boxed)]
			     [rp (test rest 'rest-parameter)] )
		    (put! db (cdr (assq rest boxedaliases)) 'boxed-rest #t) )
		  (make-node
		   '##core#closure (list (add1 csize))
		   (cons
		    (make-node
		     class
		     (list id
			   (second params)
			   (cons 
			    cvar
			    (build-lambda-list
			     (map (lambda (v)
				    (cond ((assq v boxedaliases) => cdr)
					  (else v) ) )
				  vars)
			     argc
			     (cond ((and rest (assq rest boxedaliases)) => cdr)
				   (else rest) ) ) )
			   (fourth params) )
		     (list (let ((body (transform (car subs) cvar capturedvars)))
			     (if (pair? boxedvars)
				 (fold-right
				  (lambda (alias val body) (make-node 'let (list alias) (list val body)))
				  body
				  (unzip1 boxedaliases)
				  (map (lambda (a) (make-node '##core#box '() (list (varnode (cdr a)))))
				       boxedaliases) )
				 body) ) ) )
		    (map (lambda (v) (ref-var (varnode v) here closure))
			 capturedvars) ) ) ) ) ) ) )

	  ((set!)
	   (let* ([var (first params)]
		  [val (first subs)]
		  [cval (node-class val)]
		  [immf (or (and (eq? 'quote cval) (immediate? (first (node-parameters val))))
			    (eq? '##core#undefined cval) ) ] )
	     (cond ((posq var closure)
		    => (lambda (i)
			 (if (test var 'boxed)
			     (make-node
			      (if immf '##core#updatebox_i '##core#updatebox)
			      '()
			      (list (make-node '##core#ref (list (add1 i)) (list (varnode here)))
				    (transform val here closure) ) )
			     ;; Is the following actually used???
			     (make-node
			      (if immf '##core#update_i '##core#update)
			      (list (add1 i))
			      (list (varnode here)
				    (transform val here closure) ) ) ) ) )
		   ((test var 'boxed)
		    (make-node
		     (if immf '##core#updatebox_i '##core#updatebox)
		     '()
		     (list (varnode var)
			   (transform val here closure) ) ) )
		   (else (make-node
			  'set! (list var)
			  (list (transform val here closure) ) ) ) ) ) )

	  ((##core#primitive) 
	   (make-node '##core#closure '(1) 
		      (list (make-node '##core#proc params '())) ) )

	  (else (bomb "bad node (closure2)")) ) ) )

    (define (maptransform xs here closure)
      (map (lambda (x) (transform x here closure)) xs) )
  
    (define (ref-var n here closure)
      (let ((var (first (node-parameters n))))
	(cond ((posq var closure) 
	       => (lambda (i) 
		    (make-node '##core#ref (list (+ i 1)) 
			       (list (varnode here)) ) ) )
	      (else n) ) ) )

    (define (captured-variables node env)
      (let ([vars '()])
	(let walk ([n node])
	  (let ((subs (node-subexpressions n))
		(params (node-parameters n)) )
	    (case (node-class n)
	      ((##core#variable)
	       (let ([var (first params)])
		 (when (memq var env)
		   (set! vars (lset-adjoin eq? vars var)) ) ) )
	      ((quote ##core#undefined ##core#primitive ##core#proc ##core#inline_ref) #f)
	      ((set!) 
	       (let ([var (first params)])
		 (when (memq var env) (set! vars (lset-adjoin eq? vars var)))
		 (walk (car subs)) ) )
	      (else (for-each walk subs)) ) ) )
	vars) )

    (debugging 'p "closure conversion gathering phase...")
    (gather node #f '())
    (debugging 'o "customizable procedures" customizable)
    (debugging 'p "closure conversion transformation phase...")
    (let ((node2 (transform node #f #f)))
      (unless (zero? direct-calls)
	(debugging 'o "calls to known targets" direct-calls (delay (length direct-call-ids))) )
      node2) ) )


;;; Do some preparations before code-generation can commence:

(define-record lambda-literal
  id					; symbol
  external                              ; boolean
  arguments				; (symbol...)
  argument-count			; integer
  rest-argument				; symbol | #f
  temporaries				; integer
  callee-signatures			; (integer...)
  allocated				; integer
  directly-called			; boolean
  closure-size				; integer
  looping				; boolean
  customizable				; boolean
  rest-argument-mode			; #f | LIST | VECTOR | UNUSED
  body					; expression
  direct)				; boolean
  
(define (prepare-for-code-generation node db)
  (let ([literals '()]
        [lambdas '()]
        [temporaries 0]
        [allocated 0]
	[looping 0]
        [signatures '()] 
	[fastinits 0] 
	[fastrefs 0] 
	[fastsets 0] )
    
    (define (walk n e here boxes)
      (let ((subs (node-subexpressions n))
	    (params (node-parameters n))
	    (class (node-class n)) )
	(case class

	  ((##core#undefined ##core#proc) n)

	  ((##core#variable)
	   (let ([var (first params)])
	     (cond [(posq var e) => (lambda (i) (make-node '##core#local (list i) '()))]
		   [(keyword? var) (make-node '##core#literal (list (literal var)) '())]
                   [else
		    (let* ([safe (not (or no-bound-checks
					  unsafe
					  (memq var always-bound)
					  (get db var 'standard-binding)
					  (get db var 'extended-binding) ) ) ]
			   [blockvar (memq var block-globals)] )
		      (when blockvar (set! fastrefs (add1 fastrefs)))
		      (make-node
		       '##core#global
		       (list (if blockvar
				 (blockvar-literal var)
				 (literal var) )
			     safe
			     blockvar)
		       '() ) ) ] ) ) )

	  ((##core#direct_call)
	   (set! allocated (+ allocated (fourth params)))
	   (make-node class params (mapwalk subs e here boxes)) )

	  ((##core#inline_allocate)
	   (set! allocated (+ allocated (second params)))
	   (make-node class params (mapwalk subs e here boxes)) )

	  ((##core#inline_ref)
	   (set! allocated (+ allocated (words (estimate-foreign-result-size (second params)))))
	   (make-node class params '()) )

	  ((##core#closure) 
	   (set! allocated (+ allocated (first params) 1))
	   (make-node '##core#closure params (mapwalk subs e here boxes)) )

	  ((##core#box)
	   (set! allocated (+ allocated 2))
	   (make-node '##core#box params (list (walk (first subs) e here boxes))) )

	  ((##core#updatebox)
	   (let* ([b (first subs)]
		  [subs (mapwalk subs e here boxes)] )
	     (make-node
	      (cond [(and (eq? '##core#variable (node-class b))
			  (memq (first (node-parameters b)) boxes) )
		     (set! fastinits (add1 fastinits))
		     '##core#updatebox_i]
		    [else class] )
	      '()
	      subs) ) )

	  ((##core#lambda ##core#direct_lambda) 
	   (let ([temps temporaries]
		 [sigs signatures]
		 [lping looping]
		 [alc allocated] 
		 [direct (eq? class '##core#direct_lambda)] )
	     (set! temporaries 0)
	     (set! allocated 0)
	     (set! signatures '())
	     (set! looping 0)
	     (decompose-lambda-list
	      (third params)
	      (lambda (vars argc rest)
		(let* ([id (first params)]
		       [rest-mode
			(and rest
			     (let ([rrefs (get db rest 'references)])
			       (cond [(get db rest 'assigned) 'list]
				     [(and (not (get db rest 'boxed-rest)) (or (not rrefs) (null? rrefs))) 'none] 
				     [else (get db rest 'rest-parameter)] ) ) ) ]
		       [body (walk 
			      (car subs)
			      (if (eq? 'none rest-mode)
				  (butlast vars)
				  vars)
			      id
			      '()) ] )
		  (case rest-mode
		    [(none) (debugging 'o "unused rest argument" rest id)]
		    [(vector) (debugging 'o "rest argument accessed as vector" rest id)] )
		  (when (and direct rest)
		    (bomb "bad direct lambda" id allocated rest) )
		  (set! lambdas
		    (cons (make-lambda-literal
			   id
			   (second params)
			   vars
			   argc
			   rest
			   (add1 temporaries)
			   signatures
			   allocated
			   (or direct (memq id direct-call-ids))
			   (or (get db id 'closure-size) 0)
			   (and (not rest)
				(> looping 0)
				(begin
				  (debugging 'o "identified direct recursive calls" id looping)
				  #t) )
			   (or direct (get db id 'customizable))
			   rest-mode
			   body
			   direct)
			  lambdas) )
		  (set! looping lping)
		  (set! temporaries temps)
		  (set! allocated alc)
		  (set! signatures sigs)
		  (make-node '##core#proc (list (first params)) '()) ) ) ) ) )

	  ((let)
	   (let* ([var (first params)]
		  [val (first subs)] 
		  [boxvars (if (eq? '##core#box (node-class val)) (list var) '())] )
	     (set! temporaries (add1 temporaries))
	     (make-node
	      '##core#bind (list 1)
	      (list (walk val e here boxes)
		    (walk (second subs) (append e params) here (append boxvars boxes)) ) ) ) )

	  ((set!)
	   (let ([var (first params)]
		 [val (first subs)] )
	     (cond ((posq var e)
		    => (lambda (i) 
			 (make-node '##core#setlocal (list i) (list (walk val e here boxes)) ) ) )
		   (else
		    (let* ([cval (node-class val)]
			   [safe (not (or no-bound-checks
					  unsafe
					  (memq var always-bound)
					  (get db var 'standard-binding)
					  (get db var 'extended-binding) ) ) ]
			   [blockvar (memq var block-globals)]
			   [immf (or (and (eq? cval 'quote) (immediate? (first (node-parameters val))))
				     (eq? '##core#undefined cval) ) ] )
		      (when blockvar (set! fastsets (add1 fastsets)))
		      (make-node
		       (if immf '##core#setglobal_i '##core#setglobal)
		       (list (if blockvar
				 (blockvar-literal var)
				 (literal var) )
			     blockvar)
		       (list (walk (car subs) e here boxes)) ) ) ) ) ) )

	  ((##core#call) 
	   (let ([len (length (cdr subs))])
	     (set! signatures (lset-adjoin = signatures len)) 
	     (when (and (>= (length params) 3) (eq? here (third params)))
	       (set! looping (add1 looping)) )
	     (make-node class params (mapwalk subs e here boxes)) ) )

	  ((##core#recurse)
	   (when (first params) (set! looping (add1 looping)))
	   (make-node class params (mapwalk subs e here boxes)) )

	  ((quote)
	   (let ((c (first params)))
	     (cond ((fixnum? c)
		    (if (eq? 'flonum number-type)
			(make-node '##core#literal (list (literal (exact->inexact c))) '())
			(immediate-literal c) ) )
		   ((number? c)
		    (cond ((eq? 'fixnum number-type)
			   (cond ((integer? c)
				  (warning "coerced inexact literal number `~s' to fixnum" c)
				  (immediate-literal (inexact->exact c)) )
				 (else (quit "can not coerce inexact literal `~S' to fixnum" c)) ) )
			  (else (make-node '##core#literal (list (literal c)) '())) ) )
		   ((immediate? c) (immediate-literal c))
		   (else (make-node '##core#literal (list (literal c)) '())) ) ) )

	  (else (make-node class params (mapwalk subs e here boxes)) ) ) ) )
    
    (define (mapwalk xs e here boxes)
      (map (lambda (x) (walk x e here boxes)) xs) )

    (define (literal x)
      (cond [(immediate? x) (immediate-literal x)]
	    [(and (number? x) (inexact? x) 
		  (list-index (lambda (y) (and (number? y) (inexact? y) (= x y))) literals) )
	     => values]
            [(posq x literals) => values]
	    [else (new-literal x)] ) )

    (define (new-literal x)
      (let ([i (length literals)])
	(set! literals (append literals (list x))) ; could be optimized
	i) )

    (define (blockvar-literal var)
      (or (list-index
	   (lambda (lit) 
	     (and (block-variable-literal? lit)
		  (eq? var (block-variable-literal-name lit)) ) )
	   literals)
	  (new-literal (make-block-variable-literal var)) ) )
    
    (define (immediate-literal x)
      (make-node '##core#immediate
		 (cond ((fixnum? x) `(fix ,x))
		       ((boolean? x) `(bool ,x))
		       ((char? x) `(char ,x))
		       ((null? x) '(nil))
		       (else (bomb "bad immediate (prepare)")) )
		 '() ) )
    
    (debugging 'p "preparation phase...")
    (let ((node2 (walk node '() #f '())))
      (debugging 'o "fast box initializations" fastinits)
      (debugging 'o "fast global references" fastrefs)
      (debugging 'o "fast global assignments" fastsets)
      (values node2 literals lambdas) ) ) )
