/* runtime.c - Runtime code for compiler generated executables
;
; Copyright (c) 2000-2001, Felix L. Winkelmann
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following
; conditions are met:
;
;   Redistributions of source code must retain the above copyright notice, this list of conditions and the following
;     disclaimer. 
;   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
;     disclaimer in the documentation and/or other materials provided with the distribution. 
;   Neither the name of the author nor the names of its contributors may be used to endorse or promote
;     products derived from this software without specific prior written permission. 
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
; AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
; CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
; POSSIBILITY OF SUCH DAMAGE.
;
; Send bugs, suggestions and ideas to: 
;
; felix@call-with-current-continuation.org
;
; Felix L. Winkelmann
; Steinweg 1A
; 37130 Gleichen, OT Weissenborn
; Germany
*/


#include "chicken.h"
#include <errno.h>
#include <signal.h>
#include <assert.h>
#include <limits.h>
#include <time.h>
#include <math.h>

#ifdef HAVE_SYSEXITS_H
# include <sysexits.h>
#endif

#ifndef EX_SOFTWARE
# define EX_SOFTWARE  70
#endif

#if defined(__MWERKS__) && !defined(__INTEL__)

# include <stat.h>
# ifndef MACINTOSH_GUI
#  include <console.h>
#  include <SIOUX.h>
# endif

# define NSIG                          32
# define EINVAL                        ERANGE
static int timezone = 0;

#elif !defined(C_NONUNIX)

# include <sys/types.h>
# include <sys/stat.h>
# include <sys/time.h>
# include <sys/resource.h>
# include <sys/wait.h>

#else

# include <sys/types.h>
# include <sys/stat.h>

#endif

#ifdef __DJGPP__
# include <pc.h>
static int timezone = 0;
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"

# ifndef HAVE_ALLOCA
#  error this package requires "alloca()"
# endif
#endif

#ifdef __GNUC__
# ifdef __i386__
#  define FAKE_APPLY
# endif
#endif

#ifdef _MSC_VER
# define S_IFMT             _S_IFMT
# define S_IFDIR            _S_IFDIR
# define timezone           _timezone

# ifdef _M_IX86
#  define FAKE_APPLY
# endif
#endif


/* Parameters: */

#define DEFAULT_STACK_SIZE             300000
#define DEFAULT_SYMBOL_TABLE_SIZE      2999
#define DEFAULT_HEAP_SIZE              8000000
#define DEFAULT_FINALIZER_TABLE_SIZE   64

#define WEAK_TABLE_SIZE                997
#define WEAK_HASH_ITERATIONS           4
#define WEAK_HASH_DISPLACEMENT         7
#define WEAK_COUNTER_MASK              3
#define WEAK_COUNTER_MAX               2

#define STACK_TRACE_FILE               C_text("STACKTRACE")

#define TEMPORARY_STACK_SIZE	       2048
#define TRACE_BUFFER_SIZE              32
#define STRING_BUFFER_SIZE             1024
#define DEFAULT_MUTATION_STACK_SIZE    1024
#define MUTATION_STACK_GROWTH          1024

#define FILE_INFO_SIZE                 7
#define FLONUM_PRINT_PRECISION         16

#ifdef C_SIXTY_FOUR
# define WORDS_PER_FLONUM              2
#else
# define WORDS_PER_FLONUM              4
#endif

#define MAXIMAL_NUMBER_OF_COMMAND_LINE_ARGUMENTS 32

#define INITIAL_TIMER_INTERRUPT_PERIOD 10000


/* Constants: */

#ifdef C_SIXTY_FOUR
# define ALIGNMENT_HOLE_MARKER         ((C_word)0xfffffffffffffffe)
#else
# define ALIGNMENT_HOLE_MARKER         ((C_word)0xfffffffe)
#endif

#define GC_MINOR           0
#define GC_MAJOR           1


/* Macros: */

#ifdef PARANOIA
# define check_reserve(n)             assert(C_stack_pointer - (n) > (C_word *)((C_byte *)C_stack_limit - STACK_RESERVE))
# define check_alignment(p)           assert(((C_word)(p) & 3) == 0)
#else
# define NDEBUG
# define check_reserve(n)
# define check_alignment(p)
#endif

#define aligned8(n)                  ((((C_word)(n)) & 7) == 0)
#define nmax(x, y)                   ((x) > (y) ? (x) : (y))
#define nmin(x, y)                   ((x) < (y) ? (x) : (y))


/* Type definitions: */

typedef void (*TOPLEVEL)(int c, C_word self, C_word k) C_noret;
typedef void (C_fcall *TRAMPOLINE)(void *proc) C_regparm C_noret;

typedef struct lf_list_struct
{
  C_word *lf;
  int count;
  struct lf_list_struct *next;
} LF_LIST;

typedef struct weak_table_entry_struct
{
  C_word item,
         container;
} WEAK_TABLE_ENTRY;


/* Variables: */

C_word *C_temporary_stack,
       *C_temporary_stack_bottom,
       *C_temporary_stack_limit,
       *C_stack_limit;
long C_timer_interrupt_counter,
     C_initial_timer_interrupt_period;
C_byte *C_fromspace_top,
       *C_fromspace_limit;
double C_temporary_flonum;
jmp_buf C_restart;
void *C_restart_address;
void (C_fcall *C_restart_trampoline)(void *proc) C_regparm C_noret;
int C_interrupts_enabled,
    C_enable_gcweak = 0,
    C_main_argc;
char **C_main_argv;

static C_byte *fromspace_start = NULL,
              *tospace_start,
              *tospace_top,
              *tospace_limit,
              *heap_scan_top,
              *timer_start_fromspace_top;
static C_char *trace_buffer[ TRACE_BUFFER_SIZE ],
              **trace_buffer_limit,
              **trace_buffer_top,
              **host_exchange_buffer,
              buffer[ STRING_BUFFER_SIZE ];
static C_word *symbol_table,
              *saved_stack_limit,
              **mutation_stack_bottom,
              **mutation_stack_limit,
              **mutation_stack_top,
              *stack_bottom,
              *finalizer_table,
              interrupt_hook_symbol,
              error_hook_symbol,
              break_on_startup_symbol,
              pending_finalizers_symbol,
              callback_continuation_stack_symbol;
static int trace_buffer_full,
           return_to_host,
           *host_exchange_bytes,
           *host_exchange_maximum,
           stack_size,
           heap_size,
           symbol_table_size,
           fake_tty_flag = 0,
           debug_mode = 0,
           gc_report_flag,
           gc_mode,
           gc_count_1,
           gc_count_2,
           timer_start_gc_count_1,
           timer_start_gc_count_2,
           interrupt_reason,
           stack_size_changed = 0,
           heap_size_changed = 0,
           chicken_is_running,
           chicken_ran_once,
           break_on_startup = 0;
static unsigned int mutation_count,
                    timer_start_mutation_count;
static int chicken_is_initialized = 0;
static jmp_buf gc_restart;
static long timer_start_ms,
            timer_start_gc_ms,
            interrupt_time,
            last_interrupt_latency;
static LF_LIST *lf_list;
static int signal_mapping_table[ NSIG ];
static int callback_returned_flag,
           finalizer_table_size,
           finalizer_table_maximum,
           pending_finalizer_count,
           pending_finalizer_start,
           pending_finalizer_indices[ C_MAX_PENDING_FINALIZERS ];
static WEAK_TABLE_ENTRY *weak_item_table;


#ifdef __DJGPP__
unsigned _stklen = 1048576;
#endif


/* Prototypes: */

#ifdef WINDOWS_GUI
static void parse_argv(C_char *cmds);
#endif

static void initialize_symbol_table(void);
static void global_signal_handler(int signum);
static C_word arg_val(C_char *arg);
static void barf(int code, ...) C_noret;
static void panic(C_char *msg) C_noret;
static void C_fcall initial_trampoline(void *proc) C_regparm C_noret;
static void termination_continuation(int c, C_word self, C_word result) C_noret;
static void C_fcall mark_system_globals(void) C_regparm;
static void C_fcall mark(C_word *x) C_regparm;
static WEAK_TABLE_ENTRY *C_fcall lookup_weak_table_entry(C_word item, C_word container) C_regparm;
static void values_continuation(int c, C_word closure, C_word dummy, ...) C_noret;
static C_word add_symbol(C_word **ptr, C_word key, C_word string);
static int C_fcall hash_string(int len, C_char *str) C_regparm;
static C_word C_fcall lookup(C_word key, int len, C_char *str) C_regparm;
static double compute_symbol_table_load(double *avg_bucket_len, int *total);
static C_char *dump_trace(void);
static C_word C_fcall convert_string_to_number(C_char *str, int radix, C_word *fix, double *flo) C_regparm;
static long C_fcall milliseconds(void);
static long C_fcall cpu_milliseconds(void);
static void call_cc_wrapper(int c, C_word closure, C_word k, C_word result) C_noret;
static void call_cc_values_wrapper(int c, C_word closure, C_word k, ...) C_noret;
static void cons_flonum_trampoline(void *dummy) C_noret;
static void make_port_2(void *dummy) C_noret;
static void gc_2(void *dummy) C_noret;
static void allocate_vector_2(void *dummy) C_noret;
static void cons_string_trampoline(void *dummy) C_noret;
static void get_argv_2(void *dummy) C_noret;
static void make_structure_2(void *dummy) C_noret;
static void ensure_heap_reserve_2(void *dummy) C_noret;
static void call_host_2(void *dummy) C_noret;
static void file_info_2(void *dummy) C_noret;
static void get_environment_variable_2(void *dummy) C_noret;
static void handle_interrupt(void *trampoline, void *proc) C_noret;
static void callback_trampoline(void *dummy) C_noret;
static void callback_return_continuation(int c, C_word self, C_word r) C_noret;
static void register_finalizer_2(void *dummy) C_noret;


/* Startup code: */

#if defined(C_GENERIC_CONSOLE) && !defined(C_EMBEDDED)
int main(int argc, char *argv[]
#if defined(_MSC_VER) || defined(__GNUC__)
, char *env[]
#endif
	 )
{
  int h, s, n;

#if defined(__MWERKS__) && !defined(__INTEL__) && !defined(MACINTOSH_GUI)
  argc = ccommand(&argv);
  SIOUXSettings.asktosaveonclose = 0;
  SIOUXSettings.autocloseonquit = 1;
#endif

  CHICKEN_parse_command_line(argc, argv, &h, &s, &n);
  
  if(!CHICKEN_initialize(h, s, n))
    panic(C_text("out of memory"));

  CHICKEN_run(NULL, NULL, NULL);
  return 0;
}
#endif


#if defined(WINDOWS_GUI) && !defined(C_EMBEDDED)
int WINAPI WinMain(HINSTANCE me, HINSTANCE you, LPSTR cmdline, int show)
{
  int h, s, n;

  parse_argv(GetCommandLine());
  CHICKEN_parse_command_line(C_main_argc, C_main_argv, &h, &s, &n);
  
  if(!CHICKEN_initialize(h, s, n)) panic(C_text("out of memory"));

  CHICKEN_run(NULL, NULL, NULL);
  return 0;
}


void parse_argv(C_char *cmds)
{
  C_char *ptr = cmds,
         *bptr0, *bptr, *aptr;
  int n = 0;

  C_main_argv = (C_char **)malloc(MAXIMAL_NUMBER_OF_COMMAND_LINE_ARGUMENTS * sizeof(C_char *));

  if(C_main_argv == NULL)
    panic(C_text("can not allocate argument-list buffer"));

  C_main_argc = 0;

  for(;;) {
    while(isspace(*ptr)) ++ptr;

    if(*ptr == '\0') break;

    for(bptr0 = bptr = buffer; !isspace(*ptr) && *ptr != '\0'; *(bptr++) = *(ptr++))
      ++n;
    
    *bptr = '\0';
    aptr = (C_char *)malloc(sizeof(C_char) * (n + 1));
    
    if(aptr == NULL)
      panic(C_text("can not allocate argument buffer"));

    C_strcpy(aptr, bptr0);
    C_main_argv[ C_main_argc++ ] = aptr;
  }
}
#endif /* defined(WINDOWS_GUI) && !defined(C_EMBEDDED) */


#ifdef WINDOWS_DLL
BOOL APIENTRY DllMain(HANDLE module, DWORD reason, LPVOID reserved)
{
  return TRUE;
}
#endif


#if defined(__MWERKS__) && !defined(__INTEL__)
int strncasecmp(const C_char *one, const C_char *two, size_t n)
{
  int c;

  while(n--) {
    if((c = tolower(*(one++)) - tolower(*(two++))) < 0) return -1;
    else if(c > 0) return 1;
  }

  return 0;
}
#endif


int C_dllexportdecl CHICKEN_initialize(int heap, int stack, int symbols)
{
  volatile C_word dummy[3];
  int i;

  if(chicken_is_initialized) return 1;
  else chicken_is_initialized = 1;

  if(debug_mode) printf(C_text("[debug] application startup...\n"));

  symbol_table_size = symbols ? symbols : DEFAULT_SYMBOL_TABLE_SIZE;
  stack_size = stack ? stack : DEFAULT_STACK_SIZE;
  C_set_or_change_heap_size(heap ? heap : DEFAULT_HEAP_SIZE, 0);

  /* Allocate temporary stack: */
  if((C_temporary_stack_limit = (C_word *)malloc(TEMPORARY_STACK_SIZE * sizeof(C_word))) == NULL)
    return 0;
  
  C_temporary_stack_bottom = C_temporary_stack_limit + TEMPORARY_STACK_SIZE;
  C_temporary_stack = C_temporary_stack_bottom;
  
  /* Allocate symbol table: */
  if((symbol_table = (C_word *)malloc(symbol_table_size * sizeof(C_word))) == NULL)
    return 0;
  
  /* Allocate mutation stack: */
  mutation_stack_bottom = (C_word **)malloc(DEFAULT_MUTATION_STACK_SIZE * sizeof(C_word *));

  if(mutation_stack_bottom == NULL) return 0;

  mutation_stack_top = mutation_stack_bottom;
  mutation_stack_limit = mutation_stack_bottom + DEFAULT_MUTATION_STACK_SIZE;

  /* Allocate weak item table: */
  if(C_enable_gcweak) {
    if((weak_item_table = (WEAK_TABLE_ENTRY *)calloc(WEAK_TABLE_SIZE, sizeof(WEAK_TABLE_ENTRY))) == NULL)
      return 0;
  }

  /* Initialize finalizer table: */
  finalizer_table = (C_word *)C_malloc(DEFAULT_FINALIZER_TABLE_SIZE * 2 * sizeof(C_word));

  if(finalizer_table == NULL) return 0;

  for(i = 0; i < DEFAULT_FINALIZER_TABLE_SIZE * 2; ++i)
    finalizer_table[ i ] = C_SCHEME_UNDEFINED;

  finalizer_table_size = DEFAULT_FINALIZER_TABLE_SIZE;
  finalizer_table_maximum = -1;

  /* Initialize global variables: */
  gc_report_flag = 0;
  mutation_count = gc_count_1 = gc_count_2 = 0;
  lf_list = NULL;
  C_restart_address = C_toplevel;
  C_restart_trampoline = initial_trampoline;
  trace_buffer_top = trace_buffer;
  trace_buffer_limit = trace_buffer + TRACE_BUFFER_SIZE;
  trace_buffer_full = 0;
  chicken_is_running = chicken_ran_once = 0;
  interrupt_reason = 0;
  last_interrupt_latency = 0;
  C_interrupts_enabled = 1;
  C_initial_timer_interrupt_period = INITIAL_TIMER_INTERRUPT_PERIOD;
  C_timer_interrupt_counter = INITIAL_TIMER_INTERRUPT_PERIOD;
  memset(signal_mapping_table, 0, sizeof(int) * NSIG);
  initialize_symbol_table();
  return 1;
}


void initialize_symbol_table(void)
{
  int i;

  for(i = 0; i < symbol_table_size; symbol_table[ i++ ] = C_SCHEME_END_OF_LIST);

  /* Obtain reference to hooks for later: */
  interrupt_hook_symbol = C_intern(C_heaptop, C_strlen("_sysinterrupt-hook"), C_text("\003sysinterrupt-hook"));
  error_hook_symbol = C_intern(C_heaptop, C_strlen("_syserror-hook"), C_text("\003syserror-hook"));
  break_on_startup_symbol = C_intern(C_heaptop, C_strlen("_sysbreak-on-startup"), C_text("\003sysbreak-on-startup"));
  C_set_block_item(break_on_startup_symbol, 0, C_SCHEME_FALSE);
  callback_continuation_stack_symbol =
    C_intern(C_heaptop, C_strlen("_syscallback-continuation-stack"), C_text("\003syscallback-continuation-stack"));
  pending_finalizers_symbol = C_intern(C_heaptop, C_strlen("_syspending-finalizers"), C_text("\003syspending-finalizers"));

  if(break_on_startup) 
    C_set_block_item(break_on_startup_symbol, 0, C_SCHEME_TRUE);
}


void global_signal_handler(int signum)
{
  C_raise_interrupt(signal_mapping_table[ signum ]);
  signal(signum, global_signal_handler);
}


void C_set_or_change_heap_size(C_word heap, int reintern)
{
  C_byte *ptr1, *ptr2;
  C_word size = heap / 2;

  if(heap_size_changed && fromspace_start) return;

  if(fromspace_start && heap_size >= heap) return;

  if(debug_mode) printf(C_text("[debug] heap resized to %d bytes\n"), heap);

  heap_size = heap;

  if((ptr1 = (C_byte *)realloc(fromspace_start, size + sizeof(C_word))) == NULL ||
     (ptr2 = (C_byte *)realloc(tospace_start, size + sizeof(C_word))) == NULL)
    panic(C_text("out of memory - can not allocate heap"));

  fromspace_start = (C_byte *)C_align((C_uword)ptr1);
  C_fromspace_top = fromspace_start;
  C_fromspace_limit = fromspace_start + size;
  tospace_start = (C_byte *)C_align((C_uword)ptr2);
  tospace_top = tospace_start;
  tospace_limit = tospace_start + size;

  if(reintern) initialize_symbol_table();
}
 

void C_do_resize_stack(C_word stack)
{
  C_word old = stack_size,
         diff = stack - old;

  if(diff != 0 && !stack_size_changed) {
    if(debug_mode) printf(C_text("[debug] stack resized to %d bytes\n"), stack);

    stack_size = stack;
    C_stack_limit = (C_word *)((C_byte *)C_stack_limit - diff);
  }
}


void CHICKEN_parse_command_line(int argc, char *argv[], C_word *heap, C_word *stack, C_word *symbols)
{
  int i;
  char *ptr;
  C_word x;

  C_main_argc = argc;
  C_main_argv = argv;
  *heap = DEFAULT_HEAP_SIZE;
  *stack = DEFAULT_STACK_SIZE;
  *symbols = DEFAULT_SYMBOL_TABLE_SIZE;

  for(i = 1; i < C_main_argc; ++i)
    if(!strncmp(C_main_argv[ i ], C_text("-:"), 2)) {
      for(ptr = &C_main_argv[ i ][ 2 ]; *ptr != '\0';) {
	switch(*(ptr++)) {
	case 'h':
	  *heap = arg_val(ptr); 
	  heap_size_changed = 1;
	  goto next;

	case 's':
	  *stack = arg_val(ptr);
	  stack_size_changed = 1;
	  goto next;

	case 't':
	  *symbols = arg_val(ptr);
	  ptr = NULL;
	  break;

	case 'c':
	  fake_tty_flag = 1;
	  break;

	case 'd':
	  debug_mode = 1;
	  break;

	case 'w':
	  C_enable_gcweak = 1;
	  break;

	case 'b':
	  break_on_startup = 1;
	  break;

	default: panic(C_text("illegal runtime option"));
	}
      }

    next:;
    }
}


C_word arg_val(C_char *arg)
{
  int len = C_strlen(arg);

  if(len < 1) panic(C_text("illegal runtime-option argument"));

  switch(arg[ len - 1 ]) {
  case 'k':
  case 'K':
    return atol(arg) * 1024;

  case 'm':
  case 'M':
    return atol(arg) * 1024 * 1024;

  default:
    return atol(arg);
  }
}


void C_dllexportdecl CHICKEN_run(void **data, int *bytes, int *maxlen)
{
  if(!chicken_is_initialized && !CHICKEN_initialize(0, 0, 0))
    panic(C_text("could not initialize"));

  if(chicken_is_running)
    panic(C_text("re-invocation of Scheme world while process is already running"));

  chicken_is_running = chicken_ran_once = 1;
  return_to_host = 0;
  C_stack_limit = (C_word *)((C_byte *)C_stack_pointer - stack_size);
  host_exchange_buffer = (C_byte **)data;
  host_exchange_bytes = bytes;
  host_exchange_maximum = maxlen;
  stack_bottom = C_stack_pointer;

  if(debug_mode)
    printf(C_text("[debug] stack bottom is 0x%x.\n"), (long)stack_bottom);

  /* The point of (usually) no return... */
  setjmp(C_restart);

  if(!return_to_host)
    (C_restart_trampoline)(C_restart_address);

  chicken_is_running = 0;
}


void C_dllexportdecl CHICKEN_invoke(int id, C_parameter *data, int items)
{
  int count = C_wordstobytes(C_wordsperdouble(items));
  C_word *ptrs[ 2 ],
         cnts[ 2 ];

  if(!chicken_is_initialized || !chicken_ran_once) CHICKEN_run(NULL, NULL, NULL);

  ptrs[ 0 ] = &id;
  ptrs[ 1 ] = (C_word *)data;
  cnts[ 0 ] = sizeof(C_word);
  cnts[ 1 ] = count;
  CHICKEN_run((void **)ptrs, cnts, cnts);
}


void C_fcall initial_trampoline(void *proc)
{
  TOPLEVEL top = (TOPLEVEL)proc;
  C_word closure = (C_word)C_alloc(2);

  ((C_SCHEME_BLOCK *)closure)->header = C_CLOSURE_TYPE | 1;
  C_set_block_item(closure, 0, (C_word)termination_continuation);
  (top)(2, C_SCHEME_UNDEFINED, closure);
}


void termination_continuation(int c, C_word self, C_word result)
{
  if(debug_mode) printf(C_text("[debug] application terminated normally.\n"));

  exit(0);
}


void panic(C_char *msg)
{
  C_char *last;

#ifdef C_MICROSOFT_WINDOWS
  if((last = dump_trace()) != NULL)
    sprintf(buffer, C_text("%s\nlast traced call: %s"), msg, last);
  else C_strcpy(buffer, msg);

  MessageBox(NULL, buffer, C_text("CHICKEN runtime"), MB_OK);
  ExitProcess(1);
#else
  fprintf(stderr, C_text("\n%s - execution terminated\n"), msg);
  
  if((last = dump_trace()) != NULL) 
    fprintf(stderr, C_text("last traced call: %s\n"), last);

# if defined(__MWERKS__) && !defined(__INTEL__) && !defined(MACINTOSH_GUI)
  printf("\n[press RETURN to exit...]");
  fflush(stdout);
  getchar();
# endif

  exit(1);
#endif
}


void barf(int code, ...)
{
  C_char *msg;
  C_word err = error_hook_symbol;
  int c, i;
  va_list v;

  C_temporary_stack = C_temporary_stack_bottom;
  err = C_block_item(err, 0);

#ifdef PARANOIA
  if(C_immediatep(err))
    panic(C_text("`##sys#error-hook' is not defined"));
#endif

  switch(code) {
  case C_BAD_ARGUMENT_COUNT_ERROR:
    msg = C_text("wrong number of arguments in function call");
    c = 2;
    break;

  case C_BAD_MINIMUM_ARGUMENT_COUNT_ERROR:
    msg = C_text("too few arguments in function call");
    c = 2;
    break;
  
  case C_BAD_ARGUMENT_TYPE_ERROR:
    msg = C_text("bad argument type");
    c = 1;
    break;

  case C_UNBOUND_VARIABLE_ERROR:
    msg = C_text("unbound variable");
    c = 1;
    break;

  case C_TOO_MANY_PARAMETERS_ERROR:
    msg = C_text("parameter limit exceeded");
    c = 0;
    break;

  case C_OUT_OF_MEMORY_ERROR:
    msg = C_text("not enough memory");
    c = 0;
    break;

  case C_DIVISION_BY_ZERO_ERROR:
    msg = C_text("division by zero");
    c = 0;
    break;

  case C_OUT_OF_RANGE_ERROR:
    msg = C_text("out of range");
    c = 2;
    break;

  case C_NOT_A_CLOSURE_ERROR:
    msg = C_text("call of non-procedure");
    c = 1;
    break;

  case C_CONTINUATION_CANT_RECEIVE_VALUES_ERROR:
    msg = C_text("continuation can not receive multiple values");
    c = 1;
    break;

  case C_TOO_DEEP_RECURSION_ERROR:
    msg = C_text("recursion too deep");
    c = 0;
    break;

  case C_CANT_REPRESENT_INEXACT_ERROR:
    msg = C_text("inexact number can not be represented as an exact number");
    c = 1;
    break;

  case C_NOT_A_PROPER_LIST_ERROR:
    msg = C_text("argument is not a proper list");
    c = 1;
    break;

  case C_BAD_ARGUMENT_TYPE_NO_FIXNUM_ERROR:
    msg = C_text("bad argument type - not a fixnum");
    c = 1;
    break;

  case C_BAD_ARGUMENT_TYPE_NO_STRING_ERROR:
    msg = C_text("bad argument type - not a string");
    c = 1;
    break;

  case C_BAD_ARGUMENT_TYPE_NO_PAIR_ERROR:
    msg = C_text("bad argument type - not a pair");
    c = 1;
    break;

  case C_BAD_ARGUMENT_TYPE_NO_LIST_ERROR:
    msg = C_text("bad argument type - not a list");
    c = 1;
    break;

  case C_BAD_ARGUMENT_TYPE_NO_NUMBER_ERROR:
    msg = C_text("bad argument type - not a number");
    c = 1;
    break;

  case C_BAD_ARGUMENT_TYPE_NO_SYMBOL_ERROR:
    msg = C_text("bad argument type - not a symbol");
    c = 1;
    break;

  case C_BAD_ARGUMENT_TYPE_NO_VECTOR_ERROR:
    msg = C_text("bad argument type - not a vector");
    c = 1;
    break;

  case C_BAD_ARGUMENT_TYPE_NO_CHAR_ERROR:
    msg = C_text("bad argument type - not a character");
    c = 1;
    break;

  case C_STACK_OVERFLOW_ERROR:
    msg = C_text("stack overflow");
    c = 0;
    break;

  case C_BAD_ARGUMENT_TYPE_BAD_STRUCT_ERROR:
    msg = C_text("bad argument type - not a structure of the required type");
    c = 2;
    break;

  case C_BAD_ARGUMENT_TYPE_NO_BYTEVECTOR_ERROR:
    msg = C_text("bad argument type - not a bytevector");
    c = 1;
    break;

  default: panic(C_text("illegal internal error code"));
  }
  
  if(!C_immediatep(err)) {
    C_save(C_fix(code));
    va_start(v, code);
    i = c;

    while(i--)
      C_save(va_arg(v, C_word));

    va_end(v);
    C_do_apply(c + 1, err, C_SCHEME_UNDEFINED);  /* <- no continuation is passed: '##sys#error-hook' may not return! */
  }
  else panic(msg);
}


/* Timing routines: */

long C_fcall milliseconds(void)
{
#ifdef C_NONUNIX
    if(CLOCKS_PER_SEC == 1000) return clock();
    else return ((double)clock() / (double)CLOCKS_PER_SEC) * 1000;
#else
    struct timeval tv;

    if(gettimeofday(&tv, NULL) == -1) return 0;
    else return tv.tv_sec * 1000 + tv.tv_usec / 1000;
#endif
}


long C_fcall cpu_milliseconds(void)
{
#if defined(C_NONUNIX) || defined(__CYGWIN__)
    if(CLOCKS_PER_SEC == 1000) return clock();
    else return (long)(((double)clock() / (double)CLOCKS_PER_SEC) * 1000);
#else
    struct rusage ru;

    if(getrusage(RUSAGE_SELF, &ru) == -1) return 0;
    else return ru.ru_utime.tv_sec * 1000 + ru.ru_utime.tv_usec / 1000;
#endif
}


/* Support code for callbacks: */

int C_fcall C_save_callback_continuation(C_word **ptr, C_word k)
{
  C_word p = C_pair(ptr, k, C_block_item(callback_continuation_stack_symbol, 0));
  
  C_mutate(&C_block_item(callback_continuation_stack_symbol, 0), p);
  return 0;
}


C_word C_fcall C_restore_callback_continuation(void) 
{
  C_word p = C_block_item(callback_continuation_stack_symbol, 0),
         k;

  assert(!C_immediatep(p) && C_header_bits(p) == C_PAIR_TYPE);
  k = C_block_item(p, 0);

  C_mutate(&C_block_item(callback_continuation_stack_symbol, 0), C_block_item(p, 1));
  return k;
}


C_word C_fcall C_callback(C_word closure, int argc)
{
 jmp_buf prev;

 C_memcpy(&prev, &C_restart, sizeof(jmp_buf));
 callback_returned_flag = 0;       
 C_save(closure);
 C_save(C_fix(argc));

 if(!setjmp(C_restart)) C_reclaim(callback_trampoline, NULL);

 if(!callback_returned_flag) (C_restart_trampoline)(C_restart_address);
 else {
  C_memcpy(&C_restart, &prev, sizeof(jmp_buf));
  callback_returned_flag = 0;
  return C_restore;
 }
}


C_word C_fcall C_callback_wrapper(void *proc, int argc)
{
  C_word *a = C_alloc(2),
         closure = C_closure(&a, 1, (C_word)proc),
         result;
  
  result = C_callback(closure, argc);
  assert(C_temporary_stack == C_temporary_stack_bottom);
  return result;
}


void callback_trampoline(void *dummy)
{
 C_word *a = C_alloc(2),
        k = C_closure(&a, 1, (C_word)callback_return_continuation);
 int argc = C_unfix(C_restore);
 C_word p = C_restore;

 C_drop(1);
 C_do_apply(argc, p, k);
}


void callback_return_continuation(int c, C_word self, C_word r)
{
  assert(callback_returned_flag == 0);
  callback_returned_flag = 1;
  C_save(r);
  C_reclaim(NULL, NULL);
}


/* Zap symbol names: */

void C_zap_strings(C_word str)
{
  C_word bucket, sym;
  int i;
  
  for(i = 0; i < symbol_table_size; ++i)
    for(bucket = symbol_table[ i ]; bucket != C_SCHEME_END_OF_LIST; bucket = C_block_item(bucket, 1)) {
      sym = C_block_item(bucket, 0);
      C_set_block_item(sym, 1, str);
    }  
}


/* Register literal frame: */

void C_register_lf(C_word *lf, int count)
{
  LF_LIST *node = (LF_LIST *)malloc(sizeof(LF_LIST));
  
  node->lf = lf;
  node->count = count;
  node->next = lf_list;
  lf_list = node;
}


/* Intern symbol into symbol-table: */

C_word C_fcall C_intern(C_word **ptr, int len, C_char *str)
{
  int key = hash_string(len, str);
  C_word s;

  if(C_truep(s = lookup(key, len, str))) return s;

  s = C_string(ptr, len, str);
  return add_symbol(ptr, key, s);
}


C_word C_fcall C_intern2(C_word **ptr, C_char *str)
{
  return C_intern(ptr, C_strlen(str), str);
}


int C_fcall hash_string(int len, C_char *str)
{
  unsigned int key = 0;

  while(len--) key = (key << 4) + *(str++);

  return (int)(key % symbol_table_size);
}


C_word C_fcall lookup(C_word key, int len, C_char *str)
{
  C_word bucket, sym, s;

  for(bucket = symbol_table[ key ]; bucket != C_SCHEME_END_OF_LIST; bucket = C_block_item(bucket, 1)) {
    sym = C_block_item(bucket, 0);
    s = C_block_item(sym, 1);

    if(C_header_size(s) == (C_word)len && !C_memcmp(str, (C_char *)((C_SCHEME_BLOCK *)s)->data, len))
      return sym;
  }

  return C_SCHEME_FALSE;
}


double compute_symbol_table_load(double *avg_bucket_len, int *total_n)
{
  C_word bucket;
  int i, j, alen = 0, bcount = 0, total = 0;

  for(i = 0; i < symbol_table_size; ++i) {
    bucket = symbol_table[ i ];

    for(j = 0; bucket != C_SCHEME_END_OF_LIST; ++j)
      bucket = C_block_item(bucket, 1);

    if(j > 0) {
      alen += j;
      ++bcount;
    }

    total += j;
  }

  if(avg_bucket_len != NULL)
    *avg_bucket_len = (double)alen / (double)bcount;

  *total_n = total;

  /* return load: */
  return (double)total / (double)symbol_table_size;
}


C_word add_symbol(C_word **ptr, C_word key, C_word string)
{
  C_word bucket, sym, b2, *p;

  p = *ptr;
  sym = (C_word)p;
  p += 3;
  ((C_SCHEME_BLOCK *)sym)->header = C_SYMBOL_TYPE | 2;
  C_set_block_item(sym, 0, *((char *)C_data_pointer(string)) == 0 ? sym : C_SCHEME_UNBOUND);
  C_set_block_item(sym, 1, string);
  *ptr = p;
  b2 = symbol_table[ key ];	/* previous bucket */
  bucket = C_pair(ptr, sym, b2); /* create new bucket */
  ((C_SCHEME_BLOCK *)bucket)->header = (((C_SCHEME_BLOCK *)bucket)->header & ~C_HEADER_TYPE_BITS) | C_BUCKET_TYPE;

  if(ptr != C_heaptop) C_mutate(&symbol_table[ key ], bucket);
  else {
    /* If a stack-allocated bucket was here, and we allocate from 
       heap-top (say, in a toplevel literal frame alocation) then we have
       to inform the memory manager that a 2nd gen. block points to a 
       1st gen. block, hence the mutation: */
    C_mutate(&C_block_item(bucket, 1), b2);
    symbol_table[ key ] = bucket;
  }

  return sym;
}


/* Check block allocation: */

C_word C_fcall C_permanentp(C_word x)
{
  return C_mk_bool(!C_immediatep(x) && !C_in_stackp(x) && !C_in_heapp(x));
}


int C_in_stackp(C_word x)
{
  C_word *ptr = (C_word *)(C_uword)x;
  return ptr >= C_stack_pointer_test && ptr < stack_bottom;
}


int C_fcall C_in_heapp(C_word x)
{
  C_byte *ptr = (C_byte *)(C_uword)x;
  return (ptr >= fromspace_start && ptr < C_fromspace_limit) ||
         (ptr >= tospace_start && ptr < tospace_limit);
}


/* Cons the rest-aguments together: */

C_word C_fcall C_restore_rest(C_word *ptr, int num)
{
  C_word x = C_SCHEME_END_OF_LIST;
  C_SCHEME_BLOCK *node;

  while(num--) {
    node = (C_SCHEME_BLOCK *)ptr;
    ptr += 3;
    node->header = C_PAIR_TYPE | 2;
    node->data[ 0 ] = C_restore;
    node->data[ 1 ] = x;
    x = (C_word)node;
  }

  return x;
}


C_word C_fcall C_restore_rest_vector(C_word *ptr, int num)
{
  C_word *p0 = ptr;

  *(ptr++) = C_VECTOR_TYPE | num;
  ptr += num;

  while(num--) *(--ptr) = C_restore;

  return (C_word)p0;
}


/* Print error messages and exit: */

void C_bad_memory(void)
{
  C_strcpy(buffer, C_text("there is not enough stack-space to run this executable"));
  panic(buffer);
}


void C_bad_memory_2(void)
{
  C_strcpy(buffer, C_text("there is not enough heap-space to run this executable - try using the '-:h...' option"));
  panic(buffer);
}


void C_bad_argc(int c, int n)
{
  barf(C_BAD_ARGUMENT_COUNT_ERROR, C_fix(n - 2), C_fix(c - 2));
}


void C_bad_min_argc(int c, int n)
{
  barf(C_BAD_MINIMUM_ARGUMENT_COUNT_ERROR, C_fix(n - 2), C_fix(c - 2));
}


void C_stack_overflow(void)
{
  barf(C_STACK_OVERFLOW_ERROR);
}


void C_unbound_error(C_word sym)
{
  barf(C_UNBOUND_VARIABLE_ERROR, sym);
}


void C_no_closure_error(C_word x)
{
  barf(C_NOT_A_CLOSURE_ERROR, x);
}


/* Allocate and initialize record: */

C_word C_fcall C_string(C_word **ptr, int len, C_char *str)
{
  C_word strblock = (C_word)(*ptr);

  *ptr = (C_word *)((C_word)(*ptr) + sizeof(C_header) + C_align(len));
  ((C_SCHEME_BLOCK *)strblock)->header = C_STRING_TYPE | len;
  C_memcpy(C_data_pointer(strblock), str, len);
  return strblock;
}


C_word C_fcall C_bytevector(C_word **ptr, int len, C_char *str)
{
  C_word strblock = C_string(ptr, len, str);

  C_string_to_bytevector(strblock);
  return strblock;
}


C_word C_fcall C_pbytevector(int len, C_char *str)
{
  C_SCHEME_BLOCK *pbv = C_malloc(len + sizeof(C_header));

  if(pbv == NULL) panic(C_text("out of memory - can not allocate permanent bytevector"));

  pbv->header = C_BYTEVECTOR_TYPE | len;
  C_memcpy(pbv->data, str, len);
  return (C_word)pbv;
}


C_word C_fcall C_string_aligned8(C_word **ptr, int len, C_char *str)
{
  C_word *p = *ptr,
         *p0;

  /* Align on 8-byte boundary: */
  if(aligned8(p)) ++p;

  p0 = p;
  *ptr = p + 1 + C_bytestowords(len);
  *(p++) = C_STRING_TYPE | C_8ALIGN_BIT | len;
  C_memcpy(p, str, len);
  return (C_word)p0;
}


C_word C_fcall C_string2(C_word **ptr, C_char *str)
{
  C_word strblock = (C_word)(*ptr);
  int len;

  if(str == NULL) return C_SCHEME_FALSE;

  len = C_strlen(str);
  *ptr = (C_word *)((C_word)(*ptr) + sizeof(C_header) + C_align(len));
  ((C_SCHEME_BLOCK *)strblock)->header = C_STRING_TYPE | len;
  C_memcpy(((C_SCHEME_BLOCK *)strblock)->data, str, len);
  return strblock;
}


C_word C_fcall C_string2_safe(C_word **ptr, int max, C_char *str)
{
  C_word strblock = (C_word)(*ptr);
  int len;

  if(str == NULL) return C_SCHEME_FALSE;

  len = C_strlen(str);

  if(len >= max) {
    sprintf(buffer, C_text("foreign string result exceeded maximum of %d bytes"), max);
    panic(buffer);
  }

  *ptr = (C_word *)((C_word)(*ptr) + sizeof(C_header) + C_align(len));
  ((C_SCHEME_BLOCK *)strblock)->header = C_STRING_TYPE | len;
  C_memcpy(((C_SCHEME_BLOCK *)strblock)->data, str, len);
  return strblock;
}


C_word C_fcall C_closure(C_word **ptr, int cells, C_word proc, ...)
{
  va_list va;
  C_word *p = *ptr,
         *p0 = p;

  *p = C_CLOSURE_TYPE | cells;
  *(++p) = proc;

  for(va_start(va, proc); --cells; *(++p) = va_arg(va, C_word));

  va_end(va);
  *ptr = p + 1;
  return (C_word)p0;
}


C_word C_fcall C_pair(C_word **ptr, C_word car, C_word cdr)
{
  C_word *p = *ptr,
         *p0 = p;
 
  *(p++) = C_PAIR_TYPE | 2;
  *(p++) = car;
  *(p++) = cdr;
  *ptr = p;
  return (C_word)p0;
}


C_word C_fcall C_flonum(C_word **ptr, double n)
{
  C_word *p = *ptr,
         *p0;

  /* Align double on 8-byte boundary: */
  if(aligned8(p)) ++p;

  p0 = p;
  *(p++) = C_FLONUM_TYPE | sizeof(double);
  *((double *)p) = n;
  *ptr = p + 2;
  return (C_word)p0;
}


C_word C_fcall C_mpointer(C_word **ptr, void *mp)
{
  C_word *p = *ptr,
      *p0 = p;

  *(p++) = C_POINTER_TYPE | 1;
  *((void **)p) = mp;
  *ptr = p + 1;
  return (C_word)p0;
}


C_word C_vector(C_word **ptr, int n, ...)
{
  va_list v;
  C_word *p = *ptr,
         *p0 = p; 

  *(p++) = C_VECTOR_TYPE | n;
  va_start(v, n);

  while(n--)
    *(p++) = va_arg(v, C_word);

  *ptr = p;
  va_end(v);
  return (C_word)p0;
}


C_word C_structure(C_word **ptr, int n, ...)
{
  va_list v;
  C_word *p = *ptr,
         *p0 = p; 

  *(p++) = C_STRUCTURE_TYPE | n;
  va_start(v, n);

  while(n--)
    *(p++) = va_arg(v, C_word);

  *ptr = p;
  va_end(v);
  return (C_word)p0;
}


C_word C_fcall C_mutate(C_word *slot, C_word val)
{
  int mssize;

  if(!C_immediatep(val)) {
    if(mutation_stack_top >= mutation_stack_limit) {
      assert(mutation_stack_top == mutation_stack_limit);
      mssize = mutation_stack_top - mutation_stack_bottom;
      mutation_stack_bottom = (C_word **)realloc(mutation_stack_bottom, (mssize + MUTATION_STACK_GROWTH) * sizeof(C_word *));
      
      if(mutation_stack_bottom == NULL)
	panic(C_text("out of memory - can not re-allocate mutation stack"));

      mutation_stack_limit = mutation_stack_bottom + mssize + MUTATION_STACK_GROWTH;
      mutation_stack_top = mutation_stack_bottom + mssize;
    }

    *(mutation_stack_top++) = slot;
    ++mutation_count;
  }

  return *slot = val;
}


/* Initiate garbage collection: */

void C_fcall C_reclaim(void *trampoline, void *proc)
{
  int i, j, count, n, bytes, weakn, finalizers_checked;
  C_word *p, **msp, bucket, last, item, container;
  C_header h;
  C_byte *tmp, *start;
  LF_LIST *lfn;
  C_SCHEME_BLOCK *bp;
  WEAK_TABLE_ENTRY *wep;
  long tgc = cpu_milliseconds();

  /* assert(C_timer_interrupt_counter >= 0); */

  if(interrupt_reason && C_interrupts_enabled)
    handle_interrupt(trampoline, proc);

  finalizers_checked = 0;
  C_restart_trampoline = (TRAMPOLINE)trampoline;
  C_restart_address = proc;
  heap_scan_top = (C_byte *)C_align((C_word)C_fromspace_top);
  gc_mode = GC_MINOR;

  /* Explicit second-level GC requested? */
  if((start = C_fromspace_top) >= C_fromspace_limit) 
    goto full_gc;

  /* Entry point for second-level GC: */
  if(setjmp(gc_restart)) {
  full_gc:
    heap_scan_top = (C_byte *)C_align((C_word)tospace_top);    
    gc_mode = GC_MAJOR;

    /* Mark literal frames: */
    for(lfn = lf_list; lfn != NULL; lfn = lfn->next)
      for(i = 0; i < lfn->count; mark(&lfn->lf[ i++ ]));

    /* Mark symbol table: */
    for(i = 0; i < symbol_table_size; mark(&symbol_table[ i++ ]));

    mark_system_globals();
  }
  else {
    /* Mark mutated slots: */
    for(msp = mutation_stack_bottom; msp < mutation_stack_top; mark(*(msp++)));
  }

  /* Clear the mutated slot stack: */
  mutation_stack_top = mutation_stack_bottom;

  /* Mark live values: */
  for(p = C_temporary_stack; p < C_temporary_stack_bottom; mark(p++));

 rescan:
  /* Mark nested values in already moved (marked) blocks in breadth-first manner: */
  while(heap_scan_top < (gc_mode == GC_MINOR ? C_fromspace_top : tospace_top)) {
    bp = (C_SCHEME_BLOCK *)heap_scan_top;

    if(*((C_word *)bp) == ALIGNMENT_HOLE_MARKER) 
      bp = (C_SCHEME_BLOCK *)((C_word *)bp + 1);

    n = C_header_size(bp);
    h = bp->header;
    bytes = (h & C_BYTEBLOCK_BIT) ? n : n * sizeof(C_word);
    p = bp->data;

    if(n > 0 && (h & C_BYTEBLOCK_BIT) == 0) {
      if(h & C_SPECIALBLOCK_BIT) {
	--n;
	++p;
      }

      while(n--) mark(p++);
    }

    heap_scan_top = (C_byte *)bp + C_align(bytes) + sizeof(C_word);
  }

  if(gc_mode == GC_MINOR) {
    count = (C_uword)C_fromspace_top - (C_uword)start;
    ++gc_count_1;
  }
  else {
    if(!finalizers_checked) {
      /* Mark finalizer table and remember indices of non-forwarded items: */

      p = finalizer_table; 
      last = C_block_item(pending_finalizers_symbol, 0);

#ifdef PARANOIA
      if(C_immediatep(last))
	panic(C_text("`##sys#pending-finalizers' is not defined"));
#endif

      j = pending_finalizer_start = C_unfix(C_block_item(last, 0));
      count = weakn = 0;

      for(i = 0; i <= finalizer_table_maximum; ++i) {
	item = *p;

	if(item != C_SCHEME_UNDEFINED) {
	  ++weakn;

	  if(j < C_MAX_PENDING_FINALIZERS) {
	    if((C_header_bits(item) & C_GC_FORWARDING_BIT) == 0) {
	      pending_finalizer_indices[ j++ ] = i;
	    }
	    else if(i > count) count = i;
	  }
	  else if(i > count) count = i;

	  mark(p++);
	  mark(p++);
	}
	else p += 2;
      }

      finalizer_table_maximum = count;
      pending_finalizer_count = j;
      finalizers_checked = 1;

      if(weakn > 0 && gc_report_flag) printf(C_text("GC: %d item(s) in finalizer table\n"), weakn);

      if(pending_finalizer_count > pending_finalizer_start && gc_report_flag) {
	printf(C_text("GC: finalizer table:\t size=%d,\tmax=%d\n"), finalizer_table_size, finalizer_table_maximum);
	printf(C_text("GC: pending finalizers:\t %d -> %d\n"), pending_finalizer_start, pending_finalizer_count);
      }

      goto rescan;
    }
    else {
      /* Copy finalized items with remembered indices into `##sys#pending-finalizers' (and clear table entry): */

      last = C_block_item(pending_finalizers_symbol, 0);
      C_set_block_item(last, 0, C_fix(pending_finalizer_count));

      for(i = pending_finalizer_start; i < pending_finalizer_count; ++i) {
	j = pending_finalizer_indices[ i ];

	assert(C_block_item(last, 1 + i * 2) == C_SCHEME_UNDEFINED);
	C_set_block_item(last, 1 + i * 2, finalizer_table[ j * 2 ]);
	C_set_block_item(last, 2 + i * 2, finalizer_table[ j * 2 + 1 ]);
	finalizer_table[ j * 2 ] = C_SCHEME_UNDEFINED;
	finalizer_table[ j * 2 + 1 ] = C_SCHEME_UNDEFINED;
      }
    }

    count = (C_uword)tospace_top - (C_uword)tospace_start;
    C_fromspace_top = tospace_top;
    tmp = fromspace_start;
    fromspace_start = tospace_start;
    tospace_start = tospace_top = tmp;
    tmp = C_fromspace_limit;
    C_fromspace_limit = tospace_limit;
    tospace_limit = tmp;
    ++gc_count_2;

    if(C_enable_gcweak) {
      /* Check entries in weak item table and recover items ref'd only once and which are unbound symbols: */
      weakn = 0;
      wep = weak_item_table;

      for(i = 0; i < WEAK_TABLE_SIZE; ++i, ++wep)
	if(wep->item != 0) { 
	  if((wep->container & WEAK_COUNTER_MAX) == 0) {
	    item = C_block_header(wep->item) & ~C_GC_FORWARDING_BIT;
	    container = wep->container & ~WEAK_COUNTER_MASK;

	    if(C_header_bits(item) == C_SYMBOL_TYPE && C_block_item(item, 0) == C_SCHEME_UNBOUND) {
	      ++weakn;
#ifdef PARANOIA
	      item = C_block_item(item, 1);
	      fprintf(stderr, C_text("[recovered: %.*s]\n"), C_header_size(item), (char *)C_data_pointer(item));
#endif
	      C_set_block_item(container, 0, C_SCHEME_UNDEFINED);
	    }
	  }

	  wep->item = wep->container = 0;
	}

      /* Remove empty buckets in symbol table: */
      for(i = 0; i < symbol_table_size; ++i) {
	last = 0;

	for(bucket = symbol_table[ i ]; bucket != C_SCHEME_END_OF_LIST; bucket = C_block_item(bucket, 1))
	  if(C_block_item(bucket, 0) == C_SCHEME_UNDEFINED) {
	    if(last) C_set_block_item(last, 1, C_block_item(bucket, 1));
	    else symbol_table[ i ] = C_block_item(bucket, 1);
	  }
	  else last = bucket;
      }
    }
  }

  tgc = cpu_milliseconds() - tgc;
  timer_start_gc_ms += tgc;

  /* Display GC report: 
     Note: stubbornly writes to stdout - there is no provision for other output-ports */
  if(gc_report_flag == 1 || (gc_report_flag && gc_mode == GC_MAJOR)) {
    printf(C_text("GC: level  %d\tgcs(minor)  %d\tgcs(major)  %d\n"),
	   gc_mode, gc_count_1, gc_count_2);
    i = (C_uword)C_stack_pointer;
    printf(C_text("GC: stack\t%8x\t%8x\t%8x"), (C_uword)C_stack_limit - stack_size, i, (C_uword)C_stack_limit);

    if(gc_mode == GC_MINOR) printf(C_text("\t%d"), count);

    printf(C_text("\nGC:  from\t%8x\t%8x\t%8x"),
	   (C_uword)fromspace_start, (C_uword)C_fromspace_top, (C_uword)C_fromspace_limit);

    if(gc_mode == GC_MAJOR) printf(C_text("\t%d"), count);

    printf(C_text("\nGC:    to\t%8x\t%8x\t%8x\n"), (C_uword)tospace_start, (C_uword)tospace_top, (C_uword)tospace_limit);

    if(gc_mode == GC_MAJOR && C_enable_gcweak && weakn)
      printf(C_text("GC: %d recoverable weakly held items found\n"), weakn);
  }

  if(gc_mode == GC_MAJOR) gc_count_1 = 0;

  /* Jump from the Empire State Building... */
  longjmp(C_restart, 1);
}


void C_fcall mark_system_globals(void)
{
  mark(&interrupt_hook_symbol);
  mark(&error_hook_symbol);
  mark(&callback_continuation_stack_symbol);
  mark(&break_on_startup_symbol);
  mark(&pending_finalizers_symbol);
}


void C_fcall mark(C_word *x)
{
  C_word val, item;
  int n, bytes;
  C_header h;
  C_SCHEME_BLOCK *p, *p2;
  WEAK_TABLE_ENTRY *wep;

  val = *x;

  if(C_immediatep(val)) return;

  p = (C_SCHEME_BLOCK *)val;
  
  /* not in stack and not in heap? */
  if(p < (C_SCHEME_BLOCK *)C_stack_pointer || p >= (C_SCHEME_BLOCK *)stack_bottom)
    if((p < (C_SCHEME_BLOCK *)fromspace_start || p >= (C_SCHEME_BLOCK *)C_fromspace_limit) &&
       (p < (C_SCHEME_BLOCK *)tospace_start || p < (C_SCHEME_BLOCK *)tospace_limit) )
      return;

  h = p->header;

  if(gc_mode == GC_MINOR) {
    if(h & C_GC_FORWARDING_BIT) {
      *x = val = h & ~C_GC_FORWARDING_BIT;
      return;
    }

    if(val >= (C_word)fromspace_start && val < (C_word)C_fromspace_top) return;

    p2 = (C_SCHEME_BLOCK *)C_align((C_word)C_fromspace_top);

    if((h & C_8ALIGN_BIT) && aligned8(p2)) {
      *((C_word *)p2) = ALIGNMENT_HOLE_MARKER;
      p2 = (C_SCHEME_BLOCK *)((C_word *)p2 + 1);
    }

    n = C_header_size(p);
    bytes = (h & C_BYTEBLOCK_BIT) ? n : n * sizeof(C_word);

    if(((C_byte *)p2 + bytes + sizeof(C_word)) > C_fromspace_limit)
      longjmp(gc_restart, 1);

    C_fromspace_top = (C_byte *)p2 + C_align(bytes) + sizeof(C_word);

  scavenge:
    *x = (C_word)p2;
    p2->header = h;
    p->header = C_GC_FORWARDING_BIT | (C_word)p2;
    C_memcpy(p2->data, p->data, bytes);
  }
  else {
    /* Increase counter if weakly held item: */
    if(C_enable_gcweak && (wep = lookup_weak_table_entry(val, 0)) != NULL) {
      if((wep->container & WEAK_COUNTER_MAX) == 0) ++wep->container;
    }

    if(h & C_GC_FORWARDING_BIT) {
      val = h & ~C_GC_FORWARDING_BIT;

      if(val >= (C_word)tospace_start && val < (C_word)tospace_top) {
	*x = val;
	return;
      }

      /* Link points into fromspace: fetch new pointer + header and copy... */
      p = (C_SCHEME_BLOCK *)val;
      h = p->header;

      if(h & C_GC_FORWARDING_BIT) {
	/* Link points into fromspace and into a link which points into from- or tospace: */
	val = h & ~C_GC_FORWARDING_BIT;
	
	if(val >= (C_word)tospace_start && val < (C_word)tospace_top) {
	  *x = val;
	  return;
	}

	p = (C_SCHEME_BLOCK *)val;
	h = p->header;
      }
    }

    p2 = (C_SCHEME_BLOCK *)C_align((C_word)tospace_top);

    if((h & C_8ALIGN_BIT) && aligned8(p2)) {
      *((C_word *)p2) = ALIGNMENT_HOLE_MARKER;
      p2 = (C_SCHEME_BLOCK *)((C_word *)p2 + 1);
    }

    if(C_enable_gcweak && (h & C_HEADER_TYPE_BITS) == C_BUCKET_TYPE) {
      item = C_block_item(val, 0);

      /* Lookup item in weak item table or add entry: */
      if((wep = lookup_weak_table_entry(item, (C_word)p2)) != NULL) {
	/* If item is already forwarded, then set count to 2: */
	if(C_header_bits(item) & C_GC_FORWARDING_BIT) wep->container |= 2;
      }
    }

    n = C_header_size(p);
    bytes = (h & C_BYTEBLOCK_BIT) ? n : n * sizeof(C_word);

    if(((C_byte *)p2 + bytes + sizeof(C_word)) > tospace_limit) {
      panic(C_text("out of memory - heap full"));
    }

    tospace_top = (C_byte *)p2 + C_align(bytes) + sizeof(C_word);
    goto scavenge;
  }
}


WEAK_TABLE_ENTRY *C_fcall lookup_weak_table_entry(C_word item, C_word container)
{
  int key = item,
      disp = 0,
      n;
  WEAK_TABLE_ENTRY *wep;

  for(n = 0; n < WEAK_HASH_ITERATIONS; ++n) {
    key = (key + disp) % WEAK_TABLE_SIZE;
    wep = &weak_item_table[ key ];

    if(wep->item == 0) {
      if(container != 0) {
	/* Add fresh entry: */
	wep->item = item;
	wep->container = container;
	return wep;
      }

      return NULL;
    }
    else if(wep->item == item) return wep;
    else disp += WEAK_HASH_DISPLACEMENT;
  }

  return NULL;
}


void handle_interrupt(void *trampoline, void *proc)
{
  C_word *p, x, n;
  int i;
  long c;

  /* Build vector with context information: */
  n = C_temporary_stack_bottom - C_temporary_stack;
  p = C_alloc(19 + n); /* 19 <=> 2 headers + trampoline + proc + 1 extra slot + 9 for interning + 5 for string */
  x = (C_word)p;
  *(p++) = C_VECTOR_TYPE | C_BYTEBLOCK_BIT | (2 * sizeof(C_word));
  *(p++) = (C_word)trampoline;
  *(p++) = (C_word)proc;
  C_save(x);
  x = (C_word)p;
  *(p++) = C_VECTOR_TYPE | (n + 1);
  *(p++) = C_restore;
  C_memcpy(p, C_temporary_stack, n * sizeof(C_word));

  /* Restore state to the one at the time of the interrupt: */
  C_temporary_stack = C_temporary_stack_bottom;
  i = interrupt_reason;
  interrupt_reason = 0;
  C_stack_limit = saved_stack_limit;

  /* Invoke high-level interrupt handler: */
  C_save(C_fix(i));
  C_save(x);
  x = C_block_item(interrupt_hook_symbol, 0);

  if(C_immediatep(x))
    panic(C_text("`##sys#interrupt-hook' is not defined"));

  c = cpu_milliseconds() - interrupt_time;
  last_interrupt_latency = c;
  C_timer_interrupt_counter = C_initial_timer_interrupt_period;	/* just in case */
  C_do_apply(2, x, C_SCHEME_UNDEFINED);  /* <- no continuation is passed: "##sys#interrupt-hook" may not return! */
}


#ifndef C_INLINE_RETRIEVE
C_word C_fcall C_retrieve(C_word sym)
{
  C_word val = C_block_item(sym, 0);

  if(val == C_SCHEME_UNBOUND)
    barf(C_UNBOUND_VARIABLE_ERROR, sym);

  return val;
}


void *C_fcall C_retrieve_proc(C_word closure)
{
  if(C_immediatep(closure) || C_header_bits(closure) != C_CLOSURE_TYPE)
    barf(C_NOT_A_CLOSURE_ERROR, closure);

  return (void *)C_block_item(closure, 0);
}
#endif


void C_fcall C_trace(C_char *name)
{
  if(trace_buffer_top >= trace_buffer_limit) {
    trace_buffer_top = trace_buffer;
    trace_buffer_full = 1;
  }

  *(trace_buffer_top++) = name;
}


C_char *dump_trace(void)
{
  FILE *fp;
  C_char **ptr, *r = NULL;
  int i;

  if(trace_buffer_top > trace_buffer || trace_buffer_full) {
#ifndef C_MICROSOFT_WINDOWS
    fprintf(stderr, C_text("writing stack-trace...\n"));
#endif

    if((fp = fopen(STACK_TRACE_FILE, C_text("w"))) == NULL) {
      fputs(C_text("- can not open stack-trace file"), stderr);
      return NULL;
    }

    if(trace_buffer_full) {
      i = TRACE_BUFFER_SIZE;
      fputs(C_text("...more...\n"), fp);
    }
    else i = trace_buffer_top - trace_buffer;
    
    for(ptr = trace_buffer_full ? trace_buffer_top : trace_buffer; i--; ++ptr) {
      if(ptr >= trace_buffer_limit) ptr = trace_buffer;

      fprintf(fp, C_text("%s\n"), r = *ptr);
    }

    fclose(fp);
  }

  return r;
}


C_word C_fcall C_hash_string(C_word str)
{
  unsigned C_word key = 0;
  int len = C_header_size(str);
  C_byte *ptr = C_data_pointer(str);

  while(len--) key = (key << 4) + *(ptr++);

  return C_fix(key & 0x0fffffff);
}


void C_fcall C_toplevel_entry(C_char *name)
{
  if(debug_mode) {
    printf(C_text("[debug] entering toplevel %s...\n"), name);
    fflush(stdout);
  }
}


/* Helper routines for use by '##core#inline': */

C_word C_halt(C_word msg)
{
  C_char *last;

#ifdef C_MICROSOFT_WINDOWS
  if(msg != C_SCHEME_FALSE) {
    int n = C_header_size(msg);

    C_strncpy(buffer, (C_char *)C_data_pointer(msg), n);
    buffer[ n ] = '\0';
  }
  else C_strcpy(buffer, C_text("(aborted)"));

  if((last = dump_trace()) != NULL) {
    C_strcat(buffer, C_text("\nlast traced call: "));
    C_strcat(buffer, last);
  }

  MessageBox(NULL, buffer, C_text("CHICKEN runtime"), MB_OK);
#else
  if(msg != C_SCHEME_FALSE) {
    fwrite(C_data_pointer(msg), C_header_size(msg), sizeof(C_char), stdout);
    putchar('\n');
  }

  if((last = dump_trace()) != NULL)
    fprintf(stderr, C_text("last traced call: %s\n"), last);
#endif
  
  exit(EX_SOFTWARE);
  return 0;
}


C_word C_message(C_word msg)
{
#ifdef C_MICROSOFT_WINDOWS
  int n = C_header_size(msg);

  strncpy(buffer, (C_char *)((C_SCHEME_BLOCK *)msg)->data, n);
  buffer[ n ] = '\0';
  MessageBox(NULL, buffer, C_text("CHICKEN runtime"), MB_OK);
#else
  fwrite(((C_SCHEME_BLOCK *)msg)->data, C_header_size(msg), sizeof(C_char), stdout);
  putchar('\n');
#endif
  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_equalp(C_word x, C_word y)
{
  C_header header;
  int bits, n, i;

  C_stack_check;

 loop:
  if(x == y) return 1;

  if(C_immediatep(x) || C_immediatep(y)) return 0;

  if((header = C_block_header(x)) != C_block_header(y)) return 0;
  else if((bits = header & C_HEADER_BITS_MASK) & C_BYTEBLOCK_BIT) 
    return !C_memcmp(C_data_pointer(x), C_data_pointer(y), header & C_HEADER_SIZE_MASK);
  else if((bits & C_HEADER_TYPE_BITS) == C_SYMBOL_TYPE) return 0;
  else {
    i = 0;
    n = header & C_HEADER_SIZE_MASK;

    if(bits & C_SPECIALBLOCK_BIT) {
      if(C_block_item(x, 0) != C_block_item(y, 0)) return 0;
      else ++i;

      if(n == 1) return 1;
    }

    if(--n < 0) return 1;

    while(i < n)
      if(!C_equalp(C_block_item(x, i), C_block_item(y, i))) return 0;
      else ++i;

    x = C_block_item(x, i);
    y = C_block_item(y, i);
    goto loop;
  }
    
  return 1;
}


C_word C_fcall C_set_gc_report(C_word flag)
{
  if(flag == C_SCHEME_FALSE) gc_report_flag = 0;
  else if(flag == C_SCHEME_TRUE) gc_report_flag = 2;
  else gc_report_flag = 1;

  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_start_timer(void)
{
  timer_start_mutation_count = mutation_count;
  timer_start_gc_count_1 = gc_count_1;
  timer_start_gc_count_2 = gc_count_2;
  timer_start_fromspace_top = C_fromspace_top;
  timer_start_ms = cpu_milliseconds();
  timer_start_gc_ms = 0;
  return C_SCHEME_UNDEFINED;
}


void C_stop_timer(int c, C_word closure, C_word k)
{
  long t0 = cpu_milliseconds() - timer_start_ms;
  int gc2 = gc_count_2 - timer_start_gc_count_2,
      gc1 = gc2 ? gc_count_1 : (gc_count_1 - timer_start_gc_count_1),
      mutations = mutation_count - timer_start_mutation_count,
      from = gc2 ? ((C_uword)C_fromspace_top - (C_uword)fromspace_start)
                 : ((C_uword)C_fromspace_top - (C_uword)timer_start_fromspace_top);
  C_word ab[ WORDS_PER_FLONUM * 2 + 7 ], /* 2 flonums, 1 vector of 6 elements */
         *a = ab,
         elapsed = C_flonum(&a, (double)t0 / 1000.0),
         gc_time = C_flonum(&a, (double)timer_start_gc_ms / 1000.0),
         info;

  info = C_vector(&a, 6, elapsed, gc_time, C_fix(mutations), C_fix(gc1), C_fix(gc2), C_fix(from));
  C_kontinue(k, info);
}


C_word C_exit_runtime(int code)
{
  exit(C_unfix(code));
  return 0;			/* to please the compiler... */
}


C_word C_fcall C_display_flonum(C_word port, C_word n)
{
  FILE *fp = C_port_file(port);

#ifdef HAVE_GCVT
  fprintf(fp, C_text("%s"), gcvt(C_flonum_magnitude(n), FLONUM_PRINT_PRECISION, buffer));
#else
  fprintf(fp, C_text("%.*g"), FLONUM_PRINT_PRECISION, C_flonum_magnitude(n));
#endif
  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_read_char(C_word port)
{
  int c = fgetc(C_port_file(port));

  return c == EOF ? C_SCHEME_END_OF_FILE : C_make_character(c);
}


C_word C_fcall C_peek_char(C_word port)
{
  FILE *fp = C_port_file(port);
  int c = fgetc(fp);

  ungetc(c, fp);
  return c == EOF ? C_SCHEME_END_OF_FILE : C_make_character(c);
}


C_word C_fcall C_execute_shell_command(C_word string)
{
  int n = C_header_size(string);

  C_memcpy(buffer, ((C_SCHEME_BLOCK *)string)->data, n);
  buffer[ n ] = '\0';

  if((n = system(buffer)) == -1)
    return C_fix(errno);

#ifdef C_NONUNIX
  return C_fix(n);
#else
  return C_fix(WIFEXITED(n) ? WEXITSTATUS(n) : (WIFSIGNALED(n) ? WTERMSIG(n) : WSTOPSIG(n)));
#endif
}


C_word C_fcall C_update_pointer(C_word p, C_word ptr)
{
  C_set_block_item(ptr, 0, C_num_to_int(p));
  return ptr;
}


C_word C_fcall C_string_to_pbytevector(C_word s)
{
  return C_pbytevector(C_header_size(s), C_data_pointer(s));
}


C_word C_fcall C_char_ready_p(C_word port)
{
#if defined(__DJGPP__)

  if(isatty(fileno(C_port_file(port))))
    return C_mk_bool(kbhit());

#elif !defined(C_NONUNIX)

  fd_set fs;
  struct timeval to;
  int fd = fileno(C_port_file(port));

  FD_ZERO(&fs);
  FD_SET(fd, &fs);
  to.tv_sec = to.tv_usec = 0;
  return C_mk_bool(select(fd + 1, &fs, NULL, NULL, &to) == 1);

#endif

  return C_SCHEME_TRUE;
}


C_word C_fcall C_flush_output(C_word port)
{
  fflush(C_port_file(port));
  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_fudge(C_word fudge_factor)
{
  switch(fudge_factor) {
  case C_fix(1): return C_SCHEME_END_OF_FILE;
  case C_fix(2): return C_fix(time(NULL));
  case C_fix(3): return C_make_character(C_PLATFORM_CODE);

  case C_fix(4):
#ifdef C_GENERIC_CONSOLE
    return C_SCHEME_TRUE;
#else
    return C_SCHEME_FALSE;
#endif

  case C_fix(5):
#ifdef C_GENERIC_CONSOLE
    return C_fix(0);
#endif
#ifdef WINDOWS_GUI
    return C_fix(1);
#endif
#ifdef WINDOWS_DLL
    return C_fix(2);
#endif

  case C_fix(6): return C_fix(C_MOST_POSITIVE_FIXNUM & cpu_milliseconds());

  case C_fix(7):
    return C_fix(sizeof(C_word));

  case C_fix(8):
    return C_fix(C_wordsperdouble(1));

  case C_fix(9):
    return C_fix(last_interrupt_latency);

  case C_fix(10):
    return C_fix(CLOCKS_PER_SEC);

  case C_fix(11):
#ifdef MACINTOSH_GUI
    return C_SCHEME_TRUE;
#else
    return C_SCHEME_FALSE;
#endif

  case C_fix(12):
    return C_mk_bool(fake_tty_flag);

  case C_fix(13):
    return C_mk_bool(debug_mode);

  case C_fix(14):
    return C_mk_bool(C_interrupts_enabled);

  case C_fix(15):
    return C_mk_bool(C_enable_gcweak);

  case C_fix(16):
    return C_fix(C_MOST_POSITIVE_FIXNUM & milliseconds());

  default: return C_SCHEME_UNDEFINED;
  }
}


void C_fcall C_paranoid_check_for_interrupt(void)
{
  if(--C_timer_interrupt_counter <= 0)
    C_raise_interrupt(C_TIMER_INTERRUPT_NUMBER);
}


void C_fcall C_raise_interrupt(int reason)
{
  if(C_interrupts_enabled) {
    saved_stack_limit = C_stack_limit;
    C_stack_limit = C_stack_pointer + 1000;
    interrupt_reason = reason;
    interrupt_time = cpu_milliseconds();
  }
}


C_word C_fcall C_set_initial_timer_interrupt_period(C_word n)
{
  C_initial_timer_interrupt_period = C_unfix(n);
  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_enable_interrupts(void)
{
  C_timer_interrupt_counter = C_initial_timer_interrupt_period;
  assert(C_timer_interrupt_counter > 0);
  C_interrupts_enabled = 1;
  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_disable_interrupts(void)
{
  C_interrupts_enabled = 0;
  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_establish_signal_handler(C_word signum, C_word reason)
{
  int sig = C_unfix(signum);

  if(reason == C_SCHEME_FALSE) signal(sig, SIG_IGN);
  else {
    signal_mapping_table[ sig ] = C_unfix(reason);
    signal(sig, global_signal_handler);
  }

  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_flonum_in_fixnum_range_p(C_word n)
{
  double f = C_flonum_magnitude(n);

  return C_mk_bool(f <= (double)C_MOST_POSITIVE_FIXNUM && f >= (double)C_MOST_NEGATIVE_FIXNUM);
}


C_word C_fcall C_fits_in_int_p(C_word x)
{
  double n, m;

  if(x & C_FIXNUM_BIT) return C_SCHEME_TRUE;

  n = C_flonum_magnitude(x);
  return C_mk_bool(modf(n, &m) == 0.0 && n >= C_WORD_MIN && n <= C_WORD_MAX);
}


C_word C_fcall C_fits_in_unsigned_int_p(C_word x)
{
  double n, m;

  if(x & C_FIXNUM_BIT) return C_SCHEME_TRUE;

  n = C_flonum_magnitude(x);
  return C_mk_bool(modf(n, &m) == 0.0 && n >= 0 && n <= C_UWORD_MAX);
}


/* Copy blocks into collected or static memory: */

C_word C_fcall C_copy_block(C_word from, C_word to)
{
  int n = C_header_size(from);
  long bytes;

  if(C_header_bits(from) & C_BYTEBLOCK_BIT) bytes = n;
  else bytes = C_wordstobytes(n);

  C_memcpy((C_SCHEME_BLOCK *)to, (C_SCHEME_BLOCK *)from, bytes + sizeof(C_header));
  return to;
}


C_word C_fcall C_evict_block(C_word from, C_word ptr)
{
  int n = C_header_size(from);
  long bytes;
  C_word *p = (C_word *)C_pointer_address(ptr);

  if(C_header_bits(from) & C_BYTEBLOCK_BIT) bytes = n;
  else bytes = C_wordstobytes(n);

  C_memcpy(p, (C_SCHEME_BLOCK *)from, bytes + sizeof(C_header));
  return (C_word)p;
}


/* Conversion routines: */

double C_fcall C_c_double(C_word x)
{
  if(x & C_FIXNUM_BIT) return (double)C_unfix(x);
  else return C_flonum_magnitude(x);
}


int C_fcall C_num_to_int(C_word x)
{
  if(x & C_FIXNUM_BIT) return C_unfix(x);
  else return (int)C_flonum_magnitude(x);
}


unsigned int C_fcall C_num_to_unsigned_int(C_word x)
{
  if(x & C_FIXNUM_BIT) return C_unfix(x);
  else return (unsigned int)C_flonum_magnitude(x);
}


C_word C_fcall C_int_to_num(C_word **ptr, int n)
{
  if(C_fitsinfixnump(n)) return C_fix(n);
  else return C_flonum(ptr, (double)n);
}


C_word C_fcall C_long_to_num(C_word **ptr, long n)
{
  if(C_fitsinfixnump(n)) return C_fix(n);
  else return C_flonum(ptr, (double)n);
}


C_word C_fcall C_flonum_in_int_range_p(C_word n)
{
  double m = C_flonum_magnitude(n);

  return C_mk_bool(m >= C_WORD_MIN && m <= C_WORD_MAX);
}


C_word C_fcall C_flonum_in_uint_range_p(C_word n)
{
  double m = C_flonum_magnitude(n);

  return C_mk_bool(m >= 0 && m <= C_UWORD_MAX);
}


char *C_fcall C_string_or_null(C_word x)
{
  return C_truep(x) ? C_c_string(x) : NULL;
}


void *C_fcall C_data_pointer_or_null(C_word x) 
{
  return C_truep(x) ? C_data_pointer(x) : NULL;
}


void *C_fcall C_c_pointer_or_null(C_word x) 
{
  return C_truep(x) ? (void *)C_block_item(x, 0) : NULL;
}


long C_fcall C_num_to_long(C_word x)
{
  if(x & C_FIXNUM_BIT) return C_unfix(x);
  else return (long)C_flonum_magnitude(x);
}


/* Inline versions of some standard procedures: */

C_word C_fcall C_i_listp(C_word x)
{
  C_word fast = x, slow = x;

  while(fast != C_SCHEME_END_OF_LIST)
    if(!C_immediatep(fast) && C_header_bits(fast) == C_PAIR_TYPE) {
      fast = C_block_item(fast, 1);
      
      if(fast == C_SCHEME_END_OF_LIST) return C_SCHEME_TRUE;
      else if(!C_immediatep(fast) && C_header_bits(fast) == C_PAIR_TYPE) {
	fast = C_block_item(fast, 1);
	slow = C_block_item(slow, 1);

	if(fast == slow) return C_SCHEME_FALSE;
      }
      else return C_SCHEME_FALSE;
    }
    else return C_SCHEME_FALSE;

  return C_SCHEME_TRUE;
}


C_word C_fcall C_i_string_equal_p(C_word x, C_word y)
{
  C_word n;

  if(C_immediatep(x) || C_header_bits(x) != C_STRING_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  if(C_immediatep(y) || C_header_bits(y) != C_STRING_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, y);

  n = C_header_size(x);

  return C_mk_bool(n == C_header_size(y) && !C_memcmp((char *)C_data_pointer(x), (char *)C_data_pointer(y), n));
}


C_word C_fcall C_u_i_string_equal_p(C_word x, C_word y)
{
  C_word n;

  n = C_header_size(x);
  return C_mk_bool(n == C_header_size(y) && !C_memcmp((char *)C_data_pointer(x), (char *)C_data_pointer(y), n));
}


C_word C_fcall C_i_string_ci_equal_p(C_word x, C_word y)
{
  C_word n;
  char *p1, *p2;

  if(C_immediatep(x) || C_header_bits(x) != C_STRING_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  if(C_immediatep(y) || C_header_bits(y) != C_STRING_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, y);

  n = C_header_size(x);

  if(n != C_header_size(y)) return C_SCHEME_FALSE;

  p1 = (char *)C_data_pointer(x);
  p2 = (char *)C_data_pointer(y);

  while(n--) 
    if(tolower(*(p1++)) != tolower(*(p2++))) return C_SCHEME_FALSE;

  return C_SCHEME_TRUE;
}


C_word C_fcall C_i_eqvp(C_word x, C_word y)
{
  return
    C_mk_bool(x == y ||
	      (!C_immediatep(x) && !C_immediatep(y) &&
	       C_header_bits(x) == C_FLONUM_TYPE && C_header_bits(y) == C_FLONUM_TYPE &&
	       C_flonum_magnitude(x) == C_flonum_magnitude(y) ) );
}


C_word C_fcall C_i_symbolp(C_word x)
{
  return C_mk_bool(!C_immediatep(x) && C_header_bits(x) == C_SYMBOL_TYPE);
}


C_word C_fcall C_i_pairp(C_word x)
{
  return C_mk_bool(!C_immediatep(x) && C_header_bits(x) == C_PAIR_TYPE);
}


C_word C_fcall C_i_stringp(C_word x)
{
  return C_mk_bool(!C_immediatep(x) && C_header_bits(x) == C_STRING_TYPE);
}


C_word C_fcall C_i_vectorp(C_word x)
{
  return C_mk_bool(!C_immediatep(x) && C_header_bits(x) == C_VECTOR_TYPE);
}


C_word C_fcall C_i_portp(C_word x)
{
  return C_mk_bool(!C_immediatep(x) && C_header_bits(x) == C_PORT_TYPE);
}


C_word C_fcall C_i_closurep(C_word x)
{
  return C_mk_bool(!C_immediatep(x) && C_header_bits(x) == C_CLOSURE_TYPE);
}


C_word C_fcall C_i_numberp(C_word x)
{
  return C_mk_bool((x & C_FIXNUM_BIT) || (!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE));
}


C_word C_fcall C_i_integerp(C_word x)
{
  double dummy;

  return C_mk_bool((x & C_FIXNUM_BIT) || 
		   ((!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) &&
		    modf(C_flonum_magnitude(x), &dummy) == 0.0 ) );
}


C_word C_fcall C_i_fixnum_min(C_word x, C_word y)
{
  return ((C_word)x < (C_word)y) ? x : y;
}


C_word C_fcall C_i_fixnum_max(C_word x, C_word y)
{
  return ((C_word)x > (C_word)y) ? x : y;
}


C_word C_a_i_list(C_word **a, int c, ...)
{
  va_list v;
  C_word x, last, current,
         first = C_SCHEME_END_OF_LIST;

  va_start(v, c);

  for(last = C_SCHEME_UNDEFINED; c--; last = current) {
    x = va_arg(v, C_word);
    current = C_pair(a, x, C_SCHEME_END_OF_LIST);

    if(last != C_SCHEME_UNDEFINED)
      C_set_block_item(last, 1, current);
    else first = current;
  }

  va_end(v);
  return first;
}


C_word C_a_i_string(C_word **a, int c, ...)
{
  va_list v;
  C_word x, s = (C_word)(*a);
  char *p;

  *a = (C_word *)((C_word)(*a) + sizeof(C_header) + C_align(c));
  ((C_SCHEME_BLOCK *)s)->header = C_STRING_TYPE | c;
  p = (char *)C_data_pointer(s);
  va_start(v, c);

  while(c--) {
    x = va_arg(v, C_word);

    if((x & C_IMMEDIATE_TYPE_BITS) == C_CHARACTER_BITS)
      *(p++) = C_character_code(x);
  }

  return s;
}


C_word C_a_i_record(C_word **ptr, int n, ...)
{
  va_list v;
  C_word *p = *ptr,
         *p0 = p; 

  *(p++) = C_STRUCTURE_TYPE | n;
  va_start(v, n);

  while(n--)
    *(p++) = va_arg(v, C_word);

  *ptr = p;
  va_end(v);
  return (C_word)p0;
}


C_word C_fcall C_i_exactp(C_word x)
{
  if(x & C_FIXNUM_BIT) return C_SCHEME_TRUE;

  if(C_immediatep(x) || C_header_bits(x) != C_FLONUM_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  return C_SCHEME_FALSE;
}


C_word C_fcall C_u_i_exactp(C_word x)
{
  if(x & C_FIXNUM_BIT) return C_SCHEME_TRUE;

  return C_SCHEME_FALSE;
}


C_word C_fcall C_i_inexactp(C_word x)
{
  if(x & C_FIXNUM_BIT) return C_SCHEME_FALSE;

  if(C_immediatep(x) || C_header_bits(x) != C_FLONUM_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  return C_SCHEME_TRUE;
}


C_word C_fcall C_u_i_inexactp(C_word x)
{
  if(x & C_FIXNUM_BIT) return C_SCHEME_FALSE;

  return C_SCHEME_TRUE;
}


C_word C_fcall C_i_zerop(C_word x)
{
  if(x & C_FIXNUM_BIT) return C_mk_bool(x == C_fix(0));

  if(C_immediatep(x) || C_header_bits(x) != C_FLONUM_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  return C_mk_bool(C_flonum_magnitude(x) == 0.0);
}


C_word C_fcall C_u_i_zerop(C_word x)
{
  if(x & C_FIXNUM_BIT) return C_mk_bool(x == C_fix(0));

  return C_mk_bool(C_flonum_magnitude(x) == 0.0);
}


C_word C_fcall C_i_positivep(C_word x)
{
  if(x & C_FIXNUM_BIT) return C_mk_bool(C_unfix(x) > 0);

  if(C_immediatep(x) || C_header_bits(x) != C_FLONUM_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  return C_mk_bool(C_flonum_magnitude(x) > 0.0);
}


C_word C_fcall C_u_i_positivep(C_word x)
{
  if(x & C_FIXNUM_BIT) return C_mk_bool(C_unfix(x) > 0);

  return C_mk_bool(C_flonum_magnitude(x) > 0.0);
}


C_word C_fcall C_i_negativep(C_word x)
{
  if(x & C_FIXNUM_BIT) return C_mk_bool(C_unfix(x) < 0);

  if(C_immediatep(x) || C_header_bits(x) != C_FLONUM_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  return C_mk_bool(C_flonum_magnitude(x) < 0.0);
}


C_word C_fcall C_u_i_negativep(C_word x)
{
  if(x & C_FIXNUM_BIT) return C_mk_bool(C_unfix(x) < 0);

  return C_mk_bool(C_flonum_magnitude(x) < 0.0);
}


C_word C_fcall C_i_evenp(C_word x)
{
  if(x & C_FIXNUM_BIT) return C_mk_nbool(x & 0x02);

  if(C_immediatep(x) || C_header_bits(x) != C_FLONUM_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  return C_mk_bool(fmod(C_flonum_magnitude(x), 2.0) == 0.0);
}


C_word C_fcall C_u_i_evenp(C_word x)
{
  if(x & C_FIXNUM_BIT) return C_mk_nbool(x & 0x02);

  return C_mk_bool(fmod(C_flonum_magnitude(x), 2.0) == 0.0);
}


C_word C_fcall C_i_oddp(C_word x)
{
  if(x & C_FIXNUM_BIT) return C_mk_bool(x & 0x02);

  if(C_immediatep(x) || C_header_bits(x) != C_FLONUM_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  return C_mk_bool(fmod(C_flonum_magnitude(x), 2.0) != 0.0);
}


C_word C_fcall C_u_i_oddp(C_word x)
{
  if(x & C_FIXNUM_BIT) return C_mk_bool(x & 0x02);

  return C_mk_bool(fmod(C_flonum_magnitude(x), 2.0) != 0.0);
}


C_word C_fcall C_i_car(C_word x)
{
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  return C_block_item(x, 0);
}


C_word C_fcall C_i_cdr(C_word x)
{
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  return C_block_item(x, 1);
}


C_word C_fcall C_i_cadr(C_word x)
{
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE) {
  bad:
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);
  }

  x = C_block_item(x, 1);
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE) goto bad;

  return C_block_item(x, 0);
}


C_word C_fcall C_i_cddr(C_word x)
{
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE) {
  bad:
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);
  }

  x = C_block_item(x, 1);
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE) goto bad;

  return C_block_item(x, 1);
}


C_word C_fcall C_i_caddr(C_word x)
{
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE) {
  bad:
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);
  }

  x = C_block_item(x, 1);
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE) goto bad;
  x = C_block_item(x, 1);
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE) goto bad;

  return C_block_item(x, 0);
}


C_word C_fcall C_i_cdddr(C_word x)
{
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE) {
  bad:
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);
  }

  x = C_block_item(x, 1);
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE) goto bad;
  x = C_block_item(x, 1);
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE) goto bad;

  return C_block_item(x, 1);
}


C_word C_fcall C_i_cadddr(C_word x)
{
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE) {
  bad:
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);
  }

  x = C_block_item(x, 1);
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE) goto bad;
  x = C_block_item(x, 1);
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE) goto bad;
  x = C_block_item(x, 1);
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE) goto bad;

  return C_block_item(x, 0);
}


C_word C_fcall C_i_cddddr(C_word x)
{
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE) {
  bad:
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);
  }

  x = C_block_item(x, 1);
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE) goto bad;
  x = C_block_item(x, 1);
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE) goto bad;
  x = C_block_item(x, 1);
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE) goto bad;

  return C_block_item(x, 1);
}


C_word C_fcall C_i_list_tail(C_word lst, C_word i)
{
  C_word lst0 = lst;
  int n;

  if(i & C_FIXNUM_BIT) n = C_unfix(i);
  else barf(C_BAD_ARGUMENT_TYPE_ERROR, i);

  while(n--) {
    if(C_immediatep(lst) || C_header_bits(lst) != C_PAIR_TYPE)
      barf(C_OUT_OF_RANGE_ERROR, lst0, i);
    
    lst = C_block_item(lst, 1);
  }

  return lst;
}


C_word C_fcall C_i_vector_ref(C_word v, C_word i)
{
  int j;

  if(C_immediatep(v) || C_header_bits(v) != C_VECTOR_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, v);

  if(i & C_FIXNUM_BIT) {
    j = C_unfix(i);

    if(j < 0 || j >= C_header_size(v)) barf(C_OUT_OF_RANGE_ERROR, v, i);

    return C_block_item(v, j);
  }
  
  barf(C_BAD_ARGUMENT_TYPE_ERROR, i);
  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_i_string_set(C_word s, C_word i, C_word c)
{
  int j;

  if(C_immediatep(s) || C_header_bits(s) != C_STRING_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, s);

  if(!C_immediatep(c) || (c & C_IMMEDIATE_TYPE_BITS) != C_CHARACTER_BITS)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, c);

  if(i & C_FIXNUM_BIT) {
    j = C_unfix(i);

    if(j < 0 || j >= C_header_size(s)) barf(C_OUT_OF_RANGE_ERROR, s, i);

    return C_setsubchar(s, i, c);
  }

  barf(C_BAD_ARGUMENT_TYPE_ERROR, i);
  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_i_string_ref(C_word s, C_word i)
{
  int j;

  if(C_immediatep(s) || C_header_bits(s) != C_STRING_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, s);

  if(i & C_FIXNUM_BIT) {
    j = C_unfix(i);

    if(j < 0 || j >= C_header_size(s)) barf(C_OUT_OF_RANGE_ERROR, s, i);

    return C_subchar(s, i);
  }
  
  barf(C_BAD_ARGUMENT_TYPE_ERROR, i);
  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_i_vector_length(C_word v)
{
  if(C_immediatep(v) || C_header_bits(v) != C_VECTOR_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, v);

  return C_fix(C_header_size(v));
}


C_word C_fcall C_i_string_length(C_word s)
{
  if(C_immediatep(s) || C_header_bits(s) != C_STRING_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, s);

  return C_fix(C_header_size(s));
}


C_word C_fcall C_i_length(C_word lst)
{
  int n = 0;

  if(lst != C_SCHEME_END_OF_LIST) {
    if(!C_immediatep(lst) && C_header_bits(lst) == C_PAIR_TYPE) {
      do {
	lst = C_block_item(lst, 1);
	++n;
      } while(!C_immediatep(lst) && C_header_bits(lst) == C_PAIR_TYPE);
    }
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, lst);
  }

  return C_fix(n);
}


C_word C_fcall C_i_inexact_to_exact(C_word n)
{
  double m;
  C_word r;

  if(n & C_FIXNUM_BIT) return n;
  else if(C_immediatep(n) || C_header_bits(n) != C_FLONUM_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, n);

  if(modf(C_flonum_magnitude(n), &m) == 0.0) {
    r = (C_word)m;
    
    if(r == m && C_fitsinfixnump(r))
      return C_fix(r);
  }

  barf(C_CANT_REPRESENT_INEXACT_ERROR, n);
  return 0;
}


C_word C_fcall C_i_set_car(C_word x, C_word val)
{
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  C_mutate(&C_block_item(x, 0), val);
  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_i_set_cdr(C_word x, C_word val)
{
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  C_mutate(&C_block_item(x, 1), val);
  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_i_vector_set(C_word v, C_word i, C_word x)
{
  int j;

  if(C_immediatep(v) || C_header_bits(v) != C_VECTOR_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, v);

  if(i & C_FIXNUM_BIT) {
    j = C_unfix(i);

    if(j < 0 || j >= C_header_size(v)) barf(C_OUT_OF_RANGE_ERROR, v, i);

    C_mutate(&C_block_item(v, j), x);
  }
  else barf(C_BAD_ARGUMENT_TYPE_ERROR, i);

  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_a_i_abs(C_word **a, int c, C_word x)
{
  if(x & C_FIXNUM_BIT) return C_fix(abs(C_unfix(x)));

  if(C_immediatep(x) || C_header_bits(x) != C_FLONUM_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  return C_flonum(a, fabs(C_flonum_magnitude(x)));
}


C_word C_fcall C_i_assq(C_word x, C_word lst)
{
  C_word a;

  while(!C_immediatep(lst) && C_header_bits(lst) == C_PAIR_TYPE) {
    a = C_block_item(lst, 0);

    if(!C_immediatep(a) && C_header_bits(a) == C_PAIR_TYPE) {
      if(C_block_item(a, 0) == x) return a;
    }
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, a);
  
    lst = C_block_item(lst, 1);
  }

  return C_SCHEME_FALSE;
}


C_word C_fcall C_u_i_assq(C_word x, C_word lst)
{
  C_word a;

  while(!C_immediatep(lst)) {
    a = C_block_item(lst, 0);

    if(C_block_item(a, 0) == x) return a;
    else lst = C_block_item(lst, 1);
  }

  return C_SCHEME_FALSE;
}


C_word C_fcall C_i_assv(C_word x, C_word lst)
{
  C_word a;

  while(!C_immediatep(lst) && C_header_bits(lst) == C_PAIR_TYPE) {
    a = C_block_item(lst, 0);

    if(!C_immediatep(a) && C_header_bits(a) == C_PAIR_TYPE) {
      if(C_truep(C_i_eqvp(C_block_item(a, 0), x))) return a;
    }
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, a);
  
    lst = C_block_item(lst, 1);
  }

  return C_SCHEME_FALSE;
}


C_word C_fcall C_i_assoc(C_word x, C_word lst)
{
  C_word a;

  while(!C_immediatep(lst) && C_header_bits(lst) == C_PAIR_TYPE) {
    a = C_block_item(lst, 0);

    if(!C_immediatep(a) && C_header_bits(a) == C_PAIR_TYPE) {
      if(C_equalp(C_block_item(a, 0), x)) return a;
    }
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, a);
  
    lst = C_block_item(lst, 1);
  }

  return C_SCHEME_FALSE;
}


C_word C_fcall C_i_memq(C_word x, C_word lst)
{
  while(!C_immediatep(lst) && C_header_bits(lst) == C_PAIR_TYPE) {
    if(C_block_item(lst, 0) == x) return lst;
    else lst = C_block_item(lst, 1);
  }

  return C_SCHEME_FALSE;
}


C_word C_fcall C_u_i_memq(C_word x, C_word lst)
{
  while(!C_immediatep(lst)) {
    if(C_block_item(lst, 0) == x) return lst;
    else lst = C_block_item(lst, 1);
  }

  return C_SCHEME_FALSE;
}


C_word C_fcall C_i_memv(C_word x, C_word lst)
{
  while(!C_immediatep(lst) && C_header_bits(lst) == C_PAIR_TYPE) {
    if(C_truep(C_i_eqvp(C_block_item(lst, 0), x))) return lst;
    else lst = C_block_item(lst, 1);
  }

  return C_SCHEME_FALSE;
}


C_word C_fcall C_i_member(C_word x, C_word lst)
{
  while(!C_immediatep(lst) && C_header_bits(lst) == C_PAIR_TYPE) {
    if(C_equalp(C_block_item(lst, 0), x)) return lst;
    else lst = C_block_item(lst, 1);
  }
  
  return C_SCHEME_FALSE;
}


/* Inline routines for extended bindings: */

C_word C_fcall C_i_check_exact(C_word x)
{
  if((x & C_FIXNUM_BIT) == 0) barf(C_BAD_ARGUMENT_TYPE_NO_FIXNUM_ERROR, x);

  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_i_check_char(C_word x)
{
  if((x & C_IMMEDIATE_TYPE_BITS) != C_CHARACTER_BITS) 
    barf(C_BAD_ARGUMENT_TYPE_NO_CHAR_ERROR, x);

  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_i_check_number(C_word x)
{
  if((x & C_FIXNUM_BIT) == 0 && (C_immediatep(x) || C_header_bits(x) != C_FLONUM_TYPE))
    barf(C_BAD_ARGUMENT_TYPE_NO_NUMBER_ERROR, x);

  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_i_check_string(C_word x)
{
  if(C_immediatep(x) || C_header_bits(x) != C_STRING_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_NO_STRING_ERROR, x);

  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_i_check_bytevector(C_word x)
{
  if(C_immediatep(x) || C_header_bits(x) != C_BYTEVECTOR_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_NO_BYTEVECTOR_ERROR, x);

  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_i_check_vector(C_word x)
{
  if(C_immediatep(x) || C_header_bits(x) != C_VECTOR_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_NO_VECTOR_ERROR, x);

  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_i_check_structure(C_word x, C_word st)
{
  if(C_immediatep(x) || C_header_bits(x) != C_STRUCTURE_TYPE || C_block_item(x, 0) != st)
    barf(C_BAD_ARGUMENT_TYPE_BAD_STRUCT_ERROR, x, st);

  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_i_check_pair(C_word x)
{
  if(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_NO_PAIR_ERROR, x);

  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_i_check_symbol(C_word x)
{
  if(C_immediatep(x) || C_header_bits(x) != C_SYMBOL_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_NO_SYMBOL_ERROR, x);

  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_i_check_list(C_word x)
{
  if(x != C_SCHEME_END_OF_LIST && (C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE))
    barf(C_BAD_ARGUMENT_TYPE_NO_LIST_ERROR, x);

  return C_SCHEME_UNDEFINED;
}


C_word C_fcall C_i_not_pair_p_2(C_word x)
{
  return C_mk_bool(C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE);
}


C_word C_fcall C_i_null_list_p(C_word x)
{
  if(x == C_SCHEME_END_OF_LIST) return C_SCHEME_TRUE;
  else if(!C_immediatep(x) && C_header_bits(x) == C_PAIR_TYPE) return C_SCHEME_FALSE;
  else barf(C_BAD_ARGUMENT_TYPE_NO_LIST_ERROR, x);
}


C_word C_fcall C_i_null_pointerp(C_word x)
{
  if(!C_immediatep(x) && C_header_bits(x) == C_POINTER_TYPE)
    return C_null_pointerp(x);

  barf(C_BAD_ARGUMENT_TYPE_ERROR, x);
}


/* Primitives: */

void C_apply(int c, C_word closure, C_word k, C_word fn, ...)
{
  va_list v;
  int i, n = c - 3;
  C_word x, skip;
#ifdef FAKE_APPLY
  C_word *buf = C_temporary_stack_limit;
  void *proc;
#endif

  if(c < 4) C_bad_min_argc(c, 4);

  if(C_immediatep(fn) || C_header_bits(fn) != C_CLOSURE_TYPE)
    barf(C_NOT_A_CLOSURE_ERROR, fn);

  va_start(v, fn);

  for(i = n; i > 1; --i) {
    x = va_arg(v, C_word);
#ifdef FAKE_APPLY
    *(buf++) = x;
#else
    C_save(x);
#endif
  }

  x = va_arg(v, C_word);

  if(x != C_SCHEME_END_OF_LIST && (C_immediatep(x) || C_header_bits(x) != C_PAIR_TYPE))
    barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  for(skip = x; !C_immediatep(skip) && C_header_bits(skip) == C_PAIR_TYPE; skip = C_block_item(skip, 1)) {
    x = C_block_item(skip, 0);
#ifdef FAKE_APPLY

    if(buf >= C_temporary_stack_bottom) barf(C_TOO_MANY_PARAMETERS_ERROR);

    *(buf++) = x;
#else
    C_save(x);
#endif
    ++n;
  }

  va_end(v);
  --n;
#ifdef FAKE_APPLY
  buf = alloca(n * sizeof(C_word));
  buf[ 0 ] = n + 2;
  buf[ 1 ] = fn;
  buf[ 2 ] = k;
  memcpy(&buf[ 3 ], C_temporary_stack_limit, n * sizeof(C_word));
  proc = (void *)C_block_item(fn, 0);
# ifdef _MSC_VER
  __asm { 
    mov eax, proc
    mov esp, buf
    call eax
  }
# else /* defined(__GNUC__) */
  asm volatile("movl %0, %%eax; movl %1, %%esp; call *%%eax" : : "g" (proc), "g" (buf));
# endif
#endif
  C_do_apply(n, fn, k);
}


void C_do_apply(int n, C_word fn, C_word k)
{
  void *pr = (void *)C_block_item(fn, 0);
  C_word *ptr = C_temporary_stack = C_temporary_stack_bottom;

  switch(n) {                                                                /* Are you ready for this? */
  case 0: ((C_proc2)pr)(2, fn, k);
  case 1: ((C_proc3)pr)(3, fn, k,*(ptr-1));
  case 2: ((C_proc4)pr)(4, fn, k,*(ptr-1),*(ptr-2));
  case 3: ((C_proc5)pr)(5, fn, k,*(ptr-1),*(ptr-2),*(ptr-3));
  case 4: ((C_proc6)pr)(6, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4));
  case 5: ((C_proc7)pr)(7, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5));
  case 6: ((C_proc8)pr)(8, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6));
  case 7: ((C_proc9)pr)(9, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7));
  case 8: ((C_proc10)pr)(10, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8));
  case 9: ((C_proc11)pr)(11, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9));
  case 10: ((C_proc12)pr)(12, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10));
  case 11: ((C_proc13)pr)(13, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11));
  case 12: ((C_proc14)pr)(14, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12));
  case 13: ((C_proc15)pr)(15, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13));
  case 14: ((C_proc16)pr)(16, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14));
  case 15: ((C_proc17)pr)(17, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15));
  case 16: ((C_proc18)pr)(18, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16));
  case 17: ((C_proc19)pr)(19, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17));
  case 18: ((C_proc20)pr)(20, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18));
  case 19: ((C_proc21)pr)(21, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19));
  case 20: ((C_proc22)pr)(22, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20));
  case 21: ((C_proc23)pr)(23, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21));
  case 22: ((C_proc24)pr)(24, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22));
  case 23: ((C_proc25)pr)(25, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23));
  case 24: ((C_proc26)pr)(26, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24));
  case 25: ((C_proc27)pr)(27, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25));
  case 26: ((C_proc28)pr)(28, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26));
  case 27: ((C_proc29)pr)(29, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27));
  case 28: ((C_proc30)pr)(30, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28));
  case 29: ((C_proc31)pr)(31, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29));
  case 30: ((C_proc32)pr)(32, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30));
  case 31: ((C_proc33)pr)(33, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31));
  case 32: ((C_proc34)pr)(34, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32));
  case 33: ((C_proc35)pr)(35, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33));
  case 34: ((C_proc36)pr)(36, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34));
  case 35: ((C_proc37)pr)(37, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35));
  case 36: ((C_proc38)pr)(38, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36));
  case 37: ((C_proc39)pr)(39, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37));
  case 38: ((C_proc40)pr)(40, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38));
  case 39: ((C_proc41)pr)(41, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39));
  case 40: ((C_proc42)pr)(42, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40));
  case 41: ((C_proc43)pr)(43, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41));
  case 42: ((C_proc44)pr)(44, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42));
  case 43: ((C_proc45)pr)(45, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43));
  case 44: ((C_proc46)pr)(46, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44));
  case 45: ((C_proc47)pr)(47, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45));
  case 46: ((C_proc48)pr)(48, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46));
  case 47: ((C_proc49)pr)(49, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47));
  case 48: ((C_proc50)pr)(50, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48));
  case 49: ((C_proc51)pr)(51, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49));
  case 50: ((C_proc52)pr)(52, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50));
  case 51: ((C_proc53)pr)(53, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51));
  case 52: ((C_proc54)pr)(54, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52));
  case 53: ((C_proc55)pr)(55, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53));
  case 54: ((C_proc56)pr)(56, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54));
  case 55: ((C_proc57)pr)(57, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55));
  case 56: ((C_proc58)pr)(58, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56));
  case 57: ((C_proc59)pr)(59, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57));
  case 58: ((C_proc60)pr)(60, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58));
  case 59: ((C_proc61)pr)(61, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59));
  case 60: ((C_proc62)pr)(62, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60));
  case 61: ((C_proc63)pr)(63, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61));
  case 62: ((C_proc64)pr)(64, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62));
  case 63: ((C_proc65)pr)(65, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63));
  case 64: ((C_proc66)pr)(66, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64));
#ifdef C_128_PARAMETERS
  case 65: ((C_proc67)pr)(67, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65));
  case 66: ((C_proc68)pr)(68, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66));
  case 67: ((C_proc69)pr)(69, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67));
  case 68: ((C_proc70)pr)(70, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68));
  case 69: ((C_proc71)pr)(71, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69));
  case 70: ((C_proc72)pr)(72, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70));
  case 71: ((C_proc73)pr)(73, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71));
  case 72: ((C_proc74)pr)(74, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72));
  case 73: ((C_proc75)pr)(75, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73));
  case 74: ((C_proc76)pr)(76, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74));
  case 75: ((C_proc77)pr)(77, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75));
  case 76: ((C_proc78)pr)(78, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76));
  case 77: ((C_proc79)pr)(79, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77));
  case 78: ((C_proc80)pr)(80, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78));
  case 79: ((C_proc81)pr)(81, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79));
  case 80: ((C_proc82)pr)(82, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80));
  case 81: ((C_proc83)pr)(83, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81));
  case 82: ((C_proc84)pr)(84, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82));
  case 83: ((C_proc85)pr)(85, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83));
  case 84: ((C_proc86)pr)(86, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84));
  case 85: ((C_proc87)pr)(87, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85));
  case 86: ((C_proc88)pr)(88, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86));
  case 87: ((C_proc89)pr)(89, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87));
  case 88: ((C_proc90)pr)(90, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88));
  case 89: ((C_proc91)pr)(91, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89));
  case 90: ((C_proc92)pr)(92, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90));
  case 91: ((C_proc93)pr)(93, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91));
  case 92: ((C_proc94)pr)(94, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92));
  case 93: ((C_proc95)pr)(95, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93));
  case 94: ((C_proc96)pr)(96, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94));
  case 95: ((C_proc97)pr)(97, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95));
  case 96: ((C_proc98)pr)(98, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96));
  case 97: ((C_proc99)pr)(99, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97));
  case 98: ((C_proc100)pr)(100, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98));
  case 99: ((C_proc101)pr)(101, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99));
  case 100: ((C_proc102)pr)(102, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100));
  case 101: ((C_proc103)pr)(103, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101));
  case 102: ((C_proc104)pr)(104, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102));
  case 103: ((C_proc105)pr)(105, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103));
  case 104: ((C_proc106)pr)(106, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104));
  case 105: ((C_proc107)pr)(107, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105));
  case 106: ((C_proc108)pr)(108, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106));
  case 107: ((C_proc109)pr)(109, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107));
  case 108: ((C_proc110)pr)(110, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107),*(ptr-108));
  case 109: ((C_proc111)pr)(111, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107),*(ptr-108),*(ptr-109));
  case 110: ((C_proc112)pr)(112, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107),*(ptr-108),*(ptr-109),*(ptr-110));
  case 111: ((C_proc113)pr)(113, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107),*(ptr-108),*(ptr-109),*(ptr-110),*(ptr-111));
  case 112: ((C_proc114)pr)(114, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107),*(ptr-108),*(ptr-109),*(ptr-110),*(ptr-111),*(ptr-112));
  case 113: ((C_proc115)pr)(115, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107),*(ptr-108),*(ptr-109),*(ptr-110),*(ptr-111),*(ptr-112),*(ptr-113));
  case 114: ((C_proc116)pr)(116, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107),*(ptr-108),*(ptr-109),*(ptr-110),*(ptr-111),*(ptr-112),*(ptr-113),*(ptr-114));
  case 115: ((C_proc117)pr)(117, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107),*(ptr-108),*(ptr-109),*(ptr-110),*(ptr-111),*(ptr-112),*(ptr-113),*(ptr-114),*(ptr-115));
  case 116: ((C_proc118)pr)(118, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107),*(ptr-108),*(ptr-109),*(ptr-110),*(ptr-111),*(ptr-112),*(ptr-113),*(ptr-114),*(ptr-115),*(ptr-116));
  case 117: ((C_proc119)pr)(119, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107),*(ptr-108),*(ptr-109),*(ptr-110),*(ptr-111),*(ptr-112),*(ptr-113),*(ptr-114),*(ptr-115),*(ptr-116),*(ptr-117));
  case 118: ((C_proc120)pr)(120, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107),*(ptr-108),*(ptr-109),*(ptr-110),*(ptr-111),*(ptr-112),*(ptr-113),*(ptr-114),*(ptr-115),*(ptr-116),*(ptr-117),*(ptr-118));
  case 119: ((C_proc121)pr)(121, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107),*(ptr-108),*(ptr-109),*(ptr-110),*(ptr-111),*(ptr-112),*(ptr-113),*(ptr-114),*(ptr-115),*(ptr-116),*(ptr-117),*(ptr-118),*(ptr-119));
  case 120: ((C_proc122)pr)(122, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107),*(ptr-108),*(ptr-109),*(ptr-110),*(ptr-111),*(ptr-112),*(ptr-113),*(ptr-114),*(ptr-115),*(ptr-116),*(ptr-117),*(ptr-118),*(ptr-119),*(ptr-120));
  case 121: ((C_proc123)pr)(123, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107),*(ptr-108),*(ptr-109),*(ptr-110),*(ptr-111),*(ptr-112),*(ptr-113),*(ptr-114),*(ptr-115),*(ptr-116),*(ptr-117),*(ptr-118),*(ptr-119),*(ptr-120),*(ptr-121));
  case 122: ((C_proc124)pr)(124, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107),*(ptr-108),*(ptr-109),*(ptr-110),*(ptr-111),*(ptr-112),*(ptr-113),*(ptr-114),*(ptr-115),*(ptr-116),*(ptr-117),*(ptr-118),*(ptr-119),*(ptr-120),*(ptr-121),*(ptr-122));
  case 123: ((C_proc125)pr)(125, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107),*(ptr-108),*(ptr-109),*(ptr-110),*(ptr-111),*(ptr-112),*(ptr-113),*(ptr-114),*(ptr-115),*(ptr-116),*(ptr-117),*(ptr-118),*(ptr-119),*(ptr-120),*(ptr-121),*(ptr-122),*(ptr-123));
  case 124: ((C_proc126)pr)(126, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107),*(ptr-108),*(ptr-109),*(ptr-110),*(ptr-111),*(ptr-112),*(ptr-113),*(ptr-114),*(ptr-115),*(ptr-116),*(ptr-117),*(ptr-118),*(ptr-119),*(ptr-120),*(ptr-121),*(ptr-122),*(ptr-123),*(ptr-124));
  case 125: ((C_proc127)pr)(127, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107),*(ptr-108),*(ptr-109),*(ptr-110),*(ptr-111),*(ptr-112),*(ptr-113),*(ptr-114),*(ptr-115),*(ptr-116),*(ptr-117),*(ptr-118),*(ptr-119),*(ptr-120),*(ptr-121),*(ptr-122),*(ptr-123),*(ptr-124),*(ptr-125));
  case 126: ((C_proc128)pr)(128, fn, k,*(ptr-1),*(ptr-2),*(ptr-3),*(ptr-4),*(ptr-5),*(ptr-6),*(ptr-7),*(ptr-8),*(ptr-9),*(ptr-10),*(ptr-11),*(ptr-12),*(ptr-13),*(ptr-14),*(ptr-15),*(ptr-16),*(ptr-17),*(ptr-18),*(ptr-19),*(ptr-20),*(ptr-21),*(ptr-22),*(ptr-23),*(ptr-24),*(ptr-25),*(ptr-26),*(ptr-27),*(ptr-28),*(ptr-29),*(ptr-30),*(ptr-31),*(ptr-32),*(ptr-33),*(ptr-34),*(ptr-35),*(ptr-36),*(ptr-37),*(ptr-38),*(ptr-39),*(ptr-40),*(ptr-41),*(ptr-42),*(ptr-43),*(ptr-44),*(ptr-45),*(ptr-46),*(ptr-47),*(ptr-48),*(ptr-49),*(ptr-50),*(ptr-51),*(ptr-52),*(ptr-53),*(ptr-54),*(ptr-55),*(ptr-56),*(ptr-57),*(ptr-58),*(ptr-59),*(ptr-60),*(ptr-61),*(ptr-62),*(ptr-63),*(ptr-64),*(ptr-65),*(ptr-66),*(ptr-67),*(ptr-68),*(ptr-69),*(ptr-70),*(ptr-71),*(ptr-72),*(ptr-73),*(ptr-74),*(ptr-75),*(ptr-76),*(ptr-77),*(ptr-78),*(ptr-79),*(ptr-80),*(ptr-81),*(ptr-82),*(ptr-83),*(ptr-84),*(ptr-85),*(ptr-86),*(ptr-87),*(ptr-88),*(ptr-89),*(ptr-90),*(ptr-91),*(ptr-92),*(ptr-93),*(ptr-94),*(ptr-95),*(ptr-96),*(ptr-97),*(ptr-98),*(ptr-99),*(ptr-100),*(ptr-101),*(ptr-102),*(ptr-103),*(ptr-104),*(ptr-105),*(ptr-106),*(ptr-107),*(ptr-108),*(ptr-109),*(ptr-110),*(ptr-111),*(ptr-112),*(ptr-113),*(ptr-114),*(ptr-115),*(ptr-116),*(ptr-117),*(ptr-118),*(ptr-119),*(ptr-120),*(ptr-121),*(ptr-122),*(ptr-123),*(ptr-124),*(ptr-125),*(ptr-126));
#endif
  default: barf(C_TOO_MANY_PARAMETERS_ERROR);
  }
}


void C_call_cc(int c, C_word cl, C_word k, C_word cont)
{
  C_word *a = C_alloc(3),
         wrapper;
  void *pr = (void *)C_block_item(cont, 0);

  if(C_immediatep(cont) || C_header_bits(cont) != C_CLOSURE_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, cont);

  /* Check for values-continuation: */
  if(C_block_item(k, 0) == (C_word)values_continuation)
    wrapper = C_closure(&a, 2, (C_word)call_cc_values_wrapper, k);
  else wrapper = C_closure(&a, 2, (C_word)call_cc_wrapper, k);

  ((C_proc3)pr)(3, cont, k, wrapper);
}


void call_cc_wrapper(int c, C_word closure, C_word k, C_word result)
{
  C_word cont = C_block_item(closure, 1);

  if(c != 3) C_bad_argc(c, 3);

  C_kontinue(cont, result);
}


void call_cc_values_wrapper(int c, C_word closure, C_word k, ...)
{
  va_list v;
  C_word cont = C_block_item(closure, 1),
         x1;
  int n = c;

  va_start(v, k);

  if(c > 2) {
    x1 = va_arg(v, C_word);
    --n;
    
    while(--c > 2) C_save(va_arg(v, C_word));
  }
  else x1 = C_SCHEME_UNBOUND;

  va_end(v);
  C_do_apply(n - 2, cont, x1);
}


void C_values(int c, C_word closure, C_word k, ...)
{
  va_list v;
  int n = c;

  if(c < 2) C_bad_min_argc(c, 2);

  va_start(v, k);

  /* Check continuation wether it receives multiple values: */
  if(C_block_item(k, 0) == (C_word)values_continuation) {
    while(c-- > 2)
      C_save(va_arg(v, C_word));

    va_end(v);
    C_do_apply(n - 2, k, C_SCHEME_UNBOUND); /* unbound value marks direct invocation */
  }
  
  if(c != 3)
    barf(C_CONTINUATION_CANT_RECEIVE_VALUES_ERROR, k);

  n = va_arg(v, C_word);
  va_end(v);
  C_kontinue(k, n);
}


void C_call_with_values(int c, C_word cl, C_word k, C_word thunk, C_word kont)
{
  C_word *a = C_alloc(4),
         kk;

  if(c != 4) C_bad_argc(c, 4);

  if(C_immediatep(thunk) || C_header_bits(thunk) != C_CLOSURE_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, thunk);

  if(C_immediatep(kont) || C_header_bits(kont) != C_CLOSURE_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, kont);

  kk = C_closure(&a, 3, (C_word)values_continuation, kont, k);
  C_do_apply(0, thunk, kk);
}


void C_u_call_with_values(int c, C_word cl, C_word k, C_word thunk, C_word kont)
{
  C_word *a = C_alloc(4),
         kk;

  kk = C_closure(&a, 3, (C_word)values_continuation, kont, k);
  C_do_apply(0, thunk, kk);
}


void values_continuation(int c, C_word closure, C_word arg0, ...)
{
  C_word kont = ((C_SCHEME_BLOCK *)closure)->data[ 1 ],
         k = ((C_SCHEME_BLOCK *)closure)->data[ 2 ],
         n = c,
         *ptr;
  va_list v;

  if(arg0 == C_SCHEME_UNBOUND) { /* This continuation was called by 'values'... */
    for(va_start(v, arg0); c-- > 2; C_save(va_arg(v, C_word)));

    va_end(v);
  }
  else {			/* This continuation was captured and called explicity... */
    ++n;
    c -= 1;

    /* move temporary-stack contents upwards one slot: */
    for(ptr = C_temporary_stack - c; c--; ++ptr) *ptr = ptr[ 1 ];

    C_save(arg0);
  }

  C_do_apply(n - 2, kont, k);
}


void C_times(int c, C_word closure, C_word k, ...)
{
  va_list v;
  C_word x;
  C_word iresult = 1;
  int fflag = 0;
  double fresult = 1;

  va_start(v, k);
  c -= 2;

  while(c--) {
    x = va_arg(v, C_word);
    
    if(x & C_FIXNUM_BIT) {
	fresult *= C_unfix(x);
	
	if(!fflag) iresult *= C_unfix(x);
    }
    else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
	fresult *= C_flonum_magnitude(x);

	if(!fflag) fflag = 1;
    }
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, x);
  }

  va_end(v);
  x = C_fix(iresult);
  
  if(fflag || (double)C_unfix(x) != fresult) {
      C_temporary_flonum = fresult;
      C_cons_flonum(2, C_SCHEME_UNDEFINED, k);
  }

  C_kontinue(k, x);
}


C_word C_fcall C_2_times(C_word **ptr, C_word x, C_word y)
{
  C_word iresult;
  double fresult;
  int fflag = 0;

  if(x & C_FIXNUM_BIT) {
    if(y & C_FIXNUM_BIT) {
      iresult = C_unfix(x) * C_unfix(y);
      fresult = (double)C_unfix(x) * (double)C_unfix(y);
    }
    else if(!C_immediatep(y) && C_header_bits(y) == C_FLONUM_TYPE) {
      fresult = C_unfix(x) * C_flonum_magnitude(y);
      fflag = 1;
    }
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, y);
  }
  else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
    fflag = 1;

    if(y & C_FIXNUM_BIT) fresult = C_flonum_magnitude(x) * C_unfix(y);
    else if(!C_immediatep(y) && C_header_bits(y) == C_FLONUM_TYPE)
      fresult = C_flonum_magnitude(x) * C_flonum_magnitude(y);
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, y);
  }
  else barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  iresult = C_fix(iresult);

  if(fflag || (double)C_unfix(iresult) != fresult) return C_flonum(ptr, fresult);
  
  return iresult;
}


void C_plus(int c, C_word closure, C_word k, ...)
{
  va_list v;
  C_word x;
  C_word iresult = 0;
  int fflag = 0;
  double fresult = 0;

  va_start(v, k);
  c -= 2;

  while(c--) {
    x = va_arg(v, C_word);
    
    if(x & C_FIXNUM_BIT) {
	fresult += C_unfix(x);

	if(!fflag) iresult += C_unfix(x);
    }
    else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
      fresult += C_flonum_magnitude(x);

      if(!fflag) fflag = 1;
    }
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, x);
  }

  va_end(v);
  x = C_fix(iresult);

  if(fflag || (double)C_unfix(x) != fresult) {
    C_temporary_flonum = fresult;
    C_cons_flonum(2, C_SCHEME_UNDEFINED, k);
  }

  C_kontinue(k, x);
}


C_word C_fcall C_2_plus(C_word **ptr, C_word x, C_word y)
{
  C_word iresult;
  double fresult;
  int fflag = 0;

  if(x & C_FIXNUM_BIT) {
    if(y & C_FIXNUM_BIT) {
      iresult = C_unfix(x) + C_unfix(y);
      fresult = (double)C_unfix(x) + (double)C_unfix(y);
    }
    else if(!C_immediatep(y) && C_header_bits(y) == C_FLONUM_TYPE) {
      fresult = C_unfix(x) + C_flonum_magnitude(y);
      fflag = 1;
    }
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, y);
  }
  else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
    fflag = 1;

    if(y & C_FIXNUM_BIT) fresult = C_flonum_magnitude(x) + C_unfix(y);
    else if(!C_immediatep(y) && C_header_bits(y) == C_FLONUM_TYPE)
      fresult = C_flonum_magnitude(x) + C_flonum_magnitude(y);
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, y);
  }
  else barf(C_BAD_ARGUMENT_TYPE_ERROR, x);
  
  iresult = C_fix(iresult);

  if(fflag || (double)C_unfix(iresult) != fresult) return C_flonum(ptr, fresult);
  
  return iresult;
}


void cons_flonum_trampoline(void *dummy)
{
  C_word k = C_restore,
         *a = C_alloc(WORDS_PER_FLONUM);

  C_kontinue(k, C_flonum(&a, C_temporary_flonum));
}


void C_minus(int c, C_word closure, C_word k, C_word n1, ...)
{
  va_list v;
  C_word iresult;
  int fflag;
  double fresult;

  if(c < 3) C_bad_min_argc(c, 3);

  if(n1 & C_FIXNUM_BIT) {
    fresult = iresult = C_unfix(n1);
    fflag = 0;
  }
  else if(!C_immediatep(n1) && C_header_bits(n1) == C_FLONUM_TYPE) {
    fresult = C_flonum_magnitude(n1);
    fflag = 1;
  }
  else barf(C_BAD_ARGUMENT_TYPE_ERROR, n1);

  if(c == 3) {
    if(fflag) fresult = -fresult;
    else fresult = iresult = -iresult;

    goto cont;
  }

  va_start(v, n1);
  c -= 3;

  while(c--) {
    n1 = va_arg(v, C_word);
    
    if(n1 & C_FIXNUM_BIT) {
      fresult -= C_unfix(n1);

      if(!fflag) iresult -= C_unfix(n1);
    }
    else if(!C_immediatep(n1) && C_header_bits(n1) == C_FLONUM_TYPE) {
      fresult -= C_flonum_magnitude(n1);

      if(!fflag) fflag = 1;
    }
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, n1);
  }

  va_end(v);
  
 cont:
  n1 = C_fix(iresult);

  if(fflag || (double)C_unfix(n1) != fresult) {
    C_temporary_flonum = fresult;
    C_cons_flonum(2, C_SCHEME_UNDEFINED, k);
  }

  C_kontinue(k, n1);
}


C_word C_fcall C_2_minus(C_word **ptr, C_word x, C_word y)
{
  C_word iresult;
  double fresult;
  int fflag = 0;

  if(x & C_FIXNUM_BIT) {
    if(y & C_FIXNUM_BIT) {
      iresult = C_unfix(x) - C_unfix(y);
      fresult = (double)C_unfix(x) - (double)C_unfix(y);
    }
    else if(!C_immediatep(y) && C_header_bits(y) == C_FLONUM_TYPE) {
      fresult = C_unfix(x) - C_flonum_magnitude(y);
      fflag = 1;
    }
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, y);
  }
  else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
    fflag = 1;

    if(y & C_FIXNUM_BIT) fresult = C_flonum_magnitude(x) - C_unfix(y);
    else if(!C_immediatep(y) && C_header_bits(y) == C_FLONUM_TYPE)
      fresult = C_flonum_magnitude(x) - C_flonum_magnitude(y);
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, y);
  }
  else barf(C_BAD_ARGUMENT_TYPE_ERROR, x);
  
  iresult = C_fix(iresult);

  if(fflag || (double)C_unfix(iresult) != fresult) return C_flonum(ptr, fresult);
  
  return iresult;
}


void C_divide(int c, C_word closure, C_word k, C_word n1, ...)
{
  va_list v;
  C_word n2;
  C_word iresult;
  int fflag;
  double fresult, f2;

  if(c < 3) C_bad_min_argc(c, 3);

  if(n1 & C_FIXNUM_BIT) {
    iresult = C_unfix(n1);
    fflag = 0;
  }
  else if(!C_immediatep(n1) && C_header_bits(n1) == C_FLONUM_TYPE) {
    fresult = C_flonum_magnitude(n1);
    fflag = 1;
  }
  else barf(C_BAD_ARGUMENT_TYPE_ERROR, n1);

  if(c == 3) {
    if(fflag) {
      if(fresult == 0) barf(C_DIVISION_BY_ZERO_ERROR);

      fresult = 1.0 / fresult;
    }
    else {
      if(iresult == 0) barf(C_DIVISION_BY_ZERO_ERROR);

      fresult = 1.0 / (double)iresult;
      fflag = 1;
    }

    goto cont;
  }

  va_start(v, n1);
  c -= 3;

  while(c--) {
    n1 = va_arg(v, C_word);
    
    if(n1 & C_FIXNUM_BIT) {
      if(fflag) {
	if((n1 = C_unfix(n1)) == 0) 
	  barf(C_DIVISION_BY_ZERO_ERROR);

	fresult /= n1;
      }
      else {
	if((n2 = C_unfix(n1)) == 0)
	  barf(C_DIVISION_BY_ZERO_ERROR);

	if((fresult = (double)iresult / (double)n2) != (iresult /= n2))
	  fflag = 1;
      }
    }
    else if(!C_immediatep(n1) && C_header_bits(n1) == C_FLONUM_TYPE) {
      if(fflag) {
	if((f2 = C_flonum_magnitude(n1)) == 0)
	  barf(C_DIVISION_BY_ZERO_ERROR);

	fresult /= f2;
      }
      else {
	fflag = 1;

	if((f2 = C_flonum_magnitude(n1)) == 0)
	  barf(C_DIVISION_BY_ZERO_ERROR);

	fresult = (double)iresult / f2;
      }
    }
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, n1);
  }

  va_end(v);
  
 cont:
  if(fflag) {
    C_temporary_flonum = fresult;
    C_cons_flonum(2, C_SCHEME_UNDEFINED, k);
  }
  else n1 = C_fix(iresult);

  C_kontinue(k, n1);
}


C_word C_fcall C_2_divide(C_word **ptr, C_word x, C_word y)
{
  C_word iresult;
  double fresult;
  int fflag = 0;

  if(x & C_FIXNUM_BIT) {
    if(y & C_FIXNUM_BIT) {
      if((iresult = C_unfix(y)) == 0) barf(C_DIVISION_BY_ZERO_ERROR);

      fresult = (double)C_unfix(x) / (double)iresult;
      iresult = C_unfix(x) / iresult;
    }
    else if(!C_immediatep(y) && C_header_bits(y) == C_FLONUM_TYPE) {
      if((fresult = C_flonum_magnitude(y)) == 0.0)
	barf(C_DIVISION_BY_ZERO_ERROR);

      fresult = (double)C_unfix(x) / fresult;
      fflag = 1;
    }
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, y);
  }
  else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
    fflag = 1;

    if(y & C_FIXNUM_BIT) {
      fresult = C_flonum_magnitude(x);

      if((iresult = C_unfix(y)) == 0) barf(C_DIVISION_BY_ZERO_ERROR);

      fresult = fresult / (double)iresult;
    }
    else if(!C_immediatep(y) && C_header_bits(y) == C_FLONUM_TYPE) {
      if((fresult = C_flonum_magnitude(y)) == 0.0) barf(C_DIVISION_BY_ZERO_ERROR);
      
      fresult = C_flonum_magnitude(x) / fresult;
    }
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, y);
  }
  else barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  iresult = C_fix(iresult);

  if(fflag || (double)C_unfix(iresult) != fresult) return C_flonum(ptr, fresult);
  
  return iresult;
}


void C_nequalp(int c, C_word closure, C_word k, ...)
{
  C_word x;
  int i2, f,
      fflag,
      ilast;
  double flast, f2;
  va_list v;

  c -= 2; 
  f = 1;
  va_start(v, k);

  if(c == 0) goto cont;
  
  x = va_arg(v, C_word);

  if(x & C_FIXNUM_BIT) {
    fflag = 0;
    ilast = C_unfix(x);
  }
  else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
    fflag = 1;
    flast = C_flonum_magnitude(x);
  }
  else barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  while(--c) {
    x = va_arg(v, C_word);
    
    if(x & C_FIXNUM_BIT) {
      if(fflag) {
	f = flast == (f2 = (double)C_unfix(x));
        flast = f2;
      }
      else {
	f = ilast == (i2 = C_unfix(x));
	ilast = i2;
      }
    }
    else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
      if(fflag) {
	f = flast == (f2 = C_flonum_magnitude(x));
	flast = f2;
      }
      else {
	f = (double)ilast == (f2 = C_flonum_magnitude(x));
	flast = f2;
	fflag = 1;
      }
    }
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

    if(!f) break;
  }

 cont:
  va_end(v);
  C_kontinue(k, C_mk_bool(f));
}


C_word C_fcall C_i_nequalp(C_word x, C_word y)
{
  if(x & C_FIXNUM_BIT) {
    if(y & C_FIXNUM_BIT) return C_mk_bool(x == y);
    else if(!C_immediatep(y) && C_header_bits(y) == C_FLONUM_TYPE)
      return C_mk_bool((double)C_unfix(x) == C_flonum_magnitude(y));

    barf(C_BAD_ARGUMENT_TYPE_ERROR, y);
  }
  else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
    if(y & C_FIXNUM_BIT) return C_mk_bool(C_flonum_magnitude(x) == (double)C_unfix(y));
    else if(!C_immediatep(y) && C_header_bits(y) == C_FLONUM_TYPE)
      return C_mk_bool(C_flonum_magnitude(x) == C_flonum_magnitude(y));
    
    barf(C_BAD_ARGUMENT_TYPE_ERROR, y);
  }

  return C_SCHEME_FALSE;
}


void C_greaterp(int c, C_word closure, C_word k, ...)
{
  int i2, f,
      fflag,
      ilast;
  C_word x;
  double flast, f2;
  va_list v;

  c -= 2; 
  f = 1;
  va_start(v, k);

  if(c == 0) goto cont;
  
  x = va_arg(v, C_word);

  if(x & C_FIXNUM_BIT) {
    fflag = 0;
    ilast = C_unfix(x);
  }
  else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
    fflag = 1;
    flast = C_flonum_magnitude(x);
  }
  else barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  while(--c) {
    x = va_arg(v, C_word);
    
    if(x & C_FIXNUM_BIT) {
      if(fflag) {
	f = flast > (f2 = (double)C_unfix(x));
        flast = f2;
      }
      else {
	f = ilast > (i2 = C_unfix(x));
	ilast = i2;
      }
    }
    else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
      if(fflag) {
	f = flast > (f2 = C_flonum_magnitude(x));
	flast = f2;
      }
      else {
	f = (double)ilast > (f2 = C_flonum_magnitude(x));
	flast = f2;
	fflag = 1;
      }
    }
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

    if(!f) break;
  }

 cont:
  va_end(v);
  C_kontinue(k, C_mk_bool(f));
}


C_word C_fcall C_i_greaterp(C_word x, C_word y)
{
  if(x & C_FIXNUM_BIT) {
    if(y & C_FIXNUM_BIT) return C_mk_bool(x > y);
    else if(!C_immediatep(y) && C_header_bits(y) == C_FLONUM_TYPE)
      return C_mk_bool((double)C_unfix(x) > C_flonum_magnitude(y));

    barf(C_BAD_ARGUMENT_TYPE_ERROR, y);
  }
  else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
    if(y & C_FIXNUM_BIT) return C_mk_bool(C_flonum_magnitude(x) > (double)C_unfix(y));
    else if(!C_immediatep(y) && C_header_bits(y) == C_FLONUM_TYPE)
      return C_mk_bool(C_flonum_magnitude(x) > C_flonum_magnitude(y));
    
    barf(C_BAD_ARGUMENT_TYPE_ERROR, y);
  }

  return C_SCHEME_FALSE;
}


void C_lessp(int c, C_word closure, C_word k, ...)
{
  C_word x;
  int i2, f,
      fflag,
      ilast;
  double flast, f2;
  va_list v;

  c -= 2; 
  f = 1;
  va_start(v, k);

  if(c == 0) goto cont;
  
  x = va_arg(v, C_word);

  if(x &C_FIXNUM_BIT) {
    fflag = 0;
    ilast = C_unfix(x);
  }
  else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
    fflag = 1;
    flast = C_flonum_magnitude(x);
  }
  else barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  while(--c) {
    x = va_arg(v, C_word);
    
    if(x &C_FIXNUM_BIT) {
      if(fflag) {
	f = flast < (f2 = (double)C_unfix(x));
        flast = f2;
      }
      else {
	f = ilast < (i2 = C_unfix(x));
	ilast = i2;
      }
    }
    else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
      if(fflag) {
	f = flast < (f2 = C_flonum_magnitude(x));
	flast = f2;
      }
      else {
	f = (double)ilast < (f2 = C_flonum_magnitude(x));
	flast = f2;
	fflag = 1;
      }
    }
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

    if(!f) break;
  }

 cont:
  va_end(v);
  C_kontinue(k, C_mk_bool(f));
}


C_word C_fcall C_i_lessp(C_word x, C_word y)
{
  if(x & C_FIXNUM_BIT) {
    if(y & C_FIXNUM_BIT) return C_mk_bool(x < y);
    else if(!C_immediatep(y) && C_header_bits(y) == C_FLONUM_TYPE)
      return C_mk_bool((double)C_unfix(x) < C_flonum_magnitude(y));

    barf(C_BAD_ARGUMENT_TYPE_ERROR, y);
  }
  else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
    if(y & C_FIXNUM_BIT) return C_mk_bool(C_flonum_magnitude(x) < (double)C_unfix(y));
    else if(!C_immediatep(y) && C_header_bits(y) == C_FLONUM_TYPE)
      return C_mk_bool(C_flonum_magnitude(x) < C_flonum_magnitude(y));
    
    barf(C_BAD_ARGUMENT_TYPE_ERROR, y);
  }

  return C_SCHEME_FALSE;
}


void C_greater_or_equal_p(int c, C_word closure, C_word k, ...)
{
  C_word x;
  int i2, f,
      fflag,
      ilast;
  double flast, f2;
  va_list v;

  c -= 2; 
  f = 1;
  va_start(v, k);

  if(c == 0) goto cont;
  
  x = va_arg(v, C_word);

  if(x &C_FIXNUM_BIT) {
    fflag = 0;
    ilast = C_unfix(x);
  }
  else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
    fflag = 1;
    flast = C_flonum_magnitude(x);
  }
  else barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  while(--c) {
    x = va_arg(v, C_word);
    
    if(x &C_FIXNUM_BIT) {
      if(fflag) {
	f = flast >= (f2 = (double)C_unfix(x));
        flast = f2;
      }
      else {
	f = ilast >= (i2 = C_unfix(x));
	ilast = i2;
      }
    }
    else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
      if(fflag) {
	f = flast >= (f2 = C_flonum_magnitude(x));
	flast = f2;
      }
      else {
	f = (double)ilast >= (f2 = C_flonum_magnitude(x));
	flast = f2;
	fflag = 1;
      }
    }
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

    if(!f) break;
  }

 cont:
  va_end(v);
  C_kontinue(k, C_mk_bool(f));
}


C_word C_fcall C_i_greater_or_equalp(C_word x, C_word y)
{
  if(x & C_FIXNUM_BIT) {
    if(y & C_FIXNUM_BIT) return C_mk_bool(x >= y);
    else if(!C_immediatep(y) && C_header_bits(y) == C_FLONUM_TYPE)
      return C_mk_bool((double)C_unfix(x) >= C_flonum_magnitude(y));

    barf(C_BAD_ARGUMENT_TYPE_ERROR, y);
  }
  else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
    if(y & C_FIXNUM_BIT) return C_mk_bool(C_flonum_magnitude(x) >= (double)C_unfix(y));
    else if(!C_immediatep(y) && C_header_bits(y) == C_FLONUM_TYPE)
      return C_mk_bool(C_flonum_magnitude(x) >= C_flonum_magnitude(y));
    
    barf(C_BAD_ARGUMENT_TYPE_ERROR, y);
  }

  return C_SCHEME_FALSE;
}


void C_less_or_equal_p(int c, C_word closure, C_word k, ...)
{
  C_word x;
  int i2, f,
      fflag,
      ilast;
  double flast, f2;
  va_list v;

  c -= 2; 
  f = 1;
  va_start(v, k);

  if(c == 0) goto cont;
  
  x = va_arg(v, C_word);

  if(x &C_FIXNUM_BIT) {
    fflag = 0;
    ilast = C_unfix(x);
  }
  else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
    fflag = 1;
    flast = C_flonum_magnitude(x);
  }
  else barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

  while(--c) {
    x = va_arg(v, C_word);
    
    if(x &C_FIXNUM_BIT) {
      if(fflag) {
	f = flast <= (f2 = (double)C_unfix(x));
        flast = f2;
      }
      else {
	f = ilast <= (i2 = C_unfix(x));
	ilast = i2;
      }
    }
    else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
      if(fflag) {
	f = flast <= (f2 = C_flonum_magnitude(x));
	flast = f2;
      }
      else {
	f = (double)ilast <= (f2 = C_flonum_magnitude(x));
	flast = f2;
	fflag = 1;
      }
    }
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, x);

    if(!f) break;
  }

 cont:
  va_end(v);
  C_kontinue(k, C_mk_bool(f));
}


C_word C_fcall C_i_less_or_equalp(C_word x, C_word y)
{
  if(x & C_FIXNUM_BIT) {
    if(y & C_FIXNUM_BIT) return C_mk_bool(x <= y);
    else if(!C_immediatep(y) && C_header_bits(y) == C_FLONUM_TYPE)
      return C_mk_bool((double)C_unfix(x) <= C_flonum_magnitude(y));

    barf(C_BAD_ARGUMENT_TYPE_ERROR, y);
  }
  else if(!C_immediatep(x) && C_header_bits(x) == C_FLONUM_TYPE) {
    if(y & C_FIXNUM_BIT) return C_mk_bool(C_flonum_magnitude(x) <= (double)C_unfix(y));
    else if(!C_immediatep(y) && C_header_bits(y) == C_FLONUM_TYPE)
      return C_mk_bool(C_flonum_magnitude(x) <= C_flonum_magnitude(y));
    
    barf(C_BAD_ARGUMENT_TYPE_ERROR, y);
  }

  return C_SCHEME_FALSE;
}


void C_expt(int c, C_word closure, C_word k, C_word n1, C_word n2)
{
  double m1, m2;
  C_word r;

  if(c != 4) C_bad_argc(c, 4);

  if(n1 & C_FIXNUM_BIT) m1 = C_unfix(n1);
  else if(!C_immediatep(n1) && C_header_bits(n1) == C_FLONUM_TYPE)
    m1 = C_flonum_magnitude(n1);
  else barf(C_BAD_ARGUMENT_TYPE_ERROR, n1);

  if(n2 & C_FIXNUM_BIT) m2 = C_unfix(n2);
  else if(!C_immediatep(n2) && C_header_bits(n2) == C_FLONUM_TYPE)
    m2 = C_flonum_magnitude(n2);
  else barf(C_BAD_ARGUMENT_TYPE_ERROR, n2);

  m1 = pow(m1, m2);
  r = (C_word)m1;

  if(r == m1 && (n1 & C_FIXNUM_BIT) && (n2 & C_FIXNUM_BIT) && modf(m1, &m2) == 0.0 && C_fitsinfixnump(r))
    C_kontinue(k, C_fix(r));

  C_temporary_flonum = m1;
  C_cons_flonum(2, C_SCHEME_UNDEFINED, k);
}


void C_gc(int c, C_word closure, C_word k, ...)
{
  int f;
  va_list v;

  va_start(v, k);

  if(c == 3) f = C_truep(va_arg(v, C_word));
  else if(c != 2) C_bad_min_argc(c, 2);
  else f = 1;

  C_save(k);
  va_end(v);

  if(f) C_fromspace_top = C_fromspace_limit;

  C_reclaim(gc_2, NULL);
}


void gc_2(void *dummy)
{
  C_word k = C_restore;
  
  C_kontinue(k, C_fix((C_uword)C_fromspace_limit - (C_uword)C_fromspace_top));
}


void C_make_port(int c, C_word closure, C_word k, C_word channel, C_word size, C_word extra, C_word mode)
{
  C_save(mode);
  C_save(extra);
  C_save(channel);
  C_save(size);
  C_save(k);

  if(!C_demand(1 + C_unfix(size)))
    C_reclaim(make_port_2, NULL);

  make_port_2(NULL);
}


void make_port_2(void *dummy)
{
  C_word *a,
      k = C_restore,
      size = C_unfix(C_restore),
      channel = C_restore,
      extra = C_restore,
      mode = C_restore,
      n, port;
  FILE *fp = NULL;
  C_char fmode[ 3 ];

  a = C_alloc(1 + size);
  port = (C_word)a;
  ((C_SCHEME_BLOCK *)port)->header = C_PORT_TYPE | size;

  if(size > 0)
    for(n = 1; n < size; C_set_block_item(port, n++, C_SCHEME_FALSE));
  
  switch(channel) {
  case C_fix(0): fp = stdin; break;
  case C_fix(1): fp = stdout; break;
  case C_fix(2): fp = stderr; break;
  case C_fix(-1): fp = (FILE *)C_block_item(mode, 0); break;
  default:
    if(!C_immediatep(channel) && C_header_bits(channel) == C_STRING_TYPE) {
      n = C_header_size(channel);
      C_strncpy(buffer, C_c_string(channel), n);
      buffer[ n ] = '\0';
      C_memcpy(fmode, "\0\0\0", 3);
      fmode[ 0 ] = C_character_code(mode);

      if(C_truep(extra)) fmode[ 1 ] = 'b';

      if((fp = fopen(buffer, fmode)) == NULL) {
	port = C_SCHEME_FALSE;
	goto fini;
      }
    }
    else fp = NULL;
  }
  
  C_set_block_item(port, 0, (C_word)fp);
  
 fini:
  C_kontinue(k, port);
}


void C_allocate_vector(int c, C_word closure, C_word k, C_word size, C_word bvecf, C_word init, C_word align8)
{
  int bytes, 
      n = C_unfix(size);

  if(c != 6) C_bad_argc(c, 6);

  if(n > C_HEADER_SIZE_MASK)
    barf(C_OUT_OF_RANGE_ERROR, size, C_fix(C_HEADER_SIZE_MASK));

  if(!C_truep(bvecf)) bytes = C_wordstobytes(n) + sizeof(C_word);
  else bytes = n + sizeof(C_word);

  if(C_truep(align8)) bytes += sizeof(C_word);

  C_save(k);
  C_save(size);
  C_save(init);
  C_save(bvecf);
  C_save(align8);
  C_save(C_fix(bytes));

  if(!C_demand(C_bytestowords(bytes))) {
    /* Allocate on heap: */
    if(bytes >= (C_fromspace_limit - C_fromspace_top) - stack_size * 2)
      C_fromspace_top = C_fromspace_limit; /* trigger major GC */
  
    C_save(C_SCHEME_TRUE);
    C_reclaim(allocate_vector_2, NULL);
  }

  C_save(C_SCHEME_FALSE);
  allocate_vector_2(NULL);
}


void allocate_vector_2(void *dummy)
{
  C_word mode = C_restore;
  int bytes = C_unfix(C_restore);
  C_word align8 = C_restore,
         bvecf = C_restore,
         init = C_restore;
  int size = C_unfix(C_restore);
  C_word k = C_restore,
         *v0, v;

  assert(bytes > 0);
  
  if(C_truep(mode)) {
    if(bytes >= (C_fromspace_limit - C_fromspace_top) - stack_size) {

#ifdef PARANOIA
      fprintf(stderr, C_text("[insufficient heap space: %d, need %d]\n"), 
	      (long)C_fromspace_limit - (long)C_fromspace_top - stack_size * 2, bytes);
#endif

      barf(C_OUT_OF_MEMORY_ERROR);
    }

    v0 = (C_word *)C_align((C_word)C_fromspace_top);
    C_fromspace_top += C_align(bytes);
  }
  else v0 = C_alloc(C_bytestowords(bytes));

  if(C_truep(align8) && aligned8(v0)) ++v0;

  v = (C_word)v0;

  if(!C_truep(bvecf)) {
    *(v0++) = C_VECTOR_TYPE | size | (C_truep(align8) ? C_8ALIGN_BIT : 0);
  
    while(size--) *(v0++) = init;
  }
  else {
    *(v0++) = C_STRING_TYPE | size;

    if(C_truep(init))
      C_memset(v0, C_character_code(init), size);
  }

  C_kontinue(k, v);
}


void C_string_to_symbol(int c, C_word closure, C_word k, C_word string)
{
  int len, key;
  C_word s, *a = C_alloc(6);	/* 6 <=> 1 bucket (pair) + 1 symbol */
  C_char *name;

  if(c != 3) C_bad_argc(c, 3);

  if(C_immediatep(string) || C_header_bits(string) != C_STRING_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, string);
    
  len = C_header_size(string);
  name = (C_char *)C_data_pointer(string);
  key = hash_string(len, name);

  if(!C_truep(s = lookup(key, len, name))) s = add_symbol(&a, key, string);

  C_kontinue(k, s);
}


void C_flonum_fraction(int c, C_word closure, C_word k, C_word n)
{
  double fn = C_flonum_magnitude(n);

  C_temporary_flonum = modf(fn, &fn);
  C_cons_flonum(2, C_SCHEME_UNDEFINED, k);
}


void C_exact_to_inexact(int c, C_word closure, C_word k, C_word n)
{
  if(c != 3) C_bad_argc(c, 3);

  if(n & C_FIXNUM_BIT) {
    C_temporary_flonum = (double)C_unfix(n);
    C_cons_flonum(2, C_SCHEME_UNDEFINED, k);
  }
  else if(C_immediatep(n) || C_header_bits(n) != C_FLONUM_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, n);
 
  C_kontinue(k, n);
}


void C_flonum_floor(int c, C_word closure, C_word k, C_word n)
{
  C_temporary_flonum = floor(C_flonum_magnitude(n));
  C_cons_flonum(2, C_SCHEME_UNDEFINED, k);
}


void C_flonum_ceiling(int c, C_word closure, C_word k, C_word n)
{
  C_temporary_flonum = ceil(C_flonum_magnitude(n));
  C_cons_flonum(2, C_SCHEME_UNDEFINED, k);
}


void C_flonum_truncate(int c, C_word closure, C_word k, C_word n)
{
  modf(C_flonum_magnitude(n), &C_temporary_flonum);
  C_cons_flonum(2, C_SCHEME_UNDEFINED, k);
}


void C_flonum_round(int c, C_word closure, C_word k, C_word n)
{
  double fn, i, f, i2;

  f = fabs(modf(fn = C_flonum_magnitude(n), &i));

  if(f == 0.5) {
    i2 = fn + 0.5;

    if(modf(i2 / 2, &i) == 0) C_temporary_flonum = i2;
    else C_temporary_flonum = fn - 0.5;
  }
  else if(f > 0.5) C_temporary_flonum = (i < 0) ? i - 1 : i + 1;
  else C_temporary_flonum = i;

  C_cons_flonum(2, C_SCHEME_UNDEFINED, k);
}


void C_quotient(int c, C_word closure, C_word k, C_word n1, C_word n2)
{
  double f1, f2;
  C_word result;

  if(c != 4) C_bad_argc(c, 4);

  if(n1 &C_FIXNUM_BIT) {
    if(n2 &C_FIXNUM_BIT) {
      if((n2 = C_unfix(n2)) == 0)
	barf(C_DIVISION_BY_ZERO_ERROR);
      
      result = C_fix(C_unfix(n1) / n2);
      C_kontinue(k, result);
    }
    else if(!C_immediatep(n2) && C_header_bits(n2) == C_FLONUM_TYPE) {
      f1 = (double)C_unfix(n1);
      f2 = C_flonum_magnitude(n2);
    }
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, n2);
  }
  else if(!C_immediatep(n1) && C_header_bits(n1) == C_FLONUM_TYPE) {
    f1 = C_flonum_magnitude(n1);

    if(n2 &C_FIXNUM_BIT)
      f2 = (double)C_unfix(n2);
    else if(!C_immediatep(n2) && C_header_bits(n2) == C_FLONUM_TYPE)
      f2 = C_flonum_magnitude(n2);
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, n2);
  }
  else barf(C_BAD_ARGUMENT_TYPE_ERROR, n1);

  if(f2 == 0)
    barf(C_DIVISION_BY_ZERO_ERROR);

  modf(f1 / f2, &C_temporary_flonum);
  C_cons_flonum(2, C_SCHEME_UNDEFINED, k);
}


void C_cons_flonum(int c, C_word closure, C_word k)
{
  C_word *a = C_alloc(WORDS_PER_FLONUM);

  C_kontinue(k, C_flonum(&a, C_temporary_flonum));
}


void C_string_to_number(int c, C_word closure, C_word k, C_word str, ...)
{
  int radix, radixpf = 0, sharpf = 0, ratp = 0, exactf, exactpf = 0;
  C_word n1, n, *a = C_alloc(WORDS_PER_FLONUM);
  C_char *sptr, *eptr;
  double fn1, fn;
  va_list v;

  if(c == 3) radix = 10;	/* default radix is 10 */
  else if(c == 4) {
    va_start(v, str);
    radix = va_arg(v, C_word);
    va_end(v);
    
    if(radix & C_FIXNUM_BIT) radix = C_unfix(radix);
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, radix);
  }
  else C_bad_argc(c, 2);

  if(C_immediatep(str) || C_header_bits(str) != C_STRING_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, str);

  if((n = C_header_size(str)) == 0) {
  fail:
    n = C_SCHEME_FALSE;
    goto fini;
  }

  C_memcpy(sptr = buffer, C_c_string(str), n);
  buffer[ n ] = '\0';
  
  while(*sptr == '#') {
    switch(*(++sptr)) {
    case 'b': if(radixpf) goto fail; else { radix = 2; radixpf = 1; } break;
    case 'o': if(radixpf) goto fail; else { radix = 8; radixpf = 1; } break;
    case 'd': if(radixpf) goto fail; else { radix = 10; radixpf = 1; } break;
    case 'x': if(radixpf) goto fail; else { radix = 16; radixpf = 1; } break;
    case 'e': if(exactpf) goto fail; else { exactf = 1; exactpf = 1; } break;
    case 'i': if(exactpf) goto fail; else { exactf = 0; exactpf = 1; } break;
    default: goto fail;
    }

    ++sptr;
  }

  /* check for embedded '#'s: */
  while((eptr = strchr(sptr, '#')) != NULL) {
    if(eptr[ 1 ] == '\0' || strchr("#.0123456789", eptr[ 1 ]) != NULL) {
      sharpf = 1;
      *eptr = '0';
    }
    else {
      n = C_SCHEME_FALSE;
      goto fini;
    }
  }

  /* check for rational representation: */
  if((eptr = strchr(sptr, '/')) != NULL) {
    *eptr = '\0';
    ratp = 1;

    switch(convert_string_to_number(sptr, radix, &n1, &fn1)) {
    case 0:
      n = C_SCHEME_FALSE;
      goto fini;

    case 1:
      fn1 = (double)n1;
      break;

      /* case 2: nop */
    }

    sptr = eptr + 1;
  }    
  
  /* convert number and return result: */
  switch(convert_string_to_number(sptr, radix, &n, &fn)) {
  case 0: 
    n = C_SCHEME_FALSE;
    break;

  case 1:
    if(sharpf || ratp || (exactpf && !exactf)) {
      n = C_flonum(&a, ratp ? fn1 / (double)n : (double)n);

      if(exactpf && exactf) n = C_i_inexact_to_exact(n);
    }
    else n = C_fix(n);

    break;

  case 2:
    n = C_flonum(&a, ratp ? fn1 / fn : fn);

    if(exactpf && exactf) n = C_i_inexact_to_exact(n);

    break;
  }

 fini:
  C_kontinue(k, n);
}


C_word C_fcall convert_string_to_number(C_char *str, int radix, C_word *fix, double *flo)
{
  unsigned long ln;
  C_word n;
  C_char *eptr;
  double fn;

  if(strpbrk(str, "xX\0") != NULL) return 0;

  errno = 0;
  n = strtol(str, &eptr, radix);
  
  if(((n == LONG_MAX || n == LONG_MIN) && errno == ERANGE) || *eptr != '\0') {
    if(radix != 10) {
      errno = 0;
      ln = strtoul(str, &eptr, radix);
      
      if((ln == 0 && errno == EINVAL) || (ln == ULONG_MAX && errno == ERANGE)) return 0;

      *flo = (double)ln;
      return 2;
    }

    errno = 0;
    fn = strtod(str, &eptr);

    if((fn == HUGE_VAL && errno == ERANGE) || *eptr != '\0') return 0;

    *flo = fn;
    return 2;
  }
  else if((n & C_INT_SIGN_BIT) != ((n << 1) & C_INT_SIGN_BIT)) { /* doesn't fit into fixnum? */
    if(*eptr == '\0') {
      *flo = (double)n;
      return 2;
    }
    else return 0;
  }
  else {
    *fix = n;
    return 1;
  }
}


void C_number_to_string(int c, C_word closure, C_word k, C_word num, ...)
{
  C_word radix, *a;
  C_char *p;
  double f;
  va_list v;

  if(c == 3) radix = 10;
  else if(c == 4) {
    va_start(v, num);
    radix = va_arg(v, C_word);
    va_end(v);
    
    if(radix & C_FIXNUM_BIT) radix = C_unfix(radix);
    else barf(C_BAD_ARGUMENT_TYPE_ERROR, radix);
  }
  else C_bad_argc(c, 2);

  if(num & C_FIXNUM_BIT) {
    num = C_unfix(num);

    switch(radix) {
    case 2:
      buffer[ 65 ] = '\0';
      p = buffer + 65;

      do {
	*(--p) = (num & 1) ? '1' : '0';
	num /= 2;
      } while(num);

      break;
     
    case 8: sprintf(p = buffer, C_text("%o"), num); break;
    case 10: sprintf(p = buffer, C_text("%d"), num); break;
    case 16: sprintf(p = buffer, C_text("%x"), num); break;
    default: barf(C_BAD_ARGUMENT_TYPE_ERROR, C_fix(radix));
    }

  fini:
    radix = C_strlen(p);
    
    if(!C_demand(C_bytestowords(radix) + 1)) {
      C_save(k);
      cons_string_trampoline(NULL);
    }

    a = C_alloc((C_bytestowords(radix) + 1));
    radix = C_string(&a, radix, p);
    C_kontinue(k, radix);
  }

  if(!C_immediatep(num) && C_header_bits(num) == C_FLONUM_TYPE) {
    f = C_flonum_magnitude(num);

#ifdef HAVE_GCVT
    gcvt(f, FLONUM_PRINT_PRECISION, buffer);
#else
    sprintf(buffer, C_text("%.*g"), FLONUM_PRINT_PRECISION, f);
#endif

    if((p = strpbrk(buffer, C_text(".eE"))) == NULL)
      strcat(buffer, C_text("."));

    p = buffer;
    goto fini;
  }

  barf(C_BAD_ARGUMENT_TYPE_ERROR, num);
}


void cons_string_trampoline(void *dummy)
{
  int len = C_strlen(buffer);
  C_word k = C_restore,
      *a = C_alloc(C_bytestowords(len) + 1);

  C_kontinue(k, C_string(&a, len, buffer));
}


void C_get_argv(int c, C_word closure, C_word k)
{
  int i, cells;

  if(c != 2) C_bad_argc(c, 2);

  i = C_main_argc;
  cells = 0;

  while(i--)
    cells += 7 + C_align(C_strlen(C_main_argv[ i ]));

  C_save(k);
  C_save(C_fix(cells));

  if(!C_demand(cells)) C_reclaim(get_argv_2, NULL);

  get_argv_2(NULL);
}


void get_argv_2(void *dummy)
{
  int cells = C_unfix(C_restore),
      i = C_main_argc;
  C_word k = C_restore, 
         *a = C_alloc(cells),
         list, str;
  
  for(list = C_SCHEME_END_OF_LIST; i--; list = C_pair(&a, str, list))
    str = C_string2(&a, C_main_argv[ i ]);

  C_kontinue(k, list);
}


void C_make_structure(int c, C_word closure, C_word k, C_word type, ...)
{
  va_list v;
  int i;

  va_start(v, type);

  for(i = c - 3; i--; C_save(va_arg(v, C_word)));

  va_end(v);
  C_save(type);
  C_save(k);

  if(!C_demand(c - 1)) 
    C_reclaim(make_structure_2, NULL);

  make_structure_2(NULL);
}


void make_structure_2(void *dummy)
{
  C_word k = C_restore,
      type = C_restore,
      size = C_rest_count(0),
      *a = C_alloc(size + 2),
      *s = a,
      s0 = (C_word)s;

  *(s++) = C_STRUCTURE_TYPE | (size + 1);
  *(s++) = type;
  s += size;

  while(size--)
    *(--s) = C_restore;

  C_kontinue(k, s0);
}


void C_make_symbol(int c, C_word closure, C_word k, C_word name)
{
  C_word ab[ 3 ], *a = ab,
         s0 = (C_word)a;

  *(a++) = C_SYMBOL_TYPE | 2;
  *(a++) = C_SCHEME_UNBOUND;
  *a = name;
  C_kontinue(k, s0);
}


void C_make_pointer(int c, C_word closure, C_word k)
{
  C_word ab[ 2 ], *a = ab,
         p;

  p = C_mpointer(&a, NULL);
  C_kontinue(k, p);
}


void C_ensure_heap_reserve(int c, C_word closure, C_word k, C_word n)
{
  C_save(k);

  if(!C_demand(C_bytestowords(C_unfix(n))))
    C_reclaim(ensure_heap_reserve_2, NULL);

  ensure_heap_reserve_2(NULL);
}


void ensure_heap_reserve_2(void *dummy)
{
  C_word k = C_restore;

  C_kontinue(k, C_SCHEME_UNDEFINED);
}


void C_call_host(int c, C_word closure, C_word k)
{
  C_save(k);
  return_to_host = 1;
  C_reclaim(call_host_2, NULL);
}


void call_host_2(void *dummy)
{
  C_word k = C_restore;

  C_kontinue(k, C_SCHEME_UNDEFINED);
}


void C_host_data(int c, C_word closure, C_word k, C_word index)
{
  int n, i = C_unfix(index);
  C_word *a;
  char *ptr;

  if(host_exchange_buffer == NULL) C_kontinue(k, C_SCHEME_FALSE);

  if((ptr = host_exchange_buffer[ i ]) == NULL) C_kontinue(k, C_SCHEME_FALSE);

  n = nmin(host_exchange_bytes[ i ], STRING_BUFFER_SIZE - 1);
  a = C_alloc(C_bytestowords(n) + 2);

  C_kontinue(k, C_string_aligned8(&a, n, ptr));
}


void C_set_host_data(int c, C_word closure, C_word k, C_word index, C_word data)
{
  int n,
      i = C_unfix(index);

  if(host_exchange_buffer != NULL) {
    if(data == C_SCHEME_FALSE) {
      host_exchange_buffer[ i ] = NULL;
      host_exchange_bytes[ i ] = 0;
    }
    else {
      n = C_header_size(data);
      n = nmin(host_exchange_maximum[ i ] - 1, n);
      C_memcpy(host_exchange_buffer[ i ], C_c_string(data), n);
      host_exchange_buffer[ i ][ host_exchange_bytes[ i ] = n ] = '\0';
    }
  }
  
  C_kontinue(k, C_SCHEME_UNDEFINED);
}


void C_file_info(int c, C_word closure, C_word k, C_word name)
{
  C_save(k);
  C_save(name);
  
  if(!C_demand(FILE_INFO_SIZE + 1)) C_reclaim(file_info_2, NULL);

  file_info_2(NULL);
}


void file_info_2(void *dummy)
{
  C_word name = C_restore,
      k = C_restore,
      *a = C_alloc(FILE_INFO_SIZE + 1),
      v = C_SCHEME_FALSE,
      t;
  int len = C_header_size(name);

#ifdef _MSC_VER
  struct _stat buf;
#else
  struct stat buf;
#endif
  strncpy(buffer, C_c_string(name), len);
  buffer[ len ] = '\0';

#ifdef _MSC_VER
  if(_stat(buffer, &buf) != 0) v = C_SCHEME_FALSE;
#else
  if(stat(buffer, &buf) != 0) v = C_SCHEME_FALSE;
#endif
  else {
    switch(buf.st_mode & S_IFMT) {
    case S_IFDIR: t = 1; break;
#if !defined(_MSC_VER)
    case S_IFIFO: t = 3; break;
# if !defined(__DJGPP__) && !defined(__MINGW32__)
    case S_IFLNK: t = 2; break;
    case S_IFSOCK: t = 4; break;
# endif
#endif
    default: t = 0;
    }

    v = C_vector(&a, FILE_INFO_SIZE, C_fix(buf.st_atime), C_fix(buf.st_ctime), C_fix(buf.st_mtime),
		 C_fix(buf.st_size), C_fix(t), C_fix(buf.st_mode), C_fix(buf.st_uid) ); 
  }

  C_kontinue(k, v);
}


void C_get_environment_variable(int c, C_word closure, C_word k, C_word name)
{
  int len;
  C_char *val;

  if(c != 3) C_bad_argc(c, 3);

  if(C_immediatep(name) || C_header_bits(name) != C_STRING_TYPE)
    barf(C_BAD_ARGUMENT_TYPE_ERROR, name);
  
  if((len = C_header_size(name)) >= STRING_BUFFER_SIZE)
    C_kontinue(k, C_SCHEME_FALSE);

  strncpy(buffer, C_c_string(name), len);
  buffer[ len ] = '\0';

  if((val = getenv(buffer)) == NULL)
    C_kontinue(k, C_SCHEME_FALSE);

  C_strcpy(buffer, val);
  C_save(k);
  
  if(!C_demand(1 + C_bytestowords(len + 1)))
    C_reclaim(get_environment_variable_2, NULL);

  get_environment_variable_2(NULL);
}


void get_environment_variable_2(void *dummy)
{
  int len = C_strlen(buffer);
  C_word k = C_restore,
         *a = C_alloc(1 + C_bytestowords(len + 1)),
         str = C_string(&a, len, buffer);

  C_kontinue(k, str);
}


void C_get_symbol_table_info(int c, C_word closure, C_word k)
{
  double d1, d2;
  int total;
  C_word x, y, ab[ WORDS_PER_FLONUM * 2 + 4 ], *a = ab; /* 2 flonums + 1 vector of 3 elements */

  d1 = compute_symbol_table_load(&d2, &total);
  x = C_flonum(&a, d1);
  y = C_flonum(&a, d2);
  C_kontinue(k, C_vector(&a, 3, x, y, C_fix(total)));
}


void C_get_memory_info(int c, C_word closure, C_word k)
{
  C_word ab[ 3 ], *a = ab;

  C_kontinue(k, C_vector(&a, 2, C_fix(heap_size), C_fix(stack_size)));
}


void C_context_switch(int c, C_word closure, C_word k, C_word state)
{
  C_word n = C_header_size(state) - 1,
         adrs = C_block_item(state, 0);
  TRAMPOLINE trampoline;

  C_temporary_stack = C_temporary_stack_bottom - n;
  C_memcpy(C_temporary_stack, (C_word *)state + 2, n * sizeof(C_word));
  trampoline = (TRAMPOLINE)C_block_item(adrs, 0);
  trampoline((void *)C_block_item(adrs, 1));
}


void C_peek_signed_integer(int c, C_word closure, C_word k, C_word v, C_word index)
{
  C_word x = C_block_item(v, C_unfix(index));

  if((x & C_INT_SIGN_BIT) != ((x << 1) & C_INT_SIGN_BIT)) {
    C_save(k);
    C_temporary_flonum = (double)x;
    cons_flonum_trampoline(NULL);
  }

  C_kontinue(k, C_fix(x));
}


void C_peek_unsigned_integer(int c, C_word closure, C_word k, C_word v, C_word index)
{
  C_word x = C_block_item(v, C_unfix(index));

  if((x & C_INT_SIGN_BIT) || ((x << 1) & C_INT_SIGN_BIT)) {
    C_save(k);
    C_temporary_flonum = (double)(C_uword)x;
    cons_flonum_trampoline(NULL);
  }

  C_kontinue(k, C_fix(x));
}


void C_decode_seconds(int c, C_word closure, C_word k, C_word secs, C_word mode)
{
  time_t tsecs;
  struct tm *tmt;
  C_word ab[ 11 ], *a = ab,
         info;

  tsecs = (time_t)C_unfix(secs);
  
  if(mode == C_SCHEME_FALSE) tmt = localtime(&tsecs);
  else tmt = gmtime(&tsecs);

  if(tmt  == NULL)
    C_kontinue(k, C_SCHEME_FALSE);
  
  info = C_vector(&a, 10, C_fix(tmt->tm_sec), C_fix(tmt->tm_min), C_fix(tmt->tm_hour),
		  C_fix(tmt->tm_mday), C_fix(tmt->tm_mon), C_fix(tmt->tm_year),
		  C_fix(tmt->tm_wday), C_fix(tmt->tm_yday),
		  tmt->tm_isdst > 0 ? C_SCHEME_TRUE : C_SCHEME_FALSE,
		  C_fix(timezone) );
  C_kontinue(k, info);
}


void C_machine_type(int c, C_word closure, C_word k)
{
  C_word *a, s;

  if(c != 2) C_bad_argc(c, 2);

#if defined(__alpha__)
  a = C_alloc(2 + C_bytestowords(5));
  s = C_string2(&a, "alpha");
#elif defined(__sparc__)
  a = C_alloc(2 + C_bytestowords(5));
  s = C_string2(&a, "sparc");
#elif defined(__sparc_v9__) || defined(__sparcv9)
  a = C_alloc(2 + C_bytestowords(10));
  s = C_string2(&a, "ultrasparc");
#elif defined(_M_IX86) || defined(__i386__)
  a = C_alloc(2 + C_bytestowords(3));
  s = C_string2(&a, "x86");
#else
  a = C_alloc(2 + C_bytestowords(7));
  s = C_string2(&a, "unknown");
#endif
  
  C_kontinue(k, s);
}


void C_software_type(int c, C_word closure, C_word k)
{
  C_word *a, s;

  if(c != 2) C_bad_argc(c, 2);

#if defined(__DJGPP__)
  a = C_alloc(2 + C_bytestowords(3));
  s = C_string2(&a, "dos");
#elif defined(__CYGWIN__) || defined(_MSC_VER) || defined(__MINGW32__)
  a = C_alloc(2 + C_bytestowords(7));
  s = C_string2(&a, "windows");
#elif defined(C_MACOSX)
  a = C_alloc(2 + C_bytestowords(6));
  s = C_string2(&a, "macosx");
#elif defined(__GNUC__)
  a = C_alloc(2 + C_bytestowords(4));
  s = C_string2(&a, "unix");
#elif defined(__MWERKS__) && !defined(__INTEL__)
  a = C_alloc(2 + C_bytestowords(3));
  s = C_string2(&a, "mac");
#else
  a = C_alloc(2 + C_bytestowords(7));
  s = C_string2(&a, "unknown");
#endif

 C_kontinue(k, s);
}


/* Register finalizer: */

void C_register_finalizer(int c, C_word closure, C_word k, C_word x, C_word proc)
{
  if(C_immediatep(x)) C_kontinue(k, C_SCHEME_UNDEFINED);

  C_save(k);
  C_save(x);
  C_save(proc);
  C_reclaim(register_finalizer_2, NULL);
}


void register_finalizer_2(void *dummy)
{
  C_word proc = C_restore,
         x = C_restore,
         k = C_restore,
         *ptr;
  int n, i;

  for(n = 0; n < finalizer_table_size; ++n)
    if(finalizer_table[ n * 2 ] == C_SCHEME_UNDEFINED) goto ok;

  /* Grow finalizer table to twice it's original size: */
  n = finalizer_table_size;

  if((ptr = (C_word *)C_malloc(n * 2 * 2 * sizeof(C_word))) == NULL) /* two slots for each entry */
    panic(C_text("out of memory - can not resize finalizer table"));

  C_memcpy(ptr, finalizer_table, n * 2 * sizeof(C_word));
  C_free(finalizer_table);
  finalizer_table = ptr;
  finalizer_table_size = n * 2;

  for(i = 0; i < n; ++i) {
    finalizer_table[ (n + i) * 2 ] = C_SCHEME_UNDEFINED;
  }

 ok:
  if(n > finalizer_table_maximum) finalizer_table_maximum = n;

  assert(finalizer_table[ n * 2 ] == C_SCHEME_UNDEFINED);
  finalizer_table[ n * 2 ] = x;
  finalizer_table[ n * 2 + 1 ] = proc;
  C_kontinue(k, C_SCHEME_UNDEFINED);
}
