;;;; support.scm - Miscellaneous support code for the CHICKEN compiler
;
; Copyright (c) 2000-2002, Felix L. Winkelmann
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following
; conditions are met:
;
;   Redistributions of source code must retain the above copyright notice, this list of conditions and the following
;     disclaimer. 
;   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
;     disclaimer in the documentation and/or other materials provided with the distribution. 
;   Neither the name of the author nor the names of its contributors may be used to endorse or promote
;     products derived from this software without specific prior written permission. 
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
; AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
; CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
; POSSIBILITY OF SUCH DAMAGE.
;
; Send bugs, suggestions and ideas to: 
;
; felix@call-with-current-continuation.org
;
; Felix L. Winkelmann
; Steinweg 1A
; 37130 Gleichen, OT Weissenborn
; Germany


(declare (unit support))


#{compiler
  build-information compiler-arguments process-command-line
  default-analysis-database-size default-standard-bindings default-extended-bindings side-effecting-standard-bindings
  non-foldable-standard-bindings foldable-standard-bindings non-foldable-extended-bindings foldable-extended-bindings
  standard-bindings-that-never-return-false side-effect-free-standard-bindings-that-never-return-false
  installation-home optimization-iterations compiler-cleanup-hook decompose-lambda-list
  foreign-type-table-size file-io-only nonwinding-call/cc
  unit-name insert-timer-checks used-units inlining debug-info-index debug-info-vector-name
  foreign-declarations block-compilation analysis-database-size line-number-database-size
  target-heap-size target-stack-size try-harder default-installation-home 
  default-default-target-heap-size default-default-target-stack-size verbose-mode original-program-size
  current-program-size line-number-database-2 foreign-lambda-stubs immutable-constants foreign-variables
  rest-parameters-promoted-to-vector inline-table inline-table-used constant-table constants-used mutable-constants
  dependency-list broken-constant-nodes inline-substitutions-enabled
  always-bound-to-procedure block-variable-literal
  direct-call-ids foreign-type-table first-analysis expand-debug-call
  expand-profile-lambda profile-lambda-list profile-lambda-index profile-info-vector-name
  initialize-compiler canonicalize-expression expand-foreign-lambda update-line-number-database scan-toplevel-assignments
  perform-cps-conversion analyze-expression simplifications perform-high-level-optimizations perform-pre-optimization!
  reorganize-recursive-bindings substitution-table simplify-named-call find-inlining-candidates perform-inlining!
  perform-closure-conversion prepare-for-code-generation compiler-source-file create-foreign-stub expand-foreign-lambda*
  transform-direct-lambdas! expand-debug-lambda expand-debug-assignment debug-lambda-list debug-variable-list
  debugging-chicken warnings-enabled bomb check-signature posq stringify symbolify flonum? build-lambda-list
  string->c-identifier c-ify-string words check-and-open-input-file close-checked-input-file fold-inner constant?
  collapsable-literal? immediate? canonicalize-begin-body extract-mutable-constants string->expr get get-all
  put! collect! count! get-line get-line-2 find-lambda-container display-analysis-database varnode qnode 
  build-node-graph build-expression-tree fold-boolean inline-lambda-bindings match-node expression-has-side-effects?
  simple-lambda-node? compute-database-statistics print-program-statistics output gen gen-list 
  pprint-expressions-to-file foreign-type-check estimate-foreign-result-size scan-used-variables scan-free-variables
  topological-sort print-version print-usage initialize-analysis-database write-dependency-list
  product copyright compiler-features default-declarations units-used-by-default words-per-flonum
  foreign-string-result-reserve parameter-limit default-output-filename eq-inline-operator optimizable-rest-argument-operators
  membership-test-operators membership-unfold-limit valid-compiler-options valid-compiler-options-with-argument
  default-optimization-iterations chop-separator chop-extension current-source-file source-file-changed show-source-file
  generate-code make-variable-list make-argument-list generate-foreign-stubs foreign-type-declaration
  foreign-argument-conversion foreign-result-conversion final-foreign-type
  make-random-name foreign-type-convert-result foreign-type-convert-argument}


(include "parameters")
(include "tweaks")


;;; Debugging and error-handling stuff:

(define (compiler-cleanup-hook) #f)

(define debugging-chicken '())
(define warnings-enabled #t)
(define current-source-file #f)
(define source-file-changed #t)

(define (bomb . msg-and-args)
  (if (pair? msg-and-args)
      (apply error (string-append "[internal compiler screwup] " (car msg-and-args)) (cdr msg-and-args))
      (error "[internal compiler screwup]") ) )

(define (debugging mode msg . args)
  (and (memq mode debugging-chicken)
       (begin
	 (printf "~a" msg)
	 (if (pair? args)
	     (begin
	       (display ": ")
	       (for-each (lambda (x) (printf "~s " (force x))) args) ) )
	 (newline)
	 (flush-output)
	 #t) ) )

(define (show-source-file)
  (when (and current-source-file source-file-changed)
    (fprintf (current-error-port) "~A:~%" current-source-file)
    (set! source-file-changed #f) ) )

(define (warning msg . args)
  (when warnings-enabled
    (let ((out (current-error-port)))
      (show-source-file)
      (apply fprintf out (string-append "Warning: " msg) args)
      (newline out) ) ) )

(define quit-handler
  (make-parameter (lambda (code) (exit code))) )

(define (quit msg . args)
  (let ([out (current-error-port)])
    (show-source-file)
    (apply fprintf out (string-append "Error: " msg) args)
    (newline out)
    ((quit-handler) 1) ) )

(set! ##sys#syntax-error-hook
  (lambda (msg . args)
    (let ([out (current-error-port)])
      (show-source-file)
      (fprintf out "Syntax error: ~a~%" msg) 
      (for-each (lambda (x) (write x out) (newline out)) args)
      (exit 70) ) ) )

(define (check-signature args llist)
  (define (err)
    (quit "Arguments to inlined call do not match parameter-list ~S" llist) )
  (let loop ([as args] [ll llist])
    (cond [(null? ll) (unless (null? as) (err))]
	  [(symbol? ll)]
	  [(null? as) (err)]
	  [else (loop (cdr as) (cdr ll))] ) ) )


;;; Generic utility routines:

(define (posq x lst)
  (let loop ([lst lst] [i 0])
    (cond [(null? lst) #f]
	  [(eq? x (car lst)) i]
	  [else (loop (cdr lst) (add1 i))] ) ) )

(define (stringify x)
  (cond ((string? x) x)
	((symbol? x) (symbol->string x))
	(else (sprintf "~a" x)) ) )

(define (symbolify x)
  (cond ((symbol? x) x)
	((string? x) (string->symbol x))
	(else (string->symbol (sprintf "~a" x))) ) )

(define (flonum? x)
  (and (number? x) (not (exact? x))) )

(define (build-lambda-list vars argc rest)
  (let loop ((vars vars) (n argc))
    (cond ((or (zero? n) (null? vars)) (or rest '()))
          (else (cons (car vars) (loop (cdr vars) (sub1 n)))) ) ) )

(define (string->c-identifier str)
  (let ((lst (string->list str)))
    (list->string
     (map (lambda (c) 
	    (if (and (not (char-numeric? c)) (not (char-alphabetic? c)))
		#\_
		c) )
	  (if (not (char-alphabetic? (car lst)))
	      (cons #\_ lst)
	      lst) ) ) ) )

(define (c-ify-string str)
  (list->string
   (cons 
    #\"
    (let loop ((chars (string->list str)))
      (if (null? chars)
	  '(#\")
	  (let* ((c (car chars))
		 (code (char->integer c)) )
	    (if (or (< code 32) (memq c '(#\" #\' #\\)))
		(append '(#\\)
			(cond ((< code 8) '(#\0 #\0))
			      ((< code 64) '(#\0))
			      (else '()) )
			(string->list (number->string code 8))
			(loop (cdr chars)) )
		(cons c (loop (cdr chars))) ) ) ) ) ) ) )

(eval-when (load)
  (define words (foreign-lambda int "C_bytestowords" int)) )

(eval-when (eval)
  (define words
    (let ([wordsize (##sys#fudge 7)])
      (lambda (n)
	(+ (quotient n wordsize) (if (zero? (modulo n wordsize)) 0 1)) ) ) ) )

(define (check-and-open-input-file fname . line)
  (cond [(string=? fname "-") (current-input-port)]
	[(file-exists? fname) (open-input-file fname)]
	[(or (null? line) (not (car line))) (quit "Can not open file ~s" fname)]
	[else (quit "Can not open file ~s in line ~s" fname (car line))] ) )

(define (close-checked-input-file port fname)
  (unless (string=? fname "-") (close-input-port port)) )

(define (fold-inner proc lst)
  (if (null? (cdr lst)) 
      lst
      (let fold ((xs (reverse lst)))
	(apply
	 proc 
	 (if (null? (cddr xs))
	     (list (cadr xs) (car xs))
	     (list (fold (cdr xs)) (car xs)) ) ) ) ) )


;;; Predicates on expressions and literals:

(define (constant? x)
  (or (number? x)
      (char? x)
      (string? x)
      (boolean? x)
      (and (pair? x) (eq? 'quote (car x))) ) )

(define (collapsable-literal? x)
  (or (boolean? x)
      (char? x)
      (number? x)
      (symbol? x) ) )

(define (immediate? x)
  (or (fixnum? x)
      (eq? (##core#undefined) x)
      (null? x)
      (char? x)
      (boolean? x) ) )


;;; Expression manipulation:

(define (canonicalize-begin-body body)
  (let loop ((xs body))
    (cond ((null? xs) '(##core#undefined))
	  ((null? (cdr xs)) (car xs))
	  ((or (equal? (car xs) '(##core#undefined)) (constant? (car xs))) (loop (cdr xs)))
	  (else `(let ((,(gensym 't) ,(car xs)))
		   ,(loop (cdr xs))) ) ) ) )

(define (extract-mutable-constants exp)
  (let ([mlist '()])
    (define (walk x)
      (match x
	[(? not-pair? x) x]
	[`(quote ,c)
	 (if (not (collapsable-literal? c))
	     (let ([var (make-random-name)])
	       (set! mlist (alist-cons var c mlist))
	       var)
	     x) ]
	[`(let ((vars vals) ...) . body)
	 `(let ,(map (lambda (var val) (list var (walk val))) vars vals) ,@(map walk body)) ]
	[(op . args)
	 (case op
	   [(##core#include ##core#declare ##core#immutable ##core#undefined ##core#primitive ##core#inline_ref) x]
	   [(set! lambda ##core#inline ##core#inline_allocate ##core#inline_update)
	    (cons* op (first args) (map walk (cdr args))) ]
	   [(if ##core#compiletimeonly ##core#compiletimetoo)
	    (cons op (map walk args)) ]
	   [else (map walk x)] ) ]
	[_ x] ) ) 
    (let ([exp2 (walk exp)])
      (values exp2 mlist) ) ) )

(define (string->expr str)
  (parameterize ([error-handler
		  (lambda (msg . args)
		    (quit "can not parse expression: ~s [~a]~%" str msg) ) ] )
    (let ([xs (with-input-from-string str (lambda () (unfold eof-object? values (lambda (x) (read)) (read))))])
      (cond [(null? xs) '(##core#undefined)]
	    [(null? (cdr xs)) (car xs)]
	    [else `(begin ,@xs)] ) ) ) )

(define decompose-lambda-list ##sys#decompose-lambda-list)


;;; Debugging expression instrumentation:

(define (expand-debug-lambda name llist ollist argc rest body obody)
  (let* ([exit (gensym 'exit)]
	 [args (gensym)]
	 [source 
	  (##sys#string->pbytevector
	   (with-output-to-string
	     (lambda () (pretty-print `(lambda ,ollist ,@obody))) ) ) ] 
	 ;; #(name lambda-list source argc rest-flag break? exec?)
	 [info (vector (or name anonymous-object-identifier) ollist source argc (if rest #t #f) #f #f)] 
	 [index debug-info-index] )
    (set! debug-lambda-list (alist-cons index info debug-lambda-list))
    (set! debug-info-index (add1 index))
    `(lambda ,args
       (call-with-current-continuation
	(lambda (,exit)
	  (##sys#dynamic-wind
	   (lambda () (##sys#push-debug-frame (##core#inline "C_slot" ,debug-info-vector-name ',index) ,exit) )
	   (lambda () (apply (lambda ,llist ,body) (##sys#check-debug-entry ,args)))
	   (lambda () (##sys#pop-debug-frame)) ) ) ) ) ) )

(define (expand-debug-assignment var val oexp)
  (let* ([source 
	  (##sys#string->pbytevector
	   (with-output-to-string (lambda () (pretty-print oexp))) ) ] 
	 ;; #(name source watched?)
	 [info (vector var source #f)]
	 [index debug-info-index] )
    (set! debug-info-index (add1 index))
    (set! debug-variable-list (alist-cons index info debug-variable-list))
    `(##sys#check-debug-assignment (##core#inline "C_slot" ,debug-info-vector-name ',index) ,val) ) )

(define (expand-debug-call name exp)
  `(##sys#debug-call ',name ,@exp) )


;;; Profiling instrumentation:

(define (expand-profile-lambda name llist body)
  (let ([index profile-lambda-index] 
	[args (gensym)] )
    (set! profile-lambda-list (alist-cons index name profile-lambda-list))
    (set! profile-lambda-index (add1 index))
    `(lambda ,args
       (##sys#dynamic-wind
	(lambda () (##sys#profile-entry ',index ,profile-info-vector-name))
	(lambda () (apply (lambda ,llist ,body) ,args))
	(lambda () (##sys#profile-exit ',index ,profile-info-vector-name)) ) ) ) )


;;; Database operations:
;
; - 'get' and 'put' shadow the routines in the extras-unit, we use low-level
;   symbol-keyed hash-tables here.

(define (initialize-analysis-database db)
  (for-each
   (lambda (s) 
     (put! db s 'standard-binding #t)
     (when (memq s side-effecting-standard-bindings) (put! db s 'side-effecting #t))
     (when (memq s foldable-standard-bindings) (put! db s 'foldable #t)) )
   standard-bindings)
  (for-each
   (lambda (s)
     (put! db s 'extended-binding #t)
     (when (memq s foldable-extended-bindings) (put! db s 'foldable #t)) )
   extended-bindings)
  (for-each
   (lambda (s) (put! db (car s) 'constant #t))
   mutable-constants) )

(define (get db key prop)
  (let ((plist (##sys#hash-table-ref db key)))
    (and plist
	 (let ([a (assq prop plist)])
	   (and a (##sys#slot a 1)) ) ) ) )

(define (get-all db key . props)
  (let ((plist (##sys#hash-table-ref db key)))
    (if plist
	(filter-map (lambda (prop) (assq prop plist)) props)
	'() ) ) )

(define (put! db key prop val)
  (let ([plist (##sys#hash-table-ref db key)])
    (if plist
	(let ([a (assq prop plist)])
	  (cond [a (##sys#setslot a 1 val)]
		[val (##sys#setslot plist 1 (alist-cons prop val (##sys#slot plist 1)))] ) )
	(when val (##sys#hash-table-set! db key (list (cons prop val)))) ) ) )

(define (collect! db key prop val)
  (let ((plist (##sys#hash-table-ref db key)))
    (if plist
	(let ([a (assq prop plist)])
	  (cond [a (##sys#setslot a 1 (cons val (##sys#slot a 1)))]
		[else (##sys#setslot plist 1 (alist-cons prop (list val) (##sys#slot plist 1)))] ) )
	(##sys#hash-table-set! db key (list (list prop val)))) ) )

(define (count! db key prop . val)
  (let ([plist (##sys#hash-table-ref db key)]
	[n (if (pair? val) (car val) 1)] )
    (if plist
	(let ([a (assq prop plist)])
	  (cond [a (##sys#setslot a 1 (+ (##sys#slot a 1) n))]
		[else (##sys#setslot plist 1 (alist-cons prop n (##sys#slot plist 1)))] ) )
	(##sys#hash-table-set! db key (list (cons prop val)))) ) )

(define (get-line exp)
  (get ##sys#line-number-database (car exp) exp) )

(define (get-line-2 exp)
  (let* ((name (car exp))
	 (lst (##sys#hash-table-ref ##sys#line-number-database name)) )
    (cond ((and lst (assq exp (cdr lst)))
	   => (lambda (a) (values (car lst) (cdr a))) )
	  (else (values name #f)) ) ) )

(define (find-lambda-container id cid db)
  (let loop ([id id])
    (or (eq? id cid)
	(let ([c (get db id 'contained-in)])
	  (and c (loop c)) ) ) ) )


;;; Display analysis database:

(define display-analysis-database
  (let ((names '((captured . cpt) (assigned . set) (boxed . box) (global . glo)
		 (contractable . con) (standard-binding . stb) (foldable . fld) (simple . sim)
		 (side-effecting . sef) (collapsable . col) (removable . rem) (constant . con)
		 (undefined . und) (replacing . rpg) (unused . uud) (extended-binding . xtb)
		 (customizable . cst) (has-unused-parameters . hup) (boxed-rest . bxr) ) ) )
    (lambda (db)
      (##sys#hash-table-for-each
       (lambda (sym plist)
	 (let ((val #f)
	       (csites '())
	       (refs '()) )
	   (write sym)
	   (let loop ((es plist))
	     (if (pair? es)
		 (begin
		   (case (caar es)
		     ((captured assigned boxed global contractable standard-binding foldable
		       side-effecting collapsable removable undefined replacing unused simple
		       has-unused-parameters extended-binding customizable constant boxed-rest)
		      (printf "\t~a" (cdr (assq (caar es) names))) )
		     ((unknown)
		      (set! val 'unknown) )
		     ((value)
		      (if (not (eq? val 'unknown)) (set! val (cdar es))) )
		     ((replacable home outermost contains contained-in use-expr closure-size rest-parameter
		       o-r/access-count captured-variables explicit-rest)
		      (printf "\t~a=~s" (caar es) (cdar es)) )
		     ((references)
		      (set! refs (cdar es)) )
		     ((call-sites)
		      (set! csites (cdar es)) )
		     (else (bomb "Illegal property" (car es))) )
		   (loop (cdr es)) ) ) )
	   (if (and val (not (eq? val 'unknown)))
	       (printf "\tval=~s" (cons (node-class val) (node-parameters val))) )
	   (if (pair? refs) (printf "\trefs=~s" (length refs)))
	   (if (pair? csites) (printf "\tcss=~s" (length csites)))
	   (newline) ) )
       db) ) ) )       


;;; Node creation and -manipulation:

(define-record node
  class					; symbol
  parameters				; (value...)
  subexpressions)			; (node...)

(define (varnode var) (make-node '##core#variable (list var) '()))
(define (qnode const) (make-node 'quote (list const) '()))

(define (build-node-graph exp)
  (let ([count 0])
    (define (walk x)
      (cond ((symbol? x) (varnode x))
	    ((not-pair? x) (bomb "bad expression"))
	    ((symbol? (car x))
	     (case (car x)
	       ((if ##core#undefined) (make-node (car x) '() (map walk (cdr x))))
	       ((quote)
		(let ((c (cadr x)))
		  (qnode (if (and (number? c)
				  (eq? 'fixnum number-type)
				  (not (integer? c)) )
			     (begin
			       (warning "literal '~s' is out of range - will be truncated to integer" c)
			       (inexact->exact (truncate c)) )
			     c) ) ) )
	       ((let)
		(let ([bs (cadr x)]
		      [body (caddr x)] )
		  (if (null? bs)
		      (walk body)
		      (make-node 'let (unzip1 bs)
				 (append (map (lambda (b) (walk (cadr b))) (cadr x))
					 (list (walk body)) ) ) ) ) )
	       ((lambda) (make-node 'lambda (list (cadr x)) (list (walk (caddr x)))))
	       ((set! ##core#inline ##core#callunit ##core#primitive ##core#proc) 
		(make-node (car x) (list (cadr x)) (map walk (cddr x))) )
	       ((##core#foreign-callback-wrapper)
		(let ([name (cadr (second x))])
		  (make-node
		   '##core#foreign-callback-wrapper
		   (list name (cadr (third x)) (cadr (fourth x)) (cadr (fifth x)))
		   (list (walk (sixth x))) ) ) )
	       ((##core#inline_allocate ##core#inline_ref ##core#inline_update)
		(make-node (first x) (second x) (map walk (cddr x))) )
	       (else
		(receive
		    (name ln) (get-line-2 x)
		  (make-node
		   '##core#call
		   (list (cond [(memq name always-bound-to-procedure)
				(set! count (add1 count))
				#t]
			       [else #f] )
			 (if ln
			     (string-append (##sys#symbol->qualified-string name) "@" (number->string ln))
			     (##sys#symbol->qualified-string name) ) )
		   (map walk x) ) ) ) ) )
	    (else (make-node '##core#call '(#f) (map walk x))) ) )
    (let ([exp2 (walk exp)])
      (debugging 'o "eliminated procedure checks" count)
      exp2) ) )

(define (build-expression-tree node)
  (let walk ((n node))
    (let ((subs (node-subexpressions n))
	  (params (node-parameters n)) 
	  (class (node-class n)) )
      (case class
	((if ##core#closure ##core#box ##core#cond) (cons class (map walk subs)))
	((##core#variable) (car params))
	((quote) `(quote ,(car params)))
	((let) `(let ((,(first params) ,(walk (first subs)))) ,(walk (second subs))))
	((##core#lambda) 
	 (list (if (second params)
		   'lambda
		   '##core#lambda)
	       (third params)
	       (walk (car subs)) ) )
	((##core#call) (map walk subs))
	((##core#callunit) (cons* '##core#callunit (car params) (map walk subs)))
	((##core#undefined) (list class))
	((##core#bind) 
	 (let loop ((n (car params)) (vals subs) (bindings '()))
	   (if (zero? n)
	       `(##core#bind ,(reverse bindings) ,(walk (car vals)))
	       (loop (- n 1) (cdr vals) (cons (walk (car vals)) bindings)) ) ) )
	((##core#unbox ##core#ref ##core#update ##core#update_i)
	 (cons* class (walk (car subs)) params (map walk (cdr subs))) ) 
	(else (cons class (append params (map walk subs)))) ) ) ) )

(define (fold-boolean proc lst)
  (let fold ([vars lst])
    (if (null? (cddr vars))
	(apply proc vars)
	(make-node 
	 '##core#inline '("C_and") 
	 (list (proc (first vars) (second vars))
	       (fold (cdr vars)) ) ) ) ) )

(define (inline-lambda-bindings llist args body)
  (decompose-lambda-list
   llist
   (lambda (vars argc rest)
     (receive (largs rargs) (split-at args argc)
       (fold-right
	(lambda (var val body) (make-node 'let (list var) (list val body)) )
	(if rest
	    (make-node
	     'let (list rest)
	     (list (make-node '##core#inline_allocate (list "C_a_i_list" (* 3 (length rargs))) rargs)
		   body) )
	    body)
	(take vars argc)
	largs) ) ) ) )

(define (copy-node! from to)
  (node-class-set! to (node-class from))
  (node-parameters-set! to (node-parameters from))
  (node-subexpressions-set! to (node-subexpressions from)) )


;;; Match node-structure with pattern:

(define (match-node node pat vars)
  (let ((env '()))

    (define (resolve v x)
      (cond ((assq v env) => (lambda (a) (equal? x (cdr a))))
	    ((memq v vars)
	     (set! env (alist-cons v x env))
	     #t)
	    (else (eq? v x)) ) )

    (define (match1 x p)
      (cond ((not-pair? p) (resolve p x))
	    ((not-pair? x) #f)
	    ((match1 (car x) (car p)) (match1 (cdr x) (cdr p)))
	    (else #f) ) )
    
    (define (matchn n p)
      (if (not-pair? p)
	  (resolve p n)
	  (and (eq? (node-class n) (first p))
	       (match1 (node-parameters n) (second p))
	       (let loop ((ns (node-subexpressions n))
			  (ps (cddr p)) )
		 (cond ((null? ps) (null? ns))
		       ((not-pair? ps) (resolve ps ns))
		       ((null? ns) #f)
		       (else (and (matchn (car ns) (car ps))
				  (loop (cdr ns) (cdr ps)) ) ) ) ) ) ) )

    (let ((r (matchn node pat)))
      (and r
	   (begin
	     (debugging 'a "matched" (node-class node) (node-parameters node) pat)
	     env) ) ) ) )


;;; Test nodes for certain properties:

(define (expression-has-side-effects? node db)
  (let walk ((n node))
    (let ((subs (node-subexpressions n)))
      (case (node-class n)
	((##core#variable quote ##core#undefined ##core#lambda ##core#proc) #f)
	((if let) (any walk subs))
	(else #t) ) ) ) )

(define (simple-lambda-node? node)
  (let* ([params (node-parameters node)]
	 [k (first (third params))] )
    (and (second params)
	 (let rec ([n node])
	   (case (node-class n)
	     [(##core#call)
	      (let* ([subs (node-subexpressions n)]
		     [f (first subs)] )
		(and (eq? '##core#variable (node-class f)) 
		     (eq? k (first (node-parameters f)))
		     (every rec (cdr subs)) ) ) ]
	     [(##core#callunit) #f]
	     [else (every rec (node-subexpressions n))] ) ) ) ) )
  

;;; Compute general statistics from analysis database:
;
; - Returns:
;
;   current-program-size
;   original-program-size
;   number of known variables
;   number of known procedures
;   number of global variables
;   number of known call-sites
;   number of database entries
;   average bucket load

(define (compute-database-statistics db)
  (let ((nprocs 0)
	(nvars 0)
	(nglobs 0)
	(entries 0)
	(nsites 0) )
    (##sys#hash-table-for-each
     (lambda (sym plist)
       (for-each
	(lambda (prop)
	  (set! entries (+ entries 1))
	  (case (car prop)
	    ((global) (set! nglobs (+ nglobs 1)))
	    ((value)
	     (set! nvars (+ nvars 1))
	     (if (eq? '##core#lambda (node-class (cdr prop)))
		 (set! nprocs (+ nprocs 1)) ) )
	    ((call-sites) (set! nsites (+ nsites (length (cdr prop))))) ) )
	plist) )
     db)
    (values current-program-size
	    original-program-size
	    nvars
	    nprocs
	    nglobs
	    nsites
	    entries
	    (let ((len (vector-length db))
		  (load 0) )
	      (do ((i 0 (+ i 1)))
		  ((>= i len) (/ load len))
		(set! load (+ load (length (vector-ref db i)))) ) ) ) ) )

(define (print-program-statistics db)
  (receive
   (size osize kvars kprocs globs sites entries load) (compute-database-statistics db)
   (when (debugging 's "program statistics:")
     (printf ";   program size: \t~s \toriginal program size: \t~s\n" size osize)
     (printf ";   variables with known values: \t~s\n" kvars)
     (printf ";   known procedures: \t~s\n" kprocs)
     (printf ";   global variables: \t~s\n" globs)
     (printf ";   known call sites: \t~s\n" sites) 
     (printf ";   database entries: \t~s \tload: \t~s\n" entries load) ) ) )


;;; Pretty-print expressions:

(define (pprint-expressions-to-file exps filename)
  (let ([port (if filename (open-output-file filename) (current-output-port))])
    (with-output-to-port port
      (lambda ()
	(for-each
	 (lambda (x)
	   (pretty-print x)
	   (newline) ) 
	 exps) ) )
    (when filename (close-output-port port)) ) )


;;; Create foreign type checking expression:

(define (foreign-type-check param type)
  (case type
    [(char unsigned-char) (if unsafe param `(##sys#foreign-char-argument ,param))]
    [(int unsigned-int short unsigned-short) (if unsafe param `(##sys#foreign-fixnum-argument ,param))]
    [(float double) (if unsafe param `(##sys#foreign-flonum-argument ,param))]
    [(pointer)
     `(if ,param
	  ,(if unsafe
	       param
	       `(##sys#foreign-block-argument ,param) )
	  '#f) ]
    [(integer long) (if unsafe param `(##sys#foreign-integer-argument ,param))]
    [(unsigned-integer unsigned-long)
     (if unsafe
	 param
	 `(##sys#foreign-unsigned-integer-argument ,param) ) ]
    [(c-pointer)
     `(if ,param
	  (##sys#foreign-pointer-argument ,param)
	  '#f) ]
    [(c-string)
     `(if ,param
	  ,(if unsafe 
	       `(##sys#make-c-string ,param)
	       `(##sys#make-c-string (##sys#foreign-string-argument ,param)) )
	  '#f) ]
    [else
     (cond [(and (symbol? type) (##sys#hash-table-ref foreign-type-table type))
	    => (lambda (t)
		 (foreign-type-check param (if (vector? t) (vector-ref t 0) t)) ) ]
	   [(pair? type)
	    (case (car type)
	      [(pointer function)
	       `(if ,param
		    (##sys#foreign-pointer-argument ,param)
		    '#f) ]
	      [else param] ) ]
	   [else param] ) ] ) )


;;; Compute foreign-type conversions:

(define (foreign-type-convert-result r t)
  (or (and-let* ([(symbol? t)]
		 [ft (##sys#hash-table-ref foreign-type-table t)] 
		 [(vector? ft)] )
	(list (vector-ref ft 2) r) )
      r) )

(define (foreign-type-convert-argument a t)
  (or (and-let* ([(symbol? t)]
		 [ft (##sys#hash-table-ref foreign-type-table t)] 
		 [(vector? ft)] )
	(list (vector-ref ft 1) a) )
      a) )

(define (final-foreign-type t)
  (cond [(##sys#hash-table-ref foreign-type-table t) 
	 => (lambda (t)
	      (final-foreign-type (if (vector? t) (vector-ref t 0) t))) ]
	[else t] ) )


;;; Compute foreign result size:

(define (estimate-foreign-result-size type)
  (case type
    ((char int short bool void unsigned-short scheme-object unsigned-char unsigned-int) 0)
    ((c-string float double c-pointer unsigned-integer long integer unsigned-long) 12)
    (else
     (cond [(and (symbol? type) (##sys#hash-table-ref foreign-type-table type))
	    => (lambda (t)
		 (estimate-foreign-result-size (if (vector? t) (vector-ref t 0) t)) ) ]
	   [(pair? type)
	    (case (car type)
	      [(pointer function) 12]
	      [else 0] ) ]
	   [else 0] ) ) ) )


;;; Scan expression-node for variable usage:

(define (scan-used-variables node vars)
  (let ([used '()])
    (let walk ([n node])
      (let ([subs (node-subexpressions n)])
	(case (node-class n)
	  [(##core#variable set!) 
	   (let ([var (first (node-parameters n))])
	     (when (and (memq var vars) (not (memq var used)))
	       (set! used (cons var used)) ) 
	     (for-each walk subs) ) ]
	  [(quote ##core#undefined ##core#primitive) #f]
	  [else (for-each walk subs)] ) ) )
    used) )


;;; Scan expression-node for free variables (that are not in env):

(define (scan-free-variables node)
  (let ((vars '()))

    (define (walk n e)
      (let ([subs (node-subexpressions n)]
	    [params (node-parameters n)] )
	(case (node-class n)
	  ((quote ##core#undefined ##core#primitive ##core#proc ##core#inline_ref) #f)
	  ((##core#variable) 
	   (let ((var (first params)))
	     (unless (memq var e) (set! vars (lset-adjoin eq? vars var))) ) )
	  ((set!)
	   (let ((var (first params)))
	     (unless (memq var e) (set! vars (lset-adjoin eq? vars var)))
	     (walk (car subs) e) ) )
	  ((let) 
	   (walk (first subs) e)
	   (walk (second subs) (append params e)) )
	  ((##core#lambda)
	   (decompose-lambda-list
	    (third params)
	    (lambda (vars argc rest)
	      (walk (first subs) (append vars e)) ) ) )
	  (else (walkeach subs e)) ) ) )

    (define (walkeach ns e)
      (for-each (lambda (n) (walk n e)) ns) )

    (walk node '())
    vars) )


;;; Simple topological sort:
;
; - Taken from SLIB (slightly adapted): Copyright (C) 1995 Mikael Djurfeldt

(define (topological-sort dag pred)
  (if (null? dag)
      '()
      (let* ((adj-table '())
	     (sorted '()))

	(define (insert x y)
	  (let loop ([at adj-table])
	    (cond [(null? at) (set! adj-table (cons (cons x y) adj-table))]
		  [(pred x (caar at)) (set-cdr! (car at) y)]
		  [else (loop (cdr at))] ) ) )
	
	(define (lookup x)
	  (let loop ([at adj-table])
	    (cond [(null? at) #f]
		  [(pred x (caar at)) (cdar at)]
		  [else (loop (cdr at))] ) ) )
	
	(define (visit u adj-list)
	  ;; Color vertex u
	  (insert u 'colored)
	  ;; Visit uncolored vertices which u connects to
	  (for-each (lambda (v)
		      (let ((val (lookup v)))
			(if (not (eq? val 'colored))
			    (visit v (or val '())))))
		    adj-list)
	  ;; Since all vertices downstream u are visited
	  ;; by now, we can safely put u on the output list
	  (set! sorted (cons u sorted)) )
	
	;; Hash adjacency lists
	(for-each (lambda (def) (insert (car def) (cdr def)))
		  (cdr dag))
	;; Visit vertices
	(visit (caar dag) (cdar dag))
	(for-each (lambda (def)
		    (let ((val (lookup (car def))))
		      (if (not (eq? val 'colored))
			  (visit (car def) (cdr def)))))
		  (cdr dag)) 
	sorted) ) )


;;; Write list of dependecies for makefile:

(define (write-dependency-list strlst limit out)
  (let loop ([len 0]
	     [lst (reverse strlst)] )
    (if (null? lst)
	(newline out)
	(let* ([d (car lst)]
	       [rest (cdr lst)] 
	       [len2 (+ len (string-length d) 1)] )
	  (cond [(>= len2 limit)
		 (fprintf out "\\~%\t ~a " d)
		 (loop 0 rest) ]
		[else 
		 (fprintf out "~a " d)
		 (loop len2 rest) ] ) ) ) ) )


;;; Some pathname operations:

(define (chop-separator str)
  (let ([len (sub1 (string-length str))])
    (if (and (> len 0) (char=? (string-ref str len) pathname-directory-separator))
	(substring str 0 len)
	str) ) )

(define (chop-extension str)
  (let ([len (sub1 (string-length str))])
    (let loop ([i len])
      (cond [(zero? i) str]
	    [(char=? #\. (string-ref str i)) (substring str 0 i)]
	    [else (loop (sub1 i))] ) ) ) )


;;; Print version/usage information:

(define (print-version)
  (printf "This is ~a, ~a~%~a" product build-information copyright) )

(define (print-usage)
  (print-version)
  (newline)
  (display #<<EOF
Usage: chicken {FILENAME | OPTION}

  FILENAME should be a complete source file name with extension, or "-" for
  standard input. OPTION may be one of the following:

    -help                       display this text and exit
    -version                    display compiler version and exit
    -verbose                    display information on compilation progress
    -quiet                      do not display compile information
    -debug MODES                display debugging output for the given modes
    -explicit-use               do not use units 'library' and 'eval' by default
    -output-file FILENAME       specifies output-filename, default is 'out.c'
    -include-path PATHNAME      specifies alternative path for included files
    -database-size NUMBER       specifies size of analysis-database
    -debug-level NUMBER         set level of available debugging information
    -emit-debug-info            emit extended debugging information
    -debug-calls                emit extra code to track non-loop procedure calls
    -debug-loops                emit extra code to track all procedure calls
    -no-trace                   disable tracing information
    -profile FILENAME           executable emits profiling information 
    -no-warnings                disable warnings
    -optimize                   enable all optimizations
    -optimize-level NUMBER      enable certain sets of optimization options
    -optimize-leaf-routines     enable leaf routine optimization
    -usual-integrations         assume standard procedures are not redefined
    -unsafe                     disable safety checks
    -block                      enable block-compilation
    -inline                     enable inlining
    -inline-limit NUMBER        maximum percentage of growth through inlining
    -inline-passes NUMBER       maximal number of inlining passes
    -disable-interrupts         disable interrupts in compiled code
    -fixnum-arithmetic          assume all numbers are fixnums
    -benchmark-mode             fixnum mode, no interrupts and opt.-level 3
    -no-fancy-ports             use only file- and std-ports
    -no-winding-callcc          use non-winding semantics for call/cc
    -feature SYMBOL             register feature identifier
    -no-feature SYMBOL          unregister feature identifier
    -heap-size NUMBER           specifies heap-size of compiled executable
    -nursery NUMBER
    -stack-size NUMBER          specifies nursery size of compiled executable
    -case-sensitive             preserve case of read symbols
    -check-syntax               abort compilation after macro-expansion
    -expand-only                print macro-expanded source to file and exit
    -to-stdout                  write compiled file to stdout instead of file
    -srfi-7                     process source file as SRFI-7 configuration
    -hygienic                   use syntax-case macro package
    -extend FILENAME            load file before compilation commences
    -write-dependencies         output include-file dependencies
    -write-distribution-dependencies
                                output library unit dependencies
    -dependency-output FILENAME destination for dependency output
    -prelude EXPRESSION         add expression to front of source file
    -postlude EXPRESSION        add expression to end of source file
    -prologue FILENAME          include file before main source file
    -epilogue FILENAME          include file after main source file

EOF
) )


;;; Special block-variable literal type:

(define-record block-variable-literal 
  name)					; symbol


;;; Generation of random names:

(define (make-random-name . prefix)
  (string->symbol
   (sprintf "~A-~A~A"
	    (:optional prefix (gensym))
	    (current-seconds)
	    (random 1000) ) ) )
