;; -*- Mode: Lisp -*-
;; $Id: conditions.lisp,v 1.9 2001/01/03 20:57:01 jesse Exp $

(in-package :MAISQL-SYS)

;;; Conditions
(define-condition maisql-condition ()
  ())

(define-condition maisql-error (error maisql-condition)
  ())

(define-condition maisql-simple-error (simple-condition maisql-error)
  ())

(define-condition maisql-warning (warning maisql-condition)
  ())

(define-condition maisql-simple-warning (simple-condition maisql-warning)
  ())

(define-condition maisql-type-error (error maisql-condition)
  ((slotname :initarg :slotname
	     :reader maisql-type-error-slotname)
   (typespec :initarg :typespec
	     :reader maisql-type-error-typespec)
   (value :initarg :value
	  :reader maisql-type-error-value))
  (:report (lambda (c stream)
	     (format stream
		     "Invalid value ~A in slot ~A, not of type ~A."
		     (maisql-type-error-value c)
		     (maisql-type-error-slotname c)
		     (maisql-type-error-typespec c)))))

(define-condition maisql-sql-syntax-error (maisql-error)
  ((reason :initarg :reason
	   :reader maisql-sql-syntax-error-reason))
  (:report (lambda (c stream)
	     (format stream "Invalid SQL syntax: ~A"
		     (maisql-sql-syntax-error-reason c)))))

(define-condition maisql-invalid-spec-error (maisql-error)
  ((connection-spec :initarg :connection-spec
		    :reader maisql-invalid-spec-error-connection-spec)
   (database-type :initarg :database-type
		  :reader maisql-invalid-spec-error-database-type)
   (template :initarg :template
	     :reader maisql-invalid-spec-error-template))
  (:report (lambda (c stream)
	     (format stream "The connection specification ~A~%is invalid for database type ~A.~%The connection specification must conform to ~A"
		     (maisql-invalid-spec-error-connection-spec c)
		     (maisql-invalid-spec-error-database-type c)
		     (maisql-invalid-spec-error-template c)))))

(defmacro check-connection-spec (connection-spec database-type template)
  "Check the connection specification against the provided template,
and signal an maisql-invalid-spec-error if they don't match."
  `(handler-case
       (destructuring-bind ,template ,connection-spec
	 (declare (ignore ,@(remove '&optional template)))
	 t)
     (error () (error 'maisql-invalid-spec-error
		      :connection-spec ,connection-spec
		      :database-type ,database-type
		      :template (quote ,template)))))

(define-condition maisql-connect-error (maisql-error)
  ((database-type :initarg :database-type
		  :reader maisql-connect-error-database-type)
   (connection-spec :initarg :connection-spec
		    :reader maisql-connect-error-connection-spec)
   (errno :initarg :errno :reader maisql-connect-error-errno)
   (error :initarg :error :reader maisql-connect-error-error))
  (:report (lambda (c stream)
	     (format stream "While try to connect to database ~A~%  using database-type ~A:~%  Error ~D: ~A~%  has occurred."
		     (maisql-connect-error-connection-spec c)
		     (maisql-connect-error-database-type c)
		     (maisql-connect-error-errno c)
		     (maisql-connect-error-error c)))))

(define-condition maisql-sql-error (maisql-error)
  ((database
    :initarg :database
    :initform nil
    :reader maisql-sql-error-database)
   (expression
    :initarg :expression
    :initform nil
    :reader maisql-sql-error-expression)
   (errno
    :initarg :errno
    :initform nil
    :reader maisql-sql-error-errno)
   (error
    :initarg :error
    :initform nil
    :reader maisql-sql-error-error))
  (:report (lambda (c stream)
	     (format stream "While using database ~A:~%  The SQL statement: ~S~%  Caused this error: ~A. ~% (errno is ~A)"
		     (or (and (maisql-sql-error-database c)
			      (database-name (maisql-sql-error-database c)))
			 "[unknown]")
		     (maisql-sql-error-expression c)
		     (maisql-sql-error-error c)
                     (maisql-sql-error-errno c)))))

(define-condition maisql-exists-condition (maisql-condition)
   ((old-db :initarg :old-db :reader maisql-exists-condition-old-db)
    (new-db :initarg :new-db :reader maisql-exists-condition-new-db
	    :initform nil))
   (:report (lambda (c stream)
	      (format stream "In call to ~S:~%" 'connect)
	      (cond
		((null (maisql-exists-condition-new-db c))
		 (format stream
			 "  There is an existing connection ~A to database ~A."
			 (maisql-exists-condition-old-db c)
			 (database-name (maisql-exists-condition-old-db c))))
		((eq (maisql-exists-condition-new-db c)
		     (maisql-exists-condition-old-db c))
		 (format stream
			 "  Using existing connection ~A to database ~A."
			 (maisql-exists-condition-old-db c)
			 (database-name (maisql-exists-condition-old-db c))))
		(t
		 (format stream
			 "  Created new connection ~A to database ~A~%  ~
although there is an existing connection (~A)."
			 (maisql-exists-condition-new-db c)
			 (database-name (maisql-exists-condition-new-db c))
			 (maisql-exists-condition-old-db c)))))))

(define-condition maisql-exists-warning (maisql-exists-condition
					 maisql-warning)
  ())

(define-condition maisql-exists-error (maisql-exists-condition
				       maisql-error)
  ())

(define-condition maisql-closed-error (maisql-error)
  ((database :initarg :database :reader maisql-closed-error-database))
  (:report (lambda (c stream)
	     (format stream "The database ~A has already been closed."
		     (maisql-closed-error-database c)))))

(define-condition maisql-nodb-error (maisql-error)
  ()
  (:report (lambda (c stream)
             (declare (ignore c))
             (format stream "No database is open."))))
