;;; -*- Mode: Lisp -*-
;;; $Id: context.lisp,v 1.56 2002/03/18 22:27:22 jesse Exp $
;;;
;;; Start of WO style object editing contexts.

(in-package :maisql-sys)

(defvar *ec-unwinding* nil)

(defmacro with-context ((&optional (editing-context *default-editing-context*))
                        &body body)
  `(let ((*default-editing-context* ,editing-context)
         (sql:*default-database*    (slot-value ,editing-context 'database))
         results)
    (unwind-protect
         (progn
           (setq results (multiple-value-list (progn ,@body)))
           (ec-execute-changes *default-editing-context*)
           (values-list results))
      (handler-case
          (ec-unwind-stack maisql-sys::*default-editing-context*)
        (error (e)
          (cmsg "Error unwinding stack: ~s" e))))
    (values-list results)))
    

(defmacro without-context (&body body)
  `(let ((*default-editing-context* nil))
    ,@body))

(defclass editing-context ()
  ((database        :initarg :database)
   (supercontext    :initarg :supercontext
                    :accessor supercontext
                    :initform nil)
   (subcontext-list :initarg :subcontext-list
                    :accessor subcontext-list
                    :initform nil)
   (stack           :initform nil)
   (cache           :initarg :cache)))

(defmethod print-object ((self editing-context) stream)
  (print-unreadable-object
   (self stream :type t :identity t)
   (if (and (slot-boundp self 'stack)
            (slot-boundp self 'cache)
            (slot-boundp self 'database))
       (with-slots (stack supercontext cache database)
         self
         (if supercontext
             (progn
               (write-string "root {" stream)
               (write (kernel:get-lisp-obj-address supercontext) :stream stream
                      :radix nil :base 16)
               (write-string "} sub" stream))
             (write-string "root" stream)))
       (write-string "invalid" stream))))

(defun make-editing-context (&key (source *default-database*)
                             (cache-size 100))
  (let ((context (make-instance 'editing-context
                                :cache (make-cache :size cache-size
                                                   :test #'equal))))
    (etypecase source
      (editing-context
       (with-slots (supercontext database)
         context
         (setf supercontext source
               database (slot-value source 'database))
         (push context (subcontext-list source))
         (odcl:note-stat :ec-count :increment)))
      (database
       (with-slots (database)
         context
         (setf database source)
         (odcl:note-stat :root-ec-count :increment)
         (odcl:note-stat :ec-count :increment))))
    context))

(defun destroy-editing-context (context)
  (when context
    (with-slots (subcontext-list supercontext cache)
      context
      (when (slot-boundp context 'subcontext-list)
        (mapc #'destroy-editing-context subcontext-list)
        (setf subcontext-list nil))

      (when (slot-boundp context 'cache)
        (let ((scanner (%cache-scanner cache)))
          (do ((kv (funcall scanner) (funcall scanner)))
              ((null kv))
            (when (cdr kv)
              (setf (slot-value (cdr kv) 'editing-context) nil))))
        
        (destroy-cache cache)
        (slot-makunbound context 'database))

      (when (slot-boundp context 'supercontext)
        (if supercontext
            (setf (slot-value supercontext 'subcontext-list)
                  (remove context (slot-value supercontext 'subcontext-list)))
            (odcl:note-stat :root-ec-count :decrement))
        (slot-makunbound context 'supercontext))
      (odcl:note-stat :ec-count :decrement)))
  (values))

(defmethod ec-register ((instance standard-db-object) &optional (ec *default-editing-context*))
  (editing-context-register ec instance))

(defmethod editing-context-register ((ec editing-context) (instance standard-db-object))
  (when-bind (key (instance-key-or-nil instance))
    (setf (slot-value instance 'editing-context) ec)
    (with-slots (stack cache)
      ec
      (cache-put cache key instance)))
  instance)

(defmethod registered-in-editing-context? ((instance standard-db-object)
                                           &optional (ec *default-editing-context*))
  (and (slot-boundp instance 'editing-context)
       (equal ec (slot-value instance 'editing-context))))

(defmethod ec-reset ((context editing-context))
  (with-slots (cache stack)
    context
    (setq stack nil)
    (cache-reset cache)))

(defmethod ec-insert ((instance standard-db-object) &optional (ec *default-editing-context*))
  (when ec
    (unless (registered-in-editing-context? instance ec)
      (editing-context-register ec instance))
    (editing-context-insert ec instance)))

(defmethod ec-uninsert ((instance standard-db-object) &optional (ec *default-editing-context*))
  (when (registered-in-editing-context? instance ec)
    (editing-context-uninsert ec instance)))

(defmethod editing-context-insert ((ec editing-context) (instance standard-db-object) )
  (when-bind (key (instance-key-or-nil instance))
    (with-slots (stack cache)
      ec
      (editing-context-register ec instance)
      (cache-mark cache key)
      (push `(:insert ,@key) stack)))
  instance)

(defmethod editing-context-uninsert ((ec editing-context) (instance standard-db-object) )
  (when-bind (key (instance-key-or-nil instance))
    (with-slots (stack cache)
      ec
      (cache-delete cache key)
      (setf stack
            (remove-if #'(lambda (event)
                           (and (equal (car event) :insert)
                                (equal key (cdr event))))
                       stack))
      instance)))

(defmethod ec-delete ((instance standard-db-object) &optional (ec *default-editing-context*))
  (unless (registered-in-editing-context? instance ec)
    (editing-context-register ec instance))
  (editing-context-delete ec instance))

(defmethod editing-context-delete ((ec editing-context) (instance standard-db-object))
  (when-bind (key (instance-key-or-nil instance))
    (with-slots (stack cache)
      ec
      (editing-context-register ec instance)
      (push `(:delete ,@key) stack)))
  instance)

(defmethod ec-root ((ec editing-context))
  (if (supercontext ec)
      (ec-root (supercontext ec))
      ec))

(defmethod ec-propogate-edit-to-children (ec key (instance standard-db-object))
  (when ec
    (with-slots (cache subcontext-list)
      ec
      (multiple-value-bind (object status)
          (cache-get cache key)
        (declare (ignore status))
        (when object
          (cache-update cache key (instance-by-key-nocache key :ec ec))))
      (dolist (sub subcontext-list)
        (ec-propogate-edit-to-children sub key instance)))))

(defmethod ec-edit ((instance standard-db-object) slot-name new-value)
  (unless (or *db-initializing*
              *ec-unwinding*
              (not *default-editing-context*)
              (not (slot-boundp instance 'editing-context))
              (not (slot-value instance 'editing-context)))
    (with-slots (stack cache)
      *default-editing-context*
      (when-bind (key (instance-key-or-nil instance))
        (let ((old-value (slot-value-or-nil instance slot-name)))
          (unless (equal old-value new-value)
            (cache-put cache key instance)
            (cache-mark cache key)
            (push `(:edit ,slot-name ,old-value ,new-value ,@key) stack)))))))

(defun slot-value-or-nil (instance slot-name)
  (when (slot-boundp instance slot-name)
    (slot-value instance slot-name)))

(defvar *ec-debug* nil)

(defun ec-debug-on (flag &rest flags)
  (dolist (fl (cons flag flags))
    (when fl
      (pushnew fl *ec-debug*))))

(defun ec-debug-off (flag &rest flags)
  (dolist (fl (cons flag flags))
    (when fl
      (setf *ec-debug* (remove fl *ec-debug*)))))

(defun ec-debug (flag &rest format)
  (when (member flag *ec-debug*)
    (format t "~&;; EC~S: ~A~%" flag
            (apply #'format nil format))))


(defmethod %save-recording-state (database)
  (let ((state nil))
    (when (sql-recording-p :type :command :database database)
      (push (list :command
                  (list-sql-streams :type :command :database database))
            state))
    (when (sql-recording-p :type :result)
      (push (list :result
                  (list-sql-streams :type :result :database database))
            state))
    state))

(defmethod %restore-recoding-state (database state)
  (stop-sql-recording :type :all :database database)
  (dolist (st state)
    (start-sql-recording :type (car st) :database database)
    (dolist (str (cdr st))
      (add-sql-stream str :type (car st) :database database))))
     
    
(defmethod ec-commit (&optional (ec *default-editing-context*))
  (ec-execute-changes ec))

(defmethod ec-execute-changes ((self editing-context))
  (with-slots (database cache stack)
    self
    (when stack
      (ec-debug :exec "stack: ~s" stack)
      (let (to-insert to-delete to-update complete recording-state-command
                      recording-state-result)
        
        (when (or (member :sql *ec-debug*)
                  (member :sql-command *ec-debug*))
          (setq recording-state-command (add-sql-stream *standard-output*
                                                        :type :command
                                                        :database database)))
        (when (or (member :sql *ec-debug*)
                  (member :sql-result *ec-debug*))
          (setq recording-state-result (add-sql-stream *standard-output*
                                                       :type :result
                                                       :database database)))
        (dolist (stack-item (sort stack
                                  #'(lambda (a b)
                                      (< (or (position (car a) '(:delete :insert :edit)) 3)
                                         (or (position (car b) '(:delete :insert :edit)) 3)))))
          (ec-debug :verbose "~%Processing stack event: ~A" stack-item)

          (destructuring-bind (action &rest rest)
              stack-item
            (ecase action
              (:edit
               (unless (or (member (cdddr rest) to-insert :test #'equal)
                           (member (cdddr rest) to-delete :test #'equal))
                 (pushnew (cdddr rest) to-update :test #'equal)))
              (:insert
               (unless (member (cddr rest) to-delete :test #'equal)
                 (pushnew rest to-insert :test #'equal)))
              (:delete
               (pushnew rest to-delete :test #'equal)))))
        (unwind-protect
             (with-transaction (:database database)
               (ec-debug :exec "~%Saving changes...")
               (dolist (key  to-delete)
                 (let ((*default-editing-context* nil))
                   (ec-debug :exec "Deleting ~A with key ~A" (instance-by-key key :ec self) key)
                   (maisql-sys::delete-instance-records
                    (instance-by-key key :ec self))))
               (dolist (key to-insert)
                 (ec-debug :exec "Inserting ~A with key ~A"
                           (instance-by-key key :ec self) key)
                 (maisql-sys::install-instance (instance-by-key key :ec self) :database database)
                 (cache-unmark cache key))
               (dolist (key to-update)
                 (ec-debug :exec "Updating ~A with key ~A" (instance-by-key key :ec self) key)
                 (maisql-sys::store-instance (instance-by-key key :ec self) :database database)
                 (cache-unmark cache key))
               ;; Now clean up the context
               (dolist (key to-delete)
                 (when-bind (sup (supercontext self))
                            (cache-delete (slot-value sup 'cache) key))
                 (cache-delete cache key))
               (setq complete t))
          (if complete
              (progn
                (let ((root (ec-root self)))
                  (dolist (key to-update)
                    (ec-propogate-edit-to-children root key (instance-by-key key :ec self))))
                (setf stack nil)
                (when recording-state-command
                  (delete-sql-stream recording-state-command
                                     :type :command
                                     :database database))
                (when recording-state-result
                  (delete-sql-stream recording-state-result
                                     :type :result
                                     :database database))
                t)
              (ec-unwind-stack self)))))))

(defun ec-abort (&optional (ec *default-editing-context*))
  (ec-unwind-stack ec))

(defmethod ec-unwind-stack ((self editing-context))
  (let ((*ec-unwinding* t))
    (with-slots (cache stack)
      self
      (ec-debug :exec "~%Unwinding stack: ~A." stack)
      (do ()
          ((null stack))
        (let ((stack-item (pop stack)))
          (ecase (car stack-item)
            (:delete
             (ec-debug :exec "~%Unmarking object with key ~A." (cdr stack-item))
             (cache-unmark cache (cdr stack-item)))
            (:edit
             (destructuring-bind (slot old new &rest key)
               (cdr stack-item)
               (declare (ignore new))
               (let ((obj (instance-by-key key :ec self)))
                 ;; It may have just been inserted and already removed from our cache
                 ;; so we need to ensure it exists
                 (when obj
                   (progn
                     (ec-debug :exec "~%Resetting slot ~A to ~A in ~A."
                               slot obj old)
                     (cache-unmark cache key)
                     (setf (slot-value obj slot) old))))))
            (:insert
             (ec-debug :exec "~%Removing object ~A from cache." (cdr stack-item))
             (cache-delete cache (cdr stack-item)))))))))

(defun ec-invalidate (instance &key (ec *default-editing-context*))
  (when-bind (key (instance-key instance))
    (cache-delete (slot-value ec 'cache) key)))

(defmethod ec-cache-get ((self editing-context) key)
  (with-slots (cache supercontext)
    self
    (if-bind (object (cache-get cache key))
        (values object (ec-object-status self object))
        (when supercontext
          (ec-cache-get supercontext key)))))

(defmethod ec-scan ((self editing-context) class predicate)
  (with-slots (cache)
    self
    (let ((scanner (%cache-scanner cache))
          (entries nil))
      (do ((pair (funcall scanner) (funcall scanner)))
          ((null pair))
        (when (or (null class)
                  (equal class (caar pair)))
          (when (and (or (null predicate)
                         (funcall predicate (cdr pair)))
                     (not (context-delete-pending-p self (car pair))))
            (push (cdr pair) entries))))
      entries)))

(defmethod context-delete-pending-p ((self editing-context) key)
  (with-slots (stack)
      self
    (member-if (lambda (x) (equal x (cons :delete key))) stack)))

(defmethod ec-object-status ((self editing-context) (object standard-db-object))
  (with-slots (stack)
    self
    (cond ((member (cons :delete (instance-key object)) stack :test #'equal)
           :delete)
          (t
           :active))))

(defun view-objects-equal (a b)
  (and (eql (class-of a)
            (class-of b))
       (equal (instance-key a)
              (instance-key b))))

(defmethod coerce-to-ec-key-value (value)
  value)


(defmethod instance-key ((instance standard-db-object))
  (cons (type-of instance)
        (mapcar (lambda (slotdef)
                  (let ((name (slot-definition-name slotdef)))
                    (unless (slot-boundp instance name)
                      (return-from instance-key nil))
                    (coerce-to-ec-key-value (slot-value instance name))))
                (key-slots (class-of instance)))))

(defun instance-key-or-nil (instance)
  (handler-case
      (instance-key instance)
    (error ()
      nil)))

(defmethod instance-key-to-qualifier (class-name key)
  (let ((class (find-class class-name)))
    (mapcar #'(lambda (key value)
                (sql-operation '==
                               (sql-slot-value class-name (slot-definition-name key))
                               (sql-slot-value class-name (slot-definition-name key) value)))
            (slot-value class 'key-slots)
            (cdr key))))

(defun instance-by-key-nocache (key &key (ec *default-editing-context*))
  (destructuring-bind (class-name &rest key-values)
      key
    (let ((instance-class (find-class class-name)))
      (typecase instance-class
        (view-metaclass
         (let ((key-slots (slot-value (find-class class-name) 'key-slots)))
           (unless (= (length key-slots) (length key-values))
             (error "mismatched key count"))
           (let ((database (or (and ec (slot-value ec 'database)) *default-database*))
                 (recording-state-command nil)
                 (recording-state-result nil)
                 (result nil)
                 (qual-list (instance-key-to-qualifier (car key) key)))
             (setq qual-list
                   (if (< 1 (length qual-list))
                       (apply #'sql-and qual-list)
                       (car qual-list)))
             (when (member :sql *ec-debug*)
               (setq recording-state-command (add-sql-stream *standard-output*
                                                             :type :command
                                                             :database database))
               (setq recording-state-result (add-sql-stream *standard-output*
                                                            :type :result
                                                            :database database)))
             (setq result (car (select class-name :where qual-list)))
             (when recording-state-command
               (delete-sql-stream recording-state-command
                                  :type :command
                                  :database database))
             (when recording-state-result
               (delete-sql-stream recording-state-result
                                  :type :result
                                  :database database))
             result)))
        (t
         (make-instance class-name :oid (car key-values)))))))

(defun instance-by-key (key &key (ec *default-editing-context*))
  (if ec
      (with-slots (cache)
        ec
        (let ((instance (cache-get cache key)))
          (unless instance
            (setq instance (instance-by-key-nocache key :ec ec)))
          instance))
      (instance-by-key-nocache key)))


(defmethod cached-instance-from-children (ec key)
  (let ((instance (cache-get (slot-value ec 'cache) key)))
    (unless instance
      (dolist (sub (subcontext-list ec))
        (unless instance
          (setq instance (cached-instance-from-children sub key)))))
    instance))

(defun validate-context (context)
  (declare (ignore context))
  nil)

;;#+nil
;;  (with-slots (cache database)
;;    context
;;    (let ((scanner (%cache-scanner cache)))
;;      (do ((value (funcall scanner) (funcall scanner)))
;;          ((null value))
;;        (destructuring-bind (key . cached-object)
;;            value
;;          (let ((db-object (without-context (instance-by-key-nocache key :ec context))))
;;            (compare-prejudicially cached-object db-object)))))))

(defun compare-prejudicially (x y)
  (let ((x-type (type-of x))
        (y-type (type-of y)))
    (unless (equal x-type y-type)
      (error "dang, types aren't even the same"))
    (let ((slots (mop::class-direct-slots (mop:find-class x-type))))
      (dolist (slot slots)
        (let ((x-value (slot-value x (pcl:slot-definition-name slot)))
              (y-value (slot-value y (pcl:slot-definition-name slot))))
          (typecase x-value
            (standard-db-object
             )
            (structure-object
             )
            (t
             (unless (equal x-value y-value)
               (error "slots differ: ~s ~s" x-value y-value)))))))))
