;;; -*- Mode: Lisp -*-
;;; $Id: objects.lisp,v 1.106 2002/04/04 21:36:57 craig Exp $
;;; 
;;; Object Oriented interface for MaiSQL

(in-package :maisql-sys)

(pushnew :sql *features*)

(locally-enable-sql-reader-syntax)

(eval-when (:compile-toplevel :load-toplevel)

(defmacro when-bind ((var expr) &body body)
  `(let ((,var ,expr))
    (when ,var
      ,@body)))

(defmacro if-bind ((var expr) yes no) 
  `(let ((,var ,expr))
    (if ,var
        ,yes
        ,no)))
  
(defclass standard-db-object (standard-object)
  ((view-database
    :initform nil
    :initarg :view-database
    :db-kind :virtual)
   (editing-context
    :initarg :editing-context
    :accessor editing-context
    :db-kind :virtual
    :documentation "The editing context that provided this object."))
  (:metaclass view-metaclass)
  (:documentation "Superclass for all persistent objects"))

)

(defmethod %register-class ((self t) database)
  (declare (ignore database))
  (cmsg "Registering ~s" self))

(defmethod %deregister-class ((self t) database)
  (declare (ignore database))
  (cmsg "deregistering ~s" self))

(defmethod view-database ((self standard-db-object))
  (if (and (slot-boundp self 'editing-context)
           (slot-value self 'editing-context))
      (or (slot-value (slot-value self 'editing-context) 'database)
          (slot-value self 'view-database))
      (slot-value self 'view-database)))
    

(defvar *db-deserializing* nil)
(defvar *db-initializing* nil)

(defmethod initialize-instance :around ((class standard-db-object)
                                        &rest all-keys
                                        &key &allow-other-keys)
  (declare (ignore all-keys))
  (call-next-method)
  (when *default-editing-context*
    (unless *db-initializing*
      (ec-insert class *default-editing-context*))))


;;  Insertion into the DB and editing context must be explicit
;;  (unless *db-deserializing*
;;    (ec-create class)))

(defun sequence-from-class (view-class-name)
  (sql-escape
   (concatenate
    'string
    (symbol-name (view-table (find-class view-class-name)))
    "-SEQ")))

(defun create-sequence-from-class (view-class-name
                                   &key (database *default-database*))
  (declare (ignore database))
  (create-sequence (sequence-from-class view-class-name)))

(defun drop-sequence-from-class (view-class-name
                                 &key (database *default-database*))
  (declare (ignore database))
  (drop-sequence (sequence-from-class view-class-name)))

;;
;; Build the database tables required to store the given view class
;;

(defmethod database-pkey-constraint ((class view-metaclass) database)
  (when-bind (keylist (mapcar #'view-class-slot-column (keyslots-for-class class)))
      (format nil "CONSTRAINT ~APK PRIMARY KEY~A"
            (database-output-sql (view-table class) database)
            (database-output-sql keylist database))))

(defun ensure-schema-version-table (database)
  (unless (tablep [usql_object_v] :database database)
    (create-table [usql_object_v] '(([name] string)
                                    ([vers] integer)
                                    ([def] string))
                  :database database)))

(defun update-schema-version-records (view-class-name &key (database *default-database*))
  (let ((schemadef nil)
        (tclass (find-class view-class-name)))
    (dolist (slotdef (class-slots tclass))
      (let ((res (database-generate-column-definition view-class-name slotdef database)))
        (if res
            (setf schemadef (cons res schemadef)))))
    (when schemadef
      (delete-records :from [usql_object_v]
                      :where [= [name] (sql-escape (class-name tclass))]
                      :database database)
      (insert-records :into [usql_object_v]
                      :av-pairs `(([name] ,(sql-escape (class-name tclass)))
                                  ([vers] ,(car (object-version tclass)))
                                  ([def] ,(prin1-to-string (object-definition tclass))))
                      :database database))))

(defun create-view-from-class (view-class-name &key (database *default-database*))
  (if-bind (tclass (find-class view-class-name))
      (let ((*default-database* database))
        (%install-class tclass database)
        (%register-class view-class-name database)
        (ensure-schema-version-table database)
        (update-schema-version-records view-class-name :database database))
      (error "Class ~s not found." view-class-name)))

(defmethod %install-class ((self view-metaclass) database &aux schemadef)
  (dolist (slotdef (class-slots self))
    (when-bind (res (database-generate-column-definition (class-name self)
                                                         slotdef database))
               (push res schemadef)))
  (unless schemadef
    (error "Class ~s has no :base slots" self))
  (create-table (sql-expression :table (view-table self)) schemadef
                :database database
                :constraints (database-pkey-constraint self database))
  (push self (database-view-classes database))
  t)

(defmethod install-indices ((self t))
  )

;;
;; Drop the tables which store the given view class
;;

(defun drop-view-from-class (view-class-name &key (database *default-database*))
  (if-bind (tclass (find-class view-class-name))
      (let ((*default-database* database))
        (%deregister-class view-class-name database)
        (%uninstall-class tclass)
        (delete-records :from [usql_object_v]
                        :where [= [name] (sql-escape view-class-name)]))
      (error "Class ~s not found." view-class-name)))

(defun %uninstall-class (self &key (database *default-database*))
  (drop-table (sql-expression :table (view-table self))
              :database database)
  (setf (database-view-classes database)
        (remove self (database-view-classes database))))

;;
;; List all known view classes
;;

(defmethod list-classes (database)
  (database-view-classes database))

;;
;; Define a new view class
;;

(defmacro def-view-class (class supers slots &rest options)
  `(defclass ,class ,supers ,slots ,@options
    (:metaclass view-metaclass)))

(defun keyslots-for-class (class)
  (slot-value class 'key-slots))

(defun key-qualifier-for-instance (obj &key (database *default-database*))
  (let ((tb (view-table (class-of obj))))
    (flet ((qfk (k)
             (sql-operation '==
                            (sql-expression :attribute (view-class-slot-column k) :table tb)
                            (db-value-from-slot k (slot-value obj (slot-definition-name k)) database))))
      (let* ((keys (keyslots-for-class (class-of obj)))
	     (keyxprs (mapcar #'qfk (reverse keys))))
	(cond
          ((= (length keyxprs) 0) nil)
          ((= (length keyxprs) 1) (car keyxprs))
          ((> (length keyxprs) 1) (apply #'sql-operation 'and keyxprs)))))))

;;
;; Function used by 'generate-selection-list'
;;

(defun generate-attribute-reference (vclass slotdef)
  (cond
   ((eq (view-class-slot-db-kind slotdef) :base)
    (sql-expression :attribute (view-class-slot-column slotdef)
		    :table (view-table vclass)))
   ((eq (view-class-slot-db-kind slotdef) :key)
    (sql-expression :attribute (view-class-slot-column slotdef)
		    :table (view-table vclass)))
   (t nil)))

;;
;; Function used by 'find-all'
;;

(defun generate-selection-list (vclass)
  (let ((sels nil))
    (dolist (slotdef (class-slots vclass))
      (let ((res (generate-attribute-reference vclass slotdef)))
	(if res
	    (let ((adef (cons slotdef res)))
	      (setf sels (cons adef sels))))))
    (if sels
	sels
        (error "No slots of type :base in view-class ~A" (class-name vclass)))))

;;
;; Used by 'create-view-from-class'
;;


(defmethod database-generate-column-definition (class slotdef database)
  (declare (ignore database class))
  (when (member (view-class-slot-db-kind slotdef) '(:base :key))
    (let ((cdef (list (sql-expression :attribute (view-class-slot-column slotdef))
                      (slot-definition-type slotdef))))
      (when-bind (const (view-class-slot-db-constraints slotdef))
                 (setq cdef (append cdef (list const))))
      cdef)))


;;(mapcar #'describe (select 'test-class :where [= 1 2]))
;;[select [a] [e] :from [o]]

;;
;; Called by 'get-slot-values-from-view'
;;

(defmethod update-slot-from-db ((instance standard-db-object) slotdef value)
  (declare (optimize (speed 3) #+cmu (extensions:inhibit-warnings 3)))
  (flet ((delistify (list)
           (if (listp list)
               (car list)
               list)))
    (let ((slot-reader (view-class-slot-db-reader slotdef))
          (slot-name   (slot-definition-name slotdef))
          (slot-type   (slot-definition-type slotdef)))
      (cond ((and value (null slot-reader))
             (setf (slot-value instance slot-name)
                   (read-sql-value value (delistify slot-type)
                                   (view-database instance))))
            ((null value)
             (update-slot-with-null instance slot-name slotdef))
            ((typep slot-reader 'string)
             (setf (slot-value instance slot-name)
                   (format nil slot-reader value)))
            ((typep slot-reader 'function)
             (setf (slot-value instance slot-name)
                   (apply slot-reader (list value))))
            (t
             (error "Slot reader is of an unusual type."))))))



(defmethod key-value-from-db (slotdef value database) 
  (declare (optimize (speed 3) #+cmu (extensions:inhibit-warnings 3)))
  (flet ((delistify (list)
           (if (listp list)
               (car list)
               list)))
    (let ((slot-reader (view-class-slot-db-reader slotdef))
          (slot-name (slot-definition-name slotdef))
          (slot-type (slot-definition-type slotdef)))
      (cond ((and value (null slot-reader))
             (read-sql-value value (delistify slot-type) database))
            ((null value)
             nil)
            ((typep slot-reader 'string)
             (format nil slot-reader value))
            ((typep slot-reader 'function)
             (apply slot-reader (list value)))
            (t
             (error "Slot reader is of an unusual type."))))))

(defun db-value-from-slot (slotdef val database)
  (let ((dbwriter (view-class-slot-db-writer slotdef))
	(dbtype (slot-definition-type slotdef)))
    (typecase dbwriter
      (string (format nil dbwriter val))
      (function (apply dbwriter (list val)))
      (t
       (typecase dbtype
	 (cons
	  (database-output-sql-as-type (car dbtype) val database))
	 (t
	  (database-output-sql-as-type dbtype val database)))))))

(defun check-slot-type (slotdef val)
  (let* ((slot-type (slot-definition-type slotdef))
	(basetype (if (listp slot-type) (car slot-type) slot-type)))
    (if (and slot-type val)
	(unless (typep val basetype)
	  (error 'maisql-type-error
		 :slotname (slot-definition-name slotdef)
		 :typespec slot-type
		 :value val)))))

;;
;; Called by find-all
;;

(defmethod get-slot-values-from-view (obj slotdeflist values)
    (flet ((update-slot (slot-def values)
	     (update-slot-from-db obj slot-def values)))
      (mapc #'update-slot slotdeflist values)
      obj))

;;
;; JMM - Can't go around trying to slot-access a symbol!  Guess in
;; CMUCL slot-name is the actual slot _object_, while in lispworks it
;; is a lowly symbol (the variable is called slot-name after all) so
;; the object (or in MOP terminology- the "slot definition") has to be
;; retrieved using find-slot-definition
;;

(defmethod slot-value-using-class ((class view-metaclass) instance slot-def)
  (declare (optimize (speed 3)))
  (let ((slot-name (%slot-def-name slot-def))
        (slot-kind (view-class-slot-db-kind slot-def)))
    (when (and (eql slot-kind :join)
               (not (slot-boundp instance slot-name)))
      (let ((*db-deserializing* t))
        (setf (slot-value instance slot-name)
              (fault-join-slot class instance slot-def)))))
  (call-next-method))

(defmethod (setf slot-value-using-class) (new-value (class view-metaclass) instance slot-def)
  (let ((slot-name (%slot-def-name slot-def))
        (slot-kind (view-class-slot-db-kind slot-def)))
    (unless (or (equal slot-name 'editing-context)
                ;(eql slot-kind :join)
                (eql slot-kind :virtual)
                *db-deserializing*)
      ;; Notate slot-value change in editing context.
      (ec-edit instance slot-name new-value))
    (call-next-method)))

(defun synchronize-keys (src srckey dest destkey)
  (let ((skeys (if (listp srckey) srckey (list srckey)))
	(dkeys (if (listp destkey) destkey (list destkey))))
    (mapcar #'(lambda (sk dk)
		(setf (slot-value dest dk)
		      (typecase sk
			(symbol
			 (slot-value src sk))
			(t sk))))
	    skeys dkeys)))

(defun desynchronize-keys (dest destkey)
  (let ((dkeys (if (listp destkey) destkey (list destkey))))
    (mapcar #'(lambda (dk)
		(setf (slot-value dest dk) nil))
	    dkeys)))

(defmethod add-to-relation ((target standard-db-object)
			    slot-name
			    (value standard-db-object))
  (let* ((objclass (class-of target))
	 (sdef (slotdef-for-slot-with-class slot-name objclass))
	 (dbinfo (view-class-slot-db-info sdef))
	 (homekey (gethash :home-key dbinfo))
	 (foreignkey (gethash :foreign-key dbinfo))
	 (to-many (gethash :set dbinfo)))
    (if (gethash :target-slot dbinfo)
	(error "add-to-relation does not work with many-to-many relations yet."))
    (if to-many
	(progn
	  (synchronize-keys target homekey value foreignkey)
	  (if (slot-boundp target slot-name)
	      (setf (slot-value target slot-name)
		    (append (slot-value target slot-name)
			    (list value)))
              (setf (slot-value target slot-name) (list value))))
        (progn
          (synchronize-keys value foreignkey target homekey)
          (setf (slot-value target slot-name) value)))))

(defmethod remove-from-relation ((target standard-db-object)
			    slot-name (value standard-db-object))
  (let* ((objclass (class-of target))
	 (sdef (slotdef-for-slot-with-class slot-name objclass))
	 (dbinfo (view-class-slot-db-info sdef))
	 (homekey (gethash :home-key dbinfo))
	 (foreignkey (gethash :foreign-key dbinfo))
	 (to-many (gethash :set dbinfo)))
    (when (gethash :target-slot dbinfo)
      (error "remove-relation does not work with many-to-many relations yet."))
    (if to-many
	(progn
	  (desynchronize-keys value foreignkey)
	  (if (slot-boundp target slot-name)
	      (setf (slot-value target slot-name)
		    (remove value
			    (slot-value target slot-name)
			    :test #'view-objects-equal))))
        (progn
          (desynchronize-keys target homekey)
          (setf (slot-value target slot-name)
                nil)))))

(defun %slot-def-name (slot-name)
  #+lispworks slot-name
  #+(or cmu sbcl) (slot-definition-name slot-name))

(defun modified-p (object)
  (declare (ignore object))
  t)                                    ; ???

(defmethod update-record-from-slot ((obj standard-db-object) slot &key
					(database *default-database*))
  (if (or (not (view-database obj)) (modified-p obj))
      (let* ((vct (view-table (class-of obj)))
	     (sd (slotdef-for-slot-with-class slot (class-of obj))))
	(check-slot-type sd (slot-value obj slot))
	(let* ((att (view-class-slot-column sd))
	       (val (db-value-from-slot sd (slot-value obj slot) database)))
	  (cond ((and vct sd (view-database obj))
		 (update-records :table (sql-expression :table vct)
				 :attributes (list (sql-expression :attribute att))
				 :values (list val)
				 :where (key-qualifier-for-instance obj :database database)
				 :database (view-database obj)))
		((and vct sd (not (view-database obj)))
                 (install-instance obj :database database))
		(t
		 (error "Unable to update record")))))
    t))
  
(defmethod update-record-from-slots ((obj standard-db-object) slots &key
                                     (database *default-database*))
  (if (or (not (view-database obj)) (modified-p obj))
      (let* ((vct (view-table (class-of obj)))
	     (sds (slotdefs-for-slots-with-class slots (class-of obj)))
	     (avps (mapcar #'(lambda (s)
			       (let ((val (slot-value obj (slot-definition-name s))))
				 (check-slot-type s val)
				 (list (sql-expression
					:attribute (view-class-slot-column s))
				       (db-value-from-slot
					s
					val
					database))))
			   sds)))
	(cond ((and avps (view-database obj))
	       (update-records :table (sql-expression :table vct)
			       :av-pairs avps
			       :where (key-qualifier-for-instance
				       obj
				       :database database)
			       :database (view-database obj)))
	      ((and avps (not (view-database obj)))
	       (insert-records :into (sql-expression :table vct)
			       :av-pairs avps
			       :database database)
	       (setf (slot-value obj 'view-database) database))
	      (t
	       (error "Unable to update records"))))
      t))

(defgeneric update-records-from-instance (object &key database)
  (:documentation
   "Using an instance of a view class, update the database table that
stores its instance data."))

(defmethod update-records-from-instance ((obj standard-db-object) &key (database *default-database*))
  (labels ((slot-storedp (slot)
	     (and (member (view-class-slot-db-kind slot) '(:base :key))
		  (slot-boundp obj (slot-definition-name slot))))
	   (slot-value-list (slot)
	     (let ((value (slot-value obj (slot-definition-name slot))))
	       (check-slot-type slot value)
	       (list (sql-expression :attribute (view-class-slot-column slot))
		     (db-value-from-slot slot value database)))))
    (let* ((view-class (class-of obj))
	   (view-class-table (view-table view-class))
	   (slots (remove-if-not #'slot-storedp (class-slots view-class)))
	   (record-values (mapcar #'slot-value-list slots)))
      (unless record-values
        (error "No settable slots."))
      (if (view-database obj)
	  (update-records :table (sql-expression :table view-class-table)
			  :av-pairs record-values
			  :where (key-qualifier-for-instance obj :database database)
			  :database (view-database obj))
	  (progn
	    (insert-records :into (sql-expression :table view-class-table)
			    :av-pairs record-values
			    :database database)
	    (setf (slot-value obj 'view-database) database)))
      t)))

(defmethod install-instance ((obj standard-db-object) &key (database *default-database*))
  (labels ((slot-storedp (slot)
	     (and (member (view-class-slot-db-kind slot) '(:base :key))
		  (slot-boundp obj (slot-definition-name slot))))
	   (slot-value-list (slot)
	     (let ((value (slot-value obj (slot-definition-name slot))))
	       (check-slot-type slot value)
	       (list (sql-expression :attribute (view-class-slot-column slot))
		     (db-value-from-slot slot value database)))))
    (let* ((view-class (class-of obj))
	   (view-class-table (view-table view-class))
	   (slots (remove-if-not #'slot-storedp (class-slots view-class)))
	   (record-values (mapcar #'slot-value-list slots)))


      (unless record-values
        (error "No settable slots."))
      (unless (when-bind (obj-db (slot-value obj 'view-database))
                (equal obj-db database))
        (insert-records :into (sql-expression :table view-class-table)
                        :av-pairs record-values
                        :database database)
        (setf (slot-value obj 'view-database) database))
      (values))))

(setf (symbol-function (intern "STORE-INSTANCE"))
      (symbol-function 'update-records-from-instance))

;; Perhaps the slot class is not correct in all CLOS implementations,
;; tho I have not run across a problem yet.

(defmethod handle-cascade-delete-rule ((instance standard-db-object)
				       (slot view-class-effective-slot-definition))
  (let ((val (slot-value instance (slot-definition-name slot))))
    (typecase val
      (list
       (if (gethash :target-slot (view-class-slot-db-info slot))
           ;; For relations with target-slot, we delete just the join instance
           (mapcar #'(lambda (obj)
                       (delete-instance-records obj))
                   (fault-join-slot-raw (class-of instance) instance slot))
           (dolist (obj val)
             (delete-instance-records obj))))
      (standard-db-object
       (delete-instance-records val)))))

(defmethod nullify-join-foreign-keys ((instance standard-db-object) slot)
    (let* ((dbi (view-class-slot-db-info slot))
	   (fkeys (gethash :foreign-keys dbi)))
      (mapcar #'(lambda (fk)
		  (if (view-class-slot-nulls-ok slot)
		      (setf (slot-value instance fk) nil)
		      (warn "Nullify delete rule cannot set slot not allowing nulls to nil")))
	      (if (listp fkeys) fkeys (list fkeys)))))

(defmethod handle-nullify-delete-rule ((instance standard-db-object)
				       (slot view-class-effective-slot-definition))
    (let ((dbi (view-class-slot-db-info slot)))
      (if (gethash :set dbi)
	  (if (gethash :target-slot (view-class-slot-db-info slot))
	      ;; For relations with target-slot, we delete just the join instance
	      (mapcar #'(lambda (obj)
			  (nullify-join-foreign-keys obj slot))
		      (fault-join-slot-raw (class-of instance) instance slot))
	      (dolist (obj (slot-value instance (slot-definition-name slot)))
		(nullify-join-foreign-keys obj slot)))
	  (nullify-join-foreign-keys (slot-value instance (slot-definition-name slot)) slot))))

(defmethod propogate-deletes ((instance standard-db-object))
  (let* ((view-class (class-of instance))
	 (joins (remove-if #'(lambda (sd)
			       (not (equal (view-class-slot-db-kind sd) :join)))
			   (class-slots view-class))))
    (dolist (slot joins)
      (let* ((delete-rule (gethash :delete-rule (view-class-slot-db-info slot))))
	(cond
	  ((eql delete-rule :cascade)
	   (handle-cascade-delete-rule instance slot))
	  ((eql delete-rule :deny)
	   (if (slot-value instance (slot-definition-name slot))
	       (error "Unable to delete ~A, because the ~A slot has a deny delete rule.")))
	  ((eql delete-rule :nullify)
	   (handle-nullify-delete-rule instance slot))
	  (t t))))))

(defmethod delete-instance-records ((instance standard-db-object))
  (let ((vt (sql-expression :table (view-table (class-of instance))))
	(vd (or (view-database instance) *default-database*)))
    (cond (*default-editing-context*
           (ec-delete instance))
          (vd
           (let ((qualifier (key-qualifier-for-instance instance :database vd)))
             (with-transaction ()
               (propogate-deletes instance)
               (delete-records :from vt :where qualifier :database vd)))))))


(defmethod update-instance-from-db ((instance standard-db-object))
  (let* ((view-class (find-class (class-name (class-of instance))))
         (view-table (sql-expression :table (view-table view-class)))
         (view-qual  (key-qualifier-for-instance instance :database (view-database instance)))
         (sels       (generate-selection-list view-class))
         (res (apply #'select (append (mapcar #'cdr sels) (list :from  view-table
                                                                :where view-qual)))))
    (when res
      (get-slot-values-from-view instance (mapcar #'car sels) (car res))
      res)))


(defgeneric database-null-value (type)
  (:documentation "Return an expression of type TYPE which SQL NULL values
will be converted into."))

(defmethod database-null-value ((type t))
    (cond
     ((subtypep type 'string) "")
     ((subtypep type 'integer) 0)
     ((subtypep type 'list) nil)
     ((subtypep type 'boolean) nil)
     (t
      (error "Unable to handle null for type ~A" type))))

(defgeneric update-slot-with-null (instance slotname slotdef)
  (:documentation "Called to update a slot when its column has a NULL value.  If
nulls are allowed for the column, the slot's value will be nil, otherwise its value
will be set to the result of calling DATABASE-NULL-VALUE on the type of the slot."))

(defmethod update-slot-with-null ((instance standard-db-object)
				  slotname
				  slotdef)
  (let ((st (slot-definition-type slotdef))
        (allowed (slot-value slotdef 'nulls-ok)))
    (if allowed
        (setf (slot-value instance slotname) nil)
        (setf (slot-value instance slotname)
              (database-null-value st)))))

(defvar +no-slot-value+ '+no-slot-value+)

(eval-when (:compile-toplevel :load-toplevel)
(defsql sql-slot-value (:symbol "slot-value") (classname slot &optional (value +no-slot-value+) (database *default-database*))
  (let* ((class (find-class classname))
	 (sld (slotdef-for-slot-with-class slot class)))
    (if sld
	(if (eq value +no-slot-value+)
	    (sql-expression :attribute (view-class-slot-column sld)
			    :table (view-table class))
            (db-value-from-slot
             sld
             value
             database))
        (error "Unknown slot ~A for class ~A" slot classname))))

(defsql sql-view-class (:symbol "view-class") (classname &optional (database *default-database*))
	(declare (ignore database))
	(let* ((class (find-class classname)))
	  (unless (view-table class)
	    (error "No view-table for class ~A"  classname))
	  (sql-expression :table (view-table class))))

)

#+nil
(progn
  (defstruct sql-type
  lisp-types
  sql-type
  width
  precision)

(defun get-type-specifier (type)
  (etypecase type
    (atom
     (ecase type
       ('integer
        "INTEGER")
       ('(string simple-base-string simple-string)
        "VARCHAR")))
    (cons
     "COMPLEX TYPE")))


(defvar *simple-types* (make-hash-table :test 'eql))

(defsql-type "INTEGER" (integer bignum))

(defsql-type "VARCHAR" (string simple-string))

)

(defmethod database-get-type-specifier (type args database)
  (declare (ignore type args database))
  "VARCHAR")

(defmethod database-get-type-specifier ((type (eql 'integer)) args database)
  (declare (ignore args database))
  "INT8")

(defmethod database-get-type-specifier ((type (eql 'simple-base-string)) args database)
  (declare (ignore database))
  (if args
      (format nil "VARCHAR(~A)" (car args))
      "VARCHAR"))

(defmethod database-get-type-specifier ((type (eql 'simple-string)) args database)
  (declare (ignore database))
  (if args
      (format nil "VARCHAR(~A)" (car args))
      "VARCHAR"))

(defmethod database-get-type-specifier ((type (eql 'string)) args database)
  (declare (ignore database))
  (if args
      (format nil "VARCHAR(~A)" (car args))
      "VARCHAR"))

(deftype raw-string (&optional len)
  "A string which is not trimmed when retrieved from the database"
  `(string ,len))

(defmethod database-get-type-specifier ((type (eql 'raw-string)) args database)
  (declare (ignore database))
  (if args
      (format nil "VARCHAR(~A)" (car args))
      "VARCHAR"))

(defmethod database-get-type-specifier ((type (eql 'float)) args database)
  (declare (ignore args database))
  "FLOAT4")

(defmethod database-get-type-specifier ((type (eql 'long-float)) args database)
  (declare (ignore args database))
  "FLOAT4")

(defmethod database-get-type-specifier ((type (eql 'boolean)) args database)
  (declare (ignore args database))
  "BOOL")

(defmethod database-output-sql-as-type (type val database)
  (declare (ignore type database))
  val)

(defmethod database-output-sql-as-type ((type (eql 'list)) val database)
  (declare (ignore database))
  (progv '(*print-circle* *print-array*) '(t t)
    (prin1-to-string val)))

(defmethod database-output-sql-as-type ((type (eql 'symbol)) val database)
  (declare (ignore database))
  (if val
      (concatenate 'string
                   (package-name (symbol-package val))
                   "::"
                   (symbol-name val))
      ""))

(defmethod database-output-sql-as-type ((type (eql 'keyword)) val database)
  (declare (ignore database))
  (if val
      (symbol-name val)
      ""))

(defmethod database-output-sql-as-type ((type (eql 'vector)) val database)
  (declare (ignore database))
  (progv '(*print-circle* *print-array*) '(t t)
    (prin1-to-string val)))

(defmethod database-output-sql-as-type ((type (eql 'array)) val database)
  (declare (ignore database))
  (progv '(*print-circle* *print-array*) '(t t)
    (prin1-to-string val)))

(defmethod database-output-sql-as-type (type val database)
  (declare (ignore type database))
  val)

(defmethod database-output-sql-as-type ((type (eql 'boolean)) val database)
  (declare (ignore database))
  (if val "t" "f"))

(defmethod database-output-sql-as-type ((type (eql 'string)) val database)
  (declare (ignore database))
  val)

(defmethod database-output-sql-as-type ((type (eql 'simple-string))
					val database)
  (declare (ignore database))
  val)

(defmethod database-output-sql-as-type ((type (eql 'simple-base-string))
					val database)
  (declare (ignore database))
  val)

(defmethod read-sql-value (val type database)
  (declare (ignore type database))
  (read-from-string val))

(defmethod read-sql-value (val (type (eql 'string)) database)
  (declare (ignore database))
  val)

(defmethod read-sql-value (val (type (eql 'simple-string)) database)
  (declare (ignore database))
  val)

(defmethod read-sql-value (val (type (eql 'simple-base-string)) database)
  (declare (ignore database))
  val)

(defmethod read-sql-value (val (type (eql 'raw-string)) database)
  (declare (ignore database))
  val)

(defmethod read-sql-value (val (type (eql 'keyword)) database)
  (declare (ignore database))
  (when (< 0 (length val))
    (intern (string-upcase val) "KEYWORD")))

(defmethod read-sql-value (val (type (eql 'integer)) database)
  (declare (ignore database))
  (etypecase val
    (string
     (read-from-string val))
    (number val)))

(defmethod read-sql-value (val (type (eql 'float)) database)
  (declare (ignore database))
  (float (read-from-string val))) ; writing 1.0 writes 1, so we we *really* want a float, must do (float ...)

(defmethod read-sql-value (val (type (eql 'boolean)) database)
  (declare (ignore database))
  (equal "t" val))

;; ------------------------------------------------------------
;; Logic for 'faulting in' :join slots

(defun fault-join-slot-raw (class instance slot-def)
  (let* ((dbi (view-class-slot-db-info slot-def))
	 (jc (gethash :join-class dbi)))
    (if (and *default-editing-context* (gethash :key-join dbi)
             (not (gethash :set dbi)))
        (list (instance-by-key (append (list jc)
                                       (mapcar #'(lambda (sn)
                                                   (slot-value instance sn))
                                               (listify (gethash :home-key dbi))))))
        (when-bind (jq (join-qualifier class instance slot-def))
                   (select jc :where jq)))))


(defun fault-join-slot (class instance slot-def)
  (let* ((dbi (view-class-slot-db-info slot-def))
	 (ts (gethash :target-slot dbi))
	 (res (fault-join-slot-raw class instance slot-def)))
    (when res
      (cond
	((and ts (gethash :set dbi))
	 (mapcar (lambda (obj)
		   (cons obj (slot-value obj ts))) res))
	((and ts (not (gethash :set dbi)))
	 (mapcar (lambda (obj) (slot-value obj ts)) res))
	((and (not ts) (not (gethash :set dbi)))
	 (car res))
	((and (not ts) (gethash :set dbi))
	 res)))))

(defun join-qualifier (class instance slot-def)
    (declare (ignore class))
    (let* ((dbi (view-class-slot-db-info slot-def))
	   (jc (find-class (gethash :join-class dbi)))
	   ;;(ts (gethash :target-slot dbi))
	   ;;(tsdef (if ts (slotdef-for-slot-with-class ts jc)))
	   (foreign-keys (gethash :foreign-key dbi))
	   (home-keys (gethash :home-key dbi)))
      (when (every #'(lambda (slt)
		       (and (slot-boundp instance slt)
                            (not (null (slot-value instance slt)))))
		   (if (listp home-keys) home-keys (list home-keys)))
	(mapcar #'(lambda (hk fk)
		    (let ((fksd (slotdef-for-slot-with-class fk jc)))
		      (sql-operation '==
				     (typecase fk
				       (symbol
					(sql-expression
					 :attribute (view-class-slot-column fksd)
					 :table (view-table jc)))
				       (t fk))
				     (typecase hk
				       (symbol
					(slot-value instance hk))
				       (t
					hk)))))
		(if (listp home-keys) home-keys (list home-keys))
		(if (listp foreign-keys) foreign-keys (list foreign-keys))))))

(defmethod postinitialize ((self t))
  )

(defun find-all (view-classes &rest args &key all set-operation distinct from
                 where group-by having order-by order-by-descending offset limit
                 (editing-context sql-sys::*default-editing-context*)
                 (database *default-database*))
  "tweeze me apart someone pleeze"
  (declare (ignore all set-operation from group-by having
                   offset limit)
           (optimize (debug 3) (speed 1)))
  (let* ((*db-deserializing* t)
         (*default-database* (or database
                                 (and editing-context
                                      (slot-value editing-context 'database))
                                 (error 'maisql-nodb-error))))
    (flet ((table-sql-expr (table)
             (sql-expression :table (view-table table)))
           (ref-equal (ref1 ref2)
             (equal (sql ref1)
                    (sql ref2)))
           (tables-equal (table-a table-b)
             (string= (string (slot-value table-a 'name))
                      (string (slot-value table-b 'name)))))

      (let* ((sclasses (mapcar #'find-class view-classes))
             (sels (mapcar #'generate-selection-list sclasses))
             (fullsels (apply #'append sels))
             (sel-tables (collect-table-refs where))
             (tables (remove-duplicates (append (mapcar #'table-sql-expr sclasses) sel-tables)
                                        :test #'tables-equal))
             (res nil))
        (dolist (ob (listify order-by))
          (when (and ob (not (member ob (mapcar #'cdr fullsels)
                                     :test #'ref-equal)))
            (setq fullsels (append fullsels (mapcar #'(lambda (att) (cons nil att))
                                                    (listify ob))))))
        (dolist (ob (listify order-by-descending))
          (when (and ob (not (member ob (mapcar #'cdr fullsels)
                                     :test #'ref-equal)))
            (setq fullsels (append fullsels (mapcar #'(lambda (att) (cons nil att))
                                                    (listify ob))))))
        (dolist (ob (listify distinct))
          (when (and (typep ob 'sql-ident) (not (member ob (mapcar #'cdr fullsels)
                                                        :test #'ref-equal)))
            (setq fullsels (append fullsels (mapcar #'(lambda (att) (cons nil att))
                                                    (listify ob))))))
        ;;(format t "~%fullsels is : ~A" fullsels)
        (setq res (apply #'select (append (mapcar #'cdr fullsels) (cons :from (list tables)) args)))
        (flet ((build-instance (vals)
                 (flet ((%build-instance (vclass selects)
                          (let ((class-name (class-name vclass))
                                (db-vals    (butlast vals (- (list-length vals)
                                                             (list-length selects))))
                                cache-key)
                            (setf vals (nthcdr (list-length selects) vals))
                            (loop for select in selects
                                  for value in db-vals
                                  do
                                  (when (eql (slot-value (car select) 'db-kind) :key)
                                    (push
                                     (coerce-to-ec-key-value
                                      (key-value-from-db (car select) value *default-database*)) cache-key)))
                            (push class-name cache-key)
                            (when editing-context
                              (multiple-value-bind (object status)
                                  (ec-cache-get editing-context cache-key)
                                (if object
                                    (return-from %build-instance
                                      (if (eql status :delete)
                                          nil
                                          (return-from %build-instance object))))))
                            (%make-fresh-object class-name editing-context (mapcar #'car selects) db-vals))))
                   (let ((instances (mapcar #'%build-instance sclasses sels)))
                     (if (= (length sclasses) 1)
                         (car instances)
                         instances)))))
          (remove-if #'null (mapcar #'build-instance res)))))))

(defun %make-fresh-object (class-name editing-context slots values)
  (let* ((*db-initializing* t)
         (obj (make-instance class-name
                             :view-database *default-database*
                             :editing-context editing-context)))
    (setf obj (get-slot-values-from-view obj slots values))
    (when editing-context
      (ec-register obj editing-context))
    (postinitialize obj)
    obj))
  

(defun select (&rest select-all-args)
  "Selects data from database given the constraints specified. Returns
a list of lists of record values as specified by select-all-args. By
default, the records are each represented as lists of attribute
values. The selections argument may be either db-identifiers, literal
strings or view classes.  If the argument consists solely of view
classes, the return value will be instances of objects rather than raw
tuples."
  (flet ((select-objects (target-args)
           (and target-args
                (every #'(lambda (arg)
                           (and (symbolp arg)
                                (find-class arg nil)))
                       target-args))))
    (multiple-value-bind (target-args qualifier-args)
        (query-get-selections select-all-args)
      (if (select-objects target-args)
          (apply #'find-all target-args qualifier-args)
          (let ((expr (apply #'make-query select-all-args)))
            (destructuring-bind (&key (flatp nil)
				      (database *default-database*)
                                      &allow-other-keys)
                qualifier-args
              (let ((res (query expr :database database)))
		(if (and flatp
			 (= (length (slot-value expr 'selections)) 1))
		    (mapcar #'car res)
		  res))))))))
