// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// button.cpp
//
// History:
//  01-12-2000 - created
////////////////////////////////////////////////////////////////////////

#include "button.h"
#include "misc.h"
#include "globals.h"

////////////////////////////////////////////////////////////////////////
// NAME       : ButtonWidget::ButtonWidget
// DESCRIPTION: Create a button widget.
// PARAMETERS : id     - widget identifier
//              x      - left edge of widget relative to window border
//              y      - top edge of widget
//              w      - widget width
//              h      - widget height
//              key    - key equivalent for mouse clicks
//              flags  - see widget.h for details
//              title  - widget title (may be NULL)
//              window - widget parent window
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

ButtonWidget::ButtonWidget( short id, short x, short y, unsigned short w,
                       unsigned short h, short key, unsigned short flags,
                       const char *title, Window *window ) :
     Widget( id, x, y, w, h, key, flags, title, window ) {
  hook = NULL;
  image[0] = image[1] = NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : ButtonWidget::~ButtonWidget
// DESCRIPTION: Destroy the button widget.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

ButtonWidget::~ButtonWidget( void ) {
  delete image[0];
  delete image[1];
}

////////////////////////////////////////////////////////////////////////
// NAME       : ButtonWidget::Draw
// DESCRIPTION: Draw the button widget.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
//   29-04-2001 - also accept WIDGET_STYLE_GFX in combination with
//                WIDGET_STYLE_NOBORDER
//   27-07-2001 - new flag WIDGET_STYLE_HIGHLIGHT
//   13-08-2001 - use FillRectAlpha() in a few places
////////////////////////////////////////////////////////////////////////

void ButtonWidget::Draw( void ) {
  surface->DrawBack( *this );

  if ( (flags & WIDGET_STYLE_MENU) && Clicked() ) {
    surface->DrawBox( *this, BOX_RECESSED );
    surface->FillRectAlpha( x+1, y+1, w-2, h-2, Color(CF_COLOR_SHADOW) );
  } else if ( flags & WIDGET_STYLE_HIGHLIGHT )
    surface->FillRectAlpha( *this, ColLight );
  else if ( !(flags & WIDGET_STYLE_NOBORDER) ) {
    surface->DrawBox( *this, BOX_RECESSED );
    surface->DrawBox( Rect(x+1,y+1,w-2,h-2), Clicked() ? BOX_RECESSED : BOX_RAISED );
    if ( Clicked() )
      surface->FillRectAlpha( x+2, y+2, w-4, h-4, Color(CF_COLOR_SHADOW) );
  }

  if ( flags & WIDGET_STYLE_GFX ) {
    Image *img = image[Clicked()];
    img->Draw( surface, x + (w - img->Width()) / 2, y + (h - img->Height()) / 2 );
  }

  if ( title ) {                      // print widget title
    short xoff, yoff = y + (h - font->Height()) / 2;
    if ( flags & WIDGET_ALIGN_LEFT )
      xoff = x - font->TextWidth(title) - 5;
    else if ( flags & WIDGET_ALIGN_RIGHT )
      xoff = x + w + 5;
    else                      // WIDGET_ALIGN_CENTER and default alignment
      xoff = x + (w - font->TextWidth( title )) / 2;

    if ( Disabled() ) font->Write( title, surface, xoff, yoff, Color(CF_COLOR_GHOSTED) );
    else {
      font->Write( title, surface, xoff, yoff );

      if ( key ) {                      // highlight keystroke
        char const *p = title;
        while ( *p != '\0' ) {
          if ( tolower( *p ) == key ) {
            font->Write( *p, surface, xoff, yoff,
                         (flags & WIDGET_STYLE_HIGHLIGHT) && !Clicked() ? ColDark : ColLight );
            break;
          }
          xoff += font->CharWidth( *p );
          p++;
        }
      }
    }
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : ButtonWidget::SetImage
// DESCRIPTION: For widgets which have the WIDGET_STYLE_GFX flag set,
//              define the images to use for displaying them.
// PARAMETERS : image  - the surface containing the images
//              state1 - rectangle containing the position of the image
//                       to show when the widget is not selected
//              state2 - rectangle containing the position of the image
//                       to show when the widget is selected
// RETURNS    : -
//
// HISTORY
//   29-04-2001 - use new Image class
////////////////////////////////////////////////////////////////////////

void ButtonWidget::SetImage( Surface *image,
                             const Rect &state1, const Rect &state2 ) {
  delete this->image[0];
  delete this->image[1];
  this->image[0] = new Image( image, state1.x, state1.y, state1.w, state1.h );
  this->image[1] = new Image( image, state2.x, state2.y, state2.w, state2.h );
} 

////////////////////////////////////////////////////////////////////////
// NAME       : ButtonWidget::MouseDown
// DESCRIPTION: Show the widget in depressed state if it was selected.
// PARAMETERS : button - SDL_MouseButtonEvent received from the event
//                       handler
// RETURNS    : GUI_OK
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status ButtonWidget::MouseDown( const SDL_MouseButtonEvent &button ) {
  if ( (button.button == SDL_BUTTON_LEFT) &&
       Contains( button.x - surface->LeftEdge(),
                 button.y - surface->TopEdge() ) ) Push();
  else if ( Clicked() ) Release();
  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : ButtonWidget::MouseUp
// DESCRIPTION: Release the button and activate it if the mouse pointer
//              is still over it.
// PARAMETERS : button - SDL_MouseButtonEvent received from the event
//                       handler
// RETURNS    : GUI status
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status ButtonWidget::MouseUp( const SDL_MouseButtonEvent &button ) {
  if ( Clicked() && (button.button == SDL_BUTTON_LEFT) ) {
    Release();
    if ( Contains( button.x - surface->LeftEdge(),
                   button.y - surface->TopEdge() ) ) return Activate();
  }
  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : ButtonWidget::KeyDown
// DESCRIPTION: Depress button if the correct key was hit.
// PARAMETERS : key - SDL_keysym received from the event handler
// RETURNS    : GUI_OK
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status ButtonWidget::KeyDown( const SDL_keysym &key ) {
  if ( (key.sym == this->key) ||
       ((flags & WIDGET_DEFAULT) && (key.sym == SDLK_RETURN)) ) Push();
  else if ( Clicked() ) Release();
  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : ButtonWidget::KeyUp
// DESCRIPTION: Activate widget if correct key was released.
// PARAMETERS : key - SDL_keysym received from the event handler
// RETURNS    : GUI status
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status ButtonWidget::KeyUp( const SDL_keysym &key ) {
  if ( Clicked() && ((key.sym == this->key) ||
       ((flags & WIDGET_DEFAULT) && (key.sym == SDLK_RETURN))) ) {
    Release();
    return Activate();
  }
  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : ButtonWidget::Activate
// DESCRIPTION: Activate widget, i.e. call the activation function of
//              the hook class.
// PARAMETERS : -
// RETURNS    : return code of the activation function
//
// HISTORY
//   03-04-2001 - pass a pointer to the button to the activation hook,
//                not the button identifier
////////////////////////////////////////////////////////////////////////

GUI_Status ButtonWidget::Activate( void ) {
  if ( hook ) return hook->Activate( this, surface );
  return (GUI_Status)id;
}


////////////////////////////////////////////////////////////////////////
// NAME       : CheckboxWidget::CheckboxWidget
// DESCRIPTION: Create a checkbox widget.
// PARAMETERS : id     - widget identifier
//              x      - left edge of widget relative to window border
//              y      - top edge of widget
//              w      - widget width
//              h      - widget height
//              state  - initial state (checked or unchecked)
//              key    - key equivalent for mouse clicks
//              flags  - see widget.h for details
//              title  - widget title (may be NULL)
//              window - widget parent window
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

#define GFX_CHECK_SIZE    15
#define GFX_CHECK_OFF_X   254
#define GFX_CHECK_OFF_Y   14
#define GFX_CHECK_ON_X    269
#define GFX_CHECK_ON_Y    14

CheckboxWidget::CheckboxWidget( short id, short x, short y, unsigned short w,
                       unsigned short h, bool state, short key,
                       unsigned short flags, const char *title, Window *window ) :
     ButtonWidget( id, x, y, w, h, key, flags, title, window ) {
  clicked = state;

  // set default graphics
  if ( flags & WIDGET_STYLE_GFX ) {
    image[0] = new Image( Icons, GFX_CHECK_OFF_X, GFX_CHECK_OFF_Y,
                          GFX_CHECK_SIZE, GFX_CHECK_SIZE );
    image[1] = new Image( Icons, GFX_CHECK_ON_X, GFX_CHECK_ON_Y,
                          GFX_CHECK_SIZE, GFX_CHECK_SIZE );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : CheckboxWidget::MouseDown
// DESCRIPTION: Check or uncheck the widget, depending on its current
//              state.
// PARAMETERS : button - SDL_MouseButtonEvent received from the event
//                       handler
// RETURNS    : GUI status
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status CheckboxWidget::MouseDown( const SDL_MouseButtonEvent &button ) {
  if ( (button.button == SDL_BUTTON_LEFT) &&
       Contains( button.x - surface->LeftEdge(),
                 button.y - surface->TopEdge() ) ) {
    if ( Clicked() ) Release();
    else Push();
    return Activate();
  }
  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : CheckboxWidget::KeyDown
// DESCRIPTION: Toggle the widget status when the corresponding key was
//              hit.
// PARAMETERS : key - SDL_keysym received from the event handler
// RETURNS    : GUI status
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status CheckboxWidget::KeyDown( const SDL_keysym &key ) {
  if ( (key.sym == this->key) ||
       ((flags & WIDGET_DEFAULT) && (key.sym == SDLK_RETURN)) ) {
    if ( Clicked() ) Release();
    else Push();
    return Activate();
  }
  return GUI_OK;
}


////////////////////////////////////////////////////////////////////////
// NAME       : MenuButtonWidget::MouseMove
// DESCRIPTION: When the mouse is moved over the button, highlight it.
// PARAMETERS : motion - SDL_MouseMotionEvent received from the event
//                       handler
// RETURNS    : GUI_OK
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status MenuButtonWidget::MouseMove( const SDL_MouseMotionEvent &motion ) {
  bool contain = Contains( motion.x - surface->LeftEdge(), motion.y - surface->TopEdge() );

  if ( ((flags & WIDGET_STYLE_HIGHLIGHT) && !contain) ||
       (!(flags & WIDGET_STYLE_HIGHLIGHT) && contain) ) {
    flags ^= WIDGET_STYLE_HIGHLIGHT;
    Draw();
    Show();
  }

  return GUI_OK;
}

