// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000-2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// fileio.cpp
//
// History:
//  21-02-2001 - created
//  24-07-2001 - changes for Win32 (Adam Gates <radad@xoasis.com>)
////////////////////////////////////////////////////////////////////////

#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "globals.h"
#include "fileio.h"
#include "textbox.h"      // TLWNode class


////////////////////////////////////////////////////////////////////////
// NAME       : Directory::Directory
// DESCRIPTION: Open a directory and initialize the first directory
//              entry for reading. Whether the directory was
//              successfully opened can be checked using
//              Directory::IsValid().
// PARAMETERS : dir - name of the directory to open
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Directory::Directory( const char *dir ) {
#ifdef WIN32
  string search( dir );
  append_path_delim( search );
  search += "*.*";
  m_Dir = FindFirstFile( search.c_str(), &m_Entry );
#else
  m_Entry = NULL;
  m_Dir = opendir( dir );

  if ( m_Dir ) m_Entry = readdir( m_Dir );
#endif
}

////////////////////////////////////////////////////////////////////////
// NAME       : Directory::~Directory
// DESCRIPTION: Close a directory.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Directory::~Directory( void ) {
#ifdef WIN32
  if ( m_Dir ) FindClose( m_Dir );
#else
  if ( m_Dir ) closedir( m_Dir );
#endif
}

////////////////////////////////////////////////////////////////////////
// NAME       : Directory::GetFileName
// DESCRIPTION: Get the name of the currently selected file in the
//              directory.
// PARAMETERS : -
// RETURNS    : pointer to file name or NULL if end of directory reached
//              (or not opened).
//
// HISTORY
////////////////////////////////////////////////////////////////////////

const char *Directory::GetFileName( void ) const {
#ifdef WIN32
  return m_Entry.cFileName;
#else
  if ( m_Entry ) return m_Entry->d_name;
  else return NULL;
#endif
}

////////////////////////////////////////////////////////////////////////
// NAME       : Directory::GetFileNameLen
// DESCRIPTION: Get the length of the currently selected file name.
// PARAMETERS : -
// RETURNS    : length of file name in characters
//
// HISTORY
////////////////////////////////////////////////////////////////////////

size_t Directory::GetFileNameLen( void ) const {
#ifdef WIN32
  return strlen( m_Entry.cFileName );
#else
  if ( m_Entry ) {
# if HAVE_DIRENT_H
    return strlen( m_Entry->d_name );
# else
    return m_Entry->d_namlen;
# endif
  }
  return 0;
#endif
}

////////////////////////////////////////////////////////////////////////
// NAME       : Directory::IsFileHidden
// DESCRIPTION: Determine whether the currently selected file is a
//              hidden file.
// PARAMETERS : -
// RETURNS    : TRUE if file is a hidden file, FALSE otherwise
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool Directory::IsFileHidden( void ) const {
#ifdef WIN32
  return (m_Entry.dwFileAttributes & FILE_ATTRIBUTE_HIDDEN) != 0;
#else
  if ( m_Entry ) return ( m_Entry->d_name[0] == '.' );
  else return false;
#endif
}

////////////////////////////////////////////////////////////////////////
// NAME       : Directory::NextFile
// DESCRIPTION: Initialize the next file in the directory for
//              examination.
// PARAMETERS : -
// RETURNS    : TRUE if another file was found, FALSE if the end of the
//              directory was reached.
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool Directory::NextFile( void ) {
#ifdef WIN32
  return FindNextFile( m_Dir, &m_Entry ) != FALSE;
#else
  if ( m_Dir ) m_Entry = readdir( m_Dir );
  return m_Entry != NULL;
#endif
}


////////////////////////////////////////////////////////////////////////
// NAME       : append_path_delim
// DESCRIPTION: Append a path delimiter to the end of the string if
//              it isn't already there.
// PARAMETERS : path - string containing a path
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void append_path_delim( string &path ) {
  if ( path.at( path.length()  - 1 ) != PATHDELIM ) path += PATHDELIM;
}

////////////////////////////////////////////////////////////////////////
// NAME       : make_dir
// DESCRIPTION: Create a directory.
// PARAMETERS : dir - name of the directory to create
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void make_dir( const char *dir ) {
#ifdef WIN32
  CreateDirectory( dir, NULL );
#else
  mkdir( dir, S_IRWXU|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH );
#endif
}

////////////////////////////////////////////////////////////////////////
// NAME       : create_config_dir
// DESCRIPTION: On *nix-like systems, create a configuration and save
//              directory in the user's home dir.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void create_config_dir( void ) {
  string confd = get_config_dir();

  // does the directory exist?
  Directory cfdir( confd.c_str() );
  if ( !cfdir.IsValid() ) make_dir( confd.c_str() );

  // now check for the saved games directory
  append_path_delim( confd );
  confd.append( "games" );

  Directory svdir( confd.c_str() );
  if ( !cfdir.IsValid() ) make_dir( confd.c_str() );
}

////////////////////////////////////////////////////////////////////////
// NAME       : get_home_dir
// DESCRIPTION: Get the name of the user's home directory.
// PARAMETERS : -
// RETURNS    : the user's home directory if any
//
// HISTORY
//   11-11-2001 - return "", not NULL when HOME is not set on Win32
////////////////////////////////////////////////////////////////////////

string get_home_dir( void ) {
#ifdef WIN32
  char dir[MAX_PATH];
  int length = GetEnvironmentVariable("HOME", dir, MAX_PATH);
  if ( length == 0 ) {
    length = GetEnvironmentVariable("HOMEDRIVE", dir, MAX_PATH);
    length += GetEnvironmentVariable("HOMEPATH", dir + length, MAX_PATH - length);
  }

  if ( length == 0 ) return "";
  else return dir;
#else
  return getenv( "HOME" );
#endif
}

////////////////////////////////////////////////////////////////////////
// NAME       : get_config_dir
// DESCRIPTION: Get the name of the configuration directory.
// PARAMETERS : -
// RETURNS    : config directory
//
// HISTORY
////////////////////////////////////////////////////////////////////////

string get_config_dir( void ) {
  string confd;
  string homed( get_home_dir() );

#ifdef WIN32
  if ( homed.empty() ) confd = get_data_dir();
#else
  if ( homed.empty() ) confd.append( CURRENTDIR );
#endif
  else {
    confd.append( homed );
    append_path_delim( confd );
    confd += '.';
    confd.append( PACKAGE );
  }
  append_path_delim( confd );

  return confd;
}

////////////////////////////////////////////////////////////////////////
// NAME       : get_data_dir
// DESCRIPTION: Get the name of the directory containing the data files.
// PARAMETERS : -
// RETURNS    : data directory name
//
// HISTORY
////////////////////////////////////////////////////////////////////////

string get_data_dir( void ) {

#ifdef CF_LIB_PATH
  return CF_LIB_PATH;
#elif defined WIN32
  char dir[MAX_PATH];
  GetModuleFileName( NULL, dir, MAX_PATH );
  {
    // Remove the file name
    char *l = dir;
    char *c = dir;
    while( *c != '\0' ) {
      if ( *c == PATHDELIM ) l = c;
      ++c;
    }
    ++l;
    *l = '\0';
  }
  return dir;
#else
  return CURRENTDIR;
#endif
}

////////////////////////////////////////////////////////////////////////
// NAME       : get_save_dir
// DESCRIPTION: Get the name of the directory to save games in.
// PARAMETERS : -
// RETURNS    : saved games directory
//
// HISTORY
////////////////////////////////////////////////////////////////////////

string get_save_dir( void ) {
  string saved( get_config_dir() );

  saved.append( "games" );
  append_path_delim( saved );

  return saved;
}

////////////////////////////////////////////////////////////////////////
// NAME       : get_sounds_dir
// DESCRIPTION: Get the name of the directory containing the sound
//              effects.
// PARAMETERS : -
// RETURNS    : sound effects directory
//
// HISTORY
////////////////////////////////////////////////////////////////////////

string get_sounds_dir( void ) {
  string sndd( get_data_dir() );
  append_path_delim( sndd );
  sndd.append( "sound" );
  append_path_delim( sndd );
  return sndd;
}

////////////////////////////////////////////////////////////////////////
// NAME       : get_levels_dir
// DESCRIPTION: Get the name of the directory containing the map files.
// PARAMETERS : -
// RETURNS    : levels directory name
//
// HISTORY
////////////////////////////////////////////////////////////////////////

string get_levels_dir( void ) {
  string levd( get_data_dir() );
  append_path_delim( levd );
  levd.append( "levels" );
  append_path_delim( levd );
  return levd;
}

////////////////////////////////////////////////////////////////////////
// DESCRIPTION: Check whether a given filename is already in use.
// PARAMETERS : file - filename (including full path) to check for
// RETURNS    : TRUE if file exists, FALSE otherwise
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool exists( const char *file ) {
  // try to open the file
  FILE *fd = fopen( file, "r" );
  if ( fd ) fclose( fd );
  return fd != NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : create_files_list
// DESCRIPTION: Read the names of all files with a specified suffix in
//              a directory and put them into a list.
// PARAMETERS : dirname - name of the directory to go through
//              suffix  - suffix for the files to grab (may be NULL)
//              list    - list to append nodes to
// RETURNS    : number of files found or -1 on error; the nodes added
//              to the end of the list are of the TLWNode class.
//
// HISTORY
//   04-06-2001 - suffix may be NULL
//              - suffix is not cut from the filename
////////////////////////////////////////////////////////////////////////

short create_files_list( const char *dirname, const char *suffix, List &list ) {
  TLWNode *node;
  short num = 0, suflen = 0;
  if ( suffix ) suflen = strlen( suffix );

  Directory dir( dirname );
  if ( dir.IsValid() ) {
    do {
      int len = dir.GetFileNameLen();

      if ( (suflen <= len) &&
           (strcasecmp( &(dir.GetFileName()[len-suflen]), suffix ) == 0) ) {

        node = new TLWNode;
        list.AddTail( node );
        node->name = new char[len+1];
        if ( !node->name ) {
          num = -1;
          break;
        }
        strcpy( node->name, dir.GetFileName() );
        num++;
      }
    }
    while ( dir.NextFile() );
  }
  return num;
}

