// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// game.h
//
// Changes:
//   17-06-2001 - FILE_VERSION incremented to incorporate turn history
//   09-03-2001 - incremented FILE_VERSION
//              - removed FID_SAVE, GI_PLAYERS, added GI_SAVEFILE
//   29-03-2001 - added g_level_info and ShowLevelInfo()
//              - bumped FILE_VERSION again
//   01-04-2001 - added UnitsList(), BuildingsList(), and Player()
//   30-04-2001 - bumped FILE_VERSION to include player passwords
//   08-11-2001 - Game now a subclass of ButtonHook (deleted GameHook)
////////////////////////////////////////////////////////////////////////

#ifndef _INCLUDE_GAME_H
#define _INCLUDE_GAME_H

#include <string>
using namespace std;

#include "player.h"
#include "mapwindow.h"
#include "extwindow.h"
#include "unit.h"
#include "list.h"

#define FILE_VERSION  4

#define FID_MISSION   MakeID('M','S','S','N')  // mission
#define FID_SET       MakeID('M','S','E','T')  // mission set

#define TURN_START		0
#define TURN_IN_PROGRESS	1
#define TURN_END		2

#define DEFAULT_DELAY	(5 * ANIM_SPEED_UNIT)

class MessageWindow;
class History;

class Game : public MapWindow, public ButtonHook {
public:
  Game( unsigned short flags, View *view );
  ~Game( void );

  int Load( const char *file );
  int Save( const char *file ) const;
  void Quit( void ) const;
  Unit *LoadUnit( SDL_RWops *file );

  void NextTurn( void );
  void StartTurn( void );
  GUI_Status EndTurn( void );

  Building *GetBuildingByID( unsigned short id ) const;
  Unit *GetUnitByID( unsigned short id ) const;

  Unit *MoveUnit( Unit *u, short hx, short hy );
  int MoveUnit( Unit *u, Direction dir );
  void SelectUnit( Unit *u );
  void DeselectUnit( bool update = true );
  unsigned short UnitTargets( Unit *u ) const;
  Unit *CreateUnit( const UnitType *type, short x, short y, Player *player );
  void CreateMoveMap( void );
  void RegisterCombat( Unit *att, Unit *def );

  virtual GUI_Status HandleEvent( const SDL_Event &event );
  void MoveCommand( int key );
  void SelectCommand( void );

  Player *CurrentPlayer( void ) const { return player; }
  Player *NextPlayer( void ) const { return g_players[player->ID()^1]; }
  unsigned short Turn( void ) const { return g_turn; }
  List *UnitsList( void ) { return &g_units; }
  List *BuildingsList( void ) { return &g_buildings; }
  Player *GetPlayer( unsigned char id ) const { return g_players[id]; }

  bool HaveWinner( void );
  GUI_Status CheckEvents( void );
  History *GetHistory( void ) { return g_history; }

  UnitType *GetUnitType( unsigned short type ) const { return &g_utypes[type]; }
  void UnitInfo( Unit *unit );
  void ShowBriefing( void ) const;
  void ShowDebriefing( unsigned char status, bool restart ) const;
  void ShowLevelInfo( void ) const;
  void ContainerContent( Transport *t, Building *b );
  void GameMenu( void );
  void UnitMenu( Unit *unit );

  void BeginReplay( List &hunits, List &backup );
  void EndReplay( List &backup );

  GUI_Status Activate( ButtonWidget *button, Window *win );

private:
  int LoadSet( const char *file );
  string CreateSaveFileName( const char *filename ) const;

  void EnableMapDisplay( void ) { g_display_map = true; }
  void DisableMapDisplay( void ) { g_display_map = false; }

  unsigned short g_flags;
  unsigned short g_turn;
  unsigned short g_limit;
  unsigned short g_msg_num;
  unsigned long g_msg_len;
  bool g_display_map;

  List g_units;            // list of units
  List g_buildings;        // list of structures
  List g_events;           // list of events
  List g_combat;           // list of combat records

  Player *g_players[2];   // array of players
  UnitType *g_utypes;     // array of unit types

  char **g_messages;

  char *g_mission;        // mission title
  char *g_set;            // mission set name

  char g_turn_phase;
  char g_level_info;
  char g_password[8];

  History *g_history;
  SoundEffect **g_set_sounds;  // sfx defined by the current mission set.
                               // if not NULL, last entry must be NULL
  Unit *g_tmp_prv_unit;
};


#define GI_SAVEFILE      0x0001   // for saved games
#define GI_PASSWORD      0x0002   // protected
#define GI_PBEM          0x0004   // email game
#define GI_AI            0x0008   // play against computer

#define VIC_VICTORY	0
#define VIC_DEFEAT	1
#define VIC_DRAW	2

#endif	/* _INCLUDE_GAME_H */

