// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// listselect.cpp
//
// History:
//  03-12-2000 - created
////////////////////////////////////////////////////////////////////////

#include "listselect.h"

////////////////////////////////////////////////////////////////////////
// NAME       : ListWidget::ListWidget
// DESCRIPTION: Create a new list widget. All subclasses of ListWidget
//              must call ListWidget::Update() when initializing.
// PARAMETERS : id       - widget identifier
//              x        - left edge of widget
//              y        - top edge of widget
//              w        - widget width
//              h        - widget height
//              list     - list nodes
//              selected - node highlighted by default (-1 == none)
//              flags    - widget flags (see widget.h for details)
//              title    - widget title (currently unused)
//              window   - widget parent window
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

ListWidget::ListWidget( short id, short x, short y, unsigned short w,
    unsigned short h, List *list, short selected, unsigned short flags,
    const char *title, Window *window ) :
    Widget( id, x, y, w, h, 0, flags, title, window ) {
  this->window = window;
  this->list = list;
  slider = NULL;
  hook = NULL;
  current = selected;
  spacing = 2;
  toprow = 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : ListWidget::Draw
// DESCRIPTION: Draw the list widget.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void ListWidget::Draw( void ) {
  surface->DrawBox( *this, BOX_RECESSED );
  DrawNodes();
}

////////////////////////////////////////////////////////////////////////
// NAME       : ListWidget::Update
// DESCRIPTION: Update the internal variables if the list was
//              modified (added/removed nodes, changed names, or even
//              put in an entirely different list).
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void ListWidget::Update( void ) {
  nodes = list->CountNodes();
  if ( current >= nodes ) {
    current = nodes - 1;
    if ( hook ) hook->Activate( this, Selected(), surface );
  }

  rows = ItemHeight() * nodes;
  visrows = h - 2 * spacing;
  if ( visrows > rows ) visrows = rows;

  if ( current == -1 ) toprow = 0;
  else {
    short selrow = current * ItemHeight();
    if ( selrow < toprow ) toprow = selrow;
    else if ( selrow + ItemHeight() >= toprow + visrows )
      toprow = selrow - visrows + ItemHeight();
  }

  if ( rows > visrows ) {
    if ( slider ) slider->Adjust( 0, rows - visrows, visrows );
    else { 
      // create the corresponding slider widget
      slider = new SliderWidget( id + 1, x + w - DEFAULT_SLIDER_SIZE, y,
         DEFAULT_SLIDER_SIZE, h, 0, rows - visrows, toprow, visrows, 0,
         WIDGET_VSCROLL | (flags & WIDGET_ARROWSCROLL ? 0 : WIDGET_ARROWSCROLL),
         NULL, window );
      slider->SetScrollable( this );
      SetSize( x, y, w - DEFAULT_SLIDER_SIZE, h );   // adjust list box size
    }
  } else if ( slider ) {	// delete the slider; we don't need it any more
    delete window->RemoveWidget( slider );
    slider = NULL;
    SetSize( x, y, w + DEFAULT_SLIDER_SIZE, h );   // adjust list box size
  }
} 

////////////////////////////////////////////////////////////////////////
// NAME       : ListWidget::SwitchList
// DESCRIPTION: Replace the current list by another one to be displayed.
// PARAMETERS : newlist - pointer to the list to replace the current
//              select  - number of the node to select initially for the
//                        new list
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void ListWidget::SwitchList( List *newlist, short select ) {
  list = newlist;
  current = select;
  Update();
  Draw();
  Show();
  if ( slider ) {
    slider->Draw();
    slider->Show();
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : ListWidget::Select
// DESCRIPTION: Adjust the list view so that the selected item is
//              visible and highlight it.
// PARAMETERS : item - number of the item to select
// RETURNS    : GUI status
//
// HISTORY
//   03-04-2001 - call the activation hook even if the item selected was
//                already active before. This way we can react to double
//                clicks, etc.
////////////////////////////////////////////////////////////////////////

GUI_Status ListWidget::Select( short item ) {
  if ( item < 0 ) item = -1;
  else if ( item >= nodes ) item = nodes - 1;

  if ( item != current ) {
    short newtop = toprow;
    current = item;

    if ( current >= 0 ) {
      short selrow = current * ItemHeight();
      if ( selrow < toprow ) newtop = selrow;
      else if ( selrow + ItemHeight() >= newtop + visrows )
        newtop = selrow - visrows + ItemHeight();
    }

    if ( toprow != newtop ) slider->ScrollTo( newtop );
    else {
      DrawNodes();
      Show();
    }
  }
  if ( hook ) return hook->Activate( this, Selected(), surface );
  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : ListWidget::Set
// DESCRIPTION: Adjust the list view so that row is the top row.
// PARAMETERS : row - first visible row
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void ListWidget::Set( short row ) {
  toprow = row;
  DrawNodes();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : ListWidget::MouseDown
// DESCRIPTION: Select a node.
// PARAMETERS : button - SDL_MouseButtonEvent received from the event
//                       handler
// RETURNS    : GUI status
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status ListWidget::MouseDown( const SDL_MouseButtonEvent &button ) {

  if ( button.button == SDL_BUTTON_LEFT ) {
    short mx = button.x - surface->LeftEdge();
    short my = button.y - surface->TopEdge();
    Rect sensitive( x + 2, y + spacing, w - 4, h - 2 * spacing );

    if ( sensitive.Contains( mx, my ) ) {
      short item = (my - y - spacing + toprow) / ItemHeight();
      if ( (item < nodes) ) return Select( item );
    }
  }
  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : ListWidget::KeyDown
// DESCRIPTION: If the widget has WIDGET_ARROWSCROLL set, the user can
//              shuffle through the items with the cursor keys.
// PARAMETERS : key - SDL_keysym received from the event handler
// RETURNS    : GUI status
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status ListWidget::KeyDown( const SDL_keysym &key ) {
  if ( flags & WIDGET_ARROWSCROLL ) {
    switch ( key.sym ) {
    case SDLK_UP:
    case SDLK_LEFT:
      if ( current > 0 ) return Select( current - 1 );
      break;
    case SDLK_DOWN:
    case SDLK_RIGHT:
      if ( current < nodes - 1 ) return Select( current + 1 );
      break;
    default:
      break;
    }
  }

  return GUI_OK;
}

