// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// sound.cpp
//
// History:
//  22-04-2001 - created
//  02-06-2001 - sound can be turned off using toggle_audio()
//  01-08-2001 - almost entirely rewritten
////////////////////////////////////////////////////////////////////////


#include <string>

#include "sound.h"
#include "fileio.h"

#ifndef DISABLE_SOUND
# define DISABLE_SOUND       // we don't have any sound effects anyway
#endif                       // and VC++ chokes on arrays with 0 elements...

#ifndef DISABLE_SOUND
# define NUM_SFX   0       // system effects; they have nothing to do with
                           // the sounds defined by a mission set

 static void fill_audio( void *unused, unsigned char *stream, int len );

 static bool audio_init = false;
 static SoundEffect *sfx[NUM_SFX];
 static const char *sfx_files[] = { NULL };

 static SoundEffect *sfx_playing = NULL;
 static Uint32 audio_len;
 static Uint8 *audio_pos;
 static Uint16 audio_flags;
 static SDL_AudioSpec audio_dev;

#endif

static bool audio_on;

////////////////////////////////////////////////////////////////////////
// NAME       : init_audio
// DESCRIPTION: Initialize the necessary audio structures and load the
//              sound effects.
// PARAMETERS : state - initial state of sound effects (on/off)
// RETURNS    : 0 on complete success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int init_audio( bool state ) {
  audio_on = state;

#ifndef DISABLE_SOUND
  audio_len = 0;
  audio_flags = 0;

  SDL_AudioSpec ask;
  string sndd( get_sounds_dir() );

  ask.freq = 22050;
  ask.format = AUDIO_S16;
  ask.channels = 2;
  ask.samples = 1024;
  ask.callback = fill_audio;
  ask.userdata = NULL;

  if ( SDL_OpenAudio( &ask, &audio_dev ) < 0 ) {
    fprintf( stderr, "Couldn't open audio: %s\n", SDL_GetError() );
    return -1;
  } else {
    audio_init = true;

    for ( int i = 0; i < NUM_SFX; i++ ) {
      sfx[i] = new SoundEffect( (sndd + sfx_files[i]).c_str() );
    }

  }
#endif

  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : get_audio
// DESCRIPTION: Get the current audio setting (on/off).
// PARAMETERS : -
// RETURNS    : current setting
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool get_audio( void ) {
  return audio_on;
}

////////////////////////////////////////////////////////////////////////
// NAME       : toggle_audio
// DESCRIPTION: Toggle sound effects on/off.
// PARAMETERS : -
// RETURNS    : new setting
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool toggle_audio( void ) {
#ifndef DISABLE_SOUND
  if ( audio_init && sfx_playing ) {
    sfx_playing->Stop();
    sfx_playing = NULL;
  }
#endif
  audio_on ^= 1;
  return audio_on;
}

////////////////////////////////////////////////////////////////////////
// NAME       : play_audio
// DESCRIPTION: Play a sound effect by its identifier.
// PARAMETERS : sfxid - sound effect identifier
//              flags - see sound.h for details
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void play_audio( unsigned short sfxid, unsigned short flags ) {
#ifndef DISABLE_SOUND
  if ( sfxid < NUM_SFX ) sfx[sfxid]->Play( flags );
#endif
}


////////////////////////////////////////////////////////////////////////
// NAME       : shutdown_audio
// DESCRIPTION: Shutdown the audio system.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void shutdown_audio( void ) {
#ifndef DISABLE_SOUND
  audio_init = false;
  SDL_CloseAudio();

  for ( int i = 0; i < NUM_SFX; i++ ) delete sfx[i];
#endif
}

////////////////////////////////////////////////////////////////////////
// NAME       : fill_audio
// DESCRIPTION: This is the callback function responsible for the
//              filling of the audio buffer.
// PARAMETERS : unused - n/a
//              stream - pointer to the audio buffer to be filled
//              len    - length of the audio buffer (in bytes)
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void fill_audio( void *unused, unsigned char *stream, int len ) {

#ifndef DISABLE_SOUND
  if ( audio_len == 0 ) {
    assert( sfx_playing != NULL );
    if ( audio_flags & SFX_LOOP ) sfx_playing->FillBuffer();
    else {
      sfx_playing->Stop();
      sfx_playing = NULL;
      return;
    }
  }

  len = (len > (int)audio_len ? audio_len : len );
  SDL_MixAudio( stream, audio_pos, len, SDL_MIX_MAXVOLUME );
  audio_pos += len;
  audio_len -= len;
#endif

}

////////////////////////////////////////////////////////////////////////
// NAME       : wait_audio
// DESCRIPTION: Wait for the current effect to be finished and close the
//              audio device. If no sound is currently being played,
//              this function does nothing.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void wait_audio( void ) {
#ifndef DISABLE_SOUND
  while ( audio_len > 0 ) SDL_Delay( 100 );
#endif
}


////////////////////////////////////////////////////////////////////////
// NAME       : SoundEffect::SoundEffect
// DESCRIPTION: Load a sound effect from a WAVE file.
// PARAMETERS : file - name of the WAVE file to load
//              dev  - audio spec of the (already open) audio device
// RETURNS    : -
//
// HISTORY
//   04-08-2001 - skip conversion if sound already has required format
////////////////////////////////////////////////////////////////////////

SoundEffect::SoundEffect( const char *file ) {

#ifndef DISABLE_SOUND
  ok = false;
  data = NULL;
  len = 0;

  if ( audio_init ) {
    Uint8 *wavdata;
    if ( SDL_LoadWAV( file, &spec, &wavdata, &len ) == NULL ) {
      fprintf( stderr, "Failed to load sound: %s\n", SDL_GetError() );
      return;
    }

    // convert data to current audio format if necessary
    if ( (spec.format != audio_dev.format) || (spec.channels != audio_dev.channels)
         || (spec.freq != audio_dev.freq ) ) {
      SDL_AudioCVT cvt;
      int samplesize;
      if ( SDL_BuildAudioCVT( &cvt, spec.format, spec.channels, spec.freq,
                    audio_dev.format, audio_dev.channels, audio_dev.freq) < 0 ) {
        SDL_FreeWAV( wavdata );
        return;
      }
      samplesize = ((spec.format & 0xFF)/8) * spec.channels;
      cvt.len = len & ~(samplesize-1);
      cvt.buf = new Uint8 [ cvt.len * cvt.len_mult ];
      if ( !cvt.buf ) {
        SDL_SetError( "Out of memory" );
        SDL_FreeWAV( wavdata );
        return;
      }
      memcpy( cvt.buf, wavdata, len );
      SDL_FreeWAV( wavdata );

      if ( SDL_ConvertAudio( &cvt ) < 0 ) {
        delete [] cvt.buf;
        return;
      }
      data = cvt.buf;
      len = cvt.len_cvt;
    } else data = wavdata;
    ok = true;
  }
#endif

}

////////////////////////////////////////////////////////////////////////
// NAME       : SoundEffect::~SoundEffect
// DESCRIPTION: Free the audio data.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

SoundEffect::~SoundEffect( void ) {
#ifndef DISABLE_SOUND
  if ( data ) SDL_FreeWAV( data );
#endif
}

////////////////////////////////////////////////////////////////////////
// NAME       : SoundEffect::Play
// DESCRIPTION: Play a sound effect.
// PARAMETERS : flags - see sound.h for details
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void SoundEffect::Play( unsigned short flags ) {

#ifndef DISABLE_SOUND
  if ( audio_init && audio_on && ok ) {
    SDL_LockAudio();
    FillBuffer();
    sfx_playing = this;
    SDL_UnlockAudio();
    SDL_PauseAudio(0);       // start playing

    if ( flags & SFX_WAIT ) wait_audio();
    else audio_flags = flags;
  }
#endif

}

////////////////////////////////////////////////////////////////////////
// NAME       : SoundEffect::Stop
// DESCRIPTION: Stop playing a sound effect and release the audio device.
// PARAMETERS : -
// RETURNS    : -
//
// NOTES      : This function simply resets the audio variables. That
//              way, you can stop ANY sound that's currently playing.
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void SoundEffect::Stop( void ) const {
#ifndef DISABLE_SOUND
  SDL_PauseAudio( 1 );
  SDL_LockAudio();
  audio_len = 0;
  audio_flags = 0;
  sfx_playing = NULL;
  SDL_UnlockAudio();
#endif
}

////////////////////////////////////////////////////////////////////////
// NAME       : SoundEffect::FillBuffer
// DESCRIPTION: (Re)Initialize the static audio variables.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void SoundEffect::FillBuffer( void ) const {
#ifndef DISABLE_SOUND
  audio_pos = data;
  audio_len = len;
#endif
}

