// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// textbox.cpp
//
// History:
//  02-12-2000 - created
////////////////////////////////////////////////////////////////////////

#include <string.h>

#include "textbox.h"
#include "misc.h"
#include "globals.h"

////////////////////////////////////////////////////////////////////////
// NAME       : TextWidget::TextWidget
// DESCRIPTION: Create a new TextWidget
// PARAMETERS : id     - widget identifier
//              x      - left edge of widget
//              y      - top edge of widget
//              w      - widget width
//              h      - widget height
//              str    - text to display (may be NULL)
//              flags  - widget flags (see widget.h for details)
//              title  - widget title (currently unused)
//              window - widget parent window
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

TextWidget::TextWidget( short id, short x, short y,
           unsigned short w, unsigned short h, const char *str,
           unsigned short flags, const char *title, Window *window ) :
    Widget( id, x, y, w, h, 0, flags, title, window ) {
  text = NULL;
  lineptr = NULL;
  lines = 0;
  spacing = 2;
  SetText( str );
}

////////////////////////////////////////////////////////////////////////
// NAME       : TextWidget::~TextWidget
// DESCRIPTION: Destroy a TextWidget
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

TextWidget::~TextWidget( void ) {
  delete [] text;
  delete [] lineptr;
}

////////////////////////////////////////////////////////////////////////
// NAME       : TextWidget::SetText
// DESCRIPTION: Prepare the text for use and format it appropriately.
// PARAMETERS : str - string to display
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void TextWidget::SetText( const char *str ) {
  delete [] text;
  delete [] lineptr;
  text = NULL;
  lineptr = NULL;
  rows = visrows = toprow = 0;

  if ( !str ) return;

  lines = 1;
  const int maxwidth = w - 10;
  int len = 1, linelen = 0;
  int lastspace = 0, lastspacewidth = 0;
  int linewidth = 0, charwidth;

  // find out length of text after we're done with it
  const char *ptr;
  for ( ptr = str; *ptr != '\0'; len++, ptr++ ) {
    linelen++;
    charwidth = font->CharWidth( *ptr );
    if ( *ptr == ' ' ) {
      lastspace = linelen;
      lastspacewidth = 0;
    } else lastspacewidth += charwidth;

    if ( (linewidth + charwidth <= maxwidth) && (*ptr != '\n') ) {
      linewidth += charwidth;
    } else {
      lines++;
      if ( *ptr == '\n' ) {
        linelen = 0;
        linewidth = 0;
      } else if ( lastspace > 0 ) {	// break line at last space
        linelen -= lastspace;
        linewidth = lastspacewidth;
      } else {				// break line in the middle of a word
        len++;				// we need to insert a NUL-byte here
        linelen = 1;
        linewidth = charwidth;
      }
      lastspace = 0;
      lastspacewidth = 0;
    }
  }

  // allocate buffer
  text = new char[ len ];
  lineptr = new char *[ lines ];
  lineptr[0] = text;

  len = linelen = 0;
  lastspace = lastspacewidth = 0;
  linewidth = 0;
  int line = 1;

  // now format the string correctly to fit into the box
  for ( ptr = str; *ptr != '\0'; ptr++ ) {
    charwidth = font->CharWidth( *ptr );
    linewidth += charwidth;

    if ( *ptr == ' ' ) {
      lastspace = linelen;
      lastspacewidth = 0;
    } else lastspacewidth += charwidth;

    if ( (linewidth > maxwidth) || (*ptr == '\n') ) {
      if ( *ptr == '\n' ) {
        text[len++] = '\0';
        lineptr[line] = &text[len];
        linelen = 0;
        linewidth = 0;
      } else if ( lastspace > 0 ) {		// break line at last space
        text[len - linelen + lastspace] = '\0';
        lineptr[line] = &text[len - linelen + lastspace + 1];
        if ( lastspace != linelen ) text[len] = *ptr;
        len++;
        linelen -= lastspace + 1;
        linewidth = lastspacewidth;
      } else {				// break line in the middle of a word
        text[len++] = '\0';		// we need to insert a NUL-byte here
        lineptr[line] = &text[len];
        linelen = 1;
        linewidth = charwidth;
        text[len++] = *ptr;
      }
      line++;
      lastspace = 0;
      lastspacewidth = 0;
    } else {
      text[len++] = *ptr;
      linelen++;
    }
  }
  text[len] = '\0';

  rows = lines * (font->Height() + spacing);
  visrows = h - 2 * spacing;
  if ( visrows > rows ) visrows = rows;
}

////////////////////////////////////////////////////////////////////////
// NAME       : TextWidget::Draw
// DESCRIPTION: Draw the widget and render the text.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void TextWidget::Draw( void ) {
  surface->DrawBack( *this );
  surface->DrawBox( *this, BOX_RECESSED );

  short xoff = x + 5;
  short yoff = y + 2 - toprow % (font->Height() + spacing);
  short topline = toprow / (font->Height() + spacing);
  Rect clip( x + 2, y + 2, w - 4, h - 4 );

  for ( int i = topline; i < lines; i++ ) {
    if ( flags & WIDGET_ALIGN_CENTER )
      xoff = x + 5 + (w - 10 - font->TextWidth( lineptr[i] )) / 2;
    font->Write( lineptr[i], surface, xoff, yoff, clip );
    yoff += font->Height() + spacing;
    if ( yoff >= y + h - 2 ) break;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : TextWidget::SetRow
// DESCRIPTION: Set the first row visible in the widget textbox.
// PARAMETERS : top - first row to be visible; the widget automatically
//                    adjusts this value to fit as much text as possible
//                    into the box
// RETURNS    : new top row set (after optimization)
//
// HISTORY
////////////////////////////////////////////////////////////////////////

unsigned short TextWidget::SetRow( unsigned short top ) {
  unsigned short maxrow = rows - visrows;
  if ( top <= maxrow ) toprow = top;
  else toprow = maxrow;
  Draw();
  Show();
  return toprow;
}


////////////////////////////////////////////////////////////////////////
// NAME       : TextScrollWidget::TextScrollWidget
// DESCRIPTION: Create a new TextScrollWidget.
// PARAMETERS : id     - widget identifier
//              x      - left edge of widget
//              y      - top edge of widget
//              w      - widget width
//              h      - widget height
//              str    - text to display (may be NULL)
//              flags  - widget flags (see widget.h for details)
//              title  - widget title (currently unused)
//              window - widget parent window
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

TextScrollWidget::TextScrollWidget( short id, short x, short y,
           unsigned short w, unsigned short h, const char *str,
           unsigned short flags, const char *title, Window *window ) :
    TextWidget( id, x, y, w - DEFAULT_SLIDER_SIZE, h, str, flags, title, window ) {
  current = 0;
  slider = new SliderWidget( id + 1, x + w - DEFAULT_SLIDER_SIZE, y,
               DEFAULT_SLIDER_SIZE, h, 0, Rows() - RowsVisible(), 0, RowsVisible(),
               0, WIDGET_VSCROLL|WIDGET_ARROWSCROLL, NULL, window );
  slider->SetScrollable( this );
}

////////////////////////////////////////////////////////////////////////
// NAME       : TextWidget::Set
// DESCRIPTION: Scroll the widget to the given row.
// PARAMETERS : level - new first row and slider level
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void TextScrollWidget::Set( short level ) {
  current = level;
  SetRow( level );
}

////////////////////////////////////////////////////////////////////////
// NAME       : TextScrollWidget::SetText
// DESCRIPTION: Set a new text to be displayed in the widget.
// PARAMETERS : str - string to display
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void TextScrollWidget::SetText( const char *str ) {
  TextWidget::SetText( str  );
  slider->Adjust( 0, Rows() - RowsVisible(), RowsVisible() );
  slider->ScrollTo( 0 );
}



////////////////////////////////////////////////////////////////////////
// NAME       : TextListWidget::DrawNodes
// DESCRIPTION: Draw the list nodes.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
//   13-08-2001 - use FillRectAlpha() for the selected item
////////////////////////////////////////////////////////////////////////

void TextListWidget::DrawNodes( void ) {
  Rect box( x + 4, y + 1 + spacing, w - 8, h - 2 - 2 * spacing );
  Rect area( x + 1, y + 1, w - 2, h - 2 );
  short num = toprow / ItemHeight();                        // number of top node
  TLWNode *n = static_cast<TLWNode *>(list->GetNode( num ));  // top node
  short xoff = box.x, yoff = box.y + (num * ItemHeight()) - toprow;

  surface->DrawBack( area );

  while ( n ) {
    if ( flags & WIDGET_ALIGN_CENTER )
      xoff = box.x + (box.w - font->TextWidth(n->name)) / 2;

    if ( num == current ) {
      Rect hilite( x + 2, yoff, w - 4, ItemHeight() );
      hilite.Clip( area );
      surface->FillRectAlpha( hilite, ColLight );
    }

    // print node name and clip to box
    font->Write( n->name, surface, xoff, yoff + 1, box );

    yoff += ItemHeight();
    if ( yoff >= box.y + box.h ) break;

    num++;
    n = static_cast<TLWNode *>( n->Next() );
  }
}


////////////////////////////////////////////////////////////////////////
// NAME       : StringWidget::StringWidget
// DESCRIPTION: Create a new StringWidget.
// PARAMETERS : id     - widget identifier
//              x      - left edge of widget
//              y      - top edge of widget
//              w      - widget width
//              h      - widget height
//              key    - keystroke to activate widget
//              str    - string to display
//              maxlen - maximum length of string in characters (not
//                       including the trailing NUL-byte)
//              flags  - widget flags (see widget.h for details)
//              title  - widget title
//              window - widget parent window
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

StringWidget::StringWidget( short id, short x, short y, unsigned short w,
              unsigned short h, short key, const char *str,
              unsigned short maxlen, unsigned short flags,
              const char *title, Window *window ) :
      Widget( id, x, y, w, h, key, flags, title, window ) {
  this->maxlen = maxlen;
  buffer = new char[maxlen+1];
  SetString( str, false );
}

////////////////////////////////////////////////////////////////////////
// NAME       : StringWidget::~StringWidget
// DESCRIPTION: Destroy the StringWidget.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

StringWidget::~StringWidget( void ) {
  delete [] buffer;
}

////////////////////////////////////////////////////////////////////////
// NAME       : StringWidget::SetString
// DESCRIPTION: Set a new string to display in the widget.
// PARAMETERS : newstr - string to display (may be NULL)
//              upd    - whether to update the display (default value
//                       is "true")
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void StringWidget::SetString( const char *newstr, bool upd /* = true */ ) {
  if ( newstr ) {
    len = strlen( newstr );
    strcpy( buffer, newstr );
    cursor = len;
  } else {
    len = 0;
    buffer[0] = '\0';
    cursor = 0;
  }
  cursorpix = font->TextWidth( buffer ) + 4;

  if ( upd ) {
    Draw();
    Show();
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : StringWidget::String
// DESCRIPTION: Get the current string.
// PARAMETERS : -
// RETURNS    : pointer to the string, or NULL if string is empty
//
// HISTORY
////////////////////////////////////////////////////////////////////////

const char *StringWidget::String( void ) const {
  if ( len > 0 ) return buffer;
  return NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : StringWidget::Draw
// DESCRIPTION: Draw the widget.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void StringWidget::Draw( void ) {
  short xoff = x + 4, yoff = y + (h - font->Height()) / 2;

  surface->DrawBack( *this );
  surface->DrawBox( *this, BOX_RECESSED );
  if ( Clicked() ) surface->FillRect( x + cursorpix, yoff, font->Width(),
                                      font->Height(), ColLight );

  if ( flags & WIDGET_STR_PASSWORD ) {
    for ( int i = 0; i < len; i++ ) {
      if ( Clicked() && (cursor == i) )
        font->Write( '*', surface, xoff, yoff, ColDark );
      else font->Write( '*', surface, xoff, yoff );
      xoff += font->CharWidth( '*' );
    }
  } else {
    font->Write( buffer, surface, xoff, yoff );
    if ( Clicked() && (cursor < len) ) {
      for ( int i = 0; i < cursor; i++ ) xoff += font->CharWidth( buffer[i] );
      font->Write( buffer[cursor], surface, xoff, yoff, ColDark );
    }
  }

  if ( title ) {
    if ( flags & WIDGET_ALIGN_ABOVE ) {
      xoff = (x + (w - font->TextWidth(title)) / 2);
      yoff = y - font->Height() - 4;
    } else xoff = x - font->TextWidth( title ) - 4;
    font->Write( title, surface, xoff, yoff );

    // highlight keyboard shortcut
    if ( key ) {
      char const *p = title;
      while ( *p != '\0' ) {
        if ( tolower( *p ) == key ) {
          font->Write( *p, surface, xoff, yoff, ColLight );
          break;
        }
        xoff += font->CharWidth( *p );
        p++;
      }
    }
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : StringWidget::MouseDown
// DESCRIPTION: React to mouse button presses.
// PARAMETERS : button - SDL_MouseButtonEvent received from the event
//                       handler
// RETURNS    : GUI status
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status StringWidget::MouseDown( const SDL_MouseButtonEvent &button ) {
  if ( (button.button == SDL_BUTTON_LEFT) && !(flags & WIDGET_STR_CONST) &&
        Contains( button.x - surface->LeftEdge(),
                  button.y - surface->TopEdge() ) ) {
    short xoff = button.x - surface->LeftEdge() - x;
    cursor = 0;
    cursorpix = 4;
    while ( (cursor < len) &&
          (cursorpix + font->CharWidth( buffer[cursor] ) < xoff) )
      cursorpix += font->CharWidth( buffer[cursor++] );
    return InputLoop();
  }
  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : StringWidget::KeyDown
// DESCRIPTION: React to key presses.
// PARAMETERS : key - SDL_keysym received from the event handler
// RETURNS    : GUI status
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status StringWidget::KeyDown( const SDL_keysym &key ) {
  if ( (key.sym == this->key) && !(flags & WIDGET_STR_CONST) )
    return InputLoop();
  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : StringWidget::InputLoop
// DESCRIPTION: After activation of the StringWidget, all events are
//              exclusively handled by this function, until either the
//              widget is deselected at which point the control is given
//              back to the main event handler.
// PARAMETERS : -
// RETURNS    : GUI status
//
// HISTORY
//   11-06-2001 - use View::FetchEvent() instead of SDL_WaitEvent()
////////////////////////////////////////////////////////////////////////

GUI_Status StringWidget::InputLoop( void ) {
  SDL_Event event;
  GUI_Status rc = GUI_OK;
  int unicode;
  bool quit = false;

  Push();
  unicode = SDL_EnableUNICODE( 1 );

  do {
    rc = surface->GetView()->FetchEvent( event );
    if ( (rc == GUI_QUIT) || (rc == GUI_ERROR) ) quit = true;
    else {
      if ( ((event.type == SDL_MOUSEBUTTONDOWN) &&
             !Contains( event.button.x - x, event.button.y - y )) ||
           ((event.type == SDL_KEYUP) && ((event.key.keysym.sym == SDLK_RETURN) ||
                                          (event.key.keysym.sym == SDLK_TAB))) ) {
        quit = true;
      } else if ( event.type == SDL_KEYDOWN )
        CharInput( event.key.keysym.sym, event.key.keysym.unicode );
    }
  } while ( !quit );

  SDL_EnableUNICODE( unicode );
  Release();
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : StringWidget::CharInput
// DESCRIPTION: Insert a new character in the input buffer.
// PARAMETERS : sym     - ASCII symbol for printable characters
//              unicode - UNICODE representation for non-printable chars
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void StringWidget::CharInput( short sym, unsigned short unicode ) {
  bool changed = false;

  if ( unicode ) {
    switch ( unicode ) {
    case '\b':                // BACKSPACE - delete char at cursor-1
      if ( cursor > 0 ) {
        cursorpix -= font->CharWidth( buffer[--cursor] );
        buffer[cursor] = '\0';
        if (cursor < --len) strcat( buffer, &buffer[cursor+1] );      // close the gap
        changed = true;
      }
      break;
    case 127:                 // DELETE - delete char at cursor
      if ( cursor < len ) {
        buffer[cursor] = '\0';
        strcat( buffer, &buffer[cursor+1]);
        len--;
        changed = true;
      }
      break;
    default:
      if ( font->Allowed( unicode ) && (len < maxlen) ) {   // insert char at cursor pos
        for ( int i = len; i >= cursor; i-- ) buffer[i+1] = buffer[i];
        buffer[cursor++] = (char)unicode;
        cursorpix += font->CharWidth( (char)unicode );
        len++;
        changed = true;
      }
    }
  } else switch ( sym ) {
    case SDLK_LEFT:                     // move cursor left
      if ( cursor > 0 ) {
        cursorpix -= font->CharWidth( buffer[--cursor] );
        changed = true;
      }
      break;
    case SDLK_RIGHT:
      if ( cursor < len ) {
        cursorpix += font->CharWidth( buffer[cursor++] );
        changed = true;
      }
      break;
    default:
      break;
  }

  if ( changed ) {
    Draw();
    Show();
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : StringWidget::SetFocus
// DESCRIPTION: Activate the widget, i.e. show the cursor and wait for
//              user input.
// PARAMETERS : -
// RETURNS    : GUI status
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status StringWidget::SetFocus( void ) {
  cursor = 0;
  cursorpix = 4;
  return InputLoop();
}

