// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

///////////////////////////////////////////////////////////////
// unit.h
//
// Changes:
//  03-12-2000 - created
//  12-06-2001 - added U_DUMMY and IsDummy()
//  07-11-2001 - added MapObject class definition
///////////////////////////////////////////////////////////////

#ifndef _INCLUDE_UNIT_H
#define _INCLUDE_UNIT_H

#include "list.h"
#include "player.h"
#include "misc.h"
#include "sound.h"

class MapObject {
public:
  MapObject( unsigned short mo_type ) : type(mo_type) {}
  unsigned short Type( void ) const { return type; }

private:
  unsigned short type;
};

#define MO_UNIT      1
#define MO_BUILDING  2

#define U_GROUND	0x00000001
#define U_SHIP		0x00000002
#define U_AIR		0x00000004
#define U_SUB		0x00000008
#define U_CONQUER	0x00000010	// unit can take over enemy buildings
#define U_SLOW		0x00000020	// can only fight OR move on one turn
#define U_TRANSPORT	0x00000040	// can carry other units,
                                        // note: transporters may NEVER have U_CONQUER set!
#define U_MINE		0x00000080	// can be cleared/reused by minesweepers
#define U_MINESWEEPER	0x00000100

#define U_DUMMY		0x00020000	// used for units generated for replays
#define U_BUSY		0x00040000	// used by computer player
#define U_DONE		0x00080000
#define U_MOVED		0x00100000	// unit moved this turn
#define U_ATTACKED	0x00200000	// unit initiated combat this turn
#define U_SHELTERED	0x00400000	// unit is currently inside a building/transporter
#define U_DESTROYED	0x00800000

#define MAX_GROUP_SIZE		6

#define XP_MAX_LEVEL		6
#define XP_PER_LEVEL		3

#define UT_NO_SOUND		255

class UnitType {
public:
  UnitType( void ) {}
  int Load( SDL_RWops *file, SoundEffect **sounds );

  unsigned short ut_terrain;		// wherever it may roam...
  unsigned short ut_image;
  unsigned char ut_moves;
  unsigned char ut_weight;		// used by transports
  unsigned char ut_flags;
  unsigned char ut_defence;		// defensive strength
  unsigned char ut_pow_ground;		// offensive strengths
  unsigned char ut_pow_ship;
  unsigned char ut_pow_air;
  unsigned char ut_pow_sub;
  unsigned char ut_min_range_ground;
  unsigned char ut_max_range_ground;
  unsigned char ut_min_range_ship;
  unsigned char ut_max_range_ship;
  unsigned char ut_min_range_air;
  unsigned char ut_max_range_air;
  unsigned char ut_min_range_sub;
  unsigned char ut_max_range_sub;
  unsigned char ut_points;		// base score gained for destroying this unit
  unsigned char ut_build;		// resources needed to build this unit type
  unsigned char ut_trans_slots;		// max number of units the transport can carry
  unsigned char ut_trans_weight;	// max weight the transport can carry
  unsigned char ut_typeid;
  char ut_name[20];
  SoundEffect *ut_snd_move;
  SoundEffect *ut_snd_fire;
};

class Unit : public Node, public MapObject {
public:
  Unit( const UnitType *type, Player *player, unsigned short id, short x, short y );
  Unit( SDL_RWops *file, const UnitType *types, Player **players );

  int Save( SDL_RWops *file ) const;

  unsigned short BaseImage( void ) const { return u_type->ut_image + u_pid * 6; }
  unsigned short BuildCost( void ) const { return u_type->ut_build; }
  unsigned long Flags( void ) const { return u_flags; }
  unsigned char GroupSize( void ) const { return u_group; }
  unsigned short ID( void ) const { return u_id; }
  unsigned short Image( void ) const { return BaseImage() + u_facing; }
  unsigned char Moves( void ) const { return u_moves; }
  const char *Name( void ) const { return u_type->ut_name; }
  Player *Owner( void ) const { return u_player; }
  unsigned short Points( void ) const { return u_type->ut_points; }
  const Point &Position( void ) const { return u_pos; }
  const Point *Target( void ) const { return( (u_flags & U_ATTACKED) ? &u_target : NULL ); }
  unsigned short Terrain( void ) const { return u_type->ut_terrain; }
  const UnitType *Type( void ) const { return u_type; }
  unsigned char WeaponRange( const Unit *u ) const;
  unsigned short Weight( void ) const { return u_type->ut_weight; }
  unsigned char XPLevel( void ) const { return u_xp/XP_PER_LEVEL; }

  void AwardXP( unsigned char xp );
  void SetOwner( Player *player );
  virtual void SetPosition( short x, short y );
  void SetMoves( unsigned char moves ) { u_moves = moves; }
  void RefreshMoves( void ) { u_moves = u_type->ut_moves; }

  void SetFlags( unsigned long f ) { u_flags |= (f); }
  void UnsetFlags( unsigned long f ) { u_flags &= (~f); }

  bool IsGround( void ) const { return (u_flags & U_GROUND) != 0; }
  bool IsShip( void ) const { return (u_flags & U_SHIP) != 0; }
  bool IsAircraft( void ) const { return (u_flags & U_AIR) != 0; }
  bool IsSubmarine( void ) const { return (u_flags & U_SUB) != 0; }
  bool IsMine( void ) const { return (u_flags & U_MINE) != 0; }
  bool IsMinesweeper( void ) const { return (u_flags & U_MINESWEEPER) != 0; }
  bool IsTransport( void ) const { return (u_flags & U_TRANSPORT) != 0; }

  bool IsSlow( void ) const { return (u_flags & U_SLOW) != 0; }
  bool IsConquer( void ) const { return (u_flags & U_CONQUER) != 0; }
  bool IsSheltered( void ) const { return (u_flags & U_SHELTERED) != 0; }
  bool IsReady( void ) const { return (u_flags & (U_DESTROYED|U_DONE)) == 0; }
  bool IsAlive( void ) const { return (u_flags & U_DESTROYED) == 0; }
  bool IsBusy( void ) const { return (u_flags & U_BUSY) != 0; }
  bool IsDummy( void ) const { return (u_flags & U_DUMMY) != 0; }
  bool IsDefensive( void ) const
    { return (u_type->ut_pow_ground + u_type->ut_pow_ship +
              u_type->ut_pow_air + u_type->ut_pow_sub) != 0; }

  bool CanHit( const Unit *enemy ) const;
  bool CanHitType( const Unit *enemy ) const;
  bool CouldHit( const Unit *enemy ) const;
  void Attack( const Unit *enemy );
  virtual bool Hit( unsigned short damage );

  unsigned char OffensiveStrength( const Unit *target ) const;
  unsigned char DefensiveStrength( void ) const;

  unsigned short Repair( unsigned short resources );
  unsigned short RepairCost( void ) const;
  void Face( unsigned char dir ) { u_facing = dir; }

  SoundEffect *MoveSound( void ) { return u_type->ut_snd_move; }
  SoundEffect *FireSound( void ) { return u_type->ut_snd_fire; }

protected:
  Point u_pos;		// position on map
  unsigned long u_flags;
  unsigned short u_id;
  
  unsigned char u_moves;
  unsigned char u_facing;	// direction
  unsigned char u_group;	// group size
  unsigned char u_xp;		// experience

  unsigned char u_pid;		// player id

  Point u_target;

  const UnitType *u_type;
  Player *u_player;
};

#endif	/* _INCLUDE_UNIT_H */

