# Deejayd, a media player daemon
# Copyright (C) 2007-2008 Mickael Royer <mickael.royer@gmail.com>
#                         Alexandre Rossi <alexandre.rossi@gmail.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
# -*- coding: utf-8 -*-

import os, sys, urllib, threading, traceback
from twisted.internet import threads

from deejayd.component import SignalingComponent
from deejayd.mediadb import formats, _media
from deejayd import database
from deejayd.ui import log

class NotFoundException(Exception):pass
class NotSupportedFormat(Exception):pass

def log_traceback(func):
    def log_traceback_func(self, *__args,**__kw):
        try: func(self, *__args,**__kw)
        except Exception, ex:
            log.err(_("Unable to get video metadata from %s, see traceback\
for more information.") % self.file)
            print "---------------------Traceback lines-----------------------"
            print traceback.format_exc()
            print "-----------------------------------------------------------"
            return False
        return True

    return log_traceback_func

class _DeejaydFile:
    table = "unknown"

    def __init__(self, db_con, dir, filename, path, info):
        self.file = path
        self.db_con = db_con
        self.dir = dir
        self.filename = filename
        self.info = info

    def remove(self):
        self.db_con.remove_file(self.dir, self.file, self.table)

    def insert(self):
        raise NotImplementedError

    def update(self):
        raise NotImplementedError

    def force_update(self):
        raise NotImplementedError


class DeejaydAudioFile(_DeejaydFile):
    table = "audio_library"

    @log_traceback
    def insert(self):
        file_info = self.info.parse(self.file)
        self.db_con.insert_audio_file(self.dir,self.filename,file_info)

    @log_traceback
    def update(self):
        file_info = self.info.parse(self.file)
        self.db_con.update_audio_file(self.dir,self.filename,file_info)

    def force_update(self):pass


class DeejaydVideoFile(_DeejaydFile):
    table = "video_library"

    @log_traceback
    def insert(self):
        file_info = self.info.parse(self.file)
        self.db_con.insert_video_file(self.dir,self.filename,file_info)

    @log_traceback
    def update(self):
        file_info = self.info.parse(self.file)
        self.db_con.update_video_file(self.dir,self.filename,file_info)

    def force_update(self):
        # Update external subtitle
        file_info = self.info.parse_sub(self.file)
        self.db_con.update_video_subtitle(self.dir,self.filename,file_info)

##########################################################################
##########################################################################
def inotify_action(func):
    def inotify_action_func(*__args, **__kw):
        self = __args[0]
        try:
            name = self._encode(__args[1])
            path = self._encode(__args[2])
        except UnicodeError:
            return

        self.mutex.acquire()

        rs = func(*__args, **__kw)
        if rs: # commit change
            self.inotify_db.record_mediadb_stats(self.type)
            self.inotify_db.set_update_time(self.type)
            self.inotify_db.connection.commit()
            self.dispatch_signame(self.update_signal_name)

        self.mutex.release()

    return inotify_action_func

class _Library(SignalingComponent):
    ext_dict = {}
    table = None
    type = None
    file_class = None

    def __init__(self, db_connection, player, path, fs_charset="utf-8"):
        SignalingComponent.__init__(self)

        # init Parms
        self._fs_charset = fs_charset
        self._update_id = 0
        self._update_end = True
        self._update_error = None

        self._path = os.path.abspath(path)
        # test library path
        if not os.path.isdir(self._path):
            msg = _("Unable to find directory %s") % self._path
            raise NotFoundException(msg)

        # Connection to the database
        self.db_con = db_connection

        # init a mutex
        self.mutex = threading.Lock()

        # build supported extension list
        self._build_supported_extension(player)

        self.watcher = None

    def _encode(self, data):
        try: rs = data.decode(self._fs_charset, "strict").encode("utf-8")
        except UnicodeError:
            log.err(_("%s has wrong character") %\
                 data.decode(self._fs_charset, "replace").encode("utf-8"))
            raise UnicodeError
        return rs

    def _build_supported_extension(self, player):
        raise NotImplementedError

    def get_dir_content(self,dir):
        rs = self.db_con.get_dir_info(dir,self.table)
        if len(rs) == 0 and dir != "":
            # nothing found for this directory
            raise NotFoundException

        return self._format_db_rsp(rs)

    def get_dir_files(self,dir):
        rs = self.db_con.get_files(dir, self.table)
        if len(rs) == 0 and dir != "": raise NotFoundException
        return self._format_db_rsp(rs)["files"]

    def get_all_files(self,dir):
        rs = self.db_con.get_all_files(dir, self.table)
        if len(rs) == 0 and dir != "": raise NotFoundException
        return self._format_db_rsp(rs)["files"]

    def get_file(self,file):
        rs = self.db_con.get_file_info(file,self.table)
        if len(rs) == 0:
            # this file is not found
            raise NotFoundException
        return self._format_db_rsp(rs)["files"]

    def get_root_path(self):
        return self._path

    def get_root_paths(self, db_con=None):
        if not db_con:
            db_con = self.db_con
        root_paths = [self.get_root_path()]
        for dirlink_record in db_con.get_all_dirlinks('', self.table):
            dirlink = os.path.join(self.get_root_path(),
                                   dirlink_record[1], dirlink_record[2])
            root_paths.append(dirlink)
        return root_paths

    def get_status(self):
        status = []
        if not self._update_end:
            status.append((self.type+"_updating_db",self._update_id))
        if self._update_error:
            status.append((self.type+"_updating_error",self._update_error))
            self._update_error = None

        return status

    def close(self):
        pass

    #
    # Update process
    #
    def update(self, sync = False):
        if self._update_end:
            self._update_id += 1
            if sync: # synchrone update
                self._update()
                self._update_end = True
            else: # asynchrone update
                self.defered = threads.deferToThread(self._update)
                self.defered.pause()

                # Add callback functions
                succ = lambda *x: self.end_update()
                self.defered.addCallback(succ)

                # Add errback functions
                def error_handler(failure,db_class):
                    # Log the exception to debug pb later
                    failure.printTraceback()
                    db_class.end_update(False)
                    return False
                self.defered.addErrback(error_handler,self)

                self.defered.unpause()
            self.dispatch_signame(self.update_signal_name)
            return self._update_id

        return 0

    def strip_root(self, path):
        abs_path = os.path.abspath(path)
        rel_path = os.path.normpath(abs_path[len(self.get_root_path()):])

        if rel_path != '.': rel_path = rel_path.strip("/")
        else: rel_path = ''

        return rel_path

    def is_in_root(self, path, root=None):
        """Checks if a directory is physically in the supplied root (the library root by default)."""
        if not root:
            root = self.get_root_path()
        real_root = os.path.realpath(root)
        real_path = os.path.realpath(path)

        head = real_path
        old_head = None
        while head != old_head:
            if head == real_root:
                return True
            old_head = head
            head, tail = os.path.split(head)
        return False

    def is_in_a_root(self, path, roots):
        """Checks if a directory is physically in one of the supplied roots."""
        for root in roots:
            if self.is_in_root(path, root):
                return True
        return False

    def _update(self):
        conn = self.db_con.get_new_connection()
        conn.connect()
        self._update_end = False

        try:
            self.last_update_time = conn.get_update_time(self.type)
            library_files = [(item[1],item[2]) for item \
                                in conn.get_all_files('',self.table)]
            library_dirs = [(item[1],item[2]) for item \
                                in conn.get_all_dirs('',self.table)]
            library_dirlinks = [(item[1],item[2]) for item\
                                in conn.get_all_dirlinks('', self.table)]

            self.walk_directory(conn, self.get_root_path(),
                                library_dirs, library_files, library_dirlinks)

            self.mutex.acquire()
            # Remove unexistent files and directories from library
            for (dir,filename) in library_files:
                conn.remove_file(dir, filename, self.table)
            for (root,dirname) in library_dirs:
                conn.remove_dir(root, dirname, self.table)
            for (root, dirlinkname) in library_dirlinks:
                conn.remove_dirlink(root, dirlinkname, self.table)
                if self.watcher:
                    self.watcher.stop_watching_dir(os.path.join(root,
                                                                dirlinkname))
            # Remove empty dir
            conn.erase_empty_dir(self.table)
            # update stat values
            conn.record_mediadb_stats(self.type)
            conn.set_update_time(self.type)
            # commit changes
            conn.connection.commit()
            self.mutex.release()
        finally:
            # close the connection
            conn.close()

    def walk_directory(self, db_con, walk_root,
                       library_dirs, library_files, library_dirlinks,
                       forbidden_roots=None):
        """Walk a directory for files to update. Called recursively to carefully handle symlinks."""
        if not forbidden_roots:
            forbidden_roots = [self.get_root_path()]

        for root, dirs, files in os.walk(walk_root):
            try: root = self._encode(root)
            except UnicodeError: # skip this directory
                continue

            # first update directory
            for dir in dirs:
                try: dir = self._encode(dir)
                except UnicodeError: # skip this directory
                    continue

                tuple = (self.strip_root(root), dir)
                if tuple in library_dirs:
                    library_dirs.remove(tuple)
                else:
                    db_con.insert_dir(tuple,self.table)

                # Walk only symlinks that aren't in library root or in one of
                # the additional known root paths which consist in already
                # crawled and out-of-main-root directories
                # (i.e. other symlinks).
                dir_path = os.path.join(root, dir)
                if os.path.islink(dir_path):
                    if not self.is_in_a_root(dir_path, forbidden_roots):
                        forbidden_roots.append(dir_path)
                        if tuple in library_dirlinks:
                            library_dirlinks.remove(tuple)
                        else:
                            db_con.insert_dirlink(tuple, self.table)
                            if self.watcher:
                                self.watcher.watch_dir(dir_path, self)
                        self.walk_directory(db_con, dir_path,
                                 library_dirs, library_files, library_dirlinks,
                                 forbidden_roots)

            # else update files
            for file in files:
                try: file = self._encode(file)
                except UnicodeError: # skip this file
                    continue

                try: obj_cls = self._get_file_info(file)
                except NotSupportedFormat:
                    log.info(_("File %s not supported") % file)
                    continue

                path = os.path.join(self._path, root, file)
                dir, fn = self.strip_root(root), file
                file_object = self.file_class(db_con, dir, fn, path, obj_cls)

                tuple = (dir, fn)
                if tuple in library_files:
                    library_files.remove(tuple)
                    if os.stat(os.path.join(root,file)).st_mtime >= \
                                                     int(self.last_update_time):
                        file_object.update()
                    # Even if the media has not been modified, we may need
                    # to update some information (like external subtitle)
                    # it is the aim of this function
                    else: file_object.force_update()
                else: file_object.insert()

    def end_update(self, result = True):
        self._update_end = True
        if result: log.msg(_("The %s library has been updated") % self.type)
        else:
            msg = _("Unable to update the %s library. See log.") % self.type
            log.err(msg)
            self._update_error = msg
        return True

    def _get_file_info(self, filename):
        (base, ext) = os.path.splitext(filename)
        ext = ext.lower()
        if ext in self.ext_dict.keys():
            return self.ext_dict[ext]
        else:
            raise NotSupportedFormat

    #######################################################################
    ## Inotify actions
    #######################################################################
    def set_inotify_connection(self, db):
        self.inotify_db = db

    @inotify_action
    def add_file(self, path, name):
        try: obj_cls = self._get_file_info(name)
        except NotSupportedFormat:
            return False

        file_path = os.path.join(path, name)
        dir = self.strip_root(path)
        f_obj = self.file_class(self.inotify_db, dir, name, file_path, obj_cls)
        if not f_obj.insert(): # insert failed
            return False
        self._add_missing_dir(path)
        return True

    @inotify_action
    def update_file(self, path, name):
        try: obj_cls = self._get_file_info(name)
        except NotSupportedFormat:
            return False

        file_path = os.path.join(path, name)
        dir = self.strip_root(path)
        f_obj = self.file_class(self.inotify_db, dir, name, file_path, obj_cls)
        if not f_obj.update(): # update failed
            return False
        return True

    @inotify_action
    def remove_file(self, path, name):
        self.inotify_db.remove_file(self.strip_root(path), name, self.table)
        self._remove_empty_dir(path)
        return True

    @inotify_action
    def add_directory(self, path, name, dirlink=False):
        dir_path = os.path.join(path, name)

        if dirlink:
            tuple = (self.strip_root(path), name)
            self.inotify_db.insert_dirlink(tuple, self.table)
            self.watcher.watch_dir(dir_path, self)

        self.walk_directory(self.inotify_db, dir_path,
                            [], [], self.get_root_paths(self.inotify_db))
        self._add_missing_dir(dir_path)
        self._remove_empty_dir(path)
        return True

    @inotify_action
    def remove_directory(self, path, name, dirlink=False):
        rel_path = self.strip_root(path)

        if dirlink:
            self.inotify_db.remove_dirlink(rel_path, name, self.table)
            self.watcher.stop_watching_dir(os.path.join(path, name))

        self.inotify_db.remove_dir(rel_path, name, self.table)
        self._remove_empty_dir(path)
        return True

    def _remove_empty_dir(self, path):
        path = self.strip_root(path)
        while path != "":
            if len(self.inotify_db.get_all_files(path, self.table)) > 0:
                break
            (path, dirname) = os.path.split(path)
            self.inotify_db.remove_dir(path, dirname, self.table)

    def _add_missing_dir(self, path):
        """ add missing dir in the mediadb """
        path = self.strip_root(path)
        while path != "":
            (path, dirname) = os.path.split(path)
            if self.inotify_db.is_dir_exist(path, dirname, self.table):
                break
            self.inotify_db.insert_dir((path, dirname), self.table)


class AudioLibrary(_Library):
    table = "audio_library"
    type = "audio"
    file_class = DeejaydAudioFile
    update_signal_name = 'mediadb.aupdate'

    def _build_supported_extension(self, player):
        self.ext_dict = formats.get_audio_extensions(player)

    def search(self,type,content):
        accepted_type = ('all','title','genre','filename','artist','album')
        if type not in accepted_type:
            raise NotFoundException

        rs = self.db_con.search_audio_library(type,content)
        return self._format_db_rsp(rs)["files"]

    def find(self,type,content):
        accepted_type = ('title','genre','filename','artist','album')
        if type not in accepted_type:
            raise NotFoundException

        rs = self.db_con.find_audio_library(type,content)
        return self._format_db_rsp(rs)["files"]

    def _format_db_rsp(self,rs):
        # format correctly database result
        files = []
        dirs = []
        for song in rs:
            if song[3] == 'directory': dirs.append(song[2])
            else:
                file_info = _media.SongMedia(self.db_con, song)
                file_info["uri"] = "file://"+urllib.quote(\
                    os.path.join(self._path,song[1],song[2]))
                files.append(file_info)
        return {'files':files, 'dirs': dirs}


class VideoLibrary(_Library):
    table = "video_library"
    type = "video"
    file_class = DeejaydVideoFile
    update_signal_name = 'mediadb.vupdate'

    def search(self, content):
        rs = self.db_con.search_video_library(content)
        return self._format_db_rsp(rs)["files"]

    def _build_supported_extension(self, player):
        self.ext_dict = formats.get_video_extensions(player)

    def _format_db_rsp(self,rs):
        # format correctly database result
        files = []
        dirs = []
        for (id,dir,fn,t,ti,videow,videoh,sub,len) in rs:
            if t == 'directory': dirs.append(fn)
            else:
                file_info = {"path":os.path.join(dir,fn),"length":len,
                             "media_id":id,"filename":fn,"dir":dir,
                             "title":ti,
                             "videowidth":videow,"videoheight":videoh,
                             "uri":"file://"+os.path.join(self._path,dir,fn),
                             "external_subtitle":sub,"type":"video"}
                files.append(file_info)
        return {'files':files, 'dirs': dirs}

# vim: ts=4 sw=4 expandtab
