
#include <sys/types.h>
typedef int FILE;
#include <linux/ext2_fs.h>
#include <ext2fs/ext2fs.h>

#include "dec_prom.h"
#include "delo.h"

#define ec ,},{

static struct {
    int errnum;
    char *desc;
} ext2errors[] = { 
{ 0, "OK"
#include "ext2_err.et"
}
};

static errcode_t linux_open (const char *name, int flags, io_channel * channel);
static errcode_t linux_close (io_channel channel);
static int partition_start;

#define _BLOCK_SIZE	512
#define _PART_START	64

extern const struct callback *callv;

static errcode_t delo_set_blksize(io_channel channel, int blksize);
static errcode_t delo_open(const char *name, int flags, io_channel *channel);
static errcode_t delo_read_blk(io_channel channel,
		unsigned long block, int count, void *data);
static errcode_t delo_close(io_channel channel);
static errcode_t delo_write_blk(io_channel channel,
		unsigned long block, int count, const void *data);
static errcode_t delo_flush(io_channel channel);

static struct struct_io_manager struct_delo_io_manager = {
	EXT2_ET_MAGIC_IO_MANAGER,
	"delo i/o manager",
	delo_open,
	delo_close,
	delo_set_blksize,
	delo_read_blk,
	delo_write_blk,
	delo_flush
};

static ext2_filsys fs = 0;
static io_manager delo_io_manager = &struct_delo_io_manager;
static void *filebuffer;

static errcode_t delo_open(const char *name, int flags, io_channel *channel) {

	io_channel	io;	
	int		retval;

#ifdef DEBUG
	puts("delo_open: called");
#endif

	retval=bootinit(0);

#ifdef DEBUG
	printf("delo_open: bootinit returned %d\n",retval);
#endif

	if (retval)
		return EXT2_ET_BAD_DEVICE_NAME;

	/* This cant fail :) */
	io = (io_channel) malloc (sizeof (struct struct_io_channel));

	memset(io, 0, sizeof(struct struct_io_channel));

	io->magic = EXT2_ET_MAGIC_IO_CHANNEL;
	io->manager = delo_io_manager;
	io->name = (char *) malloc (strlen(name) + 1);
 	strcpy(io->name, name);
	io->block_size = _BLOCK_SIZE;
	io->read_error = 0;
	io->write_error = 0;

	*channel = io;
	return(0);
}

static errcode_t delo_close(io_channel channel) {
#ifdef DEBUG
	printf("delo_close: called\n");
#endif
	return(0);
}

static errcode_t delo_set_blksize(io_channel channel, int blksize) {
	channel->block_size = blksize;
#ifdef DEBUG
	printf("delo_set_blksize: called %d\n",blksize);
#endif
	return(0);
}

static errcode_t delo_read_blk(io_channel channel,
		unsigned long block, int count, void *data) {

	int retval;
	struct ext2_super_block *sb = (struct ext2_super_block *) data;

	/* FIXME: This is our partition start */
	block=block * (channel->block_size / _BLOCK_SIZE) +
				partition_start;

	/* If its negative it means bytes - Positive -> Blocks */
	count=( count < 0 ) ?
			-count :
			count * channel->block_size;

#ifdef DEBUG
	printf("delo_read_blk: called for %d count %d to %x\n",
			block, count, data);
#endif

	retval=bootread(block, data, count);
	
	return(0);
}

static errcode_t delo_write_blk(io_channel channel,
		unsigned long block, int count, const void *data) {
#ifdef DEBUG
	puts("delo_write_blk: called");
#endif
	return(0);
}

static errcode_t delo_flush(io_channel channel) {
#ifdef DEBUG
	puts("delo_flush: called");
#endif
	return(0);
}

void com_err (const char *a, long i, const char *fmt, ... ) {
	printf((char *) fmt);
}

static void ext2fs_error (int errcode)
{
    int i;
    
    for (i = 0; i < sizeof (ext2errors) / sizeof (ext2errors[0]); i++)
    	if (ext2errors [i].errnum == errcode) {
    	    printf("%s",ext2errors [i].desc);
    	    return;
    	}
    printf("Unknown ext2 error");
}

static int delo_dump_block (ext2_filsys fs, blk_t *blocknr, int blockcnt, 
			void *private) {

#ifdef DEBUG
	printf("delo_dump_block: blocknr %d blockcnt %d to %x\n",
				*blocknr, blockcnt, 
				filebuffer + blockcnt * fs->io->block_size);

	printf("%d/%d ", *blocknr, blockcnt);
#endif

	if (blockcnt < 0) 
		return 0;

	/* Some progress indicator */
	if (blockcnt % 120 == 0)
		printf(".");

	if (io_channel_read_blk(fs->io,
				*blocknr,
				1,
				filebuffer + blockcnt * fs->io->block_size))
		return BLOCK_ABORT;

	return(0);
}

void *readfile(char *filename) {
	int	retval;
	ino_t	root = EXT2_ROOT_INO,
		inode = 0;
	struct 	ext2_inode	inodebuffer;

	retval=ext2fs_namei_follow(fs, root, root, filename, &inode);

#ifdef DEBUG
	printf("readfile: ext2fs_namei_follow returned %d\n",retval);	
#endif

	if (retval != 0) {
		printf("extfs_open returned ");
		ext2fs_error(retval);
		printf("(%d)\n", retval);
		return 0;
	}

	retval=ext2fs_read_inode(fs, inode, &inodebuffer);

	if (retval != 0) {
		printf("extfs_read_inode returned ");
		ext2fs_error(retval);
		printf("(%d)\n", retval);
		return 0;
	}
#ifdef DEBUG
	printf("Allocating %d\n",inodebuffer.i_size + _SECTOR_SIZE);
#endif

	filebuffer=malloc(inodebuffer.i_size + _SECTOR_SIZE);

	/* Beginning of progress indicator */
	printf("Loading %s .", filename);

	retval=ext2fs_block_iterate(fs, inode, 0, 0, delo_dump_block, 0);

#ifdef DEBUG
	printf("readfile: ext2fs_block_iterate returned %d\n",retval);	
#endif

	if (retval != 0) {
		printf("ext2fs_block_iterate returned ");
		ext2fs_error(retval);
		printf("(%d)\n",retval);
		return 0;
	}	

	/* End of progress indicator */
	printf(" ok\n");

#ifdef DEBUG
	puts("readfile: ending");
#endif

	return filebuffer;
}

int open_partition(char *partition) {
	int retval;

#ifdef DEBUG
	printf("Getting partition info\n");
#endif

	if (!(partition_start  = getpartoffset(partition))) {
		printf("Couldnt find partition %s\n",partition);
		return(1);
	}

	if (retval=ext2fs_open(partition, EXT2_FLAG_RW,
				0, 0, delo_io_manager, &fs)) {
		printf("extfs_open returned ");
		ext2fs_error(retval);
		printf("(%d)\n", retval);
	}
	return retval;
}
