/* mousing.cpp
 * callback functions for handling mouse clicks, drags, etc.
 *
 *  for Denemo, a gtk+ frontend to GNU Lilypond
 *  (c) 2000-2005 Matthew Hiller
 */

#include "commandfuncs.h"
#include <math.h>
#include "staffops.h"
#include "utils.h"

/**
 * Get the mid_c_offset of an object or click from its height relative
 * to the top of the staff.  
 */
gint
offset_from_height (gdouble height, enum clefs clef)
{
  /* Offset from the top of the staff, in half-tones.  */
  gint half_tone_offset = ((gint) (height / HALF_LINE_SPACE+((height>0)?0.5:-0.5)));

#define R(x) return x - half_tone_offset

  switch (clef)
    {
    case DENEMO_TREBLE_CLEF:
      R (10);
      break;			/* This break and the ones following are probably gratuitous.  */
    case DENEMO_BASS_CLEF:
      R (-2);
      break;
    case DENEMO_ALTO_CLEF:
      R (4);
      break;
    case DENEMO_G_8_CLEF:
      R (3);
      break;
    case DENEMO_TENOR_CLEF:
      R (2);
      break;
    case DENEMO_SOPRANO_CLEF:
      R (8);
      break;
    }
#undef R
  return 0;
}

/**
 * Set the cursors y position from a mouse click
 *
 */
void
set_cursor_y_from_click (DenemoGUI * gui, gdouble y)
{
  /* Click height relative to the top of the staff.  */
  float click_height;
  gint staffs_from_top;

  gtk_widget_queue_draw (gui->scorearea);//??????
  staffs_from_top = gui->si->currentstaffnum - gui->si->top_staff;
  // this takes account of staff->space_above and below...

  GList *curstaff;
  DenemoStaff *staff;
  gint extra_space = 0;
  for(  curstaff = g_list_nth(gui->si->thescore,gui->si->top_staff-1) ; curstaff;curstaff=curstaff->next) {
    staff = (DenemoStaff *) curstaff->data;
    extra_space += (staff->space_above );
    if(curstaff == gui->si->currentstaff)
      break;
    extra_space += (staff->space_below + staff->haslyrics?LYRICS_HEIGHT:0);

  }
  //????????????
  //extra_space -=   (staff->space_below + staff->haslyrics?LYRICS_HEIGHT:0);
  // g_print("Check %p and %p\n", gui->si->currentstaff, g_list_nth(gui->si->thescore,gui->si->currentstaffnum - gui->si->top_staff)); 
  //g_print("Extra space %d\n",extra_space);

  click_height =
    y - (gui->si->staffspace * staffs_from_top + gui->si->staffspace / 4 + extra_space);
  gui->si->cursor_y =
    offset_from_height (click_height, (enum clefs) gui->si->cursorclef);
  gui->si->staffletter_y = offsettonumber (gui->si->cursor_y);
}

struct placement_info
{
  gint staff_number, measure_number, cursor_x;
  staffnode *the_staff;
  measurenode *the_measure;
  objnode *the_obj;
};

/* find which staff in si the height y lies in, return the staff number */

static gint staff_at (gint y, DenemoScore *si) {
  GList *curstaff;
  gint space = 0;
  gint count;
  for(curstaff = g_list_nth(si->thescore, si->top_staff-1), count=0; curstaff && y>space;curstaff=curstaff->next) {
    DenemoStaff *staff = (DenemoStaff *) curstaff->data;
    count++;
    space += (staff)->space_above +
      (staff)->space_below + si->staffspace; 
    //g_print("y %d and space %d count = %d\n",y,space, count);
  } 
  if(y<=1)
    return 1;
  return count+si->top_staff-1;
}

/**
 * Gets the position from the clicked position
 *
 */
void
get_placement_from_coordinates (struct placement_info *pi,
				gdouble x, gdouble y, DenemoScore * si)
{
  GList *mwidthiterator = g_list_nth (si->measurewidths,
				      si->leftmeasurenum - 1);
  objnode *obj_iterator;
  gint x_to_explain = (gint) (x);

  /* Calculate pi->staff_number, but don't go past si->bottom_staff.
     (It can happen if there are too few staffs to fill the drawing area
     from top to bottom.)  */
//FIXME broken when space_above, below as it assumes si->staffspace for each staff
  pi->staff_number = MIN ((si->top_staff
			   + ((gint) y) / si->staffspace), si->bottom_staff);
  
  pi->staff_number = staff_at((gint)y, si);
/*   g_print("get staff number %d\n",pi->staff_number); */
  pi->measure_number = si->leftmeasurenum;
  x_to_explain -= (KEY_MARGIN + si->maxkeywidth + SPACE_FOR_TIME);
  while (x_to_explain > GPOINTER_TO_INT (mwidthiterator->data)
	 && pi->measure_number < si->rightmeasurenum)
    {
      x_to_explain -= (GPOINTER_TO_INT (mwidthiterator->data)
		       + SPACE_FOR_BARLINE);
      mwidthiterator = mwidthiterator->next;
      pi->measure_number++;
    }

  pi->the_staff = g_list_nth (si->thescore, pi->staff_number - 1);
  pi->the_measure
    = nth_measure_node_in_staff (pi->the_staff, pi->measure_number - 1);
  if (pi->the_measure != NULL){ /*check to make sure user did not click on empty space*/
	  obj_iterator = (objnode *) pi->the_measure->data;
	  pi->cursor_x = 0;
	  pi->the_obj = NULL;
	  if (obj_iterator)
	    {
	      DenemoObject *current, *next;

	      for (; obj_iterator->next;
		   obj_iterator = obj_iterator->next, pi->cursor_x++)
		{
		  current = (DenemoObject *) obj_iterator->data;
		  next = (DenemoObject *) obj_iterator->next->data;
		  /* This comparison neatly takes care of two possibilities:

		     1) That the click was to the left of current, or

		     2) That the click was between current and next, but
		     closer to current.

		     Do the math - it really does work out.  */
		  if (x_to_explain - (current->x + current->minpixelsalloted)
		      < next->x - x_to_explain)
		    {
		      pi->the_obj = obj_iterator;
		      break;
		    }
		}
	      if (!obj_iterator->next)
		/* That is, we exited the loop normally, not through a break.  */
		{
		  DenemoObject *current = (DenemoObject *) obj_iterator->data;
		  pi->the_obj = obj_iterator;
		  /* The below makes clicking to get the object at the end of
		     a measure (instead of appending after it) require
		     precision.  This may be bad; tweak me if necessary.  */
		  if (x_to_explain > current->x + current->minpixelsalloted)
		    pi->cursor_x++;
		}
	    }
  }
}


/**
 * Mouse button press callback 
 *
 */
gint
scorearea_button_press (GtkWidget * widget, GdkEventButton * event,
			  gpointer data)
{
  struct placement_info pi;
  DenemoGUI *gui = (DenemoGUI *) data;
  if (event->y < 0)
    get_placement_from_coordinates (&pi, event->x, 0, gui->si);
  else
    get_placement_from_coordinates (&pi, event->x, event->y, gui->si);
  if (pi.the_measure != NULL){ /*don't place cursor in a place that is not there*/
	 
	  gui->si->currentstaffnum = pi.staff_number;
	  gui->si->currentstaff = pi.the_staff;
	  gui->si->currentmeasurenum = pi.measure_number;
	  gui->si->currentmeasure = pi.the_measure;
	  gui->si->currentobject = pi.the_obj;
	  gui->si->cursor_x = pi.cursor_x;
 	  gui->si->cursor_appending
	    =
	    (gui->si->cursor_x ==
	     (gint) (g_list_length ((objnode *) gui->si->currentmeasure->data)));
	  
  
	  /* Quickly redraw to reset si->cursorclef.  */
	  gtk_widget_queue_draw (gui->scorearea);
	  set_cursor_y_from_click (gui, event->y);
	  if(!(gui->mode&INPUTCLASSIC)) {
	    if(gui->si->markstaffnum)
	      unset_mark(gui);
	    else
	      set_mark(gui);
	  }
	  
  gchar *selection;
  if(gui->si->currentobject) {
    DenemoObject *curObj = gui->si->currentobject->data;
    switch(curObj->type) {
    case CHORD:
      selection = g_strdup_printf("Note/rest %s%s%s%s%s%s%s%s",((chord *) curObj->object)->slur_begin_p?", begin slur":"",
				  ((chord *) curObj->object)->slur_end_p?", end slur":"",
				  ((chord *) curObj->object)->is_tied?", tied":"",
				  ((chord *) curObj->object)->crescendo_begin_p?", begin cresc.":"",
				  ((chord *) curObj->object)->crescendo_end_p?", end cresc.":"",
				  ((chord *) curObj->object)->diminuendo_begin_p?", begin dim.":"",
				  ((chord *) curObj->object)->diminuendo_end_p?", end dim.":"",
				  ((chord *) curObj->object)->is_grace?", grace note":""
 );
      break;

    case TUPOPEN:
      selection = g_strdup_printf("Tuplet %d/%d", 	((tupopen *) curObj->object)->numerator,
				  ((tupopen *) curObj->object)->denominator);
      break;
    case TUPCLOSE: 
      selection = g_strdup_printf("End tuplet");
	break;
  case CLEF:
selection = g_strdup_printf("clef change");
break;
  case TIMESIG:
selection = g_strdup_printf("time signature change");
break;
  case KEYSIG:
selection = g_strdup_printf("key signature change");
break;
  case BARLINE:
    selection = g_strdup_printf("special barline type %d", 	((barline *) curObj->object)->type);// FIXME names for these
break;
  case STEMDIRECTIVE:
selection = g_strdup_printf("stem directive: %s",((stemdirective *) curObj->object)->type==DENEMO_STEMDOWN?
			    "stem down":((stemdirective *) curObj->object)->type==DENEMO_STEMUP?"stem up":
			    "normal stemming");
break;
  case MEASUREBREAK:
selection = g_strdup_printf("measurebreak");
break;
  case STAFFBREAK:
selection = g_strdup_printf("staffbreak");
break;
  case DYNAMIC:
selection = g_strdup_printf("Dynamic: %s", ((dynamic *) curObj->object)->type->str  );
break;
  case GRACE_START:
selection = g_strdup_printf("Grace note: %s duration %d ", ((grace *) curObj->object)->on_beat?"On beat":"Before beat",
			     ((grace *) curObj->object)->duration);
break;
  case GRACE_END:
selection = g_strdup_printf("Grace note end");
break;
  case LYRIC:
selection = g_strdup_printf("Lyric: %s",  ((lyric *) curObj->object)->lyrics->str  );
break;
  case FIGURE:
selection = g_strdup_printf("Figure");
break;

    case LILYDIRECTIVE:
      selection = g_strdup_printf("Lily directive: %s", ((GString *)((lilydirective *) curObj->object)->directive)->str);
      break;
  case FAKECHORD:
selection = g_strdup_printf("Fakechord"   );
break;
  case PARTIAL:
selection = g_strdup_printf("Partial"   );
break;
    default:
      selection = g_strdup_printf("Unlisted object selected");
    }
  } else
    selection = g_strdup_printf("No selection");
  gtk_statusbar_pop(GTK_STATUSBAR (gui->statusbar), gui->status_context_id);
  gtk_statusbar_push(GTK_STATUSBAR (gui->statusbar), gui->status_context_id,
		     selection);
  g_free(selection);

  	  gtk_widget_queue_draw (gui->scorearea);
  }
  return TRUE;
}

/**
 * Mouse button release callback 
 *
 */
gint
scorearea_button_release (GtkWidget * widget, GdkEventButton * event,
			  gpointer data)
{
  struct placement_info pi;
  DenemoGUI *gui = (DenemoGUI *) data;
  if(!(gui->mode&INPUTCLASSIC)) {
    if (gui->si->markstaffnum){
      if (event->y < 0)
	get_placement_from_coordinates (&pi, event->x, 0, gui->si);
      else
	get_placement_from_coordinates (&pi, event->x, event->y, gui->si);
      if (pi.the_measure != NULL){ /*don't place cursor in a place that is not there*/
	
	gui->si->currentstaffnum = pi.staff_number;
	gui->si->currentstaff = pi.the_staff;
	gui->si->currentmeasurenum = pi.measure_number;
	gui->si->currentmeasure = pi.the_measure;
	gui->si->currentobject = pi.the_obj;
	gui->si->cursor_x = pi.cursor_x;
	gui->si->cursor_appending
	  =
	  (gui->si->cursor_x ==
	   (gint) (g_list_length ((objnode *) gui->si->currentmeasure->data)));
	
	/* Quickly redraw to reset si->cursorclef.  */
	gtk_widget_queue_draw (gui->scorearea);
	set_cursor_y_from_click (gui, event->y);
	calcmarkboundaries (gui->si);
	gtk_widget_queue_draw (gui->scorearea);
      }
    }
  }
  return TRUE;
}

