/*
   (c) Copyright 2000  convergence integrated media GmbH.
   All rights reserved.

   Written by Denis Oliver Kropp <dok@convergence.de> and
              Andreas Hundt <andi@convergence.de>.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <directfb.h>

#include <core/coredefs.h>
#include <core/coretypes.h>

#include <core/state.h>
#include <core/gfxcard.h>
#include <core/surfaces.h>

#include <gfx/convert.h>
#include <misc/util.h>

#include "regs.h"
#include "mmio.h"
#include "matrox.h"

#include "matrox_state.h"

void matrox_set_destination( MatroxDriverData *mdrv,
                             MatroxDeviceData *mdev,
                             CoreSurface      *destination )
{
     volatile __u8 *mmio            = mdrv->mmio_base;
     SurfaceBuffer *buffer          = destination->back_buffer;
     int            bytes_per_pixel = DFB_BYTES_PER_PIXEL(destination->format);

     mdev->dst_pixelpitch  = buffer->video.pitch / bytes_per_pixel;
     mdev->dst_pixeloffset = buffer->video.offset / bytes_per_pixel;

     mga_waitfifo( mdrv, mdev, 3 );

     if (mdev->old_matrox)
          mga_out32( mmio, mdev->dst_pixeloffset, YDSTORG );
     else
          mga_out32( mmio, buffer->video.offset, DSTORG );

     mga_out32( mmio, mdev->dst_pixelpitch, PITCH );

     switch (destination->format) {
          case DSPF_A8:
               mga_out32( mmio, PW8  | NODITHER, MACCESS );
               break;
          case DSPF_RGB15:
               mga_out32( mmio, PW16 | NODITHER | DIT555, MACCESS );
               break;
          case DSPF_RGB16:
               mga_out32( mmio, PW16 | NODITHER, MACCESS );
               break;
          case DSPF_RGB24:
               mga_out32( mmio, PW24 | NODITHER, MACCESS );
               break;
          case DSPF_RGB32:
          case DSPF_ARGB:
               mga_out32( mmio, PW32 | NODITHER, MACCESS );
               break;
          default:
               BUG( "unexpected pixelformat!" );
               break;
     }
}

void matrox_set_clip( MatroxDriverData *mdrv,
                      MatroxDeviceData *mdev,
                      DFBRegion        *clip )
{
     volatile __u8 *mmio = mdrv->mmio_base;

     mga_waitfifo( mdrv, mdev, 3 );

     if (mdev->old_matrox) {
          mga_out32( mmio, (mdev->dst_pixeloffset +
                            mdev->dst_pixelpitch * clip->y1) & 0xFFFFFF, YTOP );
          mga_out32( mmio, (mdev->dst_pixeloffset +
                            mdev->dst_pixelpitch * clip->y2) & 0xFFFFFF, YBOT );
     }
     else {
          mga_out32( mmio, (mdev->dst_pixelpitch * clip->y1) & 0xFFFFFF, YTOP );
          mga_out32( mmio, (mdev->dst_pixelpitch * clip->y2) & 0xFFFFFF, YBOT );
     }

     mga_out32( mmio, ((clip->x2 & 0x0FFF) << 16) |
                       (clip->x1 & 0x0FFF), CXBNDRY );
}

void matrox_validate_Color( MatroxDriverData *mdrv,
                            MatroxDeviceData *mdev,
                            CardState        *state )
{
     volatile __u8 *mmio = mdrv->mmio_base;

     if (mdev->m_Color)
          return;

     mga_waitfifo( mdrv, mdev, 4 );

     mga_out32( mmio, U8_TO_F0915(state->color.a), ALPHASTART );
     mga_out32( mmio, U8_TO_F0915(state->color.r), DR4 );
     mga_out32( mmio, U8_TO_F0915(state->color.g), DR8 );
     mga_out32( mmio, U8_TO_F0915(state->color.b), DR12 );

     mdev->m_Color = 1;
}

void matrox_validate_color( MatroxDriverData *mdrv,
                            MatroxDeviceData *mdev,
                            CardState        *state )
{
     volatile __u8 *mmio = mdrv->mmio_base;

     __u32 color;

     if (mdev->m_color)
          return;

     switch (state->destination->format) {
          case DSPF_RGB15:
               color = PIXEL_RGB15( state->color.r,
                                    state->color.g,
                                    state->color.b );
               color |= color << 16;
               break;
          case DSPF_RGB16:
               color = PIXEL_RGB16( state->color.r,
                                    state->color.g,
                                    state->color.b );
               color |= color << 16;
               break;
          case DSPF_RGB24:
               color = PIXEL_RGB24( state->color.r,
                                    state->color.g,
                                    state->color.b );
               color |= color << 24;
               break;
          case DSPF_RGB32:
               color = PIXEL_RGB32( state->color.r,
                                    state->color.g,
                                    state->color.b );
               break;
          case DSPF_ARGB:
               color = PIXEL_ARGB( state->color.a,
                                   state->color.r,
                                   state->color.g,
                                   state->color.b );
               break;
          case DSPF_A8:
               color = state->color.a;
               color |= color << 24 | color << 16 | color << 8;
               break;
          default:
               BUG( "unexpected pixelformat!" );
               return;
     }

     mga_waitfifo( mdrv, mdev, 1 );
     mga_out32( mmio, color, FCOL );

     mdev->m_color = 1;
     mdev->m_srckey = 0;
}

static __u32 matroxSourceBlend[] = {
     SRC_ZERO,                /* DSBF_ZERO         */
     SRC_ONE,                 /* DSBF_ONE          */
     0,                       /* DSBF_SRCCOLOR     */
     0,                       /* DSBF_INVSRCCOLOR  */
     SRC_ALPHA,               /* DSBF_SRCALPHA     */
     SRC_ONE_MINUS_SRC_ALPHA, /* DSBF_INVSRCALPHA  */
     SRC_DST_ALPHA,           /* DSBF_DESTALPHA    */
     SRC_ONE_MINUS_DST_ALPHA, /* DSBF_INVDESTALPHA */
     SRC_DST_COLOR,           /* DSBF_DESTCOLOR    */
     SRC_ONE_MINUS_DST_COLOR, /* DSBF_INVDESTCOLOR */
     SRC_SRC_ALPHA_SATURATE   /* DSBF_SRCALPHASAT  */
};

static __u32 matroxDestBlend[] = {
     DST_ZERO,                /* DSBF_ZERO         */
     DST_ONE,                 /* DSBF_ONE          */
     DST_SRC_COLOR,           /* DSBF_SRCCOLOR     */
     DST_ONE_MINUS_SRC_COLOR, /* DSBF_INVSRCCOLOR  */
     DST_SRC_ALPHA,           /* DSBF_SRCALPHA     */
     DST_ONE_MINUS_SRC_ALPHA, /* DSBF_INVSRCALPHA  */
     DST_DST_ALPHA,           /* DSBF_DESTALPHA    */
     DST_ONE_MINUS_DST_ALPHA, /* DSBF_INVDESTALPHA */
     0,                       /* DSBF_DESTCOLOR    */
     0,                       /* DSBF_INVDESTCOLOR */
     0                        /* DSBF_SRCALPHASAT  */
};

static __u32 matroxModulation[] = {
     VIDEOALPHA,
     VIDEOALPHA,
     VIDEOALPHA | DIFFUSEDALPHA,
     VIDEOALPHA | MODULATEDALPHA
};

void matrox_validate_drawBlend( MatroxDriverData *mdrv,
                                MatroxDeviceData *mdev,
                                CardState        *state )
{
     volatile __u8 *mmio = mdrv->mmio_base;

     if (mdev->m_drawBlend)
          return;

     mga_waitfifo( mdrv, mdev, 1 );
     mga_out32( mmio,
                matroxSourceBlend[state->src_blend - 1] |
                matroxDestBlend  [state->dst_blend - 1], ALPHACTRL );

     mdev->m_drawBlend = 1;
     mdev->m_blitBlend = 0;
}

void matrox_validate_blitBlend( MatroxDriverData *mdrv,
                                MatroxDeviceData *mdev,
                                CardState        *state )
{
     volatile __u8 *mmio = mdrv->mmio_base;

     __u32 alphactrl = 1;

     if (mdev->m_blitBlend)
          return;

     if (state->blittingflags & (DSBLIT_BLEND_ALPHACHANNEL |
                                         DSBLIT_BLEND_COLORALPHA))
          alphactrl = matroxSourceBlend[state->src_blend - 1] |
                      matroxDestBlend  [state->dst_blend - 1];

     alphactrl |= matroxModulation [state->blittingflags & 3];

     mga_waitfifo( mdrv, mdev, 1 );
     mga_out32( mmio, alphactrl, ALPHACTRL );

     mdev->m_blitBlend = 1;
     mdev->m_drawBlend = 0;
}

void matrox_validate_Source( MatroxDriverData *mdrv,
                             MatroxDeviceData *mdev,
                             CardState        *state )
{
     volatile __u8 *mmio = mdrv->mmio_base;

     __u32 texctl, texctl2;

     CoreSurface   *surface = state->source;
     SurfaceBuffer *buffer  = surface->front_buffer;

     if (mdev->m_Source)
          return;

     mdev->src_pixelpitch = buffer->video.pitch / DFB_BYTES_PER_PIXEL(surface->format);

     mdev->matrox_w2 = log2( surface->width );
     mdev->matrox_h2 = log2( surface->height );

     if (state->blittingflags & DSBLIT_BLEND_ALPHACHANNEL)
          texctl = TAMASK;
     else
          texctl = TAKEY;

     switch (surface->format) {
          case DSPF_YUY2:
               texctl |= TW422;
               break;
          case DSPF_UYVY:
               texctl |= TW422UYVY;
               break;
          case DSPF_A8:
               texctl |= TW8A;
               break;
          case DSPF_RGB15:
               texctl |= TW15;
               break;
          case DSPF_RGB16:
               texctl |= TW16;
               break;
          case DSPF_RGB32:
          case DSPF_ARGB:
               texctl |= TW32;
               break;
          default:
               BUG( "unexpected pixelformat!" );
               break;
     }

     texctl |= CLAMPUV | ((mdev->src_pixelpitch&0x7ff)<<9) | PITCHEXT | NOPERSPECTIVE;

     if (state->blittingflags & DSBLIT_COLORIZE)
          texctl |= TMODULATE;

     if (state->blittingflags & DSBLIT_SRC_COLORKEY) {
          texctl |= DECALCKEY | STRANS;
          texctl2 = DECALDIS;
     }
     else
          texctl2 = DECALDIS | CKSTRANSDIS;

     mga_waitfifo( mdrv, mdev, 5);

     mga_out32( mmio, texctl, TEXCTL );
     mga_out32( mmio, texctl2, TEXCTL2 );
     mga_out32( mmio, ((surface->width -1)<<18) |
                      ((8-mdev->matrox_w2)&63)<<9 | mdev->matrox_w2, TEXWIDTH );
     mga_out32( mmio, ((surface->height-1)<<18) |
                      ((8-mdev->matrox_h2)&63)<<9 | mdev->matrox_h2, TEXHEIGHT );
     mga_out32( mmio, buffer->video.offset, TEXORG );

     mdev->m_Source = 1;
}

void matrox_validate_source( MatroxDriverData *mdrv,
                             MatroxDeviceData *mdev,
                             CardState        *state )
{
     volatile __u8 *mmio            = mdrv->mmio_base;
     CoreSurface   *surface         = state->source;
     SurfaceBuffer *buffer          = surface->front_buffer;
     int            bytes_per_pixel = DFB_BYTES_PER_PIXEL(surface->format);

     if (mdev->m_source)
          return;

     mdev->src_pixelpitch = buffer->video.pitch / bytes_per_pixel;

     if (mdev->old_matrox)
          mdev->src_pixeloffset = buffer->video.offset / bytes_per_pixel;
     else {
          mga_waitfifo( mdrv, mdev, 1);
          mga_out32( mmio, buffer->video.offset, SRCORG );
     }

     mdev->m_source = 1;
}

void matrox_validate_SrcKey( MatroxDriverData *mdrv,
                             MatroxDeviceData *mdev,
                             CardState        *state )
{
     volatile __u8 *mmio = mdrv->mmio_base;

     if (mdev->m_SrcKey)
          return;

     mga_waitfifo( mdrv, mdev, 2);

     if (DFB_BYTES_PER_PIXEL(state->source->format) > 2) {
          mga_out32( mmio, (0xFFFF << 16) |
                     (state->src_colorkey & 0xFFFF),
                     TEXTRANS );
          mga_out32( mmio, (((1 << (DFB_BITS_PER_PIXEL(state->source->format)-16)) - 1) << 16) |
                     ((state->src_colorkey & 0xFFFF0000) >> 16),
                     TEXTRANSHIGH );
     }
     else {
          mga_out32( mmio, (((1 << DFB_BITS_PER_PIXEL(state->source->format)) - 1) << 16) |
                     (state->src_colorkey & 0xFFFF),
                     TEXTRANS );
          mga_out32( mmio, 0, TEXTRANSHIGH );
     }

     mdev->m_SrcKey = 1;
}

void matrox_validate_srckey( MatroxDriverData *mdrv,
                             MatroxDeviceData *mdev,
                             CardState        *state )
{
     volatile __u8 *mmio    = mdrv->mmio_base;
     CoreSurface   *surface = state->source;
     __u32          mask;

     if (mdev->m_srckey)
          return;

     mask = (1 << MIN( 24, DFB_BITS_PER_PIXEL(surface->format) )) - 1;

     mga_waitfifo( mdrv, mdev, 2);
     mga_out32( mmio, state->src_colorkey, FCOL );

     if (DFB_BYTES_PER_PIXEL(state->source->format) > 2)
          mga_out32( mmio, mask, BCOL );
     else
          mga_out32( mmio, (mask << 16) | mask, BCOL );

     mdev->m_srckey = 1;
     mdev->m_color = 0;
}

