/*
   (c) Copyright 2000  convergence integrated media GmbH.
   All rights reserved.

   Written by Denis Oliver Kropp <dok@convergence.de>,
              Andreas Hundt <andi@convergence.de> and
              Sven Neumann <sven@convergence.de>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <malloc.h>

#include <freetype/ftglyph.h>

#include <directfb.h>
#include <directfb_internals.h>

#include <core/coredefs.h>
#include <core/coretypes.h>

#include <core/fonts.h>
#include <core/gfxcard.h>
#include <core/surfaces.h>
#include <core/surfacemanager.h>

#include <gfx/convert.h>

#include <media/idirectfbfont.h>

#include <misc/tree.h>
#include <misc/mem.h>


static FT_Library library = NULL;


char *get_type()
{
     return "IDirectFBFont";
}

char *get_implementation()
{
     return "FT2";
}

DFBResult Probe( void *data )
{
     return DFB_OK;
}

DFBResult render_glyph( CoreFont      *thiz,
                        unichar        glyph,
                        CoreGlyphData *info,
                        CoreSurface   *surface )
{
     __u8 *src, *dst;
     FT_Error err;
     FT_Face  face;
     FT_Int   load_flags;
     int y;
     int index;
     int pitch;

     face = thiz->impl_data;

     index = FT_Get_Char_Index(face, glyph);

     load_flags = (FT_Int) face->generic.data;
     load_flags |= FT_LOAD_RENDER;

     err = FT_Load_Glyph( face, index, load_flags );
     if (err) {
          ERRORMSG( "DirectB/FontFT2: "
                    "Could not render glyph for character #%d!\n", glyph );
          return DFB_FAILURE;
     }

     HEAVYDEBUGMSG( "loaded %d\n", glyph );

     HEAVYDEBUGMSG( "top: +++ %d +++\n", face->glyph->bitmap_top );
     HEAVYDEBUGMSG( "bitmap.rows: %d +++ metrics.height: %d +++\n",
                    face->glyph->bitmap.rows,
                    (int)(face->glyph->metrics.height >> 6) );

     surfacemanager_lock( gfxcard_surface_manager() );
     err = surface_software_lock( surface, DSLF_WRITE, (void**)&dst, &pitch, 0 );
     surfacemanager_unlock( gfxcard_surface_manager() );
     if (err) {
          ERRORMSG( "DirectB/FontFT2: Unable to lock surface!\n" );
          return err;
     }

     info->width = face->glyph->bitmap.width;
     if (info->width + thiz->next_x > surface->width)
          info->width = surface->width - thiz->next_x;

     info->height = face->glyph->bitmap.rows;
     if (info->height > surface->height)
          info->height = surface->height;

     info->left = face->glyph->bitmap_left;
     info->top  = thiz->ascender - face->glyph->bitmap_top;

     src = face->glyph->bitmap.buffer;
     dst += thiz->next_x * DFB_BYTES_PER_PIXEL(surface->format);

     for (y=0; y < info->height; y++) {

          switch (face->glyph->bitmap.pixel_mode) {
          case ft_pixel_mode_grays:
               switch (DFB_BYTES_PER_PIXEL(surface->format)) {
               case 4:
                    span_a8_to_argb( src, (__u32*) dst, info->width );
                    break;
               case 1:
                    memcpy( dst, src, info->width );
                    break;
               default:
                    break;
               }
               break;

          case ft_pixel_mode_mono:
               switch (DFB_BYTES_PER_PIXEL(surface->format)) {
               case 4:
                    span_a1_to_argb( src, (__u32*) dst, info->width );
                    break;
               case 1:
                    span_a1_to_a8( src, dst, info->width );
                    break;
               default:
                    break;
               }
               break;

          default:
               break;

          }

          src += face->glyph->bitmap.pitch;
          dst += pitch;
     }

     surface_unlock( surface, 0 );

     return DFB_OK;
}

DFBResult get_glyph_info( CoreFont      *thiz,
                          unichar        glyph,
                          CoreGlyphData *info )
{
     FT_Error err;
     FT_Face  face;
     FT_Int   load_flags;
     int index;

     face = thiz->impl_data;

     index = FT_Get_Char_Index(face, glyph);

     load_flags = (FT_Int) face->generic.data;

     err = FT_Load_Glyph( face, index, load_flags);
     if (err) {
          ERRORMSG( "DirectB/FontFT2: "
                    "Could not load glyph for character #%d!\n", glyph );
          return DFB_FAILURE;
     }

     if (face->glyph->format != ft_glyph_format_bitmap) {
          err = FT_Render_Glyph( face->glyph, ft_render_mode_normal );
          if (err) {
               ERRORMSG( "DirectFB/FontFT2: Could not "
                         "render glyph for character #%d!\n", glyph );

               return DFB_FAILURE;
          }
     }

     info->width   = face->glyph->bitmap.width;
     info->height  = face->glyph->bitmap.rows;
     info->advance = face->glyph->advance.x >> 6;

     return DFB_OK;
}

DFBResult get_kerning( CoreFont *thiz,
                       unichar   prev,
                       unichar   current,
                       int      *kerning )
{
     FT_Face   face;
     FT_Vector vector;
     int prev_index;
     int current_index;

     face = thiz->impl_data;

     prev_index    = FT_Get_Char_Index(face, prev);
     current_index = FT_Get_Char_Index(face, current);

     FT_Get_Kerning( face, prev_index, current_index, ft_kerning_default, &vector );
     *kerning = vector.x >> 6;

     return DFB_OK;
}

void IDirectFBFont_FT2_Destruct( IDirectFBFont *thiz )
{
     IDirectFBFont_data *data = (IDirectFBFont_data*)thiz->priv;

     if (data->font->impl_data)
          FT_Done_Face (data->font->impl_data);
     data->font->impl_data = NULL;

     IDirectFBFont_Destruct (thiz);
}

DFBResult IDirectFBFont_FT2_Release( IDirectFBFont *thiz )
{
     INTERFACE_GET_DATA(IDirectFBFont)

     if (--data->ref == 0) {
          IDirectFBFont_FT2_Destruct( thiz );
     }

     return DFB_OK;
}

DFBResult Construct( IDirectFBFont *thiz,
                     const char *filename,
                     DFBFontDescription *desc )
{
     CoreFont *font;
     FT_Face  face;
     FT_Error err;
     FT_Int   load_flags;

     HEAVYDEBUGMSG( "DirectFB/FontFT2: Construct font '%s' height %d\n",
                    filename,
                    (desc->flags & DFDESC_HEIGHT) ? desc->height : -1 );

     if (!library) {
          err = FT_Init_FreeType( &library );
          if (err) {
               ERRORMSG( "DirectFB/FontFT2: "
                         "Initializing the FreeType Library failed!\n" );
               return DFB_FAILURE;
          }
     }

     err = FT_New_Face( library, filename, 0, &face );
     if (err) {
          /*ERRORMSG( "DirectFB/FontFT2: "
                    "Failed loading font file `%s'!\n", filename );*/
          return DFB_FAILURE;
     }

     if (desc->flags & (DFDESC_HEIGHT | DFDESC_WIDTH)) {
          err = FT_Set_Pixel_Sizes( face,
                             (desc->flags & DFDESC_WIDTH)  ? desc->width  : 0,
                             (desc->flags & DFDESC_HEIGHT) ? desc->height : 0 );
          if (err) {
               ERRORMSG( "DirectB/FontFT2: "
                         "Could not set pixel size to %d!\n", desc->height );
               FT_Done_Face( face );
               return DFB_FAILURE;
          }
     }

     load_flags = FT_LOAD_DEFAULT;
     if ((desc->flags & DFDESC_ATTRIBUTES)) {
         if (desc->attributes & DFFA_NOHINTING)
              load_flags |= FT_LOAD_NO_HINTING;
         if (desc->attributes & DFFA_MONOCHROME)
              load_flags |= FT_LOAD_MONOCHROME;
     }
     face->generic.data = (void *) load_flags;
     face->generic.finalizer = NULL;

     font = font_create();

     font->ascender    = face->size->metrics.ascender  >> 6;
     font->descender   = face->size->metrics.descender >> 6;
     font->height      = (face->size->metrics.ascender -
                          face->size->metrics.descender) >> 6;
     font->maxadvance  = face->size->metrics.max_advance >> 6;

     HEAVYDEBUGMSG( "DirectFB/FontFT2: font->height = %d\n", font->height );
     HEAVYDEBUGMSG( "DirectFB/FontFT2: font->ascender = %d\n", font->ascender );
     HEAVYDEBUGMSG( "DirectFB/FontFT2: font->descender = %d\n",font->descender );

     font->impl_data = face;

     font->GetGlyphInfo = get_glyph_info;
     font->RenderGlyph  = render_glyph;

     if (FT_HAS_KERNING ( face )) {
          font->GetKerning = get_kerning;
     }

     if ((desc->flags & DFDESC_ATTRIBUTES) &&
         (desc->attributes & DFFA_NOKERNING)) {
          font->GetKerning = NULL;
     }

     IDirectFBFont_Construct( thiz, font );

     thiz->Release = IDirectFBFont_FT2_Release;

     return DFB_OK;
}


