#include "config.h"

#ifdef HAVE_LIBMPEG3
#include "mpeg.h"
#include <unistd.h>
#include <pthread.h>
#include <stdlib.h>
#include <qthread.h>
#include "mainwnd.h"
#include "pump.h"
#include <qapplication.h>

#define BLOCK_SKIP 1152

MpegStreamSource::MpegStreamSource(QWidget *dest) : StreamSource(dest)
{
	s_ext_framebuf=0;
	s_has_video=false;
	new_frame=false;
	fp=NULL;
	fpv=NULL;
	file_pos=0L;
	framebuf=app_window->get_frame_buffer();
	do_video=true;
	fbwidth=640;
	fbheight=480;
	int i;
	for(i=0;i<480;++i)
		memset(framebuf[i], 0, 640*3+4);
}

MpegStreamSource::~MpegStreamSource()
{
	stop();
	do_video=false;

	if(fp)
		mpeg3_close(fp);
	if(fpv)
		mpeg3_close(fpv);
}

int MpegStreamSource::attach(QString file, bool vid)
{
	do_video=vid;
	return attach(file);
}

int MpegStreamSource::attach(QString file)
{
	if(access(file, 0) < 0)
		return -1;
	
	filename=file;

	if(mpeg3_check_sig((char *)(const char *)file) == 0)
		return -1;
	
	fp=mpeg3_open((char *)(const char *)file);
	if(fp == NULL)
		return -1;
	
	if(mpeg3_has_audio(fp) == 0)
	{
		mpeg3_close(fp);
		fp=NULL;
		return -1;
	}

#ifdef USE_VIDEO
	if(do_video)
	{
		if(mpeg3_has_video(fp) != 0)
		{
			s_has_video=true;
			fpv=mpeg3_open_copy((char *)(const char *)file, fp);
		}
	}
#endif

	if(mpeg3_audio_channels(fp, 0) < 2)
	{
		mpeg3_close(fp);
		fp=NULL;
		return -1;
	}

	frame_pos=0L;
	max_frames=mpeg3_audio_samples(fp, 0);
	file_pos=0L;

	sample_rate=mpeg3_sample_rate(fp, 0);

#ifdef USE_VIDEO
	if(fpv)
	{
		vwidth=mpeg3_video_width(fpv, 0);
		vheight=mpeg3_video_height(fpv, 0);
		vrate=mpeg3_frame_rate(fpv, 0);
		vframes=mpeg3_video_frames(fpv, 0);

//printf("video %dx%d @ %f fps, %ld frames\n", vwidth, vheight, vrate, vframes);
		curframe=-1;
	}
#endif

    m_artist="";
    m_title="";

	char fnb[1024];
	strcpy(fnb, file);
	char *fn=strrchr(fnb, '/');
	if(!fn)
		fn=fnb;
	else
		++fn;
	char *dot=strrchr(fn, '.');
	if(dot)
		(*dot)='\0';
	char *sep=strstr(fn, " - ");
	if(sep)
	{
		(*sep)='\0';
		sep+=3;
	}

    if(sep)
    {
        m_artist=fn;
        m_title=sep;
    }
    else
    {
        m_artist="";
        m_title=fn;
    }

	return 0;
}

int MpegStreamSource::get_buffer(char *buf, int max)
{
	if(!fp || !playing)
		return 0;
	
	if(file_pos+max > max_frames)
		max=max_frames-file_pos;

	if(!max)
	{
		if(playing)
			QThread::postEvent(this, new QEvent(QEvent::User));
		playing=0;
		return 0;
	}

	if(play_samples)
	{
		if((unsigned long)max > play_samples)
			max=(int)play_samples;
	}
	mpeg3_read_audio(fp, NULL, (short *)buf, 0, max, 0);
	mpeg3_reread_audio(fp, NULL, (short *)(buf+sizeof(short)*max), 1, max, 0);

	file_pos+=max;

	if(play_samples)
	{
		play_samples-=max;
		if(play_samples <= 0)
		{
			if(loop_play)
				play(last_start, last_samples, true);
			else
			{
				playing=0;
				QThread::postEvent(this, new QEvent(QEvent::User));
			}
		}
	}
	
	calcAgc(buf, max);
	return max;
}

int MpegStreamSource::play(unsigned long start_frame, unsigned long samples, bool loop)
{
	if(!fp)
		return -1;
	
	if(!loop_play || !loop)
		stop();

	if(start_frame >= max_frames)
		return -1;

	loop_play=loop;
	last_start=start_frame;

	unsigned long realstart=(start_frame/1152)*1152;
	if(realstart > 2*BLOCK_SKIP)
		realstart-=2*BLOCK_SKIP;
	if(realstart < BLOCK_SKIP)
		realstart=0;
	int consume=start_frame-realstart;

	mpeg3_set_sample(fp, realstart, 0);
	if(consume)
	{
		char *buffer;

		buffer=new char[consume*sizeof(short)];
		mpeg3_read_audio(fp, NULL, (short *)buffer, 0, consume, 0);
		delete buffer;
	}
	file_pos=start_frame;
	frame_pos=start_frame;
#ifdef USE_VIDEO
	if(fpv)
	{
		if(s_ext_framebuf)
		{
			long pframe=(long)(((float)start_frame/(float)sample_rate)*vrate);
			preview_lock.lock();
			mpeg3_set_frame(fpv, pframe, 0);
			mpeg3_read_frame(fpv, s_ext_framebuf, 0, 0, vwidth, vheight, s_ext_fbwidth, s_ext_fbheight, MPEG3_RGB888, 0);
			mpeg3_set_frame(fpv, curframe, 0);
			preview_lock.unlock();
			app_window->update_display();
		}
	}
#endif
	play_samples=samples;
	last_samples=play_samples;

	playing=1;
	app_window->get_pump()->work();
	return 0;
}

int MpegStreamSource::stop(void)
{
	if(!player || !fp || !playing)
		return 0;

	StreamSource::stop();
	loop_play=false;
	playing=0;
	return 0;
}

StreamSource *MpegStreamSource::get_source(void)
{
	MpegStreamSource *s=new MpegStreamSource(app_window);
	s->attach(filename, false);
	return (StreamSource *)s;
}

void MpegStreamSource::exit_loop(void)
{
	pthread_mutex_lock(&control_lock);
	play_samples=0;
	loop_play=false;
	pthread_mutex_unlock(&control_lock);
}

void MpegStreamSource::prefetch()
{
}

void MpegStreamSource::decode_video()
{
#ifdef USE_VIDEO
	vframe=(long)(((float)frame_pos/(float)sample_rate)*vrate);

	if(fpv && playing && do_video)
	{
		if(vframe != curframe)
		{
			if(vframe == (curframe+1))
			{
//printf("Contiguous frame\n");
				++curframe;
				frame_mutex.lock();
				preview_lock.lock();
				mpeg3_read_frame(fpv, framebuf, 0, 0, vwidth, vheight, fbwidth, fbheight, MPEG3_RGB565, 0);
				preview_lock.unlock();
				frame_mutex.unlock();
			}
			else
			{
				int off=vframe-curframe;
				if(off > 0 && off < 10)
				{
					preview_lock.lock();
					mpeg3_drop_frames(fpv, off-1, 0);
					preview_lock.unlock();
					curframe+=off;
//printf("Dropped %d frames\n", off-1);
				}
				else
				{
					curframe=vframe;
					preview_lock.lock();
					mpeg3_set_frame(fpv, curframe, 0);
					preview_lock.unlock();
//printf("Disjoint frame\n");
				}
				frame_mutex.lock();
				preview_lock.lock();
				mpeg3_read_frame(fpv, framebuf, 0, 0, vwidth, vheight, fbwidth, fbheight, MPEG3_RGB565, 0);
				preview_lock.unlock();
				frame_mutex.unlock();
			}

			new_frame=true;
		}
	}
#endif
}

unsigned char **MpegStreamSource::get_frame()
{
#ifdef USE_VIDEO
	if(!new_frame)
		return 0;
	new_frame=false;
	return framebuf;
#else
	return 0;
#endif
}

void MpegStreamSource::lock_frame()
{
#ifdef USE_VIDEO
	frame_mutex.lock();
#endif
}

void MpegStreamSource::unlock_frame()
{
#ifdef USE_VIDEO
	frame_mutex.unlock();
#endif
}

bool MpegStreamSource::has_video()
{
	return s_has_video;
}

void MpegStreamSource::set_preview_framebuffer(int width, int height, unsigned char **ext_framebuf)
{
printf("Setting ext_framebuf\n");
	s_ext_fbwidth=width;
	s_ext_fbheight=height;
	s_ext_framebuf=ext_framebuf;
}

#endif /* HAVE_LIBMPEG3 */
