import gnu.regexp.*;
import java.util.*;
import java.text.*;

/**
 * Replacement of placeholders for generation of indices.
 * <p>In the settings file, the user specifies the exact html code (or
 * otherwise) to be placed in the output file. This class is responsible for
 * searching this code and replacing placeholders with their related data. For
 * example, a typical placeholder might be <code>&lt;IG Field="GenDate"
 * Format="DD MM YYYY"/&gt;</code>. This would be replaced by the generation
 * date formatted in a particular way. 
 * 
 * <p>The actual placeholders consist of a field attribute which identified them
 * and then any number of other attributes. Each of these is Localized, using
 * the methods of Project to check them (they share locale with the settings
 * file). If a placeholder is incorrectly constructed - has too many attributes,
 * for example - these methods should issue warnings. 
 *
 * <p>There are two types of contructor for this class to aid different sections
 * of the program:
 *
 * <dl>
 * <dt>Multiple pattern use
 * <dd>The single object will be used to replace placeholders in multiple
 * patterns. This is used when there are few replacements for each pattern, e.g.
 * the code at the start and end of the file.
 *
 * <dt>Single pattern use
 * <dd>When a sinlge pattern is used many times, we try to make this code as
 * fast as possible. The pattern is supplied to the constructor and analysed
 * there. First, and placeholders which are not dependant on information
 * specific to the item (e.g. generate date) is replaced with the required
 * string. Then, we take the remaining string and split it up into substrings
 * and placeholders. These are placed on a linked list so that the final output
 * string can be easily constructed.
 * 
 * <br><br>$Id: IGReplace.java,v 1.32 2002/09/11 13:22:25 howama Exp $
 * 
 * @author Mark Howard
 */
public class IGReplace{
    /**
     * The pattern to use when formatting the output. This list contins strings
     * and placeholders 
     */
    private LinkedList pattern;

    /** Current project. Often used for retrieving data */
    private Project proj;

    /** Object for logging */
    private IGLog log;

    /** LogLevel. For debugging purposes */
    private static final int LOGLEVEL=5;

    /*
     * Regular expressions and RE strings for detecting first the placeholders
     * and then attributes within the placeholders. 
     * The REs are constructed in the object constructor rather than being
     * static as they may throw exceptions on construction.
     */
    private RE phElement = null;
    private static final String phElementPattern =
        "<\\s*IG Field ?= ?\"([^\"]*)\"\\s*"
        +"([^/]*)/>";
    private RE phAttrib = null;
    private static final String phAttribPattern =
        "\\s*([^=|\\s=]*)\\s*= ?\"([^\"]*)\"\\s*";

	
	/**
	 * Determines which items need to be extracted from the files.
	 */
	public static IGKeySet getWanted(IGLog log, Project proj) 
			throws REException, IllegalVariableException{
		log.add(IGLog.PROCEDURE, "IGReplace.getWanted");
		String item = "";
		if (proj.getEnumSetting("INDEX_TYPE").equals("INDEX_TYPE_SITE_MAP")){
			item = proj.getStringSetting("SM_LINKS_TXT");
		}else if (proj.getEnumSetting("INDEX_TYPE").equals("INDEX_TYPE_ORDERED_LIST")){
			item = proj.getStringSetting("OL_LINK");
		}
		RE elements = new RE(phElementPattern, RE.REG_ICASE);
		
		REMatch match = elements.getMatch(item);
		IGKeySet wanted = new IGKeySet();
		while (match != null){		
            item = item.substring(match.getEndIndex());
			String phName = match.toString(1);
			if (phName.equalsIgnoreCase(
				    proj.getPlaceHolder("PH_FILE_AUTHOR"))) {
				wanted.setWanted(IGKey.AUTHOR);
			} else if (phName.equalsIgnoreCase(
				    proj.getPlaceHolder("PH_FILE_TITLE"))) {
				wanted.setWanted(IGKey.TITLE);
			} else if (phName.equalsIgnoreCase(
				    proj.getPlaceHolder("PH_FILE_DESCRIPTION"))){
				wanted.setWanted(IGKey.DESCRIPTION);
			}else if(phName.equalsIgnoreCase( 
					proj.getPlaceHolder("PH_FILE_KEYWORDS") )){
				wanted.setWanted(IGKey.KEYWORDS);
			}else if(phName.equalsIgnoreCase( proj.getPlaceHolder("PH_FILE_SIZE") )){
				wanted.setWanted(IGKey.FILE_SIZE);
			} else if (phName.equalsIgnoreCase(
				    proj.getPlaceHolder("PH_FILE_NAME"))) {
			    wanted.setWanted(IGKey.FILE_NAME);
			} else if (phName.equalsIgnoreCase(
				    proj.getPlaceHolder("PH_FILE_TYPE"))) {
			    wanted.setWanted(IGKey.FILE_TYPE);
			}
			match = elements.getMatch(item);
		}
		return wanted;

	}
	
    /**
     * Sets up the replacement object, ready for fast text replacement.
     */
    IGReplace(IGLog logObj, Project projObj, String pat, IGKey type, String setting)
	    throws REException, IllegalVariableException {
        log = logObj;
        log.add(IGLog.PROCEDURE,
		"Contructed replacer object for single pattern");
        proj = projObj;
        pattern = new LinkedList();
        phElement = new RE(phElementPattern, RE.REG_ICASE);
        phAttrib = new RE(phAttribPattern, RE.REG_ICASE);

        /* Part of the pattern which is still to be scanned */
        String toScan = pat;

        /*
         * Go through the remaining pattern and split it up into String segments
         * and placeholders, to be stored in a Linked list as strings and
         * IGKeySets, so that the final code can be easily constructed later.
         */
        REMatch match = phElement.getMatch(pat);

        while (match != null && ! toScan.equals("")) {
            /* Add anything before this placeholder to the pattern */
            if (match.getStartIndex() > -1)
                pattern.addLast(toScan.substring(0,  match.getStartIndex()));

            /* Update the string of data still to be scanned */
            toScan = toScan.substring(match.getEndIndex());

            /* The placeholder name */
            String phName = match.toString(1);

            /* Map for storing attributes for this placeholder */
            HashMap attr = new HashMap();
            REMatch[] attrMatches = phAttrib.getAllMatches( match.toString(2) );

            /* Loop thorugh matches and set attributes in hash map */
            for (int attrNo = 0; attrNo < attrMatches.length; attrNo++) {
                attr.put(attrMatches[attrNo].toString(1),
			attrMatches[attrNo].toString(2));
                if (LOGLEVEL>9)
                    log.add(10, "Attr " + attrMatches[attrNo].toString(1)
			    + "   =   " + attrMatches[attrNo].toString(2));
            }

            /*
             * Look at the placeholder and add an appropriate entry to the
             * pattern list.
             * Issue warnings as necessary.
             */
            if (type.equals(IGKey.REPLACE_TYPE_FILE)) {
                if (phName.equalsIgnoreCase(
			    proj.getPlaceHolder("PH_FILE_AUTHOR"))) {
                    pattern.addLast(new IGPlaceholder(IGKey.AUTHOR));

                    if (! attr.isEmpty()) {
                        log.addWarning(51, "PH_TOO_MANY_ATTR", new Object[] {phName, proj.getLocalName(setting) });
                    }
                } else if (phName.equalsIgnoreCase(
			    proj.getPlaceHolder("PH_FILE_TITLE"))) {
                    pattern.addLast(new IGPlaceholder(IGKey.TITLE));

                    if (! attr.isEmpty()) {
                        log.addWarning(51, "PH_TOO_MANY_ATTR",	new Object[] {phName, proj.getLocalName(setting) });
                    }
                } else if (phName.equalsIgnoreCase(
			    proj.getPlaceHolder("PH_FILE_DESCRIPTION"))){
                    pattern.addLast( new IGPlaceholder( IGKey.DESCRIPTION ) );
                    if (!attr.isEmpty()){
                        log.addWarning(51, "PH_TOO_MANY_ATTR", new Object[] {phName, proj.getLocalName(setting) });
                    }
                }else if(phName.equalsIgnoreCase( proj.getPlaceHolder("PH_FILE_KEYWORDS") )){
					String sep = null;
                    if (!attr.isEmpty()){
                        Object[] keys = attr.keySet().toArray();
						for (int i =0; i< keys.length; i++){
							if(proj.getPlaceHolder("PH_FILE_KEYWORDS_SEPARATOR").equalsIgnoreCase((String) keys[i])){
								if (sep == null){
									sep = (String) attr.get(keys[i]); 
								}else{
									log.addWarning(52, "PH_REPEATED_ATTR", new Object[] {phName, keys[i], proj.getLocalName(setting) });
								}
							}else{
								log.addWarning( 53, "PH_UNKNOWN_ATTR", new Object[] {phName, keys[i], proj.getLocalName(setting) });
							}
						}
                    }
					if (sep == null){
	                    pattern.addLast( new IGPlaceholder( IGKey.KEYWORDS ) );
					}else{
						HashMap attribs = new HashMap();
                        attribs.put(IGKey.LIST_SEPARATOR, sep);
                        pattern.addLast( new IGPlaceholder( IGKey.KEYWORDS,  attribs) );
					}
                }else if(phName.equalsIgnoreCase( proj.getPlaceHolder("PH_FILE_SIZE") )){
                    if (!attr.isEmpty()){
                        Object[] keys = attr.keySet().toArray();
                        IGKey format = null;
                        for (int i =0; i< keys.length; i++){
                            if(proj.getPlaceHolder("PH_FORMAT").equalsIgnoreCase((String) keys[i])){
                                if(proj.getPlaceHolder("PH_FILE_SIZE_HUMAN").equalsIgnoreCase((String) attr.get(keys[i]))){
                                    if(format == null){
                                        format = IGKey.FILE_SIZE_HUMAN;
                                    }else{
                                        log.addWarning(52, "PH_REPEATED_ATTR", new Object[] {phName, keys[i], proj.getLocalName(setting) });
                                    }
                                }else if(proj.getPlaceHolder("PH_FILE_SIZE_BYTES").equalsIgnoreCase((String) attr.get(keys[i]))){
                                    if(format == null){
                                        format = IGKey.FILE_SIZE_BYTES;
                                    }else{
                                        log.addWarning(52, "PH_REPEATED_ATTR", new Object[] {phName, keys[i], proj.getLocalName(setting) });
                                    }
                                }else if(proj.getPlaceHolder("PH_FILE_SIZE_KB").equalsIgnoreCase((String) attr.get(keys[i]))){
                                    if(format == null){
                                        format = IGKey.FILE_SIZE_KB;
                                    }else{
                                        log.addWarning(52, "PH_REPEATED_ATTR", new Object[] {phName, keys[i], proj.getLocalName(setting) });
                                    }
                                }else if(proj.getPlaceHolder("PH_FILE_SIZE_MB").equalsIgnoreCase((String) attr.get(keys[i]))){
                                    if(format == null){
                                        format = IGKey.FILE_SIZE_MB;
                                    }else{
                                        log.addWarning(52, "PH_REPEATED_ATTR", new Object[] {phName, keys[i], proj.getLocalName(setting) });
                                    }
                                }else{
                                    log.addWarning( 54, "PH_UNKNOWN_VALUE", new Object[] {phName, keys[i], attr.get(keys[i]), proj.getLocalName(setting) });
                                }
                            }else{
                                log.addWarning( 53, "PH_UNKNOWN_ATTR", new Object[] {phName, keys[i], proj.getLocalName(setting) });
                            }
                        }
                        HashMap attribs = new HashMap();
                        if (format != null){
                            attribs.put(IGKey.FORMAT, IGKey.FILE_SIZE_HUMAN);
                            pattern.addLast( new IGPlaceholder( IGKey.FILE_SIZE,  attribs) );
                        }else{
                            pattern.addLast( new IGPlaceholder( IGKey.FILE_SIZE ) );
                        }
                    }else{
                        pattern.addLast( new IGPlaceholder( IGKey.FILE_SIZE ) );
                    }
                }else if(phName.equalsIgnoreCase( proj.getPlaceHolder("PH_FILE_TYPE") )){
                    pattern.addLast( new IGPlaceholder( IGKey.FILE_TYPE ) );
                    if (!attr.isEmpty()){
                        log.addWarning(51, "PH_TOO_MANY_ATTR", new Object[] {phName, proj.getLocalName(setting) });
                    }
                }else if(phName.equalsIgnoreCase( proj.getPlaceHolder("PH_FILE_PARSER") )){
                    pattern.addLast( new IGPlaceholder( IGKey.PARSER ) );
                    if (!attr.isEmpty()){
                        log.addWarning(51, "PH_TOO_MANY_ATTR", new Object[] {phName, proj.getLocalName(setting) });
                    }
                }else if(phName.equalsIgnoreCase( proj.getPlaceHolder("PH_FILE_NAME") )){
                    pattern.addLast( new IGPlaceholder( IGKey.FILE_NAME ) );
                    if (!attr.isEmpty()){
                        log.addWarning(51, "PH_TOO_MANY_ATTR", new Object[] {phName, proj.getLocalName(setting) });
                    }
                }else if(phName.equalsIgnoreCase( proj.getPlaceHolder("PH_FILE_REAL_LOCATION") )){
                    pattern.addLast( new IGPlaceholder( IGKey.LOCATION ) );
                    if (!attr.isEmpty()){
                        log.addWarning(51, "PH_TOO_MANY_ATTR", new Object[] {phName, proj.getLocalName(setting) });
                    }
                }else if(phName.equalsIgnoreCase( proj.getPlaceHolder("PH_FILE_REL_LOCATION") )){
                    pattern.addLast( new IGPlaceholder( IGKey.REL_LOCATION ) );
                    if (!attr.isEmpty()){
                        log.addWarning(51, "PH_TOO_MANY_ATTR", new Object[] {phName, proj.getLocalName(setting) });
                    }
                    /*
                    }else if(phName.equalsIgnoreCase( proj.getPlaceHolder("") )){
                    pattern.addLast( IGKeySet. );
                    */	
                }else{
                    // This placeholder is not dependant on the file. Work out
                    // it's replacement and add this to the pattern.
                    pattern.addLast( getReplacement(phName, attr, setting) );
                }
            } else if (type.equals(IGKey.REPLACE_TYPE_DIRECTORY)) {
                if(phName.equalsIgnoreCase( proj.getPlaceHolder("PH_FULL_DIR") )){
                    pattern.addLast( new IGPlaceholder( IGKey.FULL_DIR));
                    if (!attr.isEmpty()){
                        log.addWarning(51, "PH_TOO_MANY_ATTR", new Object[] {phName, proj.getLocalName(setting) });
                    }
                }else if(phName.equalsIgnoreCase( proj.getPlaceHolder("PH_TOP_DIR") )){
                    pattern.addLast( new IGPlaceholder( IGKey.TOP_DIR) );
                    if (!attr.isEmpty()){
                        log.addWarning(51, "PH_TOO_MANY_ATTR", new Object[] {phName, proj.getLocalName(setting) });
                    }
                }else{
                    // This placeholder is not dependant on the file. Work out
                    // it's replacement and add this to the pattern.
                    pattern.addLast( getReplacement(phName, attr, setting) );
                }
            } else if (type.equals(IGKey.REPLACE_TYPE_SECTION)) {
				if (phName.equalsIgnoreCase(
			    			proj.getPlaceHolder("PH_SECTION_LETTER"))) {
                    pattern.addLast(new IGPlaceholder(IGKey.SECTION_LETTER));
                    if (!attr.isEmpty()){
			    		log.addWarning(51, "PH_TOO_MANY_ATTR",
				    		new Object[]{phName, proj.getLocalName(setting) });
                    }
				} else {
			    	pattern.addLast(getReplacement(phName, attr, setting));
				}
           	} else {
                log.addError(55, "PH_UNKNOWN_TYPE", new Object[] {proj.getLocalName(setting)});
            }
            match = phElement.getMatch(toScan);
        }
        // If there's some text left after removing all placeholders, add this
        // to the list.
        if(toScan.length() > 0)
            pattern.addLast( toScan );
    }

    /**
     * Sets up the replacement object, so that it can be used to perform simple
     * replace operations using data only from the project object.
     */
    IGReplace(IGLog logObj, Project projObj) throws REException{
        log = logObj;
        log.add(IGLog.PROCEDURE, "Constructed replacer for multiple patterns");
        proj = projObj;
        pattern = null;
        phElement = new RE(phElementPattern, RE.REG_ICASE);
        phAttrib = new RE(phAttribPattern, RE.REG_ICASE);
    }

    /**
     * Returns the edited string, using data from the given input.
     */
    protected String makeCode(IGFile file){
        log.add(IGLog.PROCEDURE, "makeCode - file "+file.getLocation());
        String result = "";
        ListIterator iter = pattern.listIterator(0);
        while (iter.hasNext()){
            Object current = iter.next();
            if (current instanceof String){
                result += current;
            }else if(current instanceof IGPlaceholder){
                if((((IGPlaceholder) current).key).equals(IGKey.KEYWORDS)){
                    String joiner = null;
                    if (((IGPlaceholder) current).attributes == null){
                        joiner = ", ";
                    }else{
                        joiner = (String) ((IGPlaceholder) current).attributes.get(IGKey.LIST_SEPARATOR);
                    }
                    // TODO: improve:
                    result += IGMisc.arrayToString( IGMisc.stringToArray( (String) file.get(IGKey.KEYWORDS), ", " ) , joiner );
                }else if (((IGPlaceholder) current).attributes == null){
                    result += file.getString( ((IGPlaceholder) current).key );
                }else{
                    // There are atributes - we have to format this in some way.
                    if((((IGPlaceholder) current).key).equals(IGKey.FILE_SIZE)){
						// TODO: add full formatting capability of
						// java.text.DecimalFormat
                        
						// there is only one possible attribute atm.
                        IGKey format = (IGKey) ((IGPlaceholder) current).attributes.get( IGKey.FORMAT );
                        double bytes = (double) Integer.parseInt(file.getString(IGKey.FILE_SIZE));
                        if(format.equals(IGKey.FILE_SIZE_BYTES)){
                            result += bytes;
                        }else if(format.equals(IGKey.FILE_SIZE_HUMAN)){
                            //todo
                            result += humanSize(bytes);
                        }else if(format.equals(IGKey.FILE_SIZE_KB)){
                            //todo
                            result += (bytes / 1024 );
                        }else if(format.equals(IGKey.FILE_SIZE_MB)){
                            //todo
                            result += (bytes / (1024 * 1024));
                        }else{
							//todo
                        }

                    }else{
                        log.addError(56, "PH_ATTR_ERR", null);
                    }
                }
            }else{
                log.addError(57, "REPLACE_INVALID_TYPE", null);
            }
        }
        return result;
    }

    protected String makeCode(IGDirectory directory){
        log.add(IGLog.PROCEDURE, "makeCode - directory "+directory.topDir);
        String result = "";
        ListIterator iter = pattern.listIterator(0);
        while (iter.hasNext()){
            Object current = iter.next();
            if (current instanceof String){
                result += current;
            }else if(current instanceof IGPlaceholder){
                if (((IGPlaceholder) current).key.equals(IGKey.FULL_DIR)){
                    result += directory.fullRelDir;
                } else if (((IGPlaceholder) current).key.equals(IGKey.TOP_DIR))
		{
                    result += directory.topDir;
                } else {
                    log.addError(58, "REPLACE_UNKNOWN_PLACEHOLDER", null);
                }
            } else {
                log.addError(57, "REPLACE_INVALID_TYPE",null);
            }
        }
        return result;
    }


    /**
     * Replace a variable with a single character
     * @param ch the character to replace with
     */
    protected String makeCode(char ch) {
	String result = "";
	for (ListIterator i = pattern.listIterator(0); i.hasNext(); ) {
	    Object currentObject = i.next();

	    if (currentObject instanceof String) {
		result += (String) currentObject;
	    } else if (currentObject instanceof IGPlaceholder) {
		if (((IGPlaceholder) currentObject).key.equals(
			    IGKey.SECTION_LETTER)) {
		    result += ch;
		} else {
			log.addError(58,"REPLACE_UNKNOWN_PLACEHOLDER", null);
		}
	    } else {
		    log.addError(57, "REPLACE_INVALID_TYPE",  null);
	    }
	}

	return result;
    }


    /**
     * Replaces placeholders in a single string. This is used for the items
     * which are only done once, e.g start text
     * @param patternResourceID Identifier for the project resource which holds
	 * the pattern. 
     */
    protected String makeCode(String patternResourceID) throws IllegalVariableException {
		String pattern = proj.getStringSetting( patternResourceID );
        log.add(IGLog.PROCEDURE, "IGReplace.makeCode(simple pattern)");

        String newPattern = pattern;
        REMatch[] matches = phElement.getAllMatches(pattern);

        for (int count = 0; count < matches.length; count++) {
            if (LOGLEVEL > 9)
                log.add(10, "RE Match " + count + ": " + matches[count]);

            String phName = matches[count].toString(1);
            HashMap attr = new HashMap();

            if (LOGLEVEL > 9) {
                log.add(10, "Type= "+matches[count].toString(1));
                log.add(10, "Attr= "+matches[count].toString(2));
            }

            REMatch[] attrMatches = phAttrib.getAllMatches(
		    matches[count].toString(2));

            for (int attrNo = 0; attrNo < attrMatches.length; attrNo++) {
		attr.put(attrMatches[attrNo].toString(1),
			attrMatches[attrNo].toString(2));

                if (LOGLEVEL > 9)
                    log.add(10, "Attr " + attrMatches[attrNo].toString(1)
			    + "   =   " + attrMatches[attrNo].toString(2));
            }

            /*
             * Now test the placeholders and make a few replacements
             */
            String replaceTxt = getReplacement(phName, attr, patternResourceID);
            newPattern = phElement.substitute(newPattern, replaceTxt);
        }

        return newPattern;
    }


    private String getReplacement(String phName, HashMap attr, String setting)
	    throws IllegalVariableException{
        String replaceTxt = null;
        if (phName.equalsIgnoreCase(proj.getPlaceHolder("PH_GENERATE_DATE"))) {
            replaceTxt = DateFormat.getDateInstance().format( new Date());

            if (!attr.isEmpty()) {
                Object[] keys = attr.keySet().toArray();

                for (int i = 0; i < keys.length; i++) {
                    if (((String) keys[i]).equalsIgnoreCase(
				proj.getPlaceHolder("PH_DATE_FORMAT"))) {
			replaceTxt = new SimpleDateFormat((String) attr.get(
				    keys[i])).format(new Date());
                    } else {
                        log.addWarning(53, "PH_UNKNOWN_ATTR",
							new Object[]{phName, keys[i], proj.getLocalName(setting)});
                    }
                }
            }
        } else if (phName.equalsIgnoreCase(proj.getPlaceHolder("PH_IG_URL"))) {
            replaceTxt = "http://tildemh.com/sw/indexgen";

            if (!attr.isEmpty()) {
                log.addWarning(51, "PH_TOO_MANY_ATTR",
			new Object[]{phName, proj.getLocalName(setting)});
            }
        } else if (phName.equalsIgnoreCase(
		    proj.getPlaceHolder("PH_IG_VERSION"))) {
            replaceTxt = IGLog.INDEXGEN_VERSION;

            if (!attr.isEmpty()) {
                log.addWarning(51, "PH_TOO_MANY_ATTR",
			new Object[]{phName, proj.getLocalName(setting)});
            }
        } else if (phName.equalsIgnoreCase(
		    proj.getPlaceHolder("PH_OUTPUT_FILENAME"))) {
            replaceTxt = "<br>Todo: Output Filename<br>";
        } else if (phName.equalsIgnoreCase(
		    proj.getPlaceHolder("PH_OUTPUT_PATH"))) {
            replaceTxt = "<br>Todo: out path<br>";
        } else if (phName.equalsIgnoreCase(
		    proj.getPlaceHolder("PH_FILE_COUNT"))) {
            replaceTxt = "<br>Todo: file count<br>";
        } else if (phName.equalsIgnoreCase(
		    proj.getPlaceHolder("PH_SECTION_LETTER"))) {
				//todo:
		    replaceTxt = "X";
		} else {
            // The placeholder is unknown.
            log.addWarning(50, "PH_UNKNOWN", new Object[]{phName, proj.getLocalName(setting) });

            // We Don't want to give the entire placeholder - that would
            // really mess things up.
            replaceTxt = "<!-- IndexGen - unknown placeholder: " + phName
		+ " -->";
        }

        return replaceTxt;
    }

	/**
	 * Formats the file size in a human readable form
	 */
	public static String humanSize(double num){
		// TODO: L10N
		
		//DecimalFormat df = new DecimalFormat("###########");
		if (num < 10000){
			return (int) num + " B";
		}else if(num < 10000000){
			return (int) (num/1024) +" KB";
		}else{
			return (int)(num / (1024*1024))+ " MB";
		}

	}

}

/*
 * $Log: IGReplace.java,v $
 * Revision 1.32  2002/09/11 13:22:25  howama
 * bug fixes
 *
 * Revision 1.31  2002/09/06 11:44:04  howama
 * merged command line options by Eugene <team_pro@gala.net>.
 * More ui work
 * Changed log position
 *
 * Revision 1.30  2002/09/05 14:01:40  howama
 * new log methods for erros
 *
 * Revision 1.29  2002/09/01 13:44:04  howama
 * removed debug code
 *
 * Revision 1.28  2002/09/01 10:59:29  howama
 * More work and random bug fixes.
 *
 * Revision 1.27  2002/08/31 22:00:20  blsecres
 * Provided fallbacks for missing ordered list sort criteria.
 * Implemented insignificant article removal for sorting purposes.
 * Modified parsers to only provide information asked for in IGKeySet.
 *
 * Revision 1.26  2002/08/31 15:27:56  howama
 * various changes
 *
 * Revision 1.25  2002/08/30 02:27:06  blsecres
 * Initial ordered list support.
 * User configurable sorting.
 *
 * Revision 1.24  2002/08/29 14:58:54  howama
 * null pointer bug fix
 *
 * Revision 1.23  2002/08/28 06:33:49  howama
 * various
 *
 * Revision 1.22  2002/08/27 00:06:33  blsecres
 * Fixed typos that prevented compilation.
 *
 * Revision 1.21  2002/08/26 13:56:26  howama
 * More work & bug fixes
 *
 * Revision 1.20  2002/08/26 12:43:55  howama
 * work on file_size placeholder
 *
 * Revision 1.19  2002/08/23 13:58:01  howama
 * propogated illegalvariable exceptions rather than catching them
 *
 * Revision 1.18  2002/08/23 12:22:07  howama
 * fixed
 *
 * Revision 1.17  2002/08/23 10:41:58  howama
 * more work
 *
 * Revision 1.16  2002/08/22 19:47:13  blsecres
 * Switched code to use IGKey values.
 *
 * Revision 1.15  2002/08/22 18:21:26  howama
 * updated comments
 *
 * Revision 1.14  2002/08/22 13:42:39  howama
 * more work
 *
 * Revision 1.13  2002/08/21 15:13:12  howama
 * more work
 *
 * Revision 1.12  2002/08/20 18:56:58  howama
 * bug fixes
 *
 * Revision 1.11  2002/08/20 13:22:31  howama
 * more work
 *
 * Revision 1.10  2002/08/20 09:18:20  howama
 * various
 *
 * Revision 1.9  2002/08/19 09:47:27  howama
 * updated placeholders and replcement code
 *
 * Revision 1.8  2002/08/18 14:15:47  howama
 * upodates
 *
 * Revision 1.7  2002/08/16 11:14:04  howama
 * IGFile changes
 *
 * Revision 1.6  2002/08/09 13:05:03  howama
 * added static proc for single items
 *
 * Revision 1.5  2002/08/09 12:06:34  howama
 * put methods in place ready for implementation
 *
 * Revision 1.4  2002/08/08 20:10:57  howama
 * updates
 *
 * Revision 1.3  2002/08/08 17:24:43  howama
 * updates
 *
 * Revision 1.2  2002/07/24 09:24:21  howama
 * updated javadoc docs
 *
 * Revision 1.1  2002/07/05 09:44:11  howama
 * Initial conversion to java.
 *
 */

