/*
The DsTool program is the property of:
 
                             Cornell University 
                        Center of Applied Mathematics 
                              Ithaca, NY 14853
                      dstool_bugs@macomb.tn.cornell.edu
 
and may be used, modified and distributed freely, subject to the following
restrictions:
 
       Any product which incorporates source code from the DsTool
       program or utilities, in whole or in part, is distributed
       with a copy of that source code, including this notice. You
       must give the recipients all the rights that you have with
       respect to the use of this software. Modifications of the
       software must carry prominent notices stating who changed
       the files and the date of any change.
 
DsTool is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of FITNESS FOR A PARTICULAR PURPOSE.
The software is provided as is without any obligation on the part of Cornell 
faculty, staff or students to assist in its use, correction, modification or
enhancement.
*/

/* 
 * rk78.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "constants.h"
#include "prop.h"
#include "complib.h"
#include "utilities.h"
#include "math_utils.h"
#include "algorithms_local.h"

#define TINY  1.e-30

static int rk78(double *out_y, double *in_y, int dim, double *param, double htry,
     double min_step, double max_step, double err_tol, double safety,
     double *hdid, double *hnext, int (*derivs)(), double *workspace);

static double alpha[] = {2.0/27.0, 1.0/9.0, 1.0/6.0, 5.0/12.0, 0.5, 5.0/6.0,
			     1.0/6.0, 2.0/3.0, 1.0/3.0, 1.0, 0.0, 1.0};
static double beta[12][13] = {
  {2.0/27.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0},
  {1.0/36.0, 1.0/12.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0},
  {1.0/24.0, 0.0, 1.0/8.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0},
  {5.0/12.0, 0.0, -25.0/16.0, 25.0/16.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
     0.0, 0.0},
  {0.05, 0.0, 0.0, 0.25, 0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0},
  {-25.0/108.0, 0.0, 0.0, 125.0/108.0, -65.0/27.0, 125.0/54.0, 0.0, 0.0, 0.0,
     0.0, 0.0, 0.0, 0.0},
  {31.0/300.0, 0.0, 0.0, 0.0, 61.0/225.0, -2.0/9.0, 13.0/900.0, 0.0, 0.0,
     0.0, 0.0, 0.0, 0.0},
  {2.0, 0.0, 0.0, -53.0/6.0, 704.0/45.0, -107.0/9.0, 67.0/90.0, 3.0, 0.0,
     0.0, 0.0, 0.0, 0.0},
  {-91.0/108.0, 0.0, 0.0, 23.0/108.0, -976.0/135.0, 311.0/54.0, -19.0/60.0,
     17.0/6.0, -1.0/12.0, 0.0, 0.0, 0.0, 0.0},
  {2383.0/4100.0, 0.0, 0.0, -341.0/164.0, 4496.0/1025.0, -301.0/82.0,
     2133.0/4100.0, 45.0/82.0, 45.0/164.0, 18.0/41.0, 0.0, 0.0, 0.0},
  {3.0/205.0, 0.0, 0.0, 0.0, 0.0, -6.0/41.0, -3.0/205.0, -3.0/41.0,
     3.0/41.0, 6.0/41.0, 0.0, 0.0, 0.0},
  {-1777.0/4100.0, 0.0, 0.0, -341.0/164.0, 4496.0/1025.0, -289.0/82.0,
     2193.0/4100.0, 51.0/82.0, 33.0/164.0, 12.0/41.0, 0.0, 1.0, 0.0}
 };

static double c[] = {0.0, 0.0, 0.0, 0.0, 0.0, 34.0/105.0, 9.0/35.0, 9.0/35.0,
		       9.0/280.0, 9.0/280.0, 0.0, 41.0/840.0, 41.0/840.0};
static double d[] = {1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0,
		       -1.0, -1.0};


int
rk78_driver(double *out_state, double *in_state, double *dwork,
	    struct Prop_DataS *integ_cntl, int mode)
{
  int		i, var_dim, status = NO_ERROR;
  double	min_step, max_step, safety;
  double	t_step, htry, hnext, err_tol;

  var_dim = integ_cntl->ph_space_dim-1;

  if (( mode == FIXED_STEP ) || (mode == FIXED_STEP_INIT))
    {
     t_step = (abs(integ_cntl->direction ) == FORWARD) ? integ_cntl->time_step:-integ_cntl->time_step;
     status = rk4(out_state, in_state, integ_cntl->parameters, t_step, var_dim, integ_cntl->function, dwork);
    }
  else
    {
     min_step = integ_cntl->panel_dp_values[2]; 
     max_step = integ_cntl->panel_dp_values[3];
     err_tol = integ_cntl->panel_dp_values[4];
     safety = integ_cntl->panel_dp_values[5];
     htry = (abs(integ_cntl->direction ) == FORWARD) ? fabs(integ_cntl->estim_step):-fabs(integ_cntl->estim_step);
     
     status = rk78(out_state, in_state, var_dim, integ_cntl->parameters, 
                   htry, min_step, max_step, err_tol, safety,
	           &t_step, &hnext, integ_cntl->function, dwork);

     if ( status == NO_ERROR ) integ_cntl->estim_step = hnext;

    }

  return( status );

}




/* ---------------------------------------------------------------------------------------------
   rk78 is the one-step routine for the seventh-order Runge-Kutta algorithm with eighth-order
   stepsize regulation.  This implementation is adapted from


   inputs: 	in_y		inital state (indep. varb value in last position!)
		dim		dimension of phase space + 1
		param		parameter values for dynamical system
		htry		initial stepsize estimate
		min_step	minimum allowed stepsize
		max_step	maximum allowed stepsize
		err_tol		error tolerance
		safety		factor to produce slight reduction of full step
		derivs		func ptr to derivative routine
		workspace	pre-allocated workspace for routine to use

   output:	out_y		final state 
		hdid		actual stepsize taken
		hnext		estimated next stepsize

   author: paw		last change: 7-24-96


   --------------------------------------------------------------------------------------------- */
   

#define N_STAGES 13
#define ORDER_RECIP 0.125

static int 
rk78(double *out_y, double *in_y, int dim, double *param, double htry,
     double min_step, double max_step, double err_tol,
     double safety, double *hdid, double *hnext,
     int (*derivs)(double *, double *, double *), double *workspace)
{
  int i, k, l;
  double *x, *error, *f[N_STAGES], delta, tau, h, h_new;
  int ERROR_OK = FALSE;
  double hsign = copysign(1.0,htry);

  /* make sure stepsize is in range */
  h = htry;
  if (h > max_step) h = max_step;
  if (h < min_step) h = min_step;

  /* find size of input for error use */
  tau = sup_norm(in_y, dim);
  if (tau < 1.0) tau = 1.0;
  tau *= err_tol;
  
  /* make space for phase space coordinate including time */
  x = workspace;
  workspace += dim+1;

  /* make space for truncation error calculation */
  error = workspace;
  workspace += dim;

  /* set up space for N_STAGES vectors */
  for (i=0; i<N_STAGES; i++)
    {
      f[i] = workspace;
      workspace += dim;
    }

  /* evaluate vector field at base point */
  derivs(f[0], in_y, param);

  do {

    /* 13 stage integrator - so do 12 more evaluations */
    for (k=0; k<(N_STAGES-1); k++)
      {
	/* reset x to initial value (including time) */
	for (i=0; i<=dim; i++) x[i] = in_y[i];
	
	/* adjust x to new position */
	for (l=0; l<k-1; l++)
	  {
	    for (i=0; i<dim; i++)
	      x[i] += h*beta[k][l]*f[l][i];
	  }
	/* add time */
	x[dim] += h*alpha[k];
	
	/* evaluate vector field */
	derivs(f[k+1], x, param);
      }

    /* compute truncation error */
    for (i=0; i<dim; i++) error[i] = 0.0;
    for (k=0; k<N_STAGES; k++)
      if (d[k] != 0.0)
	for (i=0; i<dim; i++)
	  error[i] += f[k][i]*d[k];
    delta = sup_norm(error, dim) * h * 41.0 / 840.0;

    /* check if error is acceptable */
    if (delta <= tau)
      {
	/* acceptable */
	ERROR_OK = TRUE;

	/* write out_y */
	for (i=0; i<dim; i++) out_y[i] = 0.0;
	for (k=0; k<N_STAGES; k++)
	  if (c[k] != 0)
	    for (i=0; i<dim; i++)
	      out_y[i] += f[k][i]*c[k];
	for (i=0; i<dim; i++)
	  out_y[i] = h*out_y[i]+in_y[i];
	out_y[dim] = in_y[dim]+h;

	/* return what stepsize was used */
	*hdid = h;

	/* suggest a new stepsize */
	if (delta != 0.0)
	  { 
	    h_new = safety*h*pow(tau/delta, ORDER_RECIP);
	    if (fabs(h_new) > max_step) h_new = hsign*max_step;
	    if (fabs(h_new) < min_step) h_new = hsign*min_step;
	  }
	else
	  h_new = h;
	*hnext = h_new;
      }
    else {
      /* error too big, find a smaller stepsize */
      h_new = safety*h*pow(tau/delta, ORDER_RECIP);
      if (fabs(h_new) < min_step)
	{
	  if (fabs(h) == min_step)
	    {
	      /* we already tried the min step and it was not good enough! */
	      system_mess_proc(1,"Step size too small in routine rk78!");
	      return(MINOR_ERROR);
	    }
	  else
	    /* set step to min step allowed */
	    h_new = hsign*min_step;
	}
#ifdef DEBUG
      fprintf(stderr, "rk78 - reducing stepsize %f -> %f\n", h, h_new);
#endif
      /* set stepsize to new value and try again */
      h = h_new;
    }

  } while (ERROR_OK == FALSE);

  return( NO_ERROR);
}




int
rk78_init()
{
  static int		ifields[]={15};
  static double	dfields[]={1.e-6,1.0e-5,1.0e-10,0.2,1.e-5,0.8};
  static int		sel_values[] = {0};
  static char		*ifield_names[]={"Newton iter"};
  static char		*dfield_names[]={FINITE_DIFFERENCE_STEP_STRING, STOPPING_ERROR_STRING, "Min step: ",
					 "Max step: ","Err tol: ","Safety: "};
  static char		*usr_sel_labels[] = {" "};
  static int		num_choices[] = {0};

  Int_Algol.Num_Ifields = 1;	/* these three MUST be set */
  Int_Algol.Num_Dfields = 6;
  Int_Algol.Num_Sel_Items = 0;

  Int_Algol.Ifield_Names = ifield_names;
  Int_Algol.Dfield_Names = dfield_names;
  Int_Algol.Sel_Labels = usr_sel_labels;
  Int_Algol.Ifields = ifields;
  Int_Algol.Dfields = dfields;
  Int_Algol.Num_Sel_Choices = num_choices;
  Int_Algol.Sel_Values = sel_values;

}
