/*
The DsTool program is the property of:
 
                             Cornell University 
                        Center of Applied Mathematics 
                              Ithaca, NY 14853
                      dstool_bugs@macomb.tn.cornell.edu
 
and may be used, modified and distributed freely, subject to the following
restrictions:
 
       Any product which incorporates source code from the DsTool
       program or utilities, in whole or in part, is distributed
       with a copy of that source code, including this notice. You
       must give the recipients all the rights that you have with
       respect to the use of this software. Modifications of the
       software must carry prominent notices stating who changed
       the files and the date of any change.
 
DsTool is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of FITNESS FOR A PARTICULAR PURPOSE.
The software is provided as is without any obligation on the part of Cornell 
faculty, staff or students to assist in its use, correction, modification or
enhancement.
*/

/*
 * print_procs.c 
 * code for printing a postscript file
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include <sys/types.h>
#include <math.h>
#include <signal.h>
#include <setjmp.h>

#include <constants.h>
#include <defaults.h>
#include <memory.h>
#include <pm.h>
#include <plot.h>
#include <symbols.h>

#include <unistd.h>
#include <sys/uio.h>
#include "utilities.h"
#include "modellib.h"
#include "print.h"
#include "view.h"
#include "memory.h"
#include "dscolor.h"

jmp_buf   env;		/* saves stack info in case of broken pipe */


/*
 * Forward declaration of print functions used here.
 */
int print_go( int force );

int print_to_file( int fd );

int write_ps_data_all_mem( int fd, char buf[] );
int write_ps_data_mem( int fd, char buf[], memory m );

int write_header( int fd, char buf[] );
int write_prolog( int fd, char buf[] );
int write_sys_info( int fd, char buf[], int num_pts );
int write_bbox( int fd, char buf[] );
int write_ps_prelim( int fd, char buf[] );
int write_trailer( int fd, char buf[] );

char* get_ps_symbol( int symbol_index );
char* get_plot_font( void );
void  get_plot_color(void);

int   pt_within_region( double x, 
                        double y,
                        double hor_min,
                        double hor_max,
                        double ver_min,
                        double ver_max);
void  pt_to_xy( int win_num, 
                double* state, 
                double* param, 
                double* x, 
                double* y, 
                double* f);

int  index_to_ps_rgb( void );

int   write_macros( int fd, char buf[] );
void  broken_pipe_error( void );

/*************************************************************************************** 
 * Function: print_go() 
 * Purpose:  prints postscript to printer or file. In the case of a broken pipe error
 * 	     (for example, due to an incorrect printer name) the program is interrupted,
 *	     an error message is displayed, and control returns to the window manager.
 * Arguments: TRUE means overwrite file even if it exists; 
 *            FALSE means ask user whether to overwrite or cancel print.
 * Returns:   0 = successful printing; 
 *           -1 = cannot open file/pipe; 
 *           -2 = broken pipe
 *            1 = file exists; ask for overwrite permission.
 ***************************************************************************************/
int print_go( int force )
{
  FILE          *fp;
  void		broken_pipe_error(),(*old_pipe_errfunc)();
  int		status=0;
  char          *fname =  (char *) calloc(SIZE_OF_FNAME+6,sizeof(char)); 
  char          *dirname =  (char *) calloc(SIZE_OF_DIR_PLUS_FNAME+6,sizeof(char));
                            /* 6 = strlen("lpr -P") */
 /*
  * open PS file
  */ 
  if( *((int *)pm(GET, "Print.File_Flag", NULL)) ) {

      dirname = strcat(strcat((char *)pm(GET, "Print.Directory", dirname, NULL), "/"), 
		              (char *)pm(GET, "Print.Filename", fname, NULL) );

     /* 
      * check if file exists and ask user for permission 
      */
      if (check_file_to_read(dirname) && !force ) {	
	status = 1;
      }
      else if (check_file_to_write(dirname)) {
	  fp = fopen(dirname,"w");
	  status = print_to_file(fileno(fp));
	  fclose(fp);
      }
      else {
        status = -1;
      }
  }

 /*
  * pipe data to printer
  */
  else {
   /* 
    * set up interrupt handler for broken pipe; 
    * save previous handler if not set to ignore 
    */
    if( (old_pipe_errfunc=signal(SIGPIPE,SIG_IGN)) != SIG_IGN) {
      signal(SIGPIPE, broken_pipe_error);
    }

   /* 
    * save stack (return here on error) 
    */
    if(setjmp(env) == 0) {		
      strcpy(fname, "lpr -P");
      pm(GET, "Print.Printer_Name", &(fname[6]), NULL);

     /*
      * open pipe to printer
      */
      if( !(fp=popen( fname, "w")) ) {
        status = -1;
      }
      else {
       /* 
        * fileno returns file descriptor 
        */
        status = print_to_file(fileno(fp));
	pclose(fp);
      }
    }

   /*
    * there was a broken pipe! 
    */
    else {					
      status = -2;
    }

   /* 
    * restore old value for error handler 
    */
    signal(SIGPIPE,old_pipe_errfunc);		
  }

  free(fname);
  free(dirname);
  return(status);
}

/*
 * print_to_file() calls routines to produce postscript file.  The main driver.
 * Returns 0 upon successful printing
 */
int print_to_file( int fd )
{
  char		buf[BUFFERSIZE];
  int		num_pts;

  write_header(fd, buf);

  if( write_macros(fd, buf) != 0 ) /* error: Prolog not properly read */
    return -1;

  write_prolog(fd, buf);
  write_ps_prelim(fd, buf);
  num_pts = write_ps_data_all_mem(fd, buf);

  if( *((int *)pm(GET, "Print.Show_BBox", NULL)) )
    write_bbox(fd, buf);

  if( *((int *)pm(GET, "Print.Show_Info", NULL)) )
    write_sys_info(fd, buf, num_pts);

  write_trailer(fd, buf);

  return 0;
}














/*
 * write_ps_data_all_mem() 
 */
int write_ps_data_all_mem(int fd, char buf[])
{
  char*  name;
  memory m;
  int    i=0;
  int    num_pts=0;


  sprintf(buf,
  "%% translate to user coords\ngsave\nxscale  yscale  scale x0 neg y0 neg translate\n");
  write(fd,buf,strlen(buf));

  while ( (name = (char *) pm(QUERY, "Memory", MEMRY, i, NULL)) ) {
    i++;
    m = (memory) pm(GET, name, NULL);
    num_pts += write_ps_data_mem(fd,buf,m);
  }

  sprintf(buf,"grestore\n");
  write(fd,buf,strlen(buf));

  return(num_pts);
}

/*
 * write_ps_data_mem()
 *
 * modified to interpolate points for each trajectory to the boundaries.
 * needs to be improved for case (x>hor_max) && (y>ver_max)
 * needs to be improved for periodic flows.
 */
int write_ps_data_mem(int fd, char buf[], memory m)
{
  double hor_min = *((double *)pm(GET, "Print.HorMin", NULL));
  double hor_max = *((double *)pm(GET, "Print.HorMax", NULL));
  double ver_min = *((double *)pm(GET, "Print.VerMin", NULL));
  double ver_max = *((double *)pm(GET, "Print.VerMax", NULL));   

  int    hor_type  = *((int *)pm(GET, "Print.HorType", NULL));   
  int    hor_index = *((int *)pm(GET, "Print.HorIndex", NULL));   
  int    ver_type  = *((int *)pm(GET, "Print.VerType", NULL));   
  int    ver_index = *((int *)pm(GET, "Print.VerIndex", NULL));   

  int    connect_dots = *((int*) pm(GET,"Print.Connect", NULL));
  int    use_color  = *((int*) pm(GET, "Print.Color_Flag", NULL));
  int    pick_color = *((int*) pm(GET, "Print.Pick_Color", NULL));

  int	 num_pts = 0;

  double  x;
  double  y;
  int*    color_table;
  double* points;
  double* params;

  double  xnew;
  double  ynew;

  double  xold;
  double  yold;

  double  xbnd = 0;
  double  ybnd = 0;

  char*   symbol;
  char*   line_str;
  
  double   dr, dg, db;
  DsColor* dscolor;
  int      color_last = -99;
  int      color_index;
  int      new_traj = 0;

  int      new_pt_bnd;
  int      old_pt_bnd = 1; /* first point is always in bounds*/

  int      fn = FALSE;

  if ( (hor_type == FUNCT) || (ver_type == FUNCT) ) fn = TRUE;

 
  if( memory_reset_read(m) == 0 ) {

    while( memory_read_next_flow(m, NULL, NULL, NULL, NULL, NULL) == 0 ) {

      line_str = "\0";

      while( memory_read_next_traj(m, NULL, NULL, NULL) == 0 ) {

        line_str = "\0";
        new_traj = 0;

        while( memory_read_next_point(m, &points, &params, &color_table,
                                      NULL, NULL) == 0 ) {

          if (fn == TRUE) view_auxf(view_f, points, params);

          switch( hor_type ) {
            case VARB:
              xnew = points[hor_index];
              break;
            case PARAM:
              xnew = params[hor_index];
              break;
            case FUNCT:
              xnew = view_f[hor_index];
              break;
          }

          switch ( ver_type ) {
            case VARB:
              ynew = points[ver_index];
              break;
            case PARAM:
              ynew = params[ver_index];
              break;
            case FUNCT:
              ynew = view_f[ver_index];
              break;
          }

          new_pt_bnd = pt_within_region( xnew, ynew, 
                                         hor_min, hor_max, 
                                         ver_min, ver_max);
/*
printf("\nnew_pt_bnd = %d", new_pt_bnd);
printf("\nold_pt_bnd = %d", old_pt_bnd);
printf("\n");
printf("\nx,y new = %f,%f", xnew, ynew);
printf("\nx,y old = %f,%f", xold, yold);
printf("\n");
*/

            if( use_color ) {
              color_index = DsColor_get_index( pick_color,
                                               color_table[0],
                                               color_table[1] );
              if( color_last != color_index ) {
                dscolor = DsColor_get_rgb( color_index );
                dr = (float)(dscolor->r/255.0);
                dg = (float)(dscolor->g/255.0);
                db = (float)(dscolor->b/255.0);
                sprintf(buf,"%%idx: %d color_table: %d %d %d\n",
                        color_index, color_table[0],
                        color_table[1], color_table[2]);
                write(fd,buf,strlen(buf));
                sprintf(buf,"%.6f\t%.6f\t%.6f\tsetrgbcolor ", dr,dg,db);
                write(fd,buf,strlen(buf));
                color_last = color_index;
              }
            }

            symbol = get_ps_symbol( color_table[2] );

            if ( connect_dots ) {
              line_str = "\0";

              if ( new_pt_bnd == 1 && old_pt_bnd == 1 && num_pts == 0) {
                line_str = "\0";
                x = (xnew - hor_min ) / (hor_max - hor_min);
                y = (ynew - ver_min ) / (ver_max - ver_min);
                sprintf(buf,"%.6f\t%.6f\t%s%s\n",x,y,line_str,symbol);
                write(fd,buf,strlen(buf));

                num_pts++;
              }

              if ( new_pt_bnd == 1 && old_pt_bnd == 1 && num_pts > 0) {
                if (new_traj >  0) {
                  line_str = "Line\t";
                }
                if (new_traj == 0) {
                  line_str = "\0";
                  new_traj++;
                }
                x = (xnew - hor_min ) / (hor_max - hor_min);
                y = (ynew - ver_min ) / (ver_max - ver_min);
                sprintf(buf,"%.6f\t%.6f\t%s%s\n",x,y,line_str,symbol);
                write(fd,buf,strlen(buf));

                num_pts++;
              }

              if ( new_pt_bnd == 1 && old_pt_bnd == 0 && num_pts > 0) { 
                line_str = "\0";
                if( xold > hor_max ) {
                  xbnd = hor_max;
                  ybnd = yold+(ynew-yold)*(xbnd-xold)/(xnew-xold);
                }
                if( xold < hor_min ) {
                  xbnd = hor_min;
                  ybnd = yold+(ynew-yold)*(xbnd-xold)/(xnew-xold);
                }
                if( yold > ver_max ) {
                  ybnd = ver_max;
                  xbnd = xold+(xnew-xold)*(ybnd-yold)/(ynew-yold);
                }
                if( yold < ver_min ) {
                  ybnd = ver_min;
                  xbnd = xold+(xnew-xold)*(ybnd-yold)/(ynew-yold);
                }
                x = (xbnd - hor_min ) / (hor_max - hor_min);
                y = (ybnd - ver_min ) / (ver_max - ver_min);
                sprintf(buf,"%.6f\t%.6f\t%s%s\n",x,y,line_str,symbol);
                write(fd,buf,strlen(buf));

                line_str = "Line\t";
                x = (xnew - hor_min ) / (hor_max - hor_min);
                y = (ynew - ver_min ) / (ver_max - ver_min);
                sprintf(buf,"%.6f\t%.6f\t%s%s\n",x,y,line_str,symbol);
                write(fd,buf,strlen(buf));

                num_pts++;
              }
              if (new_pt_bnd == 0 && old_pt_bnd == 1 && num_pts > 0) {
                line_str = "Line\t";
                if( xnew > hor_max ) {
                  xbnd = hor_max;
                  ybnd = yold+(ynew-yold)*(xbnd-xold)/(xnew-xold);
                }
                if( xnew < hor_min ) {
                  xbnd = hor_min;
                  ybnd = yold+(ynew-yold)*(xbnd-xold)/(xnew-xold);
                }
                if( ynew > ver_max ) {
                  ybnd = ver_max;
                  xbnd = xold+(xnew-xold)*(ybnd-yold)/(ynew-yold);
                }
                if( ynew < ver_min ) {
                  ybnd = ver_min;
                  xbnd = xold+(xnew-xold)*(ybnd-yold)/(ynew-yold);
                }
                xnew = xbnd;
                ynew = ybnd;
                x = (xbnd - hor_min ) / (hor_max - hor_min);
                y = (ybnd - ver_min ) / (ver_max - ver_min);
                sprintf(buf,"%.6f\t%.6f\t%s%s\n",x,y,line_str,symbol);
                write(fd,buf,strlen(buf));

                num_pts++;
              }

            }
            else {
              if ( new_pt_bnd == 1 ) {
                line_str = "\0";
                x = (xnew - hor_min ) / (hor_max - hor_min);
                y = (ynew - ver_min ) / (ver_max - ver_min);
                sprintf(buf,"%.6f\t%.6f\t%s%s\n",x,y,line_str,symbol);
                write(fd,buf,strlen(buf));

                num_pts++;
              }
            }

          old_pt_bnd = new_pt_bnd;
          xold = xnew;
          yold = ynew;

        }
      }
    }
  } 

  return(num_pts);
}
















/*
 * write_header() writes PostScript header comments in compliance with PostScript
 * 	structuring conventions
 */
int
write_header(fd,buf)
	int		fd;
	char		buf[];
{
  sprintf(buf,"%%!PS-Adobe-2.0\n%%%%Title: %s\n%%%%Creator: dstool\n%%%%CreationDate: %s\n%%%%For: %s\n%%%%Pages: 1\n%%%%DocumentFonts: %s\n%%%%BoundingBox: 0 0 612 792\n%%%%EndComments\n%% begin the prolog...\n", 
	  get_ds_name(),get_the_time(),get_user_info(),get_plot_font());
  write(fd,buf,strlen(buf));
}

/*
 * write_prolog() defines constants, macros, and procedures to be used by PostScript
 */
int
write_prolog(fd,buf)
	int		fd;
	char		buf[];
{

  sprintf(buf,"\n%s %d %s %d %s\n%s %s %s\n%s %s %s\n",	    /* fonts */
	  "/SetFonts { /LabelPtSize", *((int *)pm(GET, "Print.Label_Pt", NULL)),
	  "def /TitlePtSize", *((int *)pm(GET, "Print.Title_Pt", NULL)), "def",
	  "   /LabelFont   ",get_plot_font(),"findfont LabelPtSize scalefont def",
	  "   /TitleFont   ",get_plot_font(),"findfont TitlePtSize scalefont def } def");
  write(fd,buf,strlen(buf));
  sprintf(buf,"%s %d %s\t\t\t\t%s\n%s %d %s\t\t\t\t%s\n",   /* BBox sides */
	  "/xBBoxSide",*((int *)pm(GET, "Print.BBox_Hor_Length", NULL)),"def",
	  "% length (in pts) of bounding box",
	  "/yBBoxSide",*((int *)pm(GET, "Print.BBox_Ver_Length", NULL)),"def",
	  "% height (in pts) of bounding box");
  write(fd,buf,strlen(buf));
  sprintf(buf,"%s %d %s\t\t\t\t\t%s\n%s %d %s\t\t\t\t%s\n",	/* BBox origin*/
	  "/xorigin",*((int *)pm(GET, "Print.BBox_Hor_Offset", NULL)),"def",
	  "% displacement (in pts) from left side of page",
	  "/yorigin",*((int *)pm(GET, "Print.BBox_Ver_Offset", NULL)),"def",
	  "% displacement (in pts) from bottom of page");
  write(fd,buf,strlen(buf));

  /* The following are definitions which are based on the above definitions.
     They do not vary from system to system. */
  sprintf(buf, "%s\n%s\t%s\n%s\t\t%s\n%s\t\t%s\n%s\t\t\t%s\n%s\t\t\t%s\n%s\n\n",
	  "% The below are not model specific but depend on the above definitions",
	  "/Yp yBBoxSide 60 add 6 HeadPtSize mul add def",
	  "% print 6 header lines at this height",
	  "/xscale xBBoxSide xf x0 sub div def",
	  "% horizontal scaling factor",
	  "/yscale yBBoxSide yf y0 sub div def",
	  "% vertical scaling factor",
	  "/xscalei 1 xscale div def",
	  "% horizontal scaling factor inverse",
	  "/yscalei 1 yscale div def",
	  "% vertical scaling factor inverse",
	  "%%%%EndProlog");
  write(fd,buf,strlen(buf));
}

/*
 * write_sys_info() writes information (to the PS file) about the dynamical system being studied
 */
int
write_sys_info(fd,buf,num_pts)
	int		fd;
	char		buf[];
	int		num_pts;
{
  char		label[MAX_LEN_VARB_NAME];
  int		i,dim;
  int 		format = *((int *) pm(GET, "Defaults.Precision", NULL));

  sprintf(buf,"%s\n(Title: %s) Display\n(Date: %s) Display\n",
	  "gsave /Courier findfont HeadPtSize scalefont setfont",
	  get_ds_name(),get_the_time());
  write(fd,buf,strlen(buf));
  sprintf(buf,"(%s Range = [ %.*lg, %.*lg ];   ",
	  (char *) pm(GET, "Print.Hor_Label", label, NULL),
	  format,  *((double *) pm(GET, "Print.HorMin", NULL)),
	  format,  *((double *) pm(GET, "Print.HorMax", NULL)) );
  write(fd,buf,strlen(buf));
  sprintf(buf,"%s Range = [ %.*lg, %.*lg ]) Display\n",
	  (char *) pm(GET, "Print.Ver_Label", label, NULL),
	  format,  *((double *) pm(GET, "Print.VerMin", NULL)),
	  format,  *((double *) pm(GET, "Print.VerMax", NULL)) );
  write(fd,buf,strlen(buf));

  pm(GET, "Model.Varb_Names", 0, label, NULL);
  sprintf(buf,"(Initial Conditions: ( %s",label);
  write(fd,buf,strlen(buf));
  dim = *((int *)pm( GET, "Model.Varb_Dim", NULL ));
  for(i=1;i<dim;i++){
    pm(GET, "Model.Varb_Names", i, label, NULL);
    sprintf(buf,", %s",label);
    write(fd,buf,strlen(buf));
  }
  sprintf(buf," )=( %.*lg",format, *((double *) pm( GET, "Selected.Varb_Ic", 0, NULL))); 
  write(fd,buf,strlen(buf));
  for(i=1;i<dim;i++){
    sprintf(buf,", %.*lg",format, *((double *) pm( GET, "Selected.Varb_Ic", i, NULL))); 
    write(fd,buf,strlen(buf));
  }
  sprintf(buf," )) Display\n");
  write(fd,buf,strlen(buf));

  dim = *((int *)pm( GET, "Model.Param_Dim", NULL ));
  if(dim>0)
    {
      pm(GET, "Model.Param_Names", 0, label, NULL);
      sprintf(buf,"(Parameters: ( %s",label);
      write(fd,buf,strlen(buf));
      for(i=1;i<dim;i++){
	pm(GET, "Model.Param_Names", i, label, NULL);
	sprintf(buf,", %s",label);
	write(fd,buf,strlen(buf));
      }
      sprintf(buf," )=( %.*lg",format, *((double *) pm( GET, "Selected.Param_Ic", 0, NULL)));
      write(fd,buf,strlen(buf));
      for(i=1;i<dim;i++){
	sprintf(buf,", %.*lg",format, *((double *) pm( GET, "Selected.Param_Ic", i, NULL))); 
	write(fd,buf,strlen(buf));
      }
    }
  else
    {
      sprintf(buf,"(Parameters: ( none");
      write(fd,buf,strlen(buf));
    }
  sprintf(buf," )) Display\n");
  write(fd,buf,strlen(buf));

  if( *((int *)pm(GET, "Model.Mapping_Flag", NULL)) )
    sprintf(buf,"(Num Pts = %d) Display   grestore\n",num_pts);
  else
    sprintf(buf,"(Num Pts = %d;  Time Step = %.*lg) Display   grestore\n",
	    num_pts, format, *((double *) pm( GET, "Flow.Stepsize", NULL))); 
  write(fd,buf,strlen(buf));
}

/*
 * write_bbox() encodes the information for a bounding box and tick marks
 */
int
write_bbox(fd,buf)
	int		fd;
	char		buf[];
{
  sprintf(buf,"gsave 1 setlinewidth  BoundingBox grestore\n%d %d Ticks\n",
	  *((int *)pm(GET, "Print.Num_Hor_Ticks", NULL)), 
	  *((int *)pm(GET, "Print.Num_Ver_Ticks", NULL)) );
  write(fd,buf,strlen(buf));
}

/*
 * write_ps_prelim() sets up fonts and draws labels
 */
int
write_ps_prelim(fd,buf)
	int		fd;
	char		buf[];
{
  char		s[MAX_LEN_DS_TITLE]; 
  /* should really be max(MAX_LEN_DS_TITLE, MAX_LEN_VARB_NAME) */

  sprintf(buf,"gsave xorigin yorigin translate linewidth setlinewidth SetFonts\n");
  write(fd,buf,strlen(buf));

  if( *((int *)pm(GET, "Print.Landscape", NULL)) )	    /* landscape mode? */
    {
      sprintf(buf, "90 rotate\t\t\t\t%% Landscape mode\n");
      write(fd,buf,strlen(buf));
    }

  sprintf(buf, "xBBoxSide 2 div yBBoxSide TitlePtSize 2 mul add\n"); /* Title */
  write(fd,buf,strlen(buf));
  pm(GET, "Print.Title", s, NULL);			   
  sprintf(buf,"(%s) () FigureTitle\n", s );
  write(fd,buf,strlen(buf));

  if ( *((int *)pm(GET, "Print.Label_Range", NULL))	)   /* label axes range? */
    {
      sprintf(buf,"(%.6lg) (%.6lg) 2 PlaceXLabels\n(%.6lg) (%.6lg) 2 PlaceYLabels\n",
	      *((double *) pm(GET, "Print.HorMin", NULL)),
	      *((double *) pm(GET, "Print.HorMax", NULL)),
	      *((double *) pm(GET, "Print.VerMin", NULL)),
	      *((double *) pm(GET, "Print.VerMax", NULL)) );
      write(fd,buf,strlen(buf));
    }

  pm(GET, "Print.Hor_Label", s, NULL);		    /* label axes */
  sprintf(buf,"(%s) XLabel\n", s );
  write(fd,buf,strlen(buf));
  pm(GET, "Print.Ver_Label", s, NULL);
  sprintf(buf,"(%s) HorYLabel\n%s\n", s,"% Place all Labels and Annotations BEFORE this line");
  write(fd,buf,strlen(buf));
}

/*
 * write_trailer() ends the PS document
 */
int
write_trailer(fd, buf)
	int		fd;
	char		buf[];
{
  sprintf(buf,"grestore  showpage\n%%%%Trailer\n");
  write(fd,buf,strlen(buf));
}

/*
 * get_plot_font() returns a PostScript font name
 * PS_font[] is defined in print.h
 */
char *
get_plot_font()
{
  return(PS_font[*((int *)pm(GET, "Print.Font", NULL))]); 
}

/*
 * pt_within_region() returns TRUE if pt is visible on the view window; FALSE otherwise
 */
int
pt_within_region(x, y, hor_min, hor_max, ver_min, ver_max)
	double          x, y, hor_min,hor_max,ver_min,ver_max;
{
  int in_region = 0;
  in_region = (x > hor_min ) && (x < hor_max ) && (y > ver_min ) && (y < ver_max ) ;
  return(in_region);
}

/*
 * get_ps_symbol() returns encoded PostScript macro which will draw an appropriate symbol
 */
char *
get_ps_symbol(symbol_index)
	int		symbol_index;
{
  int		i;
  for(i=0; i<Num_Symbols; i++)
    if(Symbol_Codes[i] == symbol_index)
      return(Symbol_Labels[i]); /* make change to src/include/symbols_def.h */
	  
  system_mess_proc(1,"print: Symbol not found! Using default symbol");
  return(Symbol_Labels[0]);
}

/*
 * pt_to_xy() takes a window number, state, and parameters, and returns the x,y
 * coords of the projection of that pt onto the given window
 */
void
pt_to_xy(win_num, state, param, x, y, f)
	int	win_num;
	double	*state, *param, *f,
  		*x, *y;		
{
}

/*
 * returns index into cms of colortable of "current" twoD window
 */
void
get_plot_color(void)
{
}

/*
 * index_to_ps_rgb() obtains the current colormap and associates to each index
 *	the corresponding postscript (RGB) color
 */
int
index_to_ps_rgb(void)
{
 /*
  pm( GET_LIST, "Color.Red_Table", 0, n_total-1, red, NULL); 
  pm( GET_LIST, "Color.Green_Table", 0, n_total-1, green, NULL); 
  pm( GET_LIST, "Color.Blue_Table", 0, n_total-1, blue, NULL); 


  for(i=0;i<n_total;i++)			PostScript RGB is between 0 and 1 
    {
      rgb->r[i] = (double) red[i] / (double)(MAX_COLORS - 1.);
      rgb->g[i] = (double) green[i] / (double)(MAX_COLORS - 1.);
      rgb->b[i] = (double) blue[i] / (double)(MAX_COLORS - 1.); 
    }  
  free_ivector(red, 0, MAX_COLORS);
  free_ivector(green, 0, MAX_COLORS);
  free_ivector(blue, 0, MAX_COLORS);
 */

  return(0);
}

/*
 * write_macros() reads in the PostScript prolog as defined 
 * by the environmental variable DSTOOL_PS_PROLOG if defined, or, 
 * as defined in $DSTOOL/site_specific/PROLOG_FILE
 * where PROLOG_FILE is as defined in defaults.h.  Upon success, 0 is returned.
 */
int
write_macros(fd,buf)
	int		fd;
	char		buf[];
{
  FILE    *pp;
  char    *prologname;
  int	  pd,n;

  prologname = (char *) calloc(SIZE_OF_DIR_PLUS_FNAME,sizeof(char));

 /*
  * choose DSTOOL_PS_PROLOG file if defined
  */
  get_dstool_path( prologname, DSTOOL_PS_PROLOG );
  strcat(prologname,"/site_specific/");
  strcat(prologname,PROLOG_FILE);  

 /* 
  * replace this filename with prologname  
  */
  if( !(pp = fopen(prologname, "r")) ) {
    system_mess_proc(1,"print: PostScript prolog not found or permission denied");
    system_mess_proc(1,prologname);
    return -1;
  }
  else {
    pd = fileno(pp);
   /* 
    * reads the Prolog and prints it out again 
    */
    while( (n = read(pd, buf, BUFFERSIZE)) > 0)	{
	write(fd, buf, n);
    } 
  } 

  fclose(pp);
  free(prologname);
  return 0;
}


/*
 * broken_pipe_error() is called upon the error signal SIGPIPE 
 * (attempt to write on pipe or other socket with no one to read it).  
 * The function returns the stack (and flow) to the point just prior to opening the pipe.
 */
void
broken_pipe_error()
{
   longjmp(env,-1);
}

