/* 
    SPaSE SP8870  -  DVB-T COFDM demodulator used in Technotrend DVB-T cards

    Copyright (C) 2001 Holger Waechtler <holger@convergence.de>
                       for Convergence Integrated Media GmbH

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/    

#include <linux/module.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/poll.h>
#include <linux/i2c.h>
#include <linux/smp_lock.h>

#include "dvb_frontend.h"

#ifdef MODULE
MODULE_DESCRIPTION("SP8870 based DVB-T Demodulator");
MODULE_AUTHOR("Holger Waechtler");
#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif
#endif

static struct i2c_driver dvbt_driver;
static struct i2c_client client_template;

MODULE_PARM(debug,"i");
static int debug = 0;
#define dprintk	if (debug) printk


static
int write_reg(struct i2c_client *client, u16 reg, u16 data)
{
	int ret;
	u8 msg[4] = { reg >> 8, reg & 0xff, data >> 8, data & 0xff };

	if ((ret = i2c_master_send (client, msg, 4)) != 4)
		printk ("L64781: write_reg error = %02x!\n", ret);

	mdelay(10);

	return ret;
}


static
u16 read_reg(struct i2c_client *client, u16 reg)
{
	struct i2c_adapter *adap=client->adapter;
	struct i2c_msg msgs[4];
	u8 reg2 [2] = { reg >> 8, reg & 0xff };
	u8 data [2];

	msgs[0].addr = client->addr;
	msgs[0].len = 2;
	msgs[0].flags = 0;
	msgs[0].buf = reg2;
	msgs[1].addr = client->addr;
	msgs[1].len = 2;
	msgs[1].flags = I2C_M_RD;
	msgs[1].buf = data;

	i2c_transfer(adap, msgs, 2);

	return (data[0] << 8 | data[1]);
}



static
void apply_frontend_param (struct i2c_client *client, FrontendParameters *param)
{
printk("%s:\n", __FUNCTION__);        
}


static
int init (struct i2c_client *client)
{
printk("%s:\n", __FUNCTION__);        
	return 0;
}


static
int attach_adapter(struct i2c_adapter *adap)
{
        struct i2c_client *client;
#if 0
int i;
client_template.adapter=adap;
printk("%s:\n", __FUNCTION__);        
printk ("scanning i2c ...\n");
for (i=0; i<0x7f; i++) {
client_template.addr = i;
if (i2c_master_send(&client_template,NULL,0) == 0)
	printk ("found client at addr == 0x%02x\n", i);
}
#endif
        client_template.adapter=adap;
        
	if (i2c_master_send(&client_template,NULL,0)) {
		dprintk ("SP8870: no SP8870 found ...\n");
               	return -1;
	}

        if (!(client = kmalloc(sizeof(struct i2c_client), GFP_KERNEL)))
                return -ENOMEM;

        memcpy(client, &client_template, sizeof(struct i2c_client));

        i2c_attach_client(client);

printk (" reg (0x200) == 0x%04x\n", read_reg (client, 0x200));
printk (" reg (0x201) == 0x%04x\n", read_reg (client, 0x201));
printk (" reg (0x303) == 0x%04x\n", read_reg (client, 0x303));
printk (" reg (0x306) == 0x%04x\n", read_reg (client, 0x306));
printk (" reg (0x309) == 0x%04x\n", read_reg (client, 0x309));
printk (" reg (0x30a) == 0x%04x\n", read_reg (client, 0x30a));
printk (" reg (0xf00) == 0x%04x\n", read_reg (client, 0xf00));
        printk("SP8870: attaching SP8870 at 0x%02x\n", (client->addr)<<1);
        
	MOD_INC_USE_COUNT;
        
        return 0;
}


static
int detach_client(struct i2c_client *client)
{
printk("%s:\n", __FUNCTION__);        
        i2c_detach_client(client);

        kfree(client);

	MOD_DEC_USE_COUNT;
        return 0;
}


static 
int dvb_command(struct i2c_client *client, unsigned int cmd, void *arg)
{
printk("%s:\n", __FUNCTION__);        
        switch (cmd) {
	case FE_INIT:
        	init(client);
                break;

	case FE_RESET:
                break;

	case FE_READ_STATUS:
	{
		FrontendStatus *status=(FrontendStatus *) arg;
		int sync;

		*status=0;

                sync=read_reg(client,0x32);
		if (sync&0x10) // FIXME: find criteria for having signal 
			*status |= FE_HAS_SIGNAL;
		if (sync&0x02) // FIXME: find criteria
			*status |= FE_HAS_CARRIER;
		if (sync&0x20) // FIXME
			*status |= FE_HAS_VITERBI;
		if (sync&0x08) // FIXME
			*status |= FE_HAS_SYNC;
		if (sync==0x7f)
			*status |= FE_HAS_LOCK;
		break;
	}

	case FE_READ_BER:
	{
		u32 *ber=(u32 *) arg;
		*ber=0;
		break;
	}

	case FE_READ_SIGNAL_STRENGTH:
	{
		s32 *signal=(s32 *) arg;
		*signal=0;
		break;
	}

	case FE_READ_SNR:
	{
		s32 *snr=(s32 *) arg;
		*snr=0; /* average quality: read_reg(client, 0x33); */
		break;
	}

	case FE_READ_UNCORRECTED_BLOCKS: 
	{
		u32 *ublocks=(u32 *) arg;
		*ublocks=read_reg(client, 0x37) | (read_reg(client, 0x38)<<8);
		break;
	}

	case FE_READ_AFC:
	{
		s32 *afc=(s32 *) arg;
		*afc=0;
		break;
	}

        case FE_SET_FRONTEND:
        {
		FrontendParameters *param = (FrontendParameters *) arg;

		apply_frontend_param (client, param);

		break;
        }

	case FE_WRITEREG:
        {
                u8 *msg = (u8 *) arg;
                write_reg(client, msg[0], msg[1]);
                break;
        }

	case FE_READREG:
        {
                u8 *msg = (u8 *) arg;
                msg[1]=read_reg(client, msg[0]);
                break;
        }

	default:
		dprintk ("%s: unknown command !!!\n", __FUNCTION__);
                return -1;
        };
        
        return 0;
} 


void inc_use (struct i2c_client *client)
{
#ifdef MODULE
printk("%s:\n", __FUNCTION__);        
        MOD_INC_USE_COUNT;
#endif
}


void dec_use (struct i2c_client *client)
{
#ifdef MODULE
printk("%s:\n", __FUNCTION__);        
        MOD_DEC_USE_COUNT;
#endif
}


static
struct i2c_driver dvbt_driver = {
	name: 		"SP8870 DVB demodulator",
	id:		I2C_DRIVERID_SP8870,
	flags:		I2C_DF_NOTIFY,
	attach_adapter:	attach_adapter,
	detach_client:	detach_client,
	command:	dvb_command,
	inc_use:	inc_use,
	dec_use:	dec_use
};


static
struct i2c_client client_template = {
        name:    "SP8870",
        id:      I2C_DRIVERID_SP8870,
        flags:   0,
        addr:    0x71,
        adapter: NULL,
        driver:  &dvbt_driver
};


static __init
int init_SP8870 (void) {
        int res;
printk("%s:\n", __FUNCTION__);        
        if ((res = i2c_add_driver(&dvbt_driver))) 
        {
                printk("SP8870: i2c driver registration failed\n");
                return res;
        }

        return 0;
}


static __exit
void exit_SP8870 (void)
{
        int res;
printk("%s:\n", __FUNCTION__);        

        if ((res = i2c_del_driver(&dvbt_driver))) 
        {
                printk("SP8870: Driver deregistration failed, "
                       "module not removed.\n");
        }
}


module_init(init_SP8870);
module_exit(exit_SP8870);

