/*  $Header: /cvsroot/dvipdfmx/src/pdfdev.c,v 1.35 2004/04/11 05:37:59 hirata Exp $
    
    This is dvipdfmx, an eXtended version of dvipdfm by Mark A. Wicks.

    Copyright (C) 2002 by Jin-Hwan Cho and Shunsaku Hirata,
    the dvipdfmx project team <dvipdfmx@project.ktug.or.kr>
    
    Copyright (C) 1998, 1999 by Mark A. Wicks <mwicks@kettering.edu>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
*/

#if HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <ctype.h>
#include <math.h>

#include "system.h"
#include "mem.h"
#include "error.h"
#include "mfileio.h"
#include "numbers.h"
#include "dvi.h"
#include "tfm.h"
#include "pdfdoc.h"
#include "pdfobj.h"

#include "pdfresource.h"
#include "pdffont.h"

#include "type0.h"
#include "cmap.h"

#include "pdfspecial.h"
#include "pdfparse.h"
#include "tpic.h"
#include "htex.h"
#include "mpost.h"
#include "psspecial.h"
#include "colorsp.h"
#include "pdflimits.h"

#include "encodings.h"
#include "agl.h"
#include "unicode.h"
#include "fontmap.h"

#include "pdfdev.h"

static int verbose = 0;

void
pdf_dev_set_verbose (void)
{
  verbose++;
  Encoding_set_verbose();
  AGL_set_verbose();
  UC_set_verbose();
  CMap_set_verbose();
  pdf_font_set_verbose();
  Type0Font_set_verbose();
  CIDFont_set_verbose();
}

/* Internal functions */
static void dev_clear_color_stack (void);
static void dev_clear_xform_stack (void);

/*
 * Acrobat doesn't seem to like coordinate systems
 * that involve scalings around 0.01, so we use
 * a scaline of 1.0.  In other words, device units = pts
 */ 

static struct {
  double hoffset, voffset;
  double width,   height;
  int    readonly;
} dev_page = {
  72.0, 72.0,
  0.0, 0.0,
  0
};


static double
dev_page_width (void)
{
  dev_page.readonly = 1;
  return dev_page.width;
}

static double
dev_page_height (void)
{
  dev_page.readonly = 1;
  return dev_page.height;
}

static double
dev_page_hoffset (void)
{
  dev_page.readonly = 1;
  return dev_page.hoffset;
}

static double
dev_page_voffset (void)
{
  dev_page.readonly = 1;
  return dev_page.voffset;
}

void
dev_set_page_size (double width, double height)
{
  if (!dev_page.readonly) {
    dev_page.width  = width;
    dev_page.height = height;
  } else {
    WARN("Can't change page size!");
  }
}

#define TEX_ONE_HUNDRED_BP 6578176
static struct {
  double dvi2pts;
  long   min_bp_val; /* Shortest resolvable distance in the output PDF.     */
  int    precision;  /* Number of decimal digits (in fractional part) kept. */
} dev_unit = {
  0.0,
  658,
  2
};

double
dev_unit_dviunit (void)
{
  return (1.0/dev_unit.dvi2pts);
}

#define DEV_PRECISION_MAX  8
static unsigned long ten_pow[10] = {
  1ul, 10ul, 100ul, 1000ul, 10000ul, 100000ul, 1000000ul, 10000000ul, 100000000ul, 1000000000ul
};

static double ten_pow_inv[10] = {
  1.0, 0.1,  0.01,  0.001,  0.0001,  0.00001,  0.000001,  0.0000001,  0.00000001,  0.000000001
};

static int
p_itoa (long value, char *buf)
{
  int   sign, ndigits;
  char *p = buf;

  if (value < 0) {
    *p++  = '-';
    value = -value;
    sign  = 1;
  } else {
    sign  = 0;
  }
    
  ndigits = 0;
  /* Generate at least one digit in reverse order */
  do {
    p[ndigits++] = (value % 10) + '0';
    value /= 10;
  } while (value != 0);

  /* Reverse the digits */
  {
    int i;

    for (i = 0; i < ndigits / 2 ; i++) {
      char tmp = p[i];
      p[i] = p[ndigits-i-1];
      p[ndigits-i-1] = tmp;
    }
  }
  p[ndigits] = '\0';

  return (sign ? ndigits + 1 : ndigits);
}

#if 0
/* This isn't usefull at all. */

#define ROUND_TOWARD_NINF 0
#define ROUND_TOWARD_ZERO 1
#define ROUND_HALF_UP     2
#define ROUND_HALF_DOWN   3

static int __round_mode = ROUND_HALF_UP;

long
dev_round (spt_t value, spt_t min_val, spt_t *error)
{
  spt_t delta;

  switch (__round_mode) {
  case ROUND_TOWARD_NINF: /* FLOOR */
    delta = value % min_val;
    break;
  case ROUND_TOWARD_ZERO: /* DOWN */
    delta = (value < 0) ? -((-value) % min_val) : value % min_val;
    break;
  case ROUND_HALF_DOWN:   /* .5 DOWN */
    {
      stp_t r;

      r = ((value < 0) ? -value : value) % min_val;
      if (2*r <= min_val) {
	delta = (value < 0) ? -r : r;
      } else {
	delta = (value < 0) ? min_val - r : r - min_val;
      }
    }
    break;
  case ROUND_HALF_UP:     /* .5 UP */
    {
      spt_t r;

      r = ((value < 0) ? -value : value) % min_val;
      if (2*r < min_val) {
	delta = (value < 0) ? -r : r;
      } else {
	delta = (value < 0) ? min_val - r : r - min_val;
      }
    }
    break;
  }
  if (error)
    *error = delta;

  return value - delta;
}
#endif

/*
 * dvipdfmx spend most of his time in dev_set_string() where integers
 * are used. Speed is not a matter for formatting floating points.
 */
static int
p_dtoa (double value, int nddigits, char *buf)
{
  int  ntotal = 0;
  long ipart, dpart;

  if (value < 0.0) {
    buf[ntotal++] = '-';
    value = -value;
  }

  if (nddigits > DEV_PRECISION_MAX)
    nddigits = DEV_PRECISION_MAX;

  value = ROUND(value, ten_pow_inv[nddigits]);
  ipart = (long) value;
  dpart = (long) (ten_pow[nddigits]*(value - ipart));

  if (dpart == 0 && ipart == 0) {
    buf[0] = '0';
    ntotal =  1;
  } else {
    if (ipart != 0) {
      ntotal += p_itoa(ipart, buf+ntotal);
    }
    if (dpart != 0) {
      /* chop trailing zeros */
      while ((dpart % 10) == 0) {
	dpart /= 10;
	nddigits--;
      }
      if (nddigits > 0) {
	int n = nddigits;
	buf[ntotal++] = '.';
	while (n-- > 0) {
	  buf[ntotal+n] = (dpart % 10) + '0';
	  dpart /= 10;
	}
      }
      ntotal += nddigits;
    }
  }
  buf[ntotal] = '\0';

  return ntotal;
}

static int
dev_sprint_bp (char *buf, spt_t value, spt_t *error)
{
  double value_in_bp;
  double error_in_bp;

  value_in_bp = (dev_unit.dvi2pts)*value;
  if (error) {
    error_in_bp = value_in_bp - ROUND(value_in_bp, ten_pow_inv[dev_unit.precision]);
    *error = ROUND(error_in_bp/(dev_unit.dvi2pts), 1);
  }

  return p_dtoa(value_in_bp, dev_unit.precision, buf);
}

double
pdf_dev_scale (void)
{
  return 1.0;
}

#define GRAPHICS_MODE 1
#define TEXT_MODE     2
#define STRING_MODE   3

int motion_state = GRAPHICS_MODE; /* Start in graphics mode */

#define FORMAT_BUF_SIZE 4096
static char format_buffer[FORMAT_BUF_SIZE];

/*
 * In PDF, vertical text positioning is always applied when current font
 * is vertical font. While ASCII pTeX manages current writing direction
 * and font's WMode separately.
 *
 * 00/11 WMODE_HH/VV  h(v) font, h(v) direction.
 * 01    WMODE_HV    -90 deg. rotated
 * 10    WMODE_VH    +90 deg. rotated
 *
 * In MetaPost PostScript file processing (mp_mode = 1), only HH/VV mode
 * is applied.
 */
#define TEXT_WMODE_HH 0
#define TEXT_WMODE_HV 1
#define TEXT_WMODE_VH 2
#define TEXT_WMODE_VV 3

#define ANGLE_CHANGES(m1,m2) ((abs((m1)-(m2)) % 3) == 0 ? 0 : 1)
#define ROTATE_TEXT(m)       ((m) != TEXT_WMODE_HH && (m) != TEXT_WMODE_VV)

/*
 * Device coordinates are relative to upper left of page.  One of the
 * first things appearing in the page stream is a coordinate transformation
 * matrix that forces this to be true.  This coordinate
 * transformation is the only place where the paper size is required.
 * Unfortunately, positive is up, which doesn't agree with TeX's convention.
 */

static struct {
  struct {
    int    id;
    double scale;  /* unused */
  } font;

  spt_t  offset;

  struct {
    spt_t  x;
    spt_t  y;
  } origin;   /* This should be moved to matrix. */

  struct {
    double slant;
    double extend;
    int    rotate; /* TEXT_WMODE_XX */
  } matrix;

  int    dir, dir_save; /* pdf_dev shouldn't have this... */
  int    force_reset;
  int    mp_mode;
  int    is_mb;    /* From dev_font[font_id].format. */
} text_state = {
  {-1, 0.0},
  0,
  {0, 0},
  {0.0, 1.0, 0},
  0, 0,
  0,
  0,
  0
};

#define FONTTYPE_SIMPLE    1
#define FONTTYPE_BITMAP    2
#define FONTTYPE_COMPOSITE 3

static struct dev_font {
  char     short_name[7]; /* Needs to be big enough to hold name "Fxxx"
			     where xxx is number of largest font */
  int      used_on_this_page;
  char    *tex_name;
  spt_t    sptsize;

  int      font_id;
  pdf_obj *resource;
  int      format;
  char    *used_chars;

  double   extend;
  double   slant;
  int      remap;
  int      mapc;
  int      wmode; /* WMode, horizontal: 0, vertical: 1 */
  int      cmap;  /* cmap id */
} *dev_font = NULL;

static int num_dev_fonts    = 0;
static int max_device_fonts = 0;
static int num_phys_fonts   = 0;

#define CURRENTFONT() ((text_state.font.id < 0) ? NULL : &(dev_font[text_state.font.id]))

int
pdf_sprint_matrix (char *buf, const pdf_tmatrix *M)
{
  int len = 0;

  len += p_dtoa(M->a, MIN(dev_unit.precision + 2, DEV_PRECISION_MAX), buf+len);
  buf[len++] = ' ';
  len += p_dtoa(M->b, MIN(dev_unit.precision + 2, DEV_PRECISION_MAX), buf+len);
  buf[len++] = ' ';
  len += p_dtoa(M->c, MIN(dev_unit.precision + 2, DEV_PRECISION_MAX), buf+len);
  buf[len++] = ' ';
  len += p_dtoa(M->d, MIN(dev_unit.precision + 2, DEV_PRECISION_MAX), buf+len);
  buf[len++] = ' ';
  len += p_dtoa(M->e, MAX(dev_unit.precision, 2), buf+len);
  buf[len++] = ' ';
  len += p_dtoa(M->f, MAX(dev_unit.precision, 2), buf+len);
  buf[len]   = '\0'; /* xxx_sprint_xxx NULL terminates strings. */

  return len;
}

int
pdf_sprint_rect (char *buf, const pdf_rect *rect)
{
  int len = 0;

  len += p_dtoa(rect->llx, dev_unit.precision, buf+len);
  buf[len++] = ' ';
  len += p_dtoa(rect->lly, dev_unit.precision, buf+len);
  buf[len++] = ' ';
  len += p_dtoa(rect->urx, dev_unit.precision, buf+len);
  buf[len++] = ' ';
  len += p_dtoa(rect->ury, dev_unit.precision, buf+len);
  buf[len]   = '\0'; /* xxx_sprint_xxx NULL terminates strings. */

  return len;
}

static void
dev_set_text_matrix (spt_t xpos, spt_t ypos, double slant, double extend, int rotate)
{
  pdf_tmatrix tm;
  int         len = 0;

  /* slant is negated for vertical font so that right-side
   * is always lower. */
  switch (rotate) {
  case TEXT_WMODE_VH:
    /* Vertical font */
    tm.a =  slant ;   tm.b =  1.0;
    tm.c = -extend;   tm.d =  0.0   ;
    break;
  case TEXT_WMODE_HV:
    /* Horizontal font */
    tm.a =  0.0;    tm.b = -extend;
    tm.c =  1.0;    tm.d = -slant ;
    break;
  case TEXT_WMODE_HH:
    /* Horizontal font */
    tm.a =  extend; tm.b =  0.0;
    tm.c =  slant ; tm.d =  1.0;
    break;
  case TEXT_WMODE_VV:
    /* Vertical font */
    tm.a =  1.0; tm.b =  -slant;
    tm.c =  0.0; tm.d =   extend;
    break;
  }
  tm.e = xpos * dev_unit.dvi2pts;
  tm.f = ypos * dev_unit.dvi2pts;

  format_buffer[len++] = ' ';
  len += pdf_sprint_matrix(format_buffer+len, &tm);
  format_buffer[len++] = ' ';
  format_buffer[len++] = 'T';
  format_buffer[len++] = 'm';

  pdf_doc_add_to_page(format_buffer, len);

  text_state.origin.x = xpos;
  text_state.origin.y = ypos;
  text_state.matrix.slant  = slant;
  text_state.matrix.extend = extend;
  text_state.matrix.rotate = rotate;
}

/*
 * reset_text_state() outputs a BT and does any necessary coordinate
 * transformations to get ready to ship out text.
 */

static void
reset_text_state (void)
{
  /*
   * We need to reset the line matrix to handle slanted fonts.
   */
  pdf_doc_add_to_page(" BT", 3);
  /*
   * text_state.matrix is identity at top of page.
   * This sometimes write unnecessary "Tm"s when transition from
   * GRAPHICS_MODE to TEXT_MODE occurs.
   */
  if (text_state.force_reset ||
      text_state.matrix.slant  != 0.0 ||
      text_state.matrix.extend != 1.0 ||
      ROTATE_TEXT(text_state.matrix.rotate)) {
    dev_set_text_matrix(0, 0,
			text_state.matrix.slant,
			text_state.matrix.extend,
			text_state.matrix.rotate);
  }
  text_state.origin.x = 0;
  text_state.origin.y = 0;
  text_state.offset   = 0;
  text_state.force_reset = 0;
}

static
void text_mode (void)
{
  switch (motion_state) {
  case TEXT_MODE:
    break;
  case STRING_MODE:
    pdf_doc_add_to_page(text_state.is_mb ? ">]TJ" : ")]TJ", 4);
    break;
  case GRAPHICS_MODE:
    reset_text_state();
    break;
  }
  motion_state      = TEXT_MODE;
  text_state.offset = 0;
}

void
graphics_mode (void)
{
  switch (motion_state) {
  case GRAPHICS_MODE:
    break;
  case STRING_MODE:
    pdf_doc_add_to_page(text_state.is_mb ? ">]TJ" : ")]TJ", 4);
    /* continue */
  case TEXT_MODE:
    pdf_doc_add_to_page(" ET", 3);
    text_state.force_reset =  0;
    text_state.font.id     = -1;
    break;
  }
  motion_state = GRAPHICS_MODE;
}

static void
start_string (spt_t xpos, spt_t ypos, double slant, double extend, int rotate)
{
  spt_t delx, dely, error_delx, error_dely;
  spt_t desired_delx, desired_dely;
  int   len = 0;

  delx = xpos - text_state.origin.x;
  dely = ypos - text_state.origin.y;
  /*
   * Precompensating for line transformation matrix.
   */
  switch (rotate) {
  case TEXT_WMODE_VH:
    /* Vertical font */
    desired_delx = (spt_t) (-(delx - dely*slant)/extend);
    desired_dely = dely;

    format_buffer[len++] = ' ';
    len += dev_sprint_bp(format_buffer+len, desired_dely, &error_dely);
    format_buffer[len++] = ' ';
    len += dev_sprint_bp(format_buffer+len, desired_delx, &error_delx);
    error_delx = -error_delx;
    break;
  case TEXT_WMODE_HV:
    /* Horizontal font */
    desired_delx = delx;
    desired_dely = (spt_t)(-(dely + delx*slant)/extend);

    format_buffer[len++] = ' ';
    len += dev_sprint_bp(format_buffer+len, desired_dely, &error_dely);
    error_dely = -error_dely;
    format_buffer[len++] = ' ';
    len += dev_sprint_bp(format_buffer+len, desired_delx, &error_delx);
    break;
  case TEXT_WMODE_HH:
    /* Horizontal font */
    desired_delx = (spt_t)((delx - dely*slant)/extend);
    desired_dely = dely;

    format_buffer[len++] = ' ';
    len += dev_sprint_bp(format_buffer+len, desired_delx, &error_delx);
    format_buffer[len++] = ' ';
    len += dev_sprint_bp(format_buffer+len, desired_dely, &error_dely);
    break;
  case TEXT_WMODE_VV:
    /* Vertical font */
    desired_delx = delx;
    desired_dely = (spt_t)((dely + delx*slant)/extend);

    format_buffer[len++] = ' ';
    len += dev_sprint_bp(format_buffer+len, desired_delx, &error_delx);
    format_buffer[len++] = ' ';
    len += dev_sprint_bp(format_buffer+len, desired_dely, &error_dely);
    break;
  }
  pdf_doc_add_to_page(format_buffer, len);
  /*
   * dvipdfm wrongly using "TD" in place of "Td".
   * The TD operator set leading, but we are not using T* etc.
   */
  pdf_doc_add_to_page(text_state.is_mb ? " Td[<" : " Td[(", 5);

  text_state.origin.x = xpos - error_delx;
  text_state.origin.y = ypos - error_dely;
  text_state.offset   = 0;
}

static void
string_mode (spt_t xpos, spt_t ypos, double slant, double extend, int rotate)
{
  switch (motion_state) {
  case STRING_MODE:
    break;
  case GRAPHICS_MODE:
    reset_text_state();
    /* continue */
  case TEXT_MODE:
    if (text_state.force_reset) {
      dev_set_text_matrix(xpos, ypos, slant, extend, rotate);
      pdf_doc_add_to_page(text_state.is_mb ? "[<" : "[(", 2);
      text_state.force_reset = 0;
    } else {
      start_string(xpos, ypos, slant, extend, rotate);
    }
    break;
  }
  motion_state = STRING_MODE;
}

/*
 * The purpose of the following routine is to force a Tf only
 * when it's actually necessary.  This became a problem when the
 * VF code was added.  The VF spec says to instantiate the
 * first font contained in the VF file before drawing a virtual
 * character.  However, that font may not be used for
 * many characters (e.g. small caps fonts).  For this reason, 
 * dev_select_font() should not force a "physical" font selection.
 * This routine prevents a PDF Tf font selection until there's
 * really a character in that font.
 */

static void
dev_set_font (int font_id)
{
  struct dev_font *font;
  int    text_rotate;
  double font_scale;
  int    len;

  /* text_mode() must come before text_state.is_mb is changed. */
  text_mode();

  font = &(dev_font[font_id]);

  text_state.is_mb = (font->format == FONTTYPE_COMPOSITE) ? 1 : 0;
  if (text_state.mp_mode) {
    text_state.dir = font->wmode;
  }
  if (font->wmode == 0 && text_state.dir == 0)
    text_rotate = TEXT_WMODE_HH;
  else if (font->wmode == 1 && text_state.dir == 1)
    text_rotate = TEXT_WMODE_VV;
  else if (font->wmode == 1 && text_state.dir == 0)
    text_rotate = TEXT_WMODE_VH;
  else if (font->wmode == 0 && text_state.dir == 1)
    text_rotate = TEXT_WMODE_HV;
  else
    text_rotate = TEXT_WMODE_HH;

  if (font->slant  != text_state.matrix.slant  ||
      font->extend != text_state.matrix.extend ||
      ANGLE_CHANGES(text_rotate, text_state.matrix.rotate)) {
    text_state.force_reset = 1;
  }
  text_state.matrix.slant  = font->slant;
  text_state.matrix.extend = font->extend;
  text_state.matrix.rotate = text_rotate;

  if (!font->resource) {
    switch (font->format) {
    case FONTTYPE_COMPOSITE:
      font->resource   = type0_font_resource(font->font_id);
      font->used_chars = type0_font_used(font->font_id);
      break;
    case FONTTYPE_SIMPLE:
    case FONTTYPE_BITMAP:
      font->resource   = pdf_font_get_resource(font->font_id);
      font->used_chars = pdf_font_get_usedchar(font->font_id);
      break;
    default:
      ERROR("Impossible font format in dev_set_font().");
    }
  }

  if (!font->used_on_this_page) { 
    pdf_doc_add_to_page_resources("Font",
				  font->short_name,
				  pdf_link_obj(font->resource));
    font->used_on_this_page = 1;
  }

  font_scale = (double) font->sptsize * dev_unit.dvi2pts;
  len  = sprintf(format_buffer, "/%s", font->short_name); /* space not necessary. */
  format_buffer[len++] = ' ';
  len += p_dtoa(font_scale, MIN(dev_unit.precision+1, DEV_PRECISION_MAX), format_buffer+len);
  format_buffer[len++] = ' ';
  format_buffer[len++] = 'T';
  format_buffer[len++] = 'f';
  pdf_doc_add_to_page(format_buffer, len);

  text_state.font.id    = font_id;
  text_state.font.scale = font_scale;
}

void
dev_set_string (spt_t xpos, spt_t ypos, unsigned char *s,
		int length, spt_t width, int font_id, int ctype)
{
  unsigned char *str; /* Pointer to the reencoded string. */
  unsigned char  wbuf[FORMAT_BUF_SIZE];
  int   i, len = 0;
  spt_t kern = 0;
  spt_t text_xorigin = text_state.origin.x;
  spt_t text_yorigin = text_state.origin.y;

  if (font_id < 0 || font_id >= num_dev_fonts)
    ERROR("Invalid font: %d (%d)", font_id, num_dev_fonts);

  /*
   * Force a Tf since we are actually trying to write a character.
   */
  if (font_id != text_state.font.id)
    dev_set_font(font_id);

  str = s;
  if (dev_font[font_id].format == FONTTYPE_COMPOSITE) {
    unsigned char tmp[FORMAT_BUF_SIZE];

    /* Omega workaround... */
    if (ctype == 1 && dev_font[font_id].mapc >= 0) {
      /* Translate single-byte chars to double byte code space */
      for (i = 0; i < length; i++) {
	tmp[i*2]   = (unsigned char) dev_font[font_id].mapc;
	tmp[i*2+1] = s[i];
      }
      length *= 2;
      str     = tmp;
    }
    /*
     * Font is double-byte font. Output is assumed to be 16-bit fixed length
     * encoding.
     * TODO: A character decomposed to multiple characters.
     */
    if (dev_font[font_id].cmap >= 0) {
      unsigned char *inbuf, *outbuf;
      long inbytesleft, outbytesleft;
      CMap *cmap;

      cmap = CMap_cache_get(dev_font[font_id].cmap);
      inbuf  = str;
      outbuf = wbuf;
      inbytesleft  = length;
      outbytesleft = FORMAT_BUF_SIZE;
      CMap_decode(cmap,
		  (const unsigned char **) &inbuf, &inbytesleft, &outbuf, &outbytesleft);
      if (inbytesleft != 0)
	ERROR("Code conversion failed. (%d bytes remains)", inbytesleft, length);
      length = FORMAT_BUF_SIZE - outbytesleft;
      str    = wbuf;
    }

    if (dev_font[font_id].used_chars != NULL) {
      for (i = 0; i < length; i += 2)
	add_to_used_chars2(dev_font[font_id].used_chars, (str[i] << 8)| str[i+1]);
    }

  } else {
    if (dev_font[font_id].remap) {
      str = wbuf;
      for (i = 0; i < length; i++) {
	str[i] = twiddle(s[i]);
        dev_font[font_id].used_chars[str[i]] = 1;
      }
    } else {
      if (dev_font[font_id].used_chars != NULL) {
	for (i = 0; i < length; i++)
	  dev_font[font_id].used_chars[str[i]] = 1;
      }
    }
  }

  /*
   * Kern is in units of character units, i.e., 1000 = 1 em.
   *
   * The following formula is of the form a*x/b where a, x, and b are signed long
   * integers.  Since in integer arithmetic (a*x) could overflow and a*(x/b) would
   * not be accurate, we use floating point arithmetic rather than trying to do
   * this all with integer arithmetic.
   */
  if (text_state.dir == 1) {
    kern = (spt_t) ((1000.0/dev_font[font_id].extend *
		     (ypos-text_yorigin+text_state.offset))/dev_font[font_id].sptsize);
    if (text_state.force_reset ||
	labs(xpos-text_xorigin) > dev_unit.min_bp_val || abs(kern) > 3000) {
      text_mode();
      kern = 0;
    }
  } else {
    kern = (spt_t) ((1000.0/dev_font[font_id].extend *
		     (text_xorigin+text_state.offset-xpos))/dev_font[font_id].sptsize);
    if (text_state.force_reset ||
	labs(ypos-text_yorigin) > dev_unit.min_bp_val || abs(kern) > 3000) {
      text_mode();
      kern = 0;
    } 
  }

  if (motion_state != STRING_MODE)
    string_mode(xpos, ypos,
		dev_font[font_id].slant, dev_font[font_id].extend, text_state.matrix.rotate);
  else if (kern != 0) {
    text_state.offset -=
      (spt_t) (kern*dev_font[font_id].extend*(dev_font[font_id].sptsize/1000.0));
    /*
     * Same issues as earlier. Use floating point for simplicity.
     * This routine needs to be fast, so we don't call sprintf() or strcpy().
     */
    format_buffer[len++] = text_state.is_mb ? '>' : ')';
    if (dev_font[font_id].wmode == 1)
      len += p_itoa(-kern, format_buffer + len);
    else
      len += p_itoa( kern, format_buffer + len);
    format_buffer[len++] = text_state.is_mb ? '<' : '(';
    pdf_doc_add_to_page(format_buffer, len);
    len = 0;
  }

  if (text_state.is_mb) {
    if (FORMAT_BUF_SIZE - len < 2*length)
      ERROR("Buffer overflow...");
    for (i = 0; i < length; i++) {
      int first, second;

      first = (str[i] >> 4) & 0x0f;
      second = str[i] & 0x0f;
      format_buffer[len++] = ((first >= 10)  ? first  + 'W' : first  + '0');
      format_buffer[len++] = ((second >= 10) ? second + 'W' : second + '0');
    }
  } else {
    len += pdfobj_escape_str(format_buffer + len, FORMAT_BUF_SIZE - len, str, length);
  }
  pdf_doc_add_to_page(format_buffer, len);

  text_state.offset += width;
}

void
pdf_dev_init (double dvi2pts, int precision,
	      double width, double height, double hoffset, double voffset)
{
  if (precision < 0 ||
      precision > DEV_PRECISION_MAX)
    WARN("Number of decimal digits out of range [0-%d].",
	 DEV_PRECISION_MAX);

  if (precision < 0) {
    dev_unit.precision = 0;
  } else if (precision > DEV_PRECISION_MAX) {
    dev_unit.precision = DEV_PRECISION_MAX;
  } else {
    dev_unit.precision = precision;
  }
  dev_unit.dvi2pts    = dvi2pts;
  dev_unit.min_bp_val = (long) ROUND(1.0/(ten_pow[dev_unit.precision]*dvi2pts), 1);
  if (dev_unit.min_bp_val < 0)
    dev_unit.min_bp_val = -dev_unit.min_bp_val;

  dev_page.hoffset = hoffset;
  dev_page.voffset = voffset;
  dev_page.height  = height;
  dev_page.width   = width;

  graphics_mode();
  dev_clear_color_stack();
  dev_clear_xform_stack();

  PDF_resource_init();
  Encoding_cache_init();
  AGLmap_cache_init();
  CMap_cache_init();

  Type0Font_cache_init();
  pdf_font_init();
}

void
pdf_dev_close (void)
{
  int i;

  /* Close all fonts */
  for (i = 0; i < num_dev_fonts; i++) {
    RELEASE(dev_font[i].tex_name);
  }
  if (dev_font)
    RELEASE(dev_font);

  /* Release all map entries */
  release_map_record();

  /* Close the various font handlers */
  pdf_font_close();
  Type0Font_cache_close();

  AGLmap_cache_close();   /* After pdf_font_close().        */
  CMap_cache_close();     /* After Type0Font_cache_close(). */
  Encoding_cache_close(); /* After pdf_font_close().        */
  PDF_resource_close();   /* Finally close resources.       */
}

/*
 * BOP, EOP, and FONT section.
 * BOP and EOP manipulate some of the same data structures
 * as the font stuff.
 */ 

/******************************** COLOR ********************************/

#include "colors.h"

static pdf_color *
color_by_name (const char *color_name) 
{
  pdf_color *result = NULL;
  int        i;

  for (i = 0; i < NUM_NAMED_COLORS; i++) {
    if (!strcasecmp(colors_by_name[i].key, color_name)) {
      result = &(colors_by_name[i].value);
      break;
    }
  }
  if (!result)
    ERROR("Color \"%s\" not known.", color_name);

  return result;
}

#if 0
struct pdf_cs_indexed
{
  int   base_id;
  int   num_components;
  void *pallete;
};

/* Use same numbering for safety. */

#define PDF_COLOR_GRAY PDF_COLOR_DEV_GRAY
#define PDF_COLOR_RGB  PDF_COLOR_DEV_RGB
#define PDF_COLOR_CMYK PDF_COLOR_DEV_CMYK

/* CIE-BASED */
#define PDF_COLOR_CAL   1
#define PDF_COLOR_LAB   2

/* SPECIAL */
#define PDF_COLOR_INDEX 3

struct pdf_cs_cal
{
  int    subtype; /* Gray or RGB */

  int    flags;   /* HAVE_GAMMA, HAVE_BLACKPOINT, HAVE_MATRIX */

  double gamma[3];
  double whitepoint[3];
  double blackpoint[3];
  double matrix[9];
}

typedef struct pdf_colorspace
{
  char  *ident;

  int    type;
  void  *data;

  pdf_obj *resource;
  pdf_obj *obj_ref;

} pdf_colorspace;
#endif

/*
 * Having color stack here is wrong.
 * dvi.c or XXX_special should treat this.
 */
#define DEV_COLOR_STACK_MAX 256

static struct {
  int top;
  pdf_color colors[DEV_COLOR_STACK_MAX];
} color_stack;

pdf_color bg_color = {
  PDF_COLOR_DEV_GRAY,
  {1.0, 1.0, 1.0, 1.0}
};

pdf_color default_color = {
  PDF_COLOR_DEV_GRAY,
  {0.0, 0.0, 0.0, 0.0}
};

static void
dev_clear_color_stack (void)
{
  color_stack.top = 0;
}

/*
 * This routine is not a real color matching.
 */
static int
color_match (const pdf_color *color1, const pdf_color *color2)
{
  if (color1->space_id != color2->space_id)
    return 0;

  switch (color1->space_id) {
  case PDF_COLOR_DEV_RGB:
    if (color1->values[0] == color2->values[0] &&
	color1->values[1] == color2->values[1] &&
	color1->values[2] == color2->values[2])
      return 1;
    break;
  case PDF_COLOR_DEV_CMYK:
    if (color1->values[0] == color2->values[0] &&
	color1->values[1] == color2->values[1] &&
	color1->values[2] == color2->values[2] &&
	color1->values[3] == color2->values[3])
      return 1;
    break;
  case PDF_COLOR_DEV_GRAY:
    if (color1->values[0] == color2->values[0])
      return 1;
    break;
  }

  return 0;
}

/*
 * Ignore color migrated to here. This is device's capacity.
 */
static int ignore_colors = 0;

void
dev_ignore_colors (void)
{
  ignore_colors = 1;
}

static void
dev_set_color (const pdf_color *color)
{
  int len = 0;

  switch (color->space_id) {
  case PDF_COLOR_DEV_RGB:
    len = sprintf(format_buffer, " %.2f %.2f %.2f",
		  color->values[0], color->values[1], color->values[2]);
    pdf_doc_add_to_page(format_buffer, len);
    pdf_doc_add_to_page(" rg", 3);
    pdf_doc_add_to_page(format_buffer, len);
    pdf_doc_add_to_page(" RG", 3);
    break;
  case PDF_COLOR_DEV_CMYK:
    len = sprintf(format_buffer, " %.2f %.2f %.2f %.2f",
		  color->values[0], color->values[1], color->values[2], color->values[3]);
    pdf_doc_add_to_page(format_buffer, len);
    pdf_doc_add_to_page(" k", 2);
    pdf_doc_add_to_page(format_buffer, len);
    pdf_doc_add_to_page(" K ", 3);
    break;
  case PDF_COLOR_DEV_GRAY:
    len = sprintf(format_buffer, " %.2f", color->values[0]);
    pdf_doc_add_to_page(format_buffer, len);
    pdf_doc_add_to_page(" g", 2);
    pdf_doc_add_to_page(format_buffer, len);
    pdf_doc_add_to_page(" G", 2);
    break;
  case PDF_COLOR_INVALID:
    ERROR("Invalid color item.");
    break;
  default:
    ERROR("Only RGB/CMYK/Gray currently supported.");
  }
}

#define CURRENTCOLOR() (color_stack.top > 0 ? &(color_stack.colors[color_stack.top-1]) : &default_color)
#define PREVCOLOR()    (color_stack.top > 1 ? &(color_stack.colors[color_stack.top-2]) : &default_color)

void
dev_do_color (void)
{
  if (ignore_colors)
    return;

  dev_set_color(CURRENTCOLOR());
}

static void
fill_background (void)
{
  pdf_rect bbox;
  int len = 0;

  if (bg_color.space_id  == PDF_COLOR_DEV_GRAY &&
      bg_color.values[0] == 1.0)
    return;

  pdf_doc_get_mediabox(pdf_doc_current_page_no(), &bbox);

  format_buffer[len++] = 'q';
  format_buffer[len++] = ' ';
  switch (bg_color.space_id) {
  case PDF_COLOR_DEV_GRAY:
    len += sprintf(format_buffer+len, "%.2f g", bg_color.values[0]);
    break;
  case PDF_COLOR_DEV_RGB:
    len += sprintf(format_buffer+len,
		   "%.2f %.2f %.2f rg",
		   bg_color.values[0], bg_color.values[1], bg_color.values[2]);
    break;
  case PDF_COLOR_DEV_CMYK:
    len += sprintf(format_buffer+len,
		   "%.2f %.2f %.2f %.2f k",
		   bg_color.values[0], bg_color.values[1], bg_color.values[2], bg_color.values[3]);
    break;
  }
  format_buffer[len++] = ' ';
  len += sprintf(format_buffer+len,
		 "%.2f %.2f %.2f %.2f re f",
		 bbox.llx, bbox.lly, bbox.urx, bbox.ury);
  format_buffer[len++] = ' ';
  format_buffer[len++] = 'Q';

  pdf_doc_set_current_page_background(format_buffer, len);
}

void
dev_bg_dev_color (const pdf_color *colorspec)
{
  if (ignore_colors)
    return;

  bg_color.space_id  = colorspec->space_id;
  bg_color.values[0] = colorspec->values[0];
  bg_color.values[1] = colorspec->values[1];
  bg_color.values[2] = colorspec->values[2];
  bg_color.values[3] = colorspec->values[3];
}

void
dev_bg_named_color (const char *color_name)
{
  pdf_color *color;

  if (ignore_colors)
    return;

  color = color_by_name(color_name);
  dev_bg_dev_color(color);
}

void
dev_set_def_dev_color (const pdf_color *colorspec)
{
  if (ignore_colors)
    return;

  default_color.space_id  = colorspec->space_id;
  default_color.values[0] = colorspec->values[0];
  default_color.values[1] = colorspec->values[1];
  default_color.values[2] = colorspec->values[2];
  default_color.values[3] = colorspec->values[3];

  dev_do_color(); /* ??? */
}

void
dev_set_def_named_color (const char *color_name)
{
  pdf_color *color;

  if (ignore_colors)
    return;

  color = color_by_name(color_name);
  dev_set_def_dev_color(color);
}

void
dev_begin_dev_color (const pdf_color *colorspec)
{
  pdf_color *currentcolor, *prevcolor;

  if (ignore_colors)
    return;

  if (color_stack.top >= DEV_COLOR_STACK_MAX) {
    WARN("Exceeded depth of color stack.");
    return;
  }

  prevcolor    = CURRENTCOLOR();
  currentcolor = &(color_stack.colors[color_stack.top]);

  currentcolor->space_id  = colorspec->space_id;
  currentcolor->values[0] = colorspec->values[0];
  currentcolor->values[1] = colorspec->values[1];
  currentcolor->values[2] = colorspec->values[2];
  currentcolor->values[3] = colorspec->values[3];
  color_stack.top += 1;

  /*
   * This may not reselect current color when the color is changed in
   * the pdf:literal special. But users are completely responsible for
   * doing correct thing in that case; any pdf:literal must be enclosed
   * by "q" and "Q" manually.
   */
  if (!color_match(currentcolor, prevcolor)) {
    dev_set_color(currentcolor);
  }
}

void
dev_begin_named_color (const char *color_name)
{
  pdf_color *color;

  if (ignore_colors)
    return;

  color = color_by_name(color_name);
  dev_begin_dev_color(color);
}

void
dev_end_color (void)
{
  pdf_color *prevcolor;

  if (ignore_colors)
    return;

  if (color_stack.top <= 0) {
    WARN("End color with no corresponding begin color.");
  } else {
    prevcolor = PREVCOLOR();
    if (!color_match(CURRENTCOLOR(), prevcolor)) {
      dev_set_color(prevcolor);
    }
    color_stack.top -= 1;
  }
}

/******************************** GSTATE ********************************/

#if 0

/* Implementing this might be hard task... */

typedef struct pdf_gstate
{
  /* Only current transform matrix supported yet. */
  pdf_tmatrix ctm;
} pdf_gstate;

#define DEV_GSTATE_STACK_MAX 256
static struct
{
  int top;
  pdf_gstate gstates[DEV_GSTATE_STACK_MAX];
} gstate_stack;

#endif

static int num_transforms = 0;

static
void dev_clear_xform_stack (void)
{
  num_transforms = 0;
}

void
dev_begin_transform (double xscale, double yscale, double rotate,
		     double x_user, double y_user)
{
  pdf_tmatrix tm;
  double      c, s;
  int         len = 0;

  if (num_transforms >= MAX_TRANSFORMS) {
    WARN("Exceeded depth of transformation stack.");
    return;
  }

  c = cos(rotate);
  s = sin(rotate);
  tm.a =  xscale * c; tm.b =  xscale * s;
  tm.c = -yscale * s; tm.d =  yscale * c;
  tm.e = (1.0 - tm.a) * x_user - tm.c * y_user;
  tm.f = (1.0 - tm.d) * y_user - tm.b * x_user;

  if (fabs(tm.a * tm.d - tm.b * tm.c) < 1.0e-7)
    WARN("Non invertible transformation matrix.");

  format_buffer[len++] = ' ';
  format_buffer[len++] = 'q';
  format_buffer[len++] = ' ';
  len += pdf_sprint_matrix(format_buffer+len, &tm);
  format_buffer[len++] = ' ';
  format_buffer[len++] = 'c';
  format_buffer[len++] = 'm';

  pdf_doc_add_to_page(format_buffer, len);

  num_transforms++;
}

void
dev_end_transform (void)
{
  if (num_transforms == 0) {
    WARN("End transform with no corresponding begin.");
    return;
  }

  pdf_doc_add_to_page(" Q", 2);
  num_transforms--;
  /*
   * Unfortunately, the following two lines are necessary in case
   * of a font or color change inside of the save/restore pair.
   * Anything that was done there must be redone, so in effect,
   * we make no assumptions about what fonts. We act like we are
   * starting a new page.
   */
  dev_reselect_font();
  dev_do_color();
}

int
dev_xform_depth (void)
{
  return num_transforms;
}

void
dev_close_all_xforms (int depth)
{
  if (num_transforms > depth) {
    WARN("Closing pending transformations at end of page/XObject.");
    while (num_transforms > depth) {
      pdf_doc_add_to_page(" Q", 2);
      num_transforms--;
    }
    dev_reselect_font();
    dev_do_color();
  }
}

/*
 * The following routine is here for forms.  Since a form is
 * self-contained, it will need its own Tf command at the
 * beginningg even if it is continuing to set type in the
 * current font.  This routine simply forces reinstantiation
 * of the current font.
 */
void
dev_reselect_font (void)
{
  int i;

  for (i = 0; i < num_dev_fonts; i++) {
    dev_font[i].used_on_this_page = 0;
  }

  text_state.font.id    = -1;
  text_state.font.scale = 0.0;

  text_state.matrix.slant  = 0.0;
  text_state.matrix.extend = 1.0;
  text_state.matrix.rotate = TEXT_WMODE_HH;
}

static void
dev_set_origin (double orig_x, double orig_y)
{
  pdf_tmatrix  ctm;
  double       scale;
  int          len;

  scale = dvi_tell_mag() * pdf_dev_scale();
  ctm.a = scale; ctm.b = 0.0  ;
  ctm.c = 0.0  ; ctm.d = scale;
  ctm.e = orig_x;
  ctm.f = orig_y;

  len = 0;
  work_buffer[len++] = ' ';
  work_buffer[len++] = 'q';
  work_buffer[len++] = ' ';
  len += pdf_sprint_matrix(work_buffer+len, &ctm);
  work_buffer[len++] = ' ';
  work_buffer[len++] = 'c';
  work_buffer[len++] = 'm';
  work_buffer[len++] = ' ';

  pdf_doc_add_to_page(work_buffer, len);
}


void
dev_bop (void)
{
  dev_page.readonly = 0;

  pdf_doc_new_page();
  dev_set_origin(dev_page_hoffset(), dev_page_height() - dev_page_voffset());

  graphics_mode();

  text_state.mp_mode      = 0;
  text_state.force_reset  = 0;
  text_state.is_mb        = 0;

  dev_reselect_font();
  dev_do_color();
}

void
dev_eop (void)
{
  graphics_mode();
  dev_close_all_xforms(0);
  pdf_doc_add_to_page(" Q", 2);
  if (!ignore_colors)
    fill_background();
  pdf_doc_finish_page();
  /* Finish any pending PS specials */
  mp_eop_cleanup();
  dev_page.readonly = 0;
}

int
dev_locate_font (const char *tex_name, spt_t ptsize, int map_id)
{
  int id, this_font;

  if (ptsize == 0)
    ERROR("dev_locate_font() called with the zero ptsize.");

  for (id = 0; id < num_dev_fonts; id++) {
    if (!strcmp(tex_name, dev_font[id].tex_name)) {
      /*
       * Must match in name and size to resolve to the same device font.
       * Scaleable fonts must match in name; however, this routine must
       * return a different id if the ptsize is different.
       */
      if (ptsize == dev_font[id].sptsize) {
        return id;
      } else if (dev_font[id].format != FONTTYPE_BITMAP) {
        break;
      }
    }
  }

  /*
   * Make sure we have room for a new one, even though we may not
   * actually create one.
   */
  if (num_dev_fonts >= max_device_fonts) {
    max_device_fonts += MAX_FONTS;
    dev_font = RENEW(dev_font, max_device_fonts, struct dev_font);
  }
  this_font = num_dev_fonts;

  if (id < num_dev_fonts) {
    /*
     * A previously existing physical font can be used; however, this
     * routine must return a distinct ID if the ptsizes are different.
     * Copy the information from the previous record to the new record.
     */
    strcpy(dev_font[this_font].short_name, dev_font[id].short_name);
    /*
     * The value in used_on_this_page will be incorrect if the font
     * has already been used on a page in a different point size.
     * It's too hard to do right.  The only negative consequence is
     * that there will be an attempt to add the resource to the page
     * resource dict. However, the second attempt will do nothing.
     */
    dev_font[this_font].used_on_this_page = 0;
    dev_font[this_font].tex_name = NEW (strlen(tex_name)+1, char);
    strcpy(dev_font[this_font].tex_name, tex_name);
    dev_font[this_font].sptsize  = ptsize;
    /*
     * These two fonts are treated as having the same physical "used_chars".
     */
    dev_font[this_font].used_chars = dev_font[id].used_chars;
    dev_font[this_font].resource   = dev_font[id].resource;
    dev_font[this_font].format     = dev_font[id].format;
    dev_font[this_font].font_id    = dev_font[id].font_id;
    dev_font[this_font].extend = dev_font[id].extend;
    dev_font[this_font].slant  = dev_font[id].slant;
    dev_font[this_font].remap  = dev_font[id].remap;
    dev_font[this_font].mapc   = dev_font[id].mapc;
    dev_font[this_font].wmode  = dev_font[id].wmode;
    dev_font[this_font].cmap   = dev_font[id].cmap;
  } else {
    /*
     * There is no physical font we can use.
     */
    int font_id = -1, font_format = -1, tfm_id  = -1;
    int encoding_id = -1, cmap_id = -1;
    int wmode = 0, remap = 0, mapc = -1;
    double extend, slant, design_size;
    char *font_name, *enc_name;
    char  short_name[7];

    /*
     * Get appropriate info from map file. (PK fonts at two different
     * point sizes would be looked up twice unecessarily.)
     */
    font_name = (map_id < 0 ? (char *)tex_name : fontmap_font_name(map_id));
    enc_name = fontmap_enc_name(map_id);
    extend   = fontmap_extend(map_id);
    slant    = fontmap_slant(map_id);
    remap    = fontmap_remap(map_id);
    mapc     = fontmap_mapc (map_id);

    if (verbose > 1) {
      MESG("\n");
      if (map_id >= 0) {
	MESG("fontmap: %s -> %s", tex_name, font_name);
	if (enc_name != NULL) MESG("(%s)", enc_name);
	if (extend   != 1.0)  MESG("[extend=%g]", extend);
	if (slant    != 0.0)  MESG("[slant=%g]", slant);
	if (remap != 0)  MESG("[remap]");
	if (mapc  >= 0)  MESG("[map=<%02x>]", mapc);
	MESG("\n");
      } else {
	MESG("fontmap: %s (no map)\n", tex_name);
      }
    }

    /*
     * If this font has an encoding specified on the record,
     * check whether it is CMap, and otherwise get its encoding ID.
     * We assume that an encoding file without extension is a CMap.
     * The default encoding_id is -1.
     */
    if (enc_name) {
      if (strstr(enc_name, ".enc") == NULL) {
        cmap_id = CMap_cache_find(enc_name);
	if (cmap_id >= 0) {
	  CMap *cmap;
	  int   cmap_type;
	  int   minbytein;

	  cmap      = CMap_cache_get(cmap_id);
	  cmap_type = CMap_get_type(cmap);
	  minbytein = CMap_get_profile(cmap, CMAP_PROF_TYPE_INBYTES_MIN);
	  /*
	   * Check for output encoding.
	   */
	  if (cmap_type != CMAP_TYPE_IDENTITY    &&
	      cmap_type != CMAP_TYPE_CODE_TO_CID &&
	      cmap_type != CMAP_TYPE_TO_UNICODE) {
	    WARN("Only 16-bit encoding supported for output encoding.");
	  }
	  /*
	   * Turn on map option.
	   */
	  if (minbytein == 2 && mapc < 0) {
	    if (verbose > 3) {
	      MESG("\n");
	      MESG("Input encoding \"%s\" requires at least 2 bytes.", CMap_get_name(cmap));
	      MESG("\n");
	      MESG("The -m <00> option will be assumed for \"%s\".", font_name);
	      MESG("\n");
	    }
	    mapc = 0;
	  }
	}
      }
      if (cmap_id < 0) {
        encoding_id = Encoding_cache_find(enc_name);
	if (encoding_id < 0) {
	  ERROR("Could not find encoding file \"%s\".", enc_name);
	}
      }
    }

    /* This should be removed from here. */
    tfm_id = tfm_open(tex_name, 0);
    design_size = (tfm_id >= 0) ? tfm_get_design_size(tfm_id) : 0.0;

    /*
     * We assume, for now that we will find this as a physical font,
     * as opposed to a vf, so we need a device name to tell the
     * lower-level routines what we want this to be called.  We'll
     * blast this name away later if we don't need it.
     */
    short_name[0] = 'F';
    p_itoa(num_phys_fonts+1, short_name+1); /* NULL terminated here */

    if (cmap_id >= 0) {
      /*
       * Composite Font
       */
      if ((font_id = type0_font(font_name, tfm_id,
				short_name, cmap_id, remap)) >= 0) {
	font_format = FONTTYPE_COMPOSITE;
        wmode       = type0_font_wmode(font_id);
	remap       = 0;
      } else {
	return -1;
      }
    } else {
      /*
       * Simple Font - always embed.
       */
      font_id = pdf_font_findfont(font_name, encoding_id,
				  ptsize*(dev_unit.dvi2pts), design_size, 1);
      if (font_id >= 0) {
	if (pdf_font_get_fonttype(font_id) == PDFFONT_FONTTYPE_TYPE3)
	  font_format = FONTTYPE_BITMAP;
	else
	  font_format = FONTTYPE_SIMPLE;
      } else {
	return -1;
      }
    }

    /*
     * This is a new physical font and we found a physical font we can use.
     */
    strcpy(dev_font[this_font].short_name, short_name);
    dev_font[this_font].used_on_this_page = 0;

    dev_font[this_font].tex_name = NEW (strlen(tex_name)+1, char);
    strcpy(dev_font[this_font].tex_name, tex_name);
    dev_font[this_font].sptsize  = ptsize;

    dev_font[this_font].format   = font_format;
    dev_font[this_font].font_id  = font_id;
    dev_font[this_font].resource = NULL;
    dev_font[this_font].wmode    = wmode;
    dev_font[this_font].cmap     = cmap_id;

    dev_font[this_font].used_chars = NULL;

#if 0
    if (wmode) {
      if (slant  != 0.0)
	WARN("Don't know how to \"slant\" vertical fonts.");
      if (extend != 1.0)
	WARN("Don't know how to \"extend\" vertical fonts.");
      slant  = 0.0;
      extend = 1.0;
    }
#endif
    dev_font[this_font].slant  = slant;
    dev_font[this_font].extend = extend;
    dev_font[this_font].remap  = remap;
    dev_font[this_font].mapc   = mapc;
  }

  num_phys_fonts++;
  num_dev_fonts++;

  return this_font;
}
  

/* This does not remember current stroking width. */
static int
dev_sprint_line (char *buf, spt_t width, spt_t p0_x, spt_t p0_y, spt_t p1_x, spt_t p1_y)
{
  int    len = 0;
  double w;

  w = width * dev_unit.dvi2pts;

  len += p_dtoa(w, MIN(dev_unit.precision+1, DEV_PRECISION_MAX), buf+len);
  buf[len++] = ' ';
  buf[len++] = 'w';
  buf[len++] = ' ';
  len += dev_sprint_bp(buf+len, p0_x, NULL);
  buf[len++] = ' ';
  len += dev_sprint_bp(buf+len, p0_y, NULL);
  buf[len++] = ' ';
  buf[len++] = 'm';
  buf[len++] = ' ';
  len += dev_sprint_bp(buf+len, p1_x, NULL);
  buf[len++] = ' ';
  len += dev_sprint_bp(buf+len, p1_y, NULL);
  buf[len++] = ' ';
  buf[len++] = 'l';
  buf[len++] = ' ';
  buf[len++] = 'S';

  return len;
}

/* Not optimized. */
#define PDF_LINE_THICKNESS_MAX 5.0
void
dev_rule (spt_t xpos, spt_t ypos, spt_t width, spt_t height)
{
  int    len = 0;
  double width_in_bp;

  graphics_mode();

  format_buffer[len++] = ' ';
  /* Don't use too thick line. */
  width_in_bp = ((width < height) ? width : height) * dev_unit.dvi2pts;
  if (width_in_bp < 0.0 || /* Shouldn't happen */
      width_in_bp > PDF_LINE_THICKNESS_MAX) {
    pdf_rect rect;

    rect.llx =  dev_unit.dvi2pts * xpos;
    rect.lly =  dev_unit.dvi2pts * ypos;
    rect.urx =  dev_unit.dvi2pts * width;
    rect.ury =  dev_unit.dvi2pts * height;
    len += pdf_sprint_rect(format_buffer+len, &rect);
    format_buffer[len++] = ' ';
    format_buffer[len++] = 'r';
    format_buffer[len++] = 'e';
    format_buffer[len++] = ' ';
    format_buffer[len++] = 'f';
  } else {
    if (width > height) {
      /* NOTE:
       *  A line width of 0 denotes the thinnest line that can be rendered at
       *  device resolution. See, PDF Reference Manual 4th ed., sec. 4.3.2,
       *  "Details of Graphics State Parameters", p. 185.
       */
      if (height < dev_unit.min_bp_val) {
	WARN("Too thin line: height=%ld (%g bp)", height, width_in_bp);
	WARN("Please consider using \"-d\" option.");
      }
      len += dev_sprint_line(format_buffer+len,
			     height,
			     xpos,
			     ypos + height/2,
			     xpos + width,
			     ypos + height/2);
    } else {
      if (width < dev_unit.min_bp_val) {
	WARN("Too thin line: width=%ld (%g bp)", width, width_in_bp);
	WARN("Please consider using \"-d\" option.");
      }
      len += dev_sprint_line(format_buffer+len,
			     width,
			     xpos + width/2,
			     ypos,
			     xpos + width/2,
			     ypos + height);
    }
  }
  pdf_doc_add_to_page(format_buffer, len);
}

/*
 * Only DVI \special related routines using this.
 *
 *  dvi_dev_xpos()*dvi_tell_mag() + dev_page_hoffset()
 *  dvi_dev_xpos()*dvi_tell_mag() + dev_page_voffset()
 *
 * or
 *
 *  dev_phys_pos(dvi_xpos()*dvi_tell_mag(), ...)
 *
 * is enough...
 */
/*
 * The following routines tell the coordinates in true Adobe
 * points with the coordinate system having its origin at the
 * bottom left of the page.
 */

double
dev_phys_x (void)
{
  return dev_page_hoffset() + dvi_dev_xpos() * dvi_tell_mag();
}

double
dev_phys_y (void)
{
  return dev_page_height() - dev_page_voffset() + dvi_dev_ypos() * dvi_tell_mag();
}

static int
src_special (char *buffer, UNSIGNED_QUAD size)
{
  char *start = buffer;
  char *end   = buffer + size;

  skip_white(&start, end);
  if ((start < end - 3 && !strncmp("src:", start, 4)) ||
      (start < end - 2 && !strncmp("om:",  start, 3)))
    return 1;
 
  return 0;
}

/*
 * This shouldn't be placed here.
 */
void
dev_do_special (void *buffer, UNSIGNED_QUAD size, spt_t x_user, spt_t y_user)
{
  double dev_xuser, dev_yuser;

  graphics_mode();

  dev_xuser =  x_user * dev_unit.dvi2pts;
  dev_yuser = -y_user * dev_unit.dvi2pts;
  if (!pdf_parse_special (buffer, size, dev_xuser, dev_yuser) &&
      !tpic_parse_special(buffer, size, dev_xuser, dev_yuser) &&
      !htex_parse_special(buffer, size) &&
      !color_special (buffer, size) &&
      !ps_parse_special (buffer, size, dev_xuser, dev_yuser) &&
      !src_special (buffer, size)) {
    WARN("Unrecognized special ignored.");
    if (verbose)
      dump(buffer, ((char *)buffer)+size);
  }
}

static unsigned dvi_stack_depth  =  0;
static int      dvi_tagged_depth = -1;
static unsigned char link_annot  =  1;

void
dev_link_annot (unsigned char flag)
{
  link_annot = flag;
}

void
dev_stack_depth (unsigned int depth)
{
  /* If decreasing below tagged_depth */
  if (link_annot && 
      dvi_stack_depth == dvi_tagged_depth &&
      depth == dvi_tagged_depth - 1) {
  /*
   * See if this appears to be the end of a "logical unit"
   * that's been broken.  If so, flush the logical unit.
   */
    pdf_doc_flush_annot();
  }
  dvi_stack_depth = depth;
}

/*
 * The following routines setup and tear down a callback at a
 * certain stack depth. This is used to handle broken (linewise)
 * links.
 */
void
dev_tag_depth (void)
{
  dvi_tagged_depth = dvi_stack_depth;
  dvi_compute_boxes (1);
}

void
dev_untag_depth (void)
{
  dvi_tagged_depth = -1;
  dvi_compute_boxes (0);
}

void
dev_expand_box (spt_t width, spt_t height, spt_t depth)
{
  double phys_width, phys_height, phys_depth, scale;

  if (link_annot && dvi_stack_depth >= dvi_tagged_depth) {
    scale = dev_unit.dvi2pts * dvi_tell_mag();
    phys_width  = scale * width;
    phys_height = scale * height;
    phys_depth  = scale * depth;
    if (text_state.dir != 1) {
      pdf_doc_expand_box (dev_phys_x(),
			  dev_phys_y() - phys_depth,
			  dev_phys_x() + phys_width,
			  dev_phys_y() + phys_height);
    } else {
      pdf_doc_expand_box (dev_phys_x() - phys_depth,
			  dev_phys_y() - phys_width,
			  dev_phys_x() + phys_height,
			  dev_phys_y());
    }
  }
}

int
dev_wmode (void)
{
  return text_state.dir;
}

void
dev_set_wmode (int text_dir)
{
  int text_rotate, font_wmode;

  font_wmode = (text_state.font.id < 0) ? 0 : dev_font[text_state.font.id].wmode;

  text_dir = (text_dir == 1) ? 1 : 0;
  if (font_wmode == 0 && text_state.dir == 0)
    text_rotate = TEXT_WMODE_HH;
  else if (font_wmode == 1 && text_state.dir == 1)
    text_rotate = TEXT_WMODE_VV;
  else if (font_wmode == 1 && text_dir == 0)
    text_rotate = TEXT_WMODE_VH;
  else if (font_wmode == 0 && text_dir == 1)
    text_rotate = TEXT_WMODE_HV;
  else
    text_rotate = TEXT_WMODE_HH;

  if (text_state.font.id >= 0 &&
      ANGLE_CHANGES(text_rotate, text_state.matrix.rotate))
    text_state.force_reset = 1;

  text_state.matrix.rotate = text_rotate;
  text_state.dir = text_dir;
}

/* MetaPost mode (or PostScript mode) */
void
dev_start_mp_mode (void)
{
  text_state.dir_save = text_state.dir;
  text_state.dir      = 0;

  text_state.matrix.slant  = 0.0;
  text_state.matrix.extend = 1.0;
  text_state.matrix.rotate = TEXT_WMODE_HH;

  text_state.mp_mode  = 1;
}

void
dev_end_mp_mode (void)
{
  text_state.dir = text_state.dir_save;
  text_state.mp_mode = 0;
}
