#ifndef _ECS_CHAMP_VEC_INT_H_
#define _ECS_CHAMP_VEC_INT_H_

/*============================================================================
 *  Prototypes des fonctions
 *   associées à la structure `ecs_champ_t' décrivant un champ
 *   et liées à la structure `ecs_vec_int_t'
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilité
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"
#include "ecs_tab_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_vec_int.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_champ.h"



/*============================================================================
 *                       Prototypes de fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction construisant un vecteur `ecs_vec_int_t' associé au champ donné
 *  Fonction construisant un vecteur `ecs_vec_int_t' associé au champ donné
 *
 *  Si les tables sont REGLEES, les tableaux sont construits
 *
 *  Cette fonction doit être utilisée conjointement avec :
 *  - soit `ecs_champ__transfere_vec_int()'
 *                               si le contenu du vecteur   a changé
 *  - soit `ecs_champ__libere_vec_int()'
 *                               si le contenu du vecteur n'a pas été modifié
 *  Ceci afin de reconstuire, si nécessaire, les tables REGLEES
 *----------------------------------------------------------------------------*/

ecs_vec_int_t * ecs_champ__initialise_vec_int
(
 ecs_champ_t *this_champ
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui transfère le contenu d'un vecteur `ecs_vec_int_t'
 *   dans les tables d'un champ donné
 *
 *  Les tables sont transformées en REGLES si possible
 *
 *  La structure `ecs_vec_int_t' est libérée
 *----------------------------------------------------------------------------*/

void ecs_champ__transfere_vec_int
(
 ecs_champ_t    *this_champ,  /* <-> Champ dans lequel on transfère `vec_int' */
 ecs_vec_int_t  *vec_int      /* <-> Vecteur `ecs_vec_int_t' a transférer     */
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui libère le tableau d'une table REGLEE qui a été
 *   precedémment developpé par l'appel de `ecs_champ__initialise_vec_int()'
 *  et qui détruit le vecteur `ecs_vec_int_t' qui a été
 *   précédemment créé      par l'appel de `ecs_champ__initialise_vec_int()'
 *----------------------------------------------------------------------------*/

void ecs_champ__libere_vec_int
(
 ecs_champ_t    * this_champ,                 /* <-> Champ associé au vecteur */
 ecs_vec_int_t  * vec_int                     /* <-> Vecteur à détruire       */
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui crée un champ à partir
 *   du contenu d'un vecteur `ecs_vec_int_t'
 *
 *  Seuls le nom du champ et les tables du champ sont remplis
 *   (les tables sont transformées en REGLES si possible)
 *
 *  Les autres membres de la structure prennent leur valeur d'initialisation
 *
 *  La structure `ecs_vec_int_t' est libérée
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ__init_avec_vec_int
(
       ecs_vec_int_t  *      vec_int,             /* --> Vecteur à transferer */
 const char           *const nom
) ;


/*----------------------------------------------------------------------------
 *  Fonction réalisant la transformation d'un champ
 *   en appliquant directement le vecteur de transformation donné
 *   sur ses éléments
 *----------------------------------------------------------------------------*/

void ecs_champ__transforme_val
(
       ecs_champ_t    *const this_champ,
       size_t                nbr_elt_new,
 const ecs_tab_int_t         vect_transf
) ;


/*----------------------------------------------------------------------------
 *  Fonction réalisant la transformation d'un champ
 *   en appliquant directement le vecteur de transformation donné
 *   sur ses valeurs
 *----------------------------------------------------------------------------*/

void ecs_champ__transforme_pos
(
       ecs_champ_t    *const this_champ,
       size_t                nbr_elt_ref,
 const ecs_tab_int_t         vect_transf
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui incrémente les valeurs d'un champ donné
 *   d'une constante donnée
 *----------------------------------------------------------------------------*/

void ecs_champ__incremente_val
(
 ecs_champ_t  * this_champ,
 ecs_int_t      increment
) ;


/*----------------------------------------------------------------------------
 *  Fonction réalisant la transformation d'un champ
 *   en appliquant directement le vecteur de transformation donné
 *   sur les valeurs associées à ses éléments
 *----------------------------------------------------------------------------*/

void ecs_champ__renumerote
(
       ecs_champ_t    *const this_champ ,
 const ecs_tab_int_t         vect_transf
) ;


void ecs_champ__renumerote_et_sgn
(
       ecs_champ_t    *const this_champ  ,
 const ecs_tab_int_t         vect_transf ,
 const ecs_tab_int_t         signe_elt
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui retourne le tableau des valeurs du champ donné,
 *   dimensionné au nombre de références distinctes du champ,
 *   (valeurs qui correspondent a des références)
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_champ__ret_reference
(
 ecs_champ_t  *const this_champ
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui remplace les références à des éléments
 *  en des références à d'autres éléments liés aux premiers
 *  par un champ de type "définition"
 *----------------------------------------------------------------------------*/

void ecs_champ__remplace_ref
(
 ecs_champ_t *const champ_rep ,
 ecs_champ_t *const champ_def
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui construit la  table de connectivité inverse
 *  "sous-éléments -> éléments"
 *----------------------------------------------------------------------------*/

ecs_champ_t * ecs_champ__inverse
(
 ecs_champ_t  *const champ_def_elt ,
 size_t              nbr_sselt
) ;



#endif /* _ECS_CHAMP_VEC_INT_H_ */
