/*
 * EveryBuddy 
 *
 * Copyright (C) 1999, Torrey Searle <tsearle@uci.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <ltdl.h>
#include <sys/types.h>
#include <dirent.h> /* Routines to read directories to find modules */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "globals.h"
#include "value_pair.h"
#include "config.h"
#include "service.h"
#include "plugin.h"
#include "plugin_api.h"
#include "dialog.h"
#include "service.h"
#include "nomodule.h"


char *PLUGIN_TYPE_TXT[]={"SERVICE", "UTILITY", "SOUND", "LOG", "GUI", "UNKNOWN"};
char *PLUGIN_STATUS_TXT[]={"Not Loaded", "Loaded", "Cannot Load"};

PLUGIN_INFO Plugin_Cannot_Load = {PLUGIN_UNKNOWN, "Unkown", "Unknown", "Unknown", "Unknown", NULL, NULL};

gint compare_plugin_loaded_service(gconstpointer a, gconstpointer b) {
	const eb_PLUGIN_INFO *epi=a;
	if(epi->service && epi->status==PLUGIN_LOADED && !strcmp( epi->service, (char *)b))
		return(0);
	return(1);
}

gint compare_plugin_name(gconstpointer a, gconstpointer b) {
	const eb_PLUGIN_INFO *epi=a;
	if(epi->name && !strcmp( epi->name, (char *)b))
		return(0);
	return(1);
}

gint compare_plugin_handle(gconstpointer a, gconstpointer b) {
	const eb_PLUGIN_INFO *epi=a;
	if( epi->Module && epi->Module == (lt_dlhandle )b)
		return(0);
	return(1);
}


eb_PLUGIN_INFO *FindLoadedPluginByService(char *service)
{
	GList *plugins=GetPref(EB_PLUGIN_LIST);
	GList *PluginData = g_list_find_custom(plugins, service, compare_plugin_loaded_service);
	if(PluginData)
		return(PluginData->data);
	return(NULL);
}

eb_PLUGIN_INFO *FindPluginByName(char *name)
{
	GList *plugins=GetPref(EB_PLUGIN_LIST);
	GList *PluginData = g_list_find_custom(plugins, name, compare_plugin_name);
	if(PluginData)
		return(PluginData->data);
	return(NULL);
}

eb_PLUGIN_INFO *FindPluginByHandle(lt_dlhandle *Module)
{
	GList *plugins=GetPref(EB_PLUGIN_LIST);
	GList *PluginData = g_list_find_custom(plugins, Module, compare_plugin_handle);
	if(PluginData)
		return(PluginData->data);
	return(NULL);
}

/* Will add/update info about a plugin */
void SetPluginInfo(PLUGIN_INFO *pi, char *name, lt_dlhandle Module, PLUGIN_STATUS status, const char *status_desc, char *service, gboolean force)
{
	GList *plugins=NULL;
	eb_PLUGIN_INFO *epi=NULL;

	epi=FindPluginByName(name);
	if(!epi) {
		epi=g_new0(eb_PLUGIN_INFO, 1);
		plugins=GetPref(EB_PLUGIN_LIST);
		plugins=g_list_append(plugins, epi);
		SetPref(EB_PLUGIN_LIST, plugins);
	}
	else if(force==TRUE || epi->status!=PLUGIN_LOADED) {
		if(epi->service)
			free(epi->service);
		free(epi->name);
		free(epi->pi.brief_desc);
		free(epi->pi.full_desc);
		free(epi->pi.version);
		free(epi->pi.date);
	}
	else 	/* A plugin is already succesfully load */
		return;
	epi->status=status;
	epi->status_desc=status_desc;
	if(!pi)
		pi=&Plugin_Cannot_Load;
	epi->pi.type=pi->type;
	epi->pi.brief_desc=strdup(pi->brief_desc);
	epi->pi.full_desc=strdup(pi->full_desc);
	epi->pi.version=strdup(pi->version);
	epi->pi.date=strdup(pi->date);
	epi->pi.init=pi->init;
	epi->pi.finish=pi->finish;
	epi->pi.prefs=pi->prefs;
	epi->name=strdup(name);
	if(service)
		epi->service=strdup(service);
	epi->Module=Module;
}

/* Find names which end in .la, the expected module extension */
int select_module_entry(const struct dirent *dent) {
	int len=0;
	char *ext;

	len=strlen(dent->d_name);
	if(len<4)
	   return(0);
	ext=(char *)dent->d_name;
	ext+=(len-3);
	eb_debug(DBG_CORE, "select_module_entry: %s[%s]\n", dent->d_name, ext);
	if(!strncmp(ext, ".la", 3))
	   return(1);
	return(0);
}

int unload_module(eb_PLUGIN_INFO *epi)
{
	int error=0;
	char buf[1024];

	eb_debug(DBG_CORE, ">Unloading plugin %s\n", epi->name);
	/* This is a service plugin, special handling required */
	if(epi->pi.finish) {
		eb_debug(DBG_CORE, "Calling plugins finish function\n");
		error=epi->pi.finish();
		if(error) {
			sprintf(buf, "Unable to unload plugin %s, still in use?\n", epi->name);
			do_error_dialog(buf, "Error");
			eb_debug(DBG_CORE, "<Plugin failed to unload\n");
			return(-1);
		}
	}
	if(epi->service) {
		struct service SERVICE_INFO = { strdup(epi->service), -1, FALSE, FALSE, FALSE, FALSE, NULL };

		SERVICE_INFO.sc=eb_nomodule_query_callbacks();
		add_service(&SERVICE_INFO);
	}
	epi->status=PLUGIN_NOT_LOADED;
	epi->pi.prefs=NULL;
	eb_debug(DBG_CORE, "Closing plugin\n");
	if(lt_dlclose(epi->Module)) {
		fprintf(stderr, "Error closing plugin: %s\n", lt_dlerror());
	}
	eb_debug(DBG_CORE, "<Plugin unloaded\n");
	return(0);

}

int load_module(char *path, char *name)
{
	char full_path[1024];
	lt_dlhandle Module;
	PLUGIN_INFO *plugin_info=NULL;
	eb_PLUGIN_INFO *epi=NULL;

	sprintf(full_path, "%s/%s", path, name);
	eb_debug(DBG_CORE, "Opening module: %s\n", full_path);
	Module = lt_dlopen(full_path);
	eb_debug(DBG_CORE, "Module: %p\n", Module);

	/* Find out if this plugin is already loaded */
	if(!Module) {
		/* Only update status on a plugin that is not already loaded */
		SetPluginInfo(NULL, full_path, NULL, PLUGIN_CANNOT_LOAD, lt_dlerror(), NULL, FALSE);
		return(-1);
	}
	plugin_info = (PLUGIN_INFO *)lt_dlsym(Module, "plugin_info");
	if(!plugin_info) {
		lt_dlclose(Module);
		/* Only update status on a plugin that is not already loaded */
		SetPluginInfo(NULL, full_path, NULL, PLUGIN_CANNOT_LOAD, "Cannot resolve symbol plugin_info", NULL, FALSE);
		return(-1);
	}
	epi=FindPluginByName(full_path);
	if(epi && epi->status==PLUGIN_LOADED) {
		lt_dlclose(Module);
		eb_debug(DBG_CORE, "Not loading already loaded module %s\n", name);
		return(-1);
	}
	switch(plugin_info->type) {
	case PLUGIN_SERVICE:
		load_service_plugin(Module, plugin_info, full_path);
		break;
	case PLUGIN_UTILITY:
		load_utility_plugin(Module, plugin_info, full_path);
		break;
	case PLUGIN_SOUND:
		load_sound_plugin(Module, plugin_info, full_path);
		break;
	case PLUGIN_LOG:
		load_log_plugin(Module, plugin_info, full_path);
		break;
	case PLUGIN_GUI:
		load_gui_plugin(Module, plugin_info, full_path);
		break;
	default:
		break;
	}
	return(0);
}

/* This is really a modules loader now */
void load_modules()
{
	struct dirent **namelist=NULL;
	char buf[1024], *modules_path=NULL, *cur_path=NULL;
	char *tok_buf=NULL, *tok_buf_old=NULL;
	int n=0, success=0;

	eb_debug(DBG_CORE, ">Entering\n");
	modules_path=g_strdup(cGetLocalPref("modules_path"));
	tok_buf = g_new0(char, strlen(modules_path)+1);
	/* Save the old pointer, because strtok_r will change it */
	tok_buf_old=tok_buf;
	lt_dlinit();
	lt_dlsetsearchpath(modules_path);

	/* Use a thread-safe strtok */
	cur_path=strtok_r(modules_path, ":", &tok_buf);
	if(!cur_path)
		cur_path=MODULE_DIR;
	do {
		n=scandir(cur_path, &namelist, select_module_entry, alphasort);
		if(n<0) {
	    	sprintf(buf, "Looking for modules in %s", cur_path);
	    	perror(buf);
			continue;
		}
		else if(n == 0) {
	    	eb_debug(DBG_CORE, "<No modules found in %s, returning.\n", cur_path);
			continue;
		}
		while(n--) {
			success = load_module(cur_path, namelist[n]->d_name);


			free(namelist[n]);
		}
		eb_debug(DBG_CORE, "Freeing namelist\n");
		free(namelist);
	} while((cur_path=strtok_r(NULL, ":", &tok_buf)));
	g_free(modules_path);
	g_free(tok_buf_old);
	eb_debug(DBG_CORE, "Adding idle_check\n");
	add_idle_check();
	eb_debug(DBG_CORE, "<End services_init\n");
}

int load_service_plugin(lt_dlhandle Module, PLUGIN_INFO *info, char *name)
{
	struct service *Service_Info=NULL;
	struct service_callbacks *(*query_callbacks)();
	int service_id=-1;
	eb_PLUGIN_INFO *epi=NULL;

	Service_Info = lt_dlsym(Module, "SERVICE_INFO");
	eb_debug(DBG_CORE, "SERVICE_INFO: %p\n", Service_Info);
	if(!Service_Info) {
		SetPluginInfo(info, name, NULL, PLUGIN_CANNOT_LOAD, "Unable to resolve symbol SERVICE_INFO", NULL, FALSE);
		lt_dlclose(Module);
		return(-1);
	}
	/* Don't load this module if there's a service of this type already loaded */
	epi=FindLoadedPluginByService(Service_Info->name);
	if(epi && epi->status==PLUGIN_LOADED) {
		fprintf(stderr, "Not loading module %s, a module for that service is already loaded!\n", name);
		SetPluginInfo(info, name, NULL, PLUGIN_CANNOT_LOAD, "Service provided by an already loaded plugin", Service_Info->name, FALSE);
		lt_dlclose(Module);
		return(-1);
	}
	/* No more hard-coded service_id numbers */
	query_callbacks = lt_dlsym(Module, "query_callbacks");
	if(!query_callbacks) {
		SetPluginInfo(info, name, NULL, PLUGIN_CANNOT_LOAD, "Unable to resolve symbol query_callbacks", Service_Info->name, FALSE);
		lt_dlclose(Module);
		return(-1);
	}
	if(info->init) {
		eb_debug(DBG_CORE, "Executing init for %s\n", info->brief_desc);
		info->init();
	}
	Service_Info->sc=query_callbacks();
	/* The callbacks are defined by the SERVICE_INFO struct in each module */
	service_id = add_service(Service_Info);
	SetPluginInfo(info, name, Module, PLUGIN_LOADED, "", Service_Info->name, TRUE);
	eb_debug(DBG_CORE, "Added module:%s, service: %s\n", name, eb_services[service_id].name);
	eb_debug(DBG_CORE, "eb_services[%i].sc->read_local_account_config: %p\n", service_id, eb_services[service_id].sc->read_local_account_config);
	return(0);
}

int load_utility_plugin(lt_dlhandle Module, PLUGIN_INFO *info, char *name)
{
	char buf[1024];
	GList *user_prefs=NULL;

	eb_debug(DBG_CORE, ">\n");
	if(!info->init) {
		SetPluginInfo(info, name, NULL, PLUGIN_CANNOT_LOAD, "No init function defined", NULL, FALSE);
		lt_dlclose(Module);
		sprintf(buf, "init function not defined for utility module %s, unloading module\n", name);
		do_error_dialog(buf, "Warning");
		return(-1);
	}
	eb_debug(DBG_CORE, "Executing init for %s\n", info->brief_desc);
	info->init();
	if(info->prefs) {
		user_prefs=GetPref(name);
		if(user_prefs) {
			eb_update_from_value_pair(info->prefs, user_prefs);
		}
		eb_debug(DBG_MOD, "prefs name: %s\n", info->prefs->widget.entry.name);
	}
	SetPluginInfo(info, name, Module, PLUGIN_LOADED, "", NULL, TRUE);
	eb_debug(DBG_CORE, "<\n");
	return(0);
}

int load_log_plugin(lt_dlhandle Module, PLUGIN_INFO *info, char *name)
{
	return(1);
}

int load_sound_plugin(lt_dlhandle Module, PLUGIN_INFO *info, char *name)
{
	return(1);
}

int load_gui_plugin(lt_dlhandle Module, PLUGIN_INFO *info, char *name)
{
	return(1);
}

/* Make sure that all the plugin_api accessible menus are initialized */
int init_menu(char *menu_name, menu_func redraw_menu, ebmType type)
{
	menu_data *md=NULL;

	md=g_new0(menu_data, 1);
	md->menu_items=NULL;
	md->redraw_menu=redraw_menu;
	md->type=type;
	SetPref(menu_name, md);
	return(0);
}

/* Set up information about how menus are redrawn and what kind of data should be sent to callbacks */
int init_menus()
{
	init_menu(EB_IMPORT_MENU, rebuild_import_menu, ebmIMPORTDATA);
	/* The chat window menu is dynamically redrawn */
	init_menu(EB_CHAT_WINDOW_MENU, NULL, ebmCONTACTDATA);
	return(0);
}
