/*
 *   (C) Copyright IBM Corp. 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: lvm2
 * File: evms2/engine/plugins/lvm2/lvm2.h
 */

#include "vgda.h"
#include "metadata.h"

#define my_plugin_record	(&lvm2_plugin)
extern plugin_record_t		lvm2_plugin;
extern engine_functions_t	*EngFncs;
extern list_anchor_t		lvm2_containers;
extern boolean			device_size_prompt;

#define LVM2_DEV_DIRECTORY	"lvm2"

/**
 * struct metadata_location
 * @mda_start:		Starting sector of this metadata-area.
 * @mda_size:		Size (in sectors) of this metadata-area.
 * @vgda_offset:	Offset (in sectors) *within* this metadata-area to the
 *			start of the text VGDA. *Not* the absolute starting
 *			sector of the VGDA.
 * @vgda_size:		Size (in bytes, *not* sectors) of the text VGDA.
 * @vgda_crc:		Checksum of the VGDA.
 *
 * Describes the location of one metadata-area on disk. A list of these
 * structures is attached to each pv_data_t structure. This structure is
 * based on information from the PV's pv_header and mda_header structures.
 **/
typedef struct metadata_location {
	u_int64_t	mda_start;
	u_int64_t	mda_size;
	u_int64_t	vgda_offset;
	u_int64_t	vgda_size;
	u_int32_t	vgda_crc;
} metadata_location_t;

/**
 * struct physical_extent
 * @pv_data:	Back-pointer to the PV private-data.
 * @le:		The logical extent currently using this PE.
 * @number:	The index of this PE in the PV's PE-map.
 *
 * Describes one physical extent on a PV.
 **/
typedef struct physical_extent {
	struct pv_data		*pv_data;
	struct logical_extent	*le;
	u_int64_t		number;
} physical_extent_t;

/**
 * struct pv_data
 * @object:	Pointer back to PV object.
 * @pe_map:	Array of physical extents.
 * @vgda_tree:	Tree describing the VG metadata. This tree is only present
 *		during discovery.
 * @metadata_area: List of metadata_location_t structures, giving the on-disk
 *		locations of the PV's metadata areas.
 * @uuid:	Unique identifier for this PV (from pv_header).
 * @label_sector: Location of PV label.
 * @pv_index:	Each PV in a VG is identified by a "pv#" name. This field
 *		holds the "#" index for that name.
 * @pe_start:	Starting sector of the first physical extent.
 * @pe_count:	Number of physical extents.
 * @flags:
 *
 * Consumed-private-data for LVM2 PV objects.
 **/
typedef struct pv_data {
	storage_object_t	*object;
	struct physical_extent	*pe_map;
	key_value_t		*vgda_tree;
	list_anchor_t		metadata_areas;
	char			uuid[LVM2_UUID_LEN+1];
	u_int64_t		label_sector;
	u_int32_t		pv_index;
	u_int64_t		pe_start;
	u_int64_t		pe_count;
	unsigned long		flags;
} pv_data_t;

#define LVM2_PV_FLAG_NEW		(1 << 0)
#define LVM2_PV_FLAG_ALLOCATABLE	(1 << 1)
#define LVM2_PV_FLAG_EXPORTED		(1 << 2)
#define LVM2_PV_FLAG_MISSING		(1 << 3)
#define LVM2_PV_FLAG_RESIZED		(1 << 4)

#define MISSING_PV(object) \
	(((pv_data_t *)object->consuming_private_data)->flags & LVM2_PV_FLAG_MISSING)

/**
 * struct logical_extent
 * @le_map:	Back-pointer to the logical-extent-map.
 * @pe:		The physical extent that this LE maps to.
 *
 * Describes one logical extent (LE) in a region-mapping. Each LE points to
 * one physical extent (PE), and that PE points back to the LE.
 **/
typedef struct logical_extent {
	struct logical_extent_map	*le_map;
	struct physical_extent		*pe;
} logical_extent_t;

/**
 * struct logical_extent_map
 * @r_map:	Back-pointer to the region-mapping.
 * @map:	Current array of logical-extent structures defining which PV/PE
 *		each LE maps to.
 * @new_map:	Array of logical-extents where the mapping will be moving to.
 * @copy_job:	Engine copy-job that will handle moving the data.
 *
 * Describes an array of logical extents in a region-mapping.
 **/
typedef struct logical_extent_map {
	struct region_mapping	*r_map;
	struct logical_extent	*map;
	struct logical_extent	*new_map;
	copy_job_t		*copy_job;
} logical_extent_map_t;

/**
 * struct region_mapping
 * @r_data:		Back-pointer to the region's private-data.
 * @start_le:		Logical extent within the whole region where this
 *			mapping begins.
 * @le_count:		Number of logical extents in this mapping.
 * @stripe_count:	Number of stripes in this mapping. 1 == linear.
 * @stripe_size:	Size of each stripe chunk (in sectors).
 * @le_maps:		An aray of logical-extent-map structures, each of which
 *			holds an array of logical-extent structures defining
 *			the logical-to-physical mapping.
 *
 * Describes on portion of the mapping for a region. Each region contains
 * a list of one or more of these mapping structures.
 *
 * FIXME: For now, we're assuming only linear and striped mappings. We might
 *        want to allow for mirror mappings to make pv-move easier.
 **/
typedef struct region_mapping {
	struct region_data		*r_data;
	u_int64_t			start_le;
	u_int64_t			le_count;
	u_int64_t			stripe_count;
	u_int64_t			stripe_size;
	struct logical_extent_map	*le_maps;
} region_mapping_t;

/**
 * struct region_data
 * @region:	Back-pointer to the region.
 * @mappings:	List of region_mapping_t structures. Each entry in this list
 *		describes one portion of the mapping for the region.
 * @uuid:	Unique identifier for this region (from the VGDA).
 * @flags:
 *
 * Private-data for LVM2 regions.
 **/
typedef struct region_data {
	storage_object_t	*region;
	list_anchor_t		mappings;
	char			uuid[LVM2_UUID_LEN+1];
	unsigned long		flags;
} region_data_t;

#define LVM2_REGION_FLAG_READ		(1 << 0)
#define LVM2_REGION_FLAG_WRITE		(1 << 1)
#define LVM2_REGION_FLAG_FIXED_MINOR	(1 << 2)
#define LVM2_REGION_FLAG_VISIBLE	(1 << 3)
#define LVM2_REGION_FLAG_PVMOVE		(1 << 4)
#define LVM2_REGION_FLAG_LOCKED		(1 << 5)
#define LVM2_REGION_FLAG_MOVE_PENDING	(1 << 6)

#define LVM2_REGION_RENAMED(region)	((region)->dev_name[0] != 0)

/**
 * struct container_data
 * @container:	Back-pointer to the container.
 * @vgda_tree:	Tree describing the VG metadata. This tree is only present
 *		during discovery.
 * @uuid:	Unique identifier for this VG (from the VGDA).
 * @sequence:	Number of times that this container's metadata has been
 *		written to disk (from the VGDA).
 * @pe_size:	Size of each extent (in sectors).
 * @max_lvs:	Max number of regions allowed in this container (unlimited if 0).
 * @max_pvs:	Max number of PVs allowed in this container (unlimited if 0).
 * @flags:
 *
 * Private-data for LVM2 containers.
 **/
typedef struct container_data {
	storage_container_t	*container;
	key_value_t		*vgda_tree;
	char			uuid[LVM2_UUID_LEN+1];
	unsigned long		sequence;
	u_int64_t		pe_size;
	unsigned long		max_lvs;
	unsigned long		max_pvs;
	unsigned long		flags;
} container_data_t;

#define LVM2_CONTAINER_FLAG_INCOMPLETE		(1 << 0)
#define LVM2_CONTAINER_FLAG_EXPORTED		(1 << 1)
#define LVM2_CONTAINER_FLAG_RESIZEABLE		(1 << 2)
#define LVM2_CONTAINER_FLAG_PARTIAL		(1 << 3)
#define LVM2_CONTAINER_FLAG_PVMOVE		(1 << 4)
#define LVM2_CONTAINER_FLAG_READ		(1 << 5)
#define LVM2_CONTAINER_FLAG_WRITE		(1 << 6)
#define LVM2_CONTAINER_FLAG_CLUSTERED		(1 << 7)
#define LVM2_CONTAINER_FLAG_SHARED		(1 << 8)
#define LVM2_CONTAINER_FLAG_MOVE_PENDING	(1 << 9)

/* Useful inlines
 * The "size" paramaters MUST be power-of-2.
 */
static inline ulong round_down(ulong n, ulong size) {
	size--;
	return(n & ~size);
}

static inline ulong round_up(ulong n, ulong size) {
	size--;
	return((n + size) & ~size);
}

static inline ulong div_up(ulong n, ulong size) {
	return(round_up(n, size) / size);
}

#include "containers.h"
#include "objects.h"
#include "regions.h"
#include "options.h"
#include "info.h"
#include "move.h"

