/*
 * Copyright (C) 2000-2001 Chris Ross and Evan Webb
 * Copyright (C) 1999-2000 Chris Ross
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include "ferite.h"

/*!
 * This is an internal variable that holds search paths for script finding
 */
FeriteStack *search_paths = NULL;

/*!
 * \fn FeriteScript *ferite_new_script()
 * \brief This will allocate a new script and set it up ready for use.
 * \return A pointer to the script
 */
FeriteScript *ferite_new_script()
{
   FeriteScript *ptr = NULL;

   FE_ENTER_FUNCTION;
   ptr = fmalloc( sizeof( FeriteScript ) );
   ptr->filename = NULL;
   ptr->scripttext = NULL;
   ptr->exec_stack = __ferite_create_stack( FE_SCRIPT_EXEC_STACK_SIZE );
   ptr->next = NULL;
   ptr->events = NULL;
   ptr->event_count = 0;
   ptr->mainns = fmalloc( sizeof( FeriteNamespace ) );
   ptr->mainns->num = 15;
   ptr->mainns->space = __ferite_create_hash( NULL, FE_SCRIPT_TOP_LEVEL_NS_SIZE );
   ptr->error_cb = NULL;
   ptr->warning_cb = NULL;
   ptr->error_state = 0;
   FE_LEAVE_FUNCTION( ptr );
}

/*!
 * \fn FILE *__ferite_fopen( char *filename, char *mode )
 * \brief This is a special wrapper around the standard system <b>fopen</b>.
 * \param filename The file to open
 * \param mode The mode to open the file with (eg. reaed/write)
 * \return A pointer to a FILE (same as fopen)
 *
 * This function uses the search path facility of ferite. This means that it
 * will try and load the script upon each search path. This is particularily
 * useful if you know that you have items in /foo/ and you want to load 'bar'.
 * If /foo/ is in the serach path and filename='bar' it will find it.
 */
FILE *__ferite_fopen( char *filename, char *mode )
{
   FILE *fp = NULL;
   struct stat filestat;
   int i, stillNotFound = 1;
   char actual_filename[1024];

   FE_ENTER_FUNCTION;
   sprintf( actual_filename, "%s", filename );
   if( stat(actual_filename, &filestat) == -1 ) /* tried the actual file name */
   {
      if( filename[0] != '/' )
      {
		 if( search_paths != NULL )
		 {
			for( i = 0; i <= search_paths->stack_ptr; i++ )
			{
			   if( search_paths->stack[i] != NULL )
			   {
				  sprintf( actual_filename, "%s/%s", search_paths->stack[i], filename );
				  if( stat(actual_filename, &filestat) != -1 ) /* same sirectory as main script */
				  {
					 stillNotFound = 0;
					 break;
				  }
			   }
			}
		 }
      }
   }
   fp = fopen( actual_filename, mode );
   FE_LEAVE_FUNCTION( fp );
}

/*!
 * \fn int ferite_script_load( FeriteScript *script, char *filename )
 * \brief Load the text of a script so that it may be compiled.
 * \param script The script to load the text into
 * \param filename The name of the script to load.
 * \return Returns 0 on fail and 1 on success
 *
 * This function uses the search path facility of ferite.
 */
int ferite_script_load( FeriteScript *script, char *filename )
{
   FILE *scriptfile = NULL;
   struct stat filestat;
   int i, stillNotFound = 1;
   char actual_filename[1024];
   char *homedir = NULL;

   FE_ENTER_FUNCTION;
   script->filename = NULL;
   script->scripttext = NULL;

   sprintf( actual_filename, "%s", filename );
   if( stat(actual_filename, &filestat) == -1 ) /* tried the actual file name */
   {
      if( filename[0] != '/' )
      {
		 if( search_paths != NULL )
		 {
			for( i = 0; i <= search_paths->stack_ptr; i++ )
			{
			   if( search_paths->stack[i] != NULL )
			   {
				  sprintf( actual_filename, "%s/%s", search_paths->stack[i], filename );
				  if( stat(actual_filename, &filestat) != -1 ) /* same sirectory as main script */
				  {
					 stillNotFound = 0;
					 break;
				  }
			   }
			}
		 }
		 if( stillNotFound )
		 {
			sprintf( actual_filename, "%s/%s", SCRIPT_DIR, filename ); /* global lib scripts */
			if( stat(actual_filename, &filestat) == -1 )
			{
			   homedir = __ferite_user_home_dir(0);
			   sprintf( actual_filename, "%s/.ferite/scripts/%s", homedir, filename );
			   if( stat(actual_filename, &filestat) == -1 )
			   {
				  ffree( homedir );
				  FE_LEAVE_FUNCTION( 0 );
			   }
			   ffree( homedir );
			}
		 }
      }
      else
      {
		 FE_LEAVE_FUNCTION( 0 );
      }
   }

   script->filename = fstrdup( actual_filename );
   script->scripttext = fmalloc(filestat.st_size+1);
   memset( script->scripttext, '\0', filestat.st_size+1 );
   scriptfile = fopen( actual_filename, "r" );
   if( scriptfile ) /* load the file */
   {
      fread( script->scripttext, sizeof(char), filestat.st_size, scriptfile );
      fclose( scriptfile );
   }
   /* remove the #! from the file. */
   if( script->scripttext[0] == '#' )
   {
      i = 0;
      while( script->scripttext[i] != '\n' )
      {
		 script->scripttext[i] = ' ';
		 i++;
      }
   }
   FE_LEAVE_FUNCTION( 1 );
}

/*!
 * \fn void ferite_add_search_path( char *path )
 * \brief Add a search path to the system
 * \param path The path to add
 */
void ferite_add_search_path( char *path )
{
   FE_ENTER_FUNCTION;

   if( search_paths == NULL )
     search_paths = __ferite_create_stack( FE_SEARCH_PATH_STACK_SIZE );

   __ferite_stack_push( search_paths, fstrdup( path ) );

   FE_LEAVE_FUNCTION( NOWT );
}

/*!
 * \fn void ferite_delete_search_paths()
 * \brief Clean up the source paths
 */
void ferite_delete_search_paths()
{
   int i;

   FE_ENTER_FUNCTION;
   if( search_paths != NULL )
   {
      for( i = 0; i <= search_paths->stack_ptr; i++ )
      {
		 if( search_paths->stack[i] != NULL )
		   ffree( search_paths->stack[i] );
      }
      ffree( search_paths->stack );
      ffree( search_paths );
   }
   FE_LEAVE_FUNCTION(NOWT);
}

/*!
 * \fn int ferite_script_delete( FeriteScript *script )
 * \brief Delete a script - this will clean everything up
 * \param script A pointer to the script to be deleted
 * \return Returns 0 on fail and 1 on success
 */
int ferite_script_delete( FeriteScript *script )
{
   int i;

   FE_ENTER_FUNCTION;

   if( script != NULL )
   {
      for( i = 0; i <= script->exec_stack->stack_ptr; i++ )
	  {
/*	 printf( "Deleting variable hash %p\n", script->exec_stack->stack[i] );*/
		 if( script->exec_stack->stack[i] != NULL )
		 {
			__ferite_delete_variable_hash( script, ((FeriteExecuteRec*)script->exec_stack->stack[i])->variable_hash );
		 }
      }
      __ferite_delete_stack( script->exec_stack );
      __ferite_check_gc();
      __ferite_delete_namespace( script, script->mainns );

      /* delete sotred events */
      if( script->events != NULL )
      {
		 for( i = 0; i < script->event_count; i++ )
		   ffree( script->events[i] );
		 ffree( script->events );
      }

      if( script->filename != NULL )
		ffree( script->filename );
      if( script->scripttext != NULL )
		ffree( script->scripttext );
      ffree( script );
      script = NULL;
   }
   else
     ferite_warning( NULL, "Trying to free NULL script\n" );
   FE_LEAVE_FUNCTION( 1 );
}

/*!
 * \fn void __ferite_delete_scripts( FeriteScript *script )
 * \brief Recursively delete a linked list of scripts
 * \param script A pointer to the head of the list
 */
void __ferite_delete_scripts( FeriteScript *script )
{
   FE_ENTER_FUNCTION;
   if( script != NULL )
   {
      if( script->next != NULL )
		__ferite_delete_scripts( script->next );
      ferite_script_delete( script );
   }
   FE_LEAVE_FUNCTION( NOWT );
}

/* code from here on in is evil and not mine :)
 * you may thank raster for it =P */

/*!
 * \fn char *__ferite_user_home_dir(int uid)
 * \brief Allocate a string containing the users home directory
 * \param uid The user id (not used at the moment)
 * \return A string containing the home directory
 */
char *__ferite_user_home_dir(int uid)
{
   char *s;

   FE_ENTER_FUNCTION;
   if( (s = getenv("HOME")) != NULL )
   {
      FE_LEAVE_FUNCTION( fstrdup(s) );
   }
   else
   {
      if( (s = getenv("TMP")) != NULL )
      {
		 FE_LEAVE_FUNCTION( fstrdup(s) );
      }
   }
   FE_LEAVE_FUNCTION( NULL );
}

/*!
 * \fn char **__ferite_file_dir( char *dir, int *num )
 * \brief Create an array of char *'s which contains all the files within a directory
 * \param dir The directory to read
 * \param num This is a pointer to an int where the number of entries will be placed
 * \return An array containing pointers to null terminated strings
 */
char **__ferite_file_dir( char *dir, int *num )
{
   int                 i, dirlen;
   int                 done = 0;
   DIR                *dirp;
   char              **names;
   struct dirent      *dp;

   FE_ENTER_FUNCTION;

   *num = 0;
   if( (!dir) || (!*dir) )
   {
      FUD(( "dir doesn't exist (variable)\n" ));
      FE_LEAVE_FUNCTION( 0 );
   }

   dirp = opendir( dir );
   if( !dirp )
   {
      FUD(( "can't open directory %s\n", dir ));
      *num = 0;
      FE_LEAVE_FUNCTION( NULL );
   }
   /* count # of entries in dir (worst case) */
   for( dirlen = 0; (dp = readdir(dirp)) != NULL; dirlen++ );
   if (!dirlen)
   {
      FUD(( "no directory entries\n" ));
      closedir(dirp);
      *num = dirlen;
      FE_LEAVE_FUNCTION( NULL );
   }
   names = (char **)fmalloc(dirlen * sizeof(char *));

   if( !names )
   {
      FUD(( "can't allocate memory for dir entries\n" ));
      FE_LEAVE_FUNCTION( NULL );
   }

   rewinddir( dirp );
   FUD(( "scanning directories\n" ));
   for( i = 0; i < dirlen; )
   {
      dp = readdir( dirp );
      if( !dp )
		break;
      if( (strcmp(dp->d_name, ".")) && (strcmp(dp->d_name, "..")) )
      {
		 names[i] = fstrdup(dp->d_name);
		 FUD(( "Found file %s\n", dp->d_name ));
		 i++;
      }
   }

   if (i < dirlen)
     dirlen = i;                       /* dir got shorter... */
   closedir(dirp);
   *num = dirlen;
   /* do a simple bubble sort here to alphanumberic it */
   FUD(( "sorting\n" ));
   while (!done)
   {
      done = 1;
      for (i = 0; i < dirlen - 1; i++)
      {
		 if (strcmp(names[i], names[i + 1]) > 0)
		 {
			char               *tmp;

			tmp = names[i];
			names[i] = names[i + 1];
			names[i + 1] = tmp;
			done = 0;
		 }
      }
   }
   FE_LEAVE_FUNCTION( names );
}

/*!
 * \fn __ferite_file_dir_free( char **l, int num )
 * \brief Clean up a directory list handed out by __ferite_file_dir
 * \param l The list
 * \param num The number of entries
 */
void __ferite_file_dir_free( char **l, int num )
{
   FE_ENTER_FUNCTION;
   if( l != NULL )
   {
      while( num >= 0 && num-- )
      {
		 if( l[num] )
		 {
			FUD(( "ffreeing %s\n", l[num] ));
			ffree(l[num]);
		 }
      }
      ffree(l);
   }
   FE_LEAVE_FUNCTION( NOWT );
}

/*!
 * \fn int __ferite_item_in_list( char **list, int size, char *item)
 * \brief Check to see if an item is in a array of char *'s
 * \param list The list
 * \param num The number of entries
 * \param item The item to check for
 * \return Returns 0 on fail and 1 on success
 */
int __ferite_item_in_list( char **list, int size, char *item)
{
   int i;

   FE_ENTER_FUNCTION;
   if( size < 1 )
   {
      FE_LEAVE_FUNCTION( 0 );
   }
   if( list == NULL )
   {
      FE_LEAVE_FUNCTION( 0 );
   }
   if( item == NULL )
   {
      FE_LEAVE_FUNCTION( 0 );
   }

   for (i = 0; i < size; i++)
   {
      if( strcmp( list[i], item ) == 0 )
      {
		 FE_LEAVE_FUNCTION( 1 );
      }
   }
   FE_LEAVE_FUNCTION( 0 );
}
