/*
 * Copyright (C) 2000-2001 Chris Ross and Evan Webb
 * Copyright (C) 1999-2000 Chris Ross
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include "ferite.h"
#include <pcre.h> /* perl compatible regualr expressions */

/***************************************************************
 * The way some of this code operates (eg within the matches)  *
 * was semi borrowed from php, this is because the pcre module *
 * that is shiped with php is rather good.                     *
 *                                                             *
 * Those parts of the code borrowed were written by            *
 *   Andrei Zmievski <andrei@ispi.net> and are subject to the  *
 *   PHP 2.02 lisence - http://www.php.net/license/2_02.txt    *
 ***************************************************************/

/* memory functions so pcre uses ferite's memory management */
void *__ferite_regex_malloc( size_t size)
{
   void *ptr;
   FE_ENTER_FUNCTION;
   ptr = fmalloc( size );
   FE_LEAVE_FUNCTION( ptr );
}

void __ferite_regex_free( void *ptr )
{
   FE_ENTER_FUNCTION;
   ffree( ptr );
   FE_LEAVE_FUNCTION( NOWT );
}

void __ferite_init_regex()
{
   FE_ENTER_FUNCTION;
   pcre_malloc = __ferite_regex_malloc;
   pcre_free = __ferite_regex_free;
   FUD(( "REGEX, Using: PCRE %s\n", pcre_version() ));
   FE_LEAVE_FUNCTION( NOWT );
}

FeriteRegex *__ferite_create_regex()
{
   FeriteRegex *ptr = NULL;

   FE_ENTER_FUNCTION;

   ptr = fmalloc( sizeof( FeriteRegex ) );
   ptr->pattern = NULL;
   ptr->type = F_RGX_MATCH;
   ptr->pcre_options = 0;
   ptr->fergx_options = 0;
   ptr->compiled_re = NULL;
   ptr->compile_buf = NULL;
   ptr->swap_buf = NULL;
   ptr->extra_info = NULL;

   FE_LEAVE_FUNCTION( ptr );
}

void __ferite_delete_regex( FeriteRegex *rgx )
{
   FE_ENTER_FUNCTION;

   if( rgx->pattern != NULL )
     ffree( rgx->pattern );
   if( rgx->compiled_re != NULL )
     ffree( rgx->compiled_re );
   if( rgx->compile_buf != NULL )
     ffree( rgx->compile_buf );
   if( rgx->swap_buf != NULL )
     ffree( rgx->swap_buf );
   ffree( rgx );

   FE_LEAVE_FUNCTION( NOWT );
}

FeriteRegex *__ferite_generate_regex( char *pattern )
{
   FeriteRegex *ptr = NULL;
   int i = 0, keepsearching = 0, leftbound = 0, rightbound = 0;
   char *tmpbuf = 0;

   FE_ENTER_FUNCTION;

   ptr = __ferite_create_regex();
   ptr->pattern = fstrdup( pattern );

   ptr->type = F_RGX_MATCH;

   /* go from start to end picking up modifiers (eg s and m) */
   FUD(( "REGEX prefix\n" ));
   for( i = 0,keepsearching = 1; i < strlen( pattern ) && keepsearching; i++ )
   {
      switch( pattern[i] )
      {
       case 's':
       case 'S':
		 FUD(( "REGEX - recognised SWAP\n" ));
		 ptr->type = F_RGX_SWAP;
		 break;
       case 'm':
       case 'M':
		 FUD(( " - recognised MATCH\n" ));
		 ptr->type = F_RGX_MATCH;
		 break;
       case '/':
		 leftbound = i;
		 keepsearching = 0;
		 break;
       default:
		 ferite_warning( NULL, "Regex Modifier %c - UNKOWN, ignoring\n", pattern[i] );
      }
   }
   /* go from end to beginning to get the options */
   FUD(( "REGEX postfix's\n" ));
   for( keepsearching = 1,i = strlen( pattern ) - 1; i >= 0 && keepsearching; i-- )
   {
      switch( pattern[i] )
      {
		 /* pcre supported perl'isms */
       case 'x': FUD(( "REGEX: `-> Extended Line\n" ));    ptr->pcre_options |= PCRE_EXTENDED; break;
       case 's': FUD(( "REGEX: `-> Single Line Mode\n" )); ptr->pcre_options |= PCRE_DOTALL; break;
       case 'm': FUD(( "REGEX: `-> Multi Line Mode\n" ));  ptr->pcre_options |= PCRE_MULTILINE; break;
       case 'i': FUD(( "REGEX: `-> Case Insensitive\n" )); ptr->pcre_options |= PCRE_CASELESS; break;
		 /* pcre + ferite internal'isms */
       case 'g': FUD(( "REGEX: `-> Global Matching\n" ));  ptr->fergx_options |= F_RGX_GLOBAL; break;
       case 'o': FUD(( "REGEX: `-> Cache Compile\n" ));    ptr->fergx_options |= F_RGX_COMPILE; break;
       case 'e':
		 if( ptr->type != F_RGX_SWAP )
		   ferite_warning( NULL, "Script Evaluator can only be used on a swap, not a match\n" );
		 else
		 {
			FUD(( "REGEX: `-> Evaluate Swap Portion\n" ));
			ptr->fergx_options |= F_RGX_EVAL_SWP;
		 }
		 break;
		 /* pcre's specific stuff */
       case 'A': FUD(( "REGEX: `-> Anchored\n" ));         ptr->pcre_options |= PCRE_ANCHORED; break;
       case 'D': FUD(( "REGEX: `-> Dollar @ ABS End\n" )); ptr->pcre_options |= PCRE_DOLLAR_ENDONLY; break;
		 /* get out clause */
       case '/':
		 rightbound = i;
		 keepsearching = 0;
		 break;
       default:
		 ferite_warning( NULL, "Regex Option %c - UNKNOWN, ignoring\n", pattern[i] );
      }
   }

   /* setup compile buffers and pre-compile the regex */
   if( ptr->type == F_RGX_MATCH )
   {
      tmpbuf = __ferite_copy_string( ptr->pattern, leftbound+1, rightbound-1 );
      ptr->compile_buf = __ferite_replace_string( tmpbuf, "\\/", "/" );
      ffree( tmpbuf );
      if( ptr->fergx_options & F_RGX_COMPILE )
      {
		 FUD(( "REGEX: Compiling RGX: \"%s\"\n", ptr->compile_buf ));
		 ptr->compiled_re = __ferite_compile_regex( ptr->compile_buf, ptr->pcre_options );
      }
   }
   if( ptr->type == F_RGX_SWAP )
   {
      for( i = leftbound+1; i < strlen( pattern ); i++ )
      {
		 if( pattern[i] == '/' && pattern[i-1] != '\\' )
		 {
			tmpbuf = __ferite_copy_string( ptr->pattern, leftbound+1, i - 1 );
			ptr->compile_buf = __ferite_replace_string( tmpbuf, "\\/", "/" );
			ffree( tmpbuf );
			ptr->swap_buf = __ferite_copy_string( ptr->pattern, i+1, rightbound-1 );
			FUD(( "REGEX: Swap \"%s\" with \"%s\"\n", ptr->compile_buf, ptr->swap_buf ));
			break;
		 }
      }
      if( ptr->fergx_options & F_RGX_COMPILE )
      {
		 FUD(( "REGEX: Compiling RGX: \"%s\"\n", ptr->compile_buf ));
		 ptr->compiled_re = __ferite_compile_regex( ptr->compile_buf, ptr->pcre_options );
      }
   }
   FE_LEAVE_FUNCTION( ptr );
}

void *__ferite_compile_regex( char *pattern, int options )
{
   void       *ptr;
   const char *error;
   int         erroroffset;

   FE_ENTER_FUNCTION;
   ptr = pcre_compile( pattern, options, &error, &erroroffset, NULL );

   if( ptr == NULL )
   {
      ferite_warning( NULL, "Regex Compliation failed: %s at offset %d\n", error, erroroffset );
      return NULL;
   }

   FE_LEAVE_FUNCTION( ptr );
}

#define islabelstart( c )   ( c == '_' || (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') )
#define islabelchar( c )   ( c == '_' || (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') || (c >= '0' && c <= '9') )

char *__ferite_regex_swap_vars( char *rgxBuf, FeriteScript *script )
{
   FeriteVariable   *repVar = NULL;
   FeriteExecuteRec *er = script->exec_stack->stack[script->exec_stack->stack_ptr];
   char             *oldRgx, buf[1024];
   char             *newRgx = fstrdup("");
   int               i, start, len = strlen(rgxBuf), newLength = 0;
   FeriteNamespaceBucket *nsb;

   /* fprintf( stderr, "--> start\n" ); */
   for( i = 0; i < len; i++ )
   {
      memset( buf, '\0', 1024 );
      oldRgx = newRgx;
      if( rgxBuf[i] == '$' && islabelstart( rgxBuf[i+1] ) )
	  {
		 start = i++;
		 while( i < len && islabelchar( rgxBuf[i] ) )
		   i++;
		 strncpy( buf, rgxBuf+start, i - start );
/*	 fprintf( stderr, "  ---> Found Variable `%s' (%s)\n", buf, buf+1 ); */
		 repVar = __ferite_hash_get( script, er->variable_hash, buf+1 );
		 if( repVar == NULL )
		 {
			nsb = __ferite_namespace_element_exists( script, script->mainns, buf+1 );
			if( nsb != NULL )
			{
			   repVar = nsb->data;
			}
		 }
		 if( repVar != NULL && repVar->type == F_VAR_STR )
		 {
/*	    fprintf( stderr, "        value: `%s'\n", VAS(repVar) ); */
			newLength = strlen(oldRgx) + strlen(VAS(repVar)) + 1;
			newRgx = fmalloc( newLength );
			memset( newRgx, '\0', newLength );
			strcpy( newRgx, oldRgx );
			strcat( newRgx, VAS(repVar) );
			ffree( oldRgx );
/*	    fprintf( stderr, "        newRgx `%s', allocated %d, actual %d\n", newRgx, newLength, strlen(newRgx) ); */
		 }
		 else
		 {
/*	    fprintf( stderr, "        value: `%s'\n", "no value" );	    */
		 }
		 i--;
      }
	  else
	  {
		 start = i;
		 while( i < len && rgxBuf[i] != '$' )
		   i++;
/*	 strncpy( buf, rgxBuf+start, i - start );
	 fprintf( stderr, "  ---> Found Data      `%s'\n", buf );*/
		 newLength = strlen(oldRgx) + (i-start) + 1;
		 newRgx = fmalloc( newLength );
		 memset( newRgx, '\0', newLength );
		 strcpy( newRgx, oldRgx );
		 strncat( newRgx, rgxBuf+start, i - start );
		 ffree( oldRgx );
/*	 fprintf( stderr, "        newRgx `%s', allocated %d, actual %d\n", newRgx, newLength, strlen(newRgx) );*/
		 i--;
      }
   }
/*   fprintf( stderr, "--> end\n" );*/
   return newRgx;
}

FeriteVariable *__ferite_execute_regex( FeriteRegex *rgx, FeriteVariable *target, FeriteScript *script )
{
   FeriteVariable *ptr = NULL;
   char *oldRgx, *oldSwap;

   FE_ENTER_FUNCTION;
   if( rgx == NULL )
     ferite_error( script, "Trying to execute non-existant regex" );
   if( target == NULL )
     ferite_error( script, "Trying to execute regex on non-existant variable" );
   if( target->type != F_VAR_STR )
   {
      ferite_warning( script, "Trying to execute on non-string value, returning false\n" );
      FE_LEAVE_FUNCTION( __ferite_create_number_long_variable( "regex-exec-return", 0 ) );
   }

   oldRgx = rgx->compile_buf;
   oldSwap = rgx->swap_buf;

   /* as Mr. Friedl says in Mastering Regular Expressions, this does the 'Doublequotish'
    * behavior (ie variable substiution */
   rgx->compile_buf = __ferite_regex_swap_vars( rgx->compile_buf, script );
   if( rgx->type == F_RGX_SWAP && rgx->swap_buf != NULL )
     rgx->swap_buf = __ferite_regex_swap_vars( rgx->swap_buf, script );
   if( rgx->compiled_re != NULL && strcmp( rgx->compile_buf, oldRgx ) )
   {
	  ffree( rgx->compiled_re );
   }

   switch( rgx->type )
   {
    case F_RGX_MATCH:
      ptr = __ferite_execute_match_regex( rgx, target, ( rgx->fergx_options & F_RGX_GLOBAL ? 1 : 0 ), script->mainns, script );
      break;
    case F_RGX_SWAP:
      ptr = __ferite_execute_swap_regex( rgx, target, ( rgx->fergx_options & F_RGX_GLOBAL ? 1 : 0 ), script->mainns, script );
      break;
    default:
      ferite_warning( script, "Unknown regex type %d, returning false\n", rgx->type );
      ptr = __ferite_create_number_long_variable( "regex-exec-return", 0 );
   }

   ffree( rgx->compile_buf );
   if( rgx->type == F_RGX_SWAP && rgx->swap_buf != NULL )
   {
	  ffree( rgx->swap_buf );
   }
   rgx->compile_buf = oldRgx;
   rgx->swap_buf = oldSwap;

   FE_LEAVE_FUNCTION( ptr );
}

FeriteVariable *__ferite_execute_match_regex( FeriteRegex *rgx, FeriteVariable *target, int global, FeriteNamespace *ns, FeriteScript *script )
{
   FeriteVariable *retv = NULL; /* rturn value */
   FeriteVariable *ptr = NULL;  /* so we can fudge with the arrays */
   FeriteNamespaceBucket *nsb;
   int captured_str_cnt = 0; /* number of strings that will be captured in ()'s */
   int *offsets;         /* array of subpattern offsets */
   int size_offsets;     /* size of the array */
   int start_offset;     /* damn nation, this is the new start :) */
   int current_var = 0;  /* the global variable we need to update */
   int current_match = 0;
   int count = 0;        /* count the number of subexpressions */
   int i;
   char buf[10], *match;
   const char **stringlist /* the subparts () values */;
   int g_notempty = 0;

   FE_ENTER_FUNCTION;

   /* if the regex is not compiled already -> compile it */
   if( rgx->compiled_re == NULL )
   {
      rgx->compiled_re = __ferite_compile_regex( rgx->compile_buf, rgx->pcre_options );
      if( rgx->compiled_re == NULL )
      {
		 retv = __ferite_create_number_long_variable( "regex-exec-return", 0 );
		 FE_LEAVE_FUNCTION( retv );
      }
   }

   /* get the number of subparts */
   captured_str_cnt = pcre_info( rgx->compiled_re, NULL, NULL ) + 1;
   /* create an offset array */
   size_offsets = captured_str_cnt * 3;
   offsets = (int *)fmalloc(size_offsets * sizeof(int));

   /* we need to setup the global variable hash so that r1->r99 holds the number of subparts */
   for( i = 1; i <= captured_str_cnt; i++ )
   {
      memset( buf, '\0', 10 );
      sprintf( buf, "r%d", i );
      nsb = __ferite_namespace_element_exists( script, ns, buf );
      ptr = ( nsb == NULL ? NULL : nsb->data );
      if( ptr == NULL )
      {
		 ptr = __ferite_create_string_variable( buf, "" );
		 ptr->flags.constant = 2; /* we make these global variables read only */
		 __ferite_register_ns_variable( script, ns, ptr );
      }
   }

   /* setup the return value */
   if( global )
     retv = __ferite_create_uarray_variable( "regex-exec-return", 32 );
   else
     retv = __ferite_create_string_variable( "regex-exec-return", "" );
   
   start_offset = 0;
   current_var = 1;
   current_match = 1;

   do
   {
      /* execute regex */
      FUD(( "REGEX: Executing against \"%s\"\n", VAS(target) ));
      count = pcre_exec( rgx->compiled_re, rgx->extra_info, VAS(target), strlen( VAS(target) ),
						 start_offset, g_notempty, offsets, size_offsets );

      /* check for too many substrings */
      if( count == 0 )
      {
		 ferite_warning( script, "A match was found but too many substrings found.\n" );
		 count = size_offsets / 3;
      }

      FUD(( "REGEX: match count = %d\n", count ));
      /* we matched something */
      if( count >= 0 )
      {
		 match = VAS(target) + offsets[0];

		 /* get the list of substrings */
		 if( pcre_get_substring_list( VAS(target), offsets, count, &stringlist) < 0 )
		 {
			ffree(offsets);
			ferite_warning( script, "Unable to obtain captured strings in regular expression.\n");
			__ferite_variable_destroy( script, retv );
			retv = __ferite_create_number_long_variable( "regex-exec-return", 0 );
			FE_LEAVE_FUNCTION( retv );
		 }

		 if( global )
		 {
			FUD(( "REGEX: Setting index %d to \"%s\"\n", current_match, (char *)stringlist[0] ));
			memset( buf, '\0', 10 );
			sprintf( buf, "hash-%d", current_match );
			ptr = __ferite_create_string_variable( buf, (char *)stringlist[0] );
			FUD(("REGEX: Current Match: %d\n" ));
			__ferite_uarray_add( script, VAUA(retv), ptr, NULL, -1 );
			current_match++;
		 }
		 /* need to setup the r%% variables */
		 /* 0 = total matched string 1++ is subparts */
		 for( i = 1; i < count; i++ )
		 {
			memset( buf, '\0', 10 );
			sprintf( buf, "r%d", current_var );
			nsb = __ferite_namespace_element_exists( script, ns, buf );
			ptr = nsb->data /*__ferite_get_variable_from_hash( gv, buf )*/;
			ffree( VAS(ptr) );
			VAS(ptr) = fstrdup( (char *)stringlist[i] );
			FUD(( "Setting %s to %s\n", ptr->name, VAS(ptr) ));
			current_var++;
		 }
		 ffree( stringlist );
      }
      else /* we didn't match something */
      {
		 match = "\0";
		 if (g_notempty != 0 && start_offset < strlen(VAS(target)) )
		 {
			offsets[0] = start_offset;
			offsets[1] = start_offset + 1;
		 }
		 else
		   break;
      }

      g_notempty = ( offsets[1] == offsets[0] ) ? PCRE_NOTEMPTY : 0;

      /* Advance to the position right after the last full match */
      start_offset = offsets[1];

   }
   while( global );

   ffree( offsets );

   if( !global )
   {
      ffree( VAS(retv) );
      VAS(retv) = fstrdup( match );
   }
   FE_LEAVE_FUNCTION( retv );
}

FeriteVariable *__ferite_execute_swap_regex( FeriteRegex *rgx, FeriteVariable *target, int global, FeriteNamespace *ns, FeriteScript *script )
{
   FeriteVariable *retv = NULL; /* rturn value */
   FeriteVariable *ptr = NULL;  /* so we can fudge with the arrays */
   FeriteVariable *ePtr = NULL, *ePtrRV = NULL;
   FeriteNamespaceBucket *nsb;
   int captured_str_cnt = 0; /* number of strings that will be captured in ()'s */
   int *offsets;         /* array of subpattern offsets */
   int size_offsets;     /* size of the array */
   int start_offset;     /* damn nation, this is the new start :) */
   int current_var = 0;  /* the global variable we need to update */
   int current_match = 0;
   int count = 0;        /* count the number of subexpressions */
   int i, swaps = 0, walkpos = 0, target_backtick;
   int endOfLastMatch = 0;
   char buf[10], *match, *replace_buffer, *tmpbuf = NULL;
   char *newstr = NULL;
   const char **stringlist /* the subparts () values */;
   int g_notempty = 0;
   int repbuf_len = 0;

   FE_ENTER_FUNCTION;

/*   printf( "Swaping \"%s\" with \"%s\"\n", rgx->compile_buf, rgx->swap_buf );*/
   /* if the regex is not compiled already -> compile it */
   if( rgx->compiled_re == NULL )
   {
      rgx->compiled_re = __ferite_compile_regex( rgx->compile_buf, rgx->pcre_options );
      if( rgx->compiled_re == NULL )
      {
		 retv = __ferite_create_number_long_variable( "regex-exec-return", 0 );
		 FE_LEAVE_FUNCTION( retv );
      }
   }

   /* get the number of subparts */
   captured_str_cnt = pcre_info( rgx->compiled_re, NULL, NULL ) + 1;
   /* create an offset array */
   size_offsets = captured_str_cnt * 3;
   offsets = (int *)fmalloc(size_offsets * sizeof(int));

   /* we need to setup the global variable hash so that r1->r99 holds the number of subparts */
   for( i = 1; i <= captured_str_cnt; i++ )
   {
      memset( buf, '\0', 10 );
      sprintf( buf, "r%d", i );
      nsb = __ferite_namespace_element_exists( script, ns, buf );
      ptr = ( nsb == NULL ? NULL : nsb->data );
      if( ptr == NULL )
      {
		 ptr = __ferite_create_string_variable( buf, "" );
		 ptr->flags.constant = 2; /* we make these global variables read only */
		 __ferite_register_ns_variable( script, ns, ptr );
      }
   }

   start_offset = 0;
   current_var = 1;
   current_match = 0;

   newstr = fmalloc( 1 );
   *newstr = '\0';

   do
   {
      /* execute regex */
      FUD(( "REGEX: Executing against \"%s\"\n", VAS(target) ));
      count = pcre_exec( rgx->compiled_re, rgx->extra_info, VAS(target), strlen( VAS(target) ),
						 start_offset, g_notempty, offsets, size_offsets );

      /* check for too many substrings */
      if( count == 0 )
      {
		 ferite_warning( script, "A match was found but too many substrings found.\n" );
		 count = size_offsets / 3;
      }

      FUD(( "REGEX: match count = %d\n", count ));
      /* we matched something */
      if( count >= 0 )
      {
		 match = VAS(target) + offsets[0];

		 /* get the list of substrings */
		 if( pcre_get_substring_list( VAS(target), offsets, count, &stringlist) < 0 )
		 {
			ffree(offsets);
			ferite_warning( script, "Unable to obtain captured strings in regular expression.\n");
			retv = __ferite_create_number_long_variable( "regex-exec-return", 0 );
			FE_LEAVE_FUNCTION( retv );
		 }

		 /* need to setup the r%% variables we do this before doing the string stuff as
		  * it allows us to use it in the eval operator */
		 /* 0 = total matched string 1++ is subparts */
		 for( i = 1; i < count; i++ )
		 {
			memset( buf, '\0', 10 );
			sprintf( buf, "r%d", current_var );
			nsb = __ferite_namespace_element_exists( script, ns, buf );
			ptr = nsb->data;
			ffree( VAS(ptr) );
			VAS(ptr) = fstrdup( (char *)stringlist[i] );
			FUD(( "Setting %s to %s\n", ptr->name, VAS(ptr) ));
			current_var++;
		 }

		 /* build the replace buffer */
		 if( rgx->swap_buf != NULL )
		 {
			repbuf_len = strlen(rgx->swap_buf) + (strlen(rgx->compile_buf) * 10) + 1;
			replace_buffer = fcalloc(repbuf_len, sizeof(char));
			for( i = 0, walkpos = 0; i < strlen( rgx->swap_buf ) && walkpos < repbuf_len; i++ )
			{
			   /* check to see if we have a backtick */
			   if( rgx->swap_buf[i] == '\\' )
			   {
				  /* account for escaped backticks */
				  if( rgx->swap_buf[i+1] == '\\' )
				  {
					 replace_buffer[walkpos++] = rgx->swap_buf[i];
					 i++;
				  }
				  else
				  {
					 /* get the required backtick */
					 if( i < (strlen(rgx->swap_buf) - 1) )
					 {
						target_backtick = rgx->swap_buf[i+1] - '0';
						if( i < (strlen(rgx->swap_buf) - 2) )
						  target_backtick = (rgx->swap_buf[i+2] <= '9' && rgx->swap_buf[i+2] >= '0' ? (target_backtick * 10) + rgx->swap_buf[i+2] - '0' : target_backtick);
						if( target_backtick <= count )
						{
						   memcpy( replace_buffer + walkpos, stringlist[target_backtick], strlen( stringlist[target_backtick] ) );
						   walkpos += strlen(stringlist[target_backtick]);
						   if( target_backtick > 9 )
							 i += 2;
						   else
							 i++;
						}
					 }
					 else
					 {
						break;
					 }
				  }
			   }
			   else
			   {
				  replace_buffer[walkpos++] = rgx->swap_buf[i];
			   }
			}
			replace_buffer[walkpos] = '\0'; /* terminate the string */
			FUD(("RGX: Walkpos %d, repbuf_len %d, strlen, %d, content: ``%s''\n", walkpos, repbuf_len, strlen( replace_buffer ), replace_buffer ));
		 }
		 else
		 {
			replace_buffer = fstrdup("");
		 }

		 /*	 printf( "Swap at index %d len %d\n", offsets[0], strlen(stringlist[0]) ); */
		 
		 /* ok this is what we do:
		  *   setup the rx variables
		  *   call eval
		  *   use it's return as a string */
		 if( rgx->fergx_options & F_RGX_EVAL_SWP ){
			FUD(( ">>>> Swap evaluation <<<< '%s' \n", replace_buffer ));
			ePtr = fe_new_str( "regex-eval", replace_buffer );
			
			/* we dont need this any more */
			ffree( replace_buffer );
			ePtrRV = __ferite_op_eval( script, ePtr );
			FUD(( "eval returned: %s, %s\n", ePtrRV->name, __ferite_variable_id_to_str( script, ePtrRV->type ) ));
			if( ePtrRV->type != F_VAR_STR ){
			   replace_buffer = fstrdup("NON_STR_EVAL_RETURN");
			} else {
			   replace_buffer = fstrdup(VAS(ePtrRV));
			}
			/* clean up afterwards */
			__ferite_variable_destroy( script, ePtr );
			__ferite_variable_destroy( script, ePtrRV );
		 }

		 
		 /* grow the buffer */
		 FUD(( "New Str Length: %d\n", strlen( newstr ) + (offsets[0] - endOfLastMatch) + strlen( replace_buffer ) + 10 ));
		 tmpbuf = fcalloc( strlen( newstr ) + (offsets[0] - endOfLastMatch) + strlen( replace_buffer ) + 10, sizeof(char) );
		 strcpy( tmpbuf, newstr );
		 ffree( newstr );

		 newstr = tmpbuf;

		 /* ok what we do here is find out the offset within the original string and then replace that with
		  * the swap buffer */
		 strncat( newstr, VAS(target) + endOfLastMatch, offsets[0] - endOfLastMatch );
		 strcat( newstr, replace_buffer );
		 
		 FUD(( "Real Length: %d\n", strlen(newstr) ));
		 /* new offset */
		 endOfLastMatch = offsets[0] + strlen(stringlist[0]);
		 /*	 printf( "newstr: %s %d\n", newstr, endOfLastMatch );*/
		 
		 swaps++;

		 ffree( replace_buffer );

		 FUD(( "RGX: Captured String count: %d\n", captured_str_cnt ));
		 for( i = 0; i < captured_str_cnt; i++ )
		 {
			FUD(( "RGX: STRLIST: [%d] %s\n", i, stringlist[i] ));
		 }
		 ffree( stringlist );
      }
      else /* we didn't match something */
      {
		 match = "\0";
		 if (g_notempty != 0 && start_offset < strlen(VAS(target)) )
		 {
			offsets[0] = start_offset;
			offsets[1] = start_offset + 1;
		 }
		 else
		   break;
      }

      g_notempty = ( offsets[1] == offsets[0] ) ? PCRE_NOTEMPTY : 0;

      /* Advance to the position right after the last full match */
      start_offset = offsets[1];

   }
   while( global );

   /* cleaning up */
   if( endOfLastMatch < strlen(VAS(target)) )
   {
/*      printf( "Cleaning up\n" );*/
      tmpbuf = fcalloc( strlen( newstr ) + (strlen(VAS(target) + endOfLastMatch)) + 2, sizeof(char) );
      strcpy( tmpbuf, newstr );
      ffree( newstr );
      newstr = tmpbuf;
      strcat( newstr, VAS(target) + endOfLastMatch );
   }

   ffree( VAS(target) );
   VAS(target) = newstr;
   ffree( offsets );

   retv = __ferite_create_number_long_variable( "regex-swap-exec-return", swaps );
   MARK_VARIABLE_AS_DISPOSABLE( retv );
   FE_LEAVE_FUNCTION( retv );
}
