/******************************************************************** 
   Copyright (C) 2000 Bassoukos Tassos <abas@aix.meng.auth.gr>
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later
   version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*********************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <gnome.h>
#include <glib.h>


#include "network.h"
#include "protocol.h"
#include "bookmarks.h"
#include "tasks.h"
#include "threads.h"
#include "server.h"
#include "login.h"
#include "smalltrans.h"
#include "news.h"
#include "userlist.h"
#include "chat.h"
#include "guiprefs.h"
#include "privs.h"
#include "filelist.h"
#include "files.h"
#include "filetransfer.h"
#include "privchat.h"

typedef struct {
  char *name;
  gpointer data;
  void (*create)(Connection *);
  void (*destroy)(Connection *,gpointer);
} Hook;

static int hookstrcmp(Hook *h, char *s)
{return h->name ? strcmp(h->name, s) : -1;}

static Hook *hooks_find(Connection *c,char *name){
  GList *l;
  pthread_mutex_lock(c->mutex);
  l = g_list_find_custom(c->hooks, name, (GCompareFunc) hookstrcmp);
  pthread_mutex_unlock(c->mutex);
  return l ? l->data : NULL;
}

void hooks_create(Connection *c,char *name, gpointer data,
		void (*create)(Connection *),
		STransFunc destroy){
  Hook *h;

  if((h=hooks_find(c,name)))
  {
    printf(_("Overwriting old data for hook %s (while trying to create)..\n"),name);
    h->data = data;
    return;
  }
  
  h=(Hook *)malloc(sizeof(Hook));
  h->data=data;
  h->create=create;
  h->destroy=destroy;
  h->name=name;
  pthread_mutex_lock(c->mutex);
  c->hooks=g_list_prepend(c->hooks,h);
  pthread_mutex_unlock(c->mutex);
}

void hooks_set_data(Connection *c,char *name,gpointer data){
  Hook *h=hooks_find(c,name);
  if(!h){
    printf(_("Attempted to set data for hook %s without init!\n"),name);
    return;
  }
  if(h->data && data)
    printf(_("Overwriting old data for hook %s..\n"),name);
  h->data=data;
}

gpointer hooks_get_data(Connection *c,char *name){
  Hook *h=hooks_find(c,name);
  return h ? h->data : NULL;
}

static void hook_destroy(Hook *h, Connection *c)
{
    if(h->data && h->destroy)
      h->destroy(c,h->data);
    h->data=NULL;
}

static void hooks_run_destroy(Connection *c){
  g_list_foreach(c->hooks, (GFunc) hook_destroy, c);
}
/* NOT USED CURRENTLY 
static void hook_create(Hook *h, Connection *c)
{
	if(h->create)
		h->create(c);
}

static void hooks_run_create(Connection *c){
	g_list_foreach(c->hooks, (GFunc) hook_create, c);
}
*/
/* ================================== */

GSList *connections_list=NULL;

typedef struct {
  STransFunc start;
  gpointer data;
} ProcEntry;

typedef struct {
  int task;
  HLTransaction *transaction;
  gpointer data;
  void (*start)(Connection *,HLTransaction *orig,HLTransaction *reply,gpointer p);
} PendingReply;

void server_transaction_start(Connection *c,
			      STransFunc start,
			      gpointer data){
  ProcEntry *p=(ProcEntry *)malloc(sizeof(ProcEntry));
  p->start=start;
  p->data=data;
  pthread_mutex_lock(c->mutex);
  c->proclist=g_list_append(c->proclist,p);
  pthread_cond_signal(c->signal);
  pthread_mutex_unlock(c->mutex);
}

void server_transaction_reply_set(Connection *c,
				  HLTransaction *t,
				  void (*start)
				  (Connection *,HLTransaction *,HLTransaction *,gpointer),
				  gpointer data){
  PendingReply *p=(PendingReply *)malloc(sizeof(PendingReply));
  p->start=start;
  p->data=data;
  p->transaction=t;
  p->task=t->w.task;
  pthread_mutex_lock(c->mutex);
  c->replies_pending=g_list_append(c->replies_pending,p);
  pthread_cond_signal(c->signal);
  pthread_mutex_unlock(c->mutex);
}

void server_reply_default(Connection *c,HLTransaction *t,HLTransaction *reply,gpointer data){
  if(t->task)
    task_remove_when_idle(t->task);

  transaction_destroy(t);
  transaction_read_objects(c,reply);

  if(reply->w.error){
    HLObject *o=transaction_find_object(reply,HLO_ERRORMSG);
    show_error((c && c->gui && c->gui->main_window) ? c : NULL, NULL, 
    	"%s%s%s", data ? (char *)data : "", o ? "\n" : "", o ? o->data.string : _("\nunspecified error"));
  }

  if(data)
    free(data);

  transaction_destroy(reply);
}

void forall_servers(STransFunc func,gpointer data){
  g_slist_foreach(connections_list, (GFunc)func, data);
}

static void server_ended_reply_helper(PendingReply *p, gpointer dummy)
{
	if(ft_test_reply_close(p->start) && ft_is_zombie(p->data))
		free(p->data);
}

static gboolean server_thread_ended(Connection *c){

  /* FIXME: check for reconnect ??? */
  /* FIXME: remove_lists */
  connections_list=g_slist_remove(connections_list,c);
  if(c->infd!=NULL)
    fclose(c->infd);
  if(c->outfd!=NULL)
    fclose(c->outfd);
  hooks_run_destroy(c);
  if(c->gui!=NULL)
    gui_destroy(c);
  if(c->task!=NULL)
    task_remove(c->task);
  if(c->replies_pending)
  	g_list_foreach(c->replies_pending, (GFunc)server_ended_reply_helper, NULL);
  if(c->mutex)
    mutex_free(c->mutex);
  if(c->signal)
    cond_free(c->signal);
  free_bookmark(c->bookmark);
  free(c);
  tasks_count_set_title();
  return FALSE;
}

static void server_thread_end(Connection *c){
  gboolean start=FALSE;
  pthread_mutex_lock(c->mutex);
  if(!(--c->end_count))
    start=TRUE;
  pthread_mutex_unlock(c->mutex);
  if(start)
    gtk_idle_add((GtkFunction)server_thread_ended, c);
}

static gboolean server_do_stop(Connection *c){
  killThread(c->thread_in);
  killThread(c->thread_out);
  return FALSE;
}

static void server_user_cancel(Task *t){
  ((Connection *)t->user_data)->task=NULL;
  gtk_idle_add((GtkFunction)server_do_stop,t->user_data);
}

void server_cancel(Connection *c){
  gtk_idle_add((GtkFunction)server_do_stop,c);
}

static void handle_banner(Connection *c,HLTransaction *t,gpointer data){
  transaction_read_objects(c,t);
  transaction_destroy(t);
}

static struct ServerMessage {
  int type;
  void (*handle)(Connection *c,HLTransaction *t,gpointer data);
  gpointer data;
  char *message;
} server_messages[]={
  {HLSI_AGREEMENT,handle_server_agreement,NULL,N_("agreement")},
  {HLSI_NEWPOST,handle_server_new_post,NULL,N_("news post")},
  {HLSI_USERCHANGE,handle_user_change,NULL,NULL},
  {HLSI_USERLEAVE,handle_user_leave,NULL,NULL},
  {HLSI_SELFUSER,handle_privs_sent,NULL,NULL},
  {HLSI_BROADCAST,handle_server_message,NULL,NULL},
  {HLSI_RELAYCHAT,chat_receive_mesasage,NULL,NULL},
  {HLSI_SERVERQUPDATE,handle_ftq_update,NULL,NULL},
  {HLSI_BANNERURL,handle_banner,NULL,NULL},

  {HLSI_JOINEDPCHAT,privchat_handle_privchat,privchat_handle_other_join,NULL},
  {HLSI_LEFTPCHAT,privchat_handle_privchat,privchat_handle_other_leave,NULL},
  {HLSI_CHANGESUBJECT,privchat_handle_privchat,privchat_handle_subject_change,NULL},
  {HLSI_INVITEDTOPCHAT,privchat_handle_privchat,privchat_handle_invite,NULL},
  
  {0,NULL,NULL}
};

typedef struct {
  char *name;
  Connection *c;
  Task *t;
  pthread_mutex_t *m;
  pthread_cond_t *cond;
} TaskRequest;

static gboolean server_got_task(TaskRequest *tq){
  pthread_mutex_lock(tq->m);
  tq->t=task_new_sname(tq->c);
  task_add(tq->t);
  task_newtext(tq->t, 0.1, tq->name);
  pthread_cond_signal(tq->cond);
  pthread_mutex_unlock(tq->m);
  return FALSE;
}

static Task *server_get_task(Connection *c,struct ServerMessage *sm){
  TaskRequest tq;

  tq.c=c;
  tq.name=g_strdup_printf(_("Receiving %s"),sm->message);
  tq.m=mutex_new();
  tq.cond=cond_new();
  pthread_mutex_lock(tq.m);
  gtk_idle_add((GtkFunction)server_got_task,(gpointer)&tq);
  pthread_cond_wait(tq.cond,tq.m);
  pthread_mutex_unlock(tq.m);
  cond_free(tq.cond);
  mutex_free(tq.m);
  free(tq.name);
  return tq.t;
}

static void server_handle_remote_request(Connection *c,HLTransaction *t){
  struct ServerMessage *sm=server_messages;
  int i;

  for(;sm->handle!=NULL;sm++)
    if(sm->type==t->w.type){
      Task *task=NULL;
      if(sm->message!=NULL){
	task=server_get_task(c,sm);
	t->task=task;
      }
      sm->handle(c,t,sm->data);
      if(task!=NULL)
	task_remove_when_idle(task);
      return;
    } 
  printf(_("Unknown Server-initiated transaction: info=%d type=%d task=%d\n"),
	 t->w.info,t->w.type,t->w.task);
  transaction_read_objects(c,t);
  printf(_(" object_count=%d;\n"),t->w.object_count);
  for(i=0;i<t->w.object_count;i++)
    printf(_(" %d.type=%d;\n"),i,t->objects[i]->type);
  transaction_destroy(t);
}

void server_set_default_reply(Connection *c,HLTransaction *t){
  server_transaction_reply_set(c,t,server_reply_default,NULL);
}

static int compare_replies(PendingReply *a, Task *b)
{return (int)a->task - (int)b;}
	
static void server_handle_reply(Connection *c,HLTransaction *t){
  PendingReply *pr=NULL;
  GList *l;

  pthread_mutex_lock(c->mutex);
  /* This is tricky, we rely on glib's passing the task pointer as a second arg to compare_replies */
	if((l = g_list_find_custom(c->replies_pending, (gpointer)t->w.task, (GCompareFunc) compare_replies)))
	{
		c->replies_pending = g_list_remove_link(c->replies_pending, l);
		pr = l->data;
		g_list_free_1(l);
	}
  pthread_mutex_unlock(c->mutex);
  
  
  if(!pr){
    char *buf;
    transaction_read_objects(c,t);
    buf=g_strdup_printf(_("Unexpected reply to task %d"),t->w.task);
    show_message_error(c,buf);
    free(buf);
    transaction_destroy(t);
    return;
  }

  pr->start(c,pr->transaction,t,pr->data);
  free(pr);
}

static void server_do_read(Connection *c){
  HLTransaction *t;

  while((t=transaction_read_header(c)))
  {
    if(!t->w.type)
      server_handle_reply(c,t);
    else
      server_handle_remote_request(c,t); 
  }
	show_error(NULL, NULL, _("%s: Connection closed"), S_NAME(c));
  server_transaction_start(c,NULL,NULL);
}

static void login_handle_reply(Connection *c,HLTransaction *t,
			       HLTransaction *r,gpointer p){
  Task *task;
  HLObject *o;

  if((task=c->task)!=NULL){
    c->task=NULL;
    task->user_data=NULL;
    task->user_cancel=NULL;
    task_remove_when_idle(task);
  }
  t->task=NULL;
  r->task=NULL;
  if(r->w.error){
    server_reply_default(c,t,r,strdup(_("Login Error:")));
    server_cancel(c);
    return;
  }
  transaction_read_objects(c,r);
  if((o=transaction_find_object(r,HLO_SERVERNAME))!=NULL){
    free(c->bookmark->name);
    c->bookmark->name=strdup(o->data.string);
  }
  if((o=transaction_find_object(r,HLO_VERSION))!=NULL){
    c->version=o->data.number;
  }
  transaction_destroy(t);
  transaction_destroy(r);
  gtk_idle_add(login_done,(gpointer)c);
}

static void start_login(Connection *c){
  HLTransaction *t;
  
  t=transaction_new(HLCT_LOGIN,(gpointer)c,TRUE);
  if(c->bookmark->login!=NULL &&  strlen(c->bookmark->login)>0)
    transaction_add_object(t,create_string(HLO_LOGIN,c->bookmark->login));
  if(c->bookmark->passwd!=NULL && strlen(c->bookmark->passwd)>0)
    transaction_add_object(t,create_string(HLO_PASSWD,c->bookmark->passwd));
  if(client_version!=0)
    transaction_add_object(t,create_number(HLO_VERSION,client_version));
  transaction_set_task(t,c->task,0.75,1.0);
  server_transaction_reply_set(c,t,login_handle_reply,(gpointer)c);
  transaction_send(t,c);
  ++c->end_count;
  newThread(&c->thread_in,(VoidFunc)server_do_read,(VoidFunc)server_thread_end,(gpointer)c);
}

static void server_do_write(Connection *c){
  NetAddr na;
  char buf[10];
  ProcEntry *pe = NULL;

  if(isIP(&na,c->bookmark->host)==FALSE){
    if(resolveHost(&na,c->bookmark->host)==FALSE){
      show_error_errno(c,NULL,_("No IP for name %s:"),c->bookmark->host);
      return;
    }
  }
  if(c->task!=NULL)
    task_newtext(c->task,0.25,_("Connecting to %s..."),inet_ntoa(na));  
  else 
    pthread_testcancel();
  c->infd=getSocketTo(&na,c->bookmark->port);
  if(c->infd==NULL){
    show_error_errno(c,NULL,_("Could not connect to %s:%d :"),
    	c->bookmark->host,c->bookmark->port);
    return;
  }
  c->outfd=fdopen(dup(fileno(c->infd)),"w");

  if(c->task!=NULL)
    task_newtext(c->task,0.50,_("Handshaking..."));
  else 
    pthread_testcancel();
  fwrite("TRTPHOTL\00\01\00\02",12,1,c->outfd);
  fflush(c->outfd);
  if(fread(buf,8,1,c->infd)!=1 ||
     memcmp(buf,"TRTP\0\0\0\0",8)!=0){
    show_error(c,NULL,_("%s is not a hotline server!"),c->bookmark->host);
    return;
  }
  if(c->task!=NULL){
    task_newtext(c->task,0.75,_("Logging in ..."));
  } else 
    pthread_testcancel();

  start_login(c);
  
  while(1){
    pthread_mutex_lock(c->mutex);
    if(c->proclist){
      c->proclist=g_list_remove(c->proclist,pe = (ProcEntry *)c->proclist->data);
      pthread_mutex_unlock(c->mutex);
      if(pe->start){
	pe->start(c,pe->data);
	free(pe);
      } else {
	free(pe);
	return;
      }
    } else {
      pthread_cleanup_push((VoidFunc)pthread_mutex_unlock,c->mutex);
      pthread_cond_wait(c->signal,c->mutex);
      pthread_cleanup_pop(1);
    }
  }
}

void server_connect_to(Bookmark *b){
  Connection *c=(Connection *)malloc(sizeof(Connection));
  c->bookmark=dup_bookmark(b);
  c->infd=NULL;
  c->outfd=NULL;
  c->gui=NULL;
  c->version=0;
  c->signal=cond_new();
  c->mutex=mutex_new();
  c->proclist=NULL;
  c->replies_pending=NULL;
  c->hooks=NULL;
  c->task=task_new_sname(c);
  c->task->user_data=(gpointer)c;
  c->task->user_cancel=server_user_cancel;
  task_newtext(c->task, 0, _("Resolving %s ..."),b->host);
  task_add(c->task);
  connections_list=g_slist_append(connections_list,c);
  c->end_count=1;
  newThread(&c->thread_out, (VoidFunc)server_do_write, (VoidFunc)server_thread_end,c);
}
