/* C Mode */

/* load.c
   Support for loading Scheme files
   Originally implemented by Ken Haase in the Machine Understanding Group
     at the MIT Media Laboratory.

   Copyright (C) 1994-2001 Massachusetts Institute of Technology
   Copyright (C) 2001-2002 beingmeta, inc. (A Delaware Corporation)

   This program comes with absolutely NO WARRANTY, including implied
   warranties of merchantability or fitness for any particular purpose.

    Use, modification, and redistribution of this program is permitted
    under the terms of either (at the developer's discretion) the GNU
    General Public License (GPL) Version 2, the GNU Lesser General Public
    License.

    This program is based on the FramerD library released in Fall 2001 by
    MIT under both the GPL and the LGPL licenses, both of which accompany
    this distribution.  Subsequent modifications by beingmeta, inc. are
    also released under both the GPL and LGPL licenses (at the developer's
    discretion).
*/

static char vcid[] = "$Id: load.c,v 1.25 2002/07/05 14:32:09 haase Exp $";

/** Recording source files **/
/** File Loading **/
/** Evaluating things just once **/
/** Loading components **/
/** Initialization **/

#include "fdeval.h"

#if DYNAMIC_LINKING_ENABLED
#if HAVE_DLFCN_H
#include <dlfcn.h>
#endif
#endif

#define MODULE_TIMEOUT 30 /* seconds */

#if (defined(WIN32))
static char *dlerror()
{
  return "no dlerror support";
}
#endif

void free_env(lispenv env);
static lisp current_file_symbol, safe_symbol;

fd_exception fd_ModuleNotFound=_("Module file not found");
fd_exception fd_InvalidModule=_("Invalid module specifier");
static fd_exception ModuleAlreadyBound=_("Module name already bound");

static void set_encoding_from_header(FILE *fp,char *buf)
{
  char *header_start=strstr(buf,"-*-");
  if (header_start == NULL) return;
  else {
    char *coding_start=strstr(header_start+3,"coding:");
    char *header_end=NULL;
    if (coding_start) header_end=strstr(coding_start,"-*-");
    if ((header_end) && (header_end > coding_start)) {
      char buf[1024], *scan=coding_start+7, *write=buf;
      while ((scan < header_end) && (*scan != ';'))
	*write++=*scan++;
      *write++=0;
      fd_set_file_encoding(fp,buf);}}
}

#if FD_USING_THREADS
static fd_tld_key current_file_key;
static fd_tld_key current_env_key;
#define get_cur_file() ((FILE *) fd_tld_get(current_file_key))
#define set_cur_file(x) fd_tld_set(current_file_key,x)
#define get_cur_env() ((fd_lispenv) fd_tld_get(current_env_key))
#define set_cur_env(x) fd_tld_set(current_env_key,x)
#else
static FILE *current_file;
static fd_lispenv current_env;
#define get_cur_file() (current_file)
#define set_cur_file(x) current_file=x
#define get_cur_env() (current_env)
#define set_cur_env(x) current_env=x
#endif

static lisp lisp_set_encoding_cproc(lisp enc)
{
  FILE *f=get_cur_file();
  if (f) fd_set_file_encoding(f,fd_strdata(enc));
  else fd_set_default_encoding(fd_strdata(enc));
  return FD_VOID;
}

/** File Loading **/

FRAMERD_EXPORT
lisp fd_process_file(char *fname,char *enc,fd_lispenv env)
{
  FILE *in=fd_fopen(fname,"r"), *cur; fd_lispenv saved_env;
  if (in) {
    lisp result=FD_EMPTY_CHOICE, last_form=FD_VOID;
    lisp old_fname, new_fname;
    char buf[1024];
    UNWIND_PROTECT {
      char *absolute; size_t n;
      
      /* Save current file */
      old_fname=fd_thread_symeval(current_file_symbol);
      
      /* Dynamically bind *CURRENT-FILE* */
      
      absolute=fd_absolute_pathname(fname);
      new_fname=fd_make_string(absolute); free(absolute);
      fd_thread_symbind(current_file_symbol,new_fname);
      cur=get_cur_file(); set_cur_file(in);
      saved_env=get_cur_env(); set_cur_env(env);

      n=fread(buf,1,1023,in);
      buf[n]='\0';
      
      if (enc) fd_set_file_encoding(in,enc);
      else set_encoding_from_header(in,buf);

      if ((buf[0] == '#') && (buf[1] == '!')) {
	char *eol=strchr(buf,'\n');
	if (eol) fseek(in,eol-buf,SEEK_SET); /* Skip the first line */
	else fseek(in,0,SEEK_SET);}
      else fseek(in,0,SEEK_SET);

      /* Evaluating expressions */
      while (1) {
	lisp form=fd_parse_lisp_from_stream(in);
	if (FD_EOF_OBJECTP(form)) break;
	else {
	  decref(result); result=fd_eval_in_env(form,get_cur_env());
	  decref(last_form); last_form=form;
	}}}
    ON_UNWIND {
      if (fd_theException()) {
	fd_u8char *details=fd_exception_details();
	if (details)
	  fd_warn(_("Error <%m> while loading \"%s\""),
		  fd_theException(),fname);
	else fd_warn(_("Error <%m:%s> while loading \"%s\""),
		     fd_theException(),details,fname);

	fd_warn(_("Last form was %q"),last_form);}
      set_cur_file(cur); set_cur_env(saved_env);
      fd_fclose(in);
      fd_thread_symbind(current_file_symbol,old_fname);
      fd_decref(new_fname); fd_decref(old_fname);
      decref(last_form);}
    END_UNWIND;
    return result;}
  else fd_raise_detailed_exception(fd_FileOpenFailed,fname);
}

static lisp traced_process_file(char *fname,char *enc,fd_lispenv env)
{
  FILE *in=fd_fopen(fname,"r"), *cur; fd_lispenv saved_env;
  if (in) {
    lisp result=FD_EMPTY_CHOICE, last_form=FD_VOID;
    lisp old_fname, new_fname;
    char buf[1024];
    UNWIND_PROTECT {
      char *absolute; size_t n;
      
      /* Save current file */
      old_fname=fd_thread_symeval(current_file_symbol);
      
      /* Dynamically bind *CURRENT-FILE* */
      
      absolute=fd_absolute_pathname(fname);
      new_fname=fd_make_string(absolute); free(absolute);
      fd_thread_symbind(current_file_symbol,new_fname);
      cur=get_cur_file(); set_cur_file(in);
      saved_env=get_cur_env(); set_cur_env(env);

      n=fread(buf,1,1023,in);
      buf[n]='\0';
      
      if (enc) fd_set_file_encoding(in,enc);
      else set_encoding_from_header(in,buf);

      if ((buf[0] == '#') && (buf[1] == '!')) {
	char *eol=strchr(buf,'\n');
	if (eol) fseek(in,eol-buf,SEEK_SET); /* Skip the first line */
	else fseek(in,0,SEEK_SET);}
      else fseek(in,0,SEEK_SET);
      

      /* Evaluating expressions */
      while (1) {
	lisp form=fd_parse_lisp_from_stream(in);
	if (FD_EOF_OBJECTP(form)) break;
	else {
	  fd_fprintf(stderr,"eval:  %q\n",form);
	  decref(result); result=fd_eval_in_env(form,get_cur_env());
	  fd_fprintf(stderr,"value: %q\n",result);
	  decref(last_form); last_form=form;
	}}}
    ON_UNWIND {
      if (fd_theException()) {
	fd_warn(_("Error <%m> while loading \"%s\""),fd_theException(),fname);
	fd_warn(_("Last form was %q"),last_form);}
      set_cur_file(cur); set_cur_env(saved_env);
      fd_fclose(in);
      fd_thread_symbind(current_file_symbol,old_fname);
      decref(last_form);}
    END_UNWIND;
    return result;}
  else fd_raise_detailed_exception(fd_FileOpenFailed,fname);
}

FRAMERD_EXPORT
/* fd_load_file
     Arguments: a filename string, an encoding string, an environment
     Returns: a lisp pointer
 Loads a file into a particular environment with a particular encoding.
*/
lisp fd_load_file(char *filename,char *enc,fd_lispenv env)
{
  lisp result=fd_process_file(filename,enc,env);
  return result;
}

/* Locking modules while loading */

static char **loading_modules=NULL;
static int n_loading_modules=0;
static int loading_modules_len=0;
#if FD_USING_THREADS
static fd_mutex module_loading_lock;
#endif

static int loading_module(char *name)
{
  int i=0;
  lock_mutex(&module_loading_lock);
  while (i < n_loading_modules)
    if (strcmp(loading_modules[i],name) == 0) {
      unlock_mutex(&module_loading_lock);
      return 1;}
    else i++;

  /* Make sure there's enough space */
  if (loading_modules == NULL) {
    loading_modules=fd_malloc(sizeof(char *)*16);
    loading_modules_len=16;}
  else if (n_loading_modules == loading_modules_len) {
    loading_modules=fd_malloc(sizeof(char *)*(loading_modules_len+16));
    loading_modules_len=loading_modules_len+16;}

  /* Update the table */
  loading_modules[n_loading_modules++]=fd_strdup(name);
  unlock_mutex(&module_loading_lock);
  return 0;
}

static void done_loading_module(char *name)
{
  int i=0;
  lock_mutex(&module_loading_lock);
  while (i < n_loading_modules)
    if (strcmp(loading_modules[i],name) == 0) break;
    else i++;
  if (i == n_loading_modules) {
    unlock_mutex(&module_loading_lock);
    fd_raise_detailed_exception(_("Module not being loaded"),name);}
  fd_xfree(loading_modules[i]);
  memmove(&(loading_modules[i]),&(loading_modules[i+1]),
	  sizeof(char *)*(n_loading_modules-i));
  n_loading_modules--;
  unlock_mutex(&module_loading_lock);
}

/* This should really be done with condition vars, but it's too complicated to figure out right now. */
static int wait_for_module_to_load(char *name,int timeout)
{
  int i=0; 
  if (timeout == 0)
    fd_raise_detailed_exception(_("Timeout on module load"),name);
  sleep(1);
  lock_mutex(&module_loading_lock);
  while (i < n_loading_modules)
    if (strcmp(loading_modules[i],name) == 0) break;
    else i++;
  unlock_mutex(&module_loading_lock);
  if (i == n_loading_modules)
    return wait_for_module_to_load(name,timeout-1);
  else return 1;
}

/* Module functions */

/* This looks in <dir> for a module named <name> and returns its
   pathname if it exists. */
static fd_u8char *check_module(fd_u8char *dir,fd_u8char *name)
{
  int dirlen=strlen(dir), namelen=strlen(name), need_slash=1;
  /* Allocate a buffer we know will be more than big enough */
  fd_u8char *file=fd_malloc(dirlen+namelen+24), *suffix_pos;
  if (dir[dirlen-1] == '/') need_slash=0;
  strcpy(file,dir); if (need_slash) strcat(file,"/"); strcat(file,name);
  suffix_pos=file+(dirlen+namelen+need_slash);
  strcpy(suffix_pos,".fdx");
  if (fd_file_existsp(file)) return file;
  strcpy(suffix_pos,".so");
  if (fd_file_existsp(file)) return file;
  strcpy(suffix_pos,".dll");
  if (fd_file_existsp(file)) return file;
  strcpy(suffix_pos,"/module.fdx");
  if (fd_file_existsp(file)) return file;
  fd_free(file,dirlen+namelen+24);
  return NULL;
}

/* Checks if a files suffix makes it look like a DLL (shared object).  */
static int is_dllp(fd_u8char *file)
{
  fd_u8char *point_pos=strrchr(file,'.');
  if ((point_pos) &&
      ((strcmp(point_pos,".so") == 0) ||
       (strcmp(point_pos,".dll") == 0)))
    return 1;
  else return 0;
}

static lisp module_lookup(lisp symbol,fd_lispenv env)
{
  struct FD_MODULE *m=env->module;
  if (m == NULL) return FD_VOID;
  else if (fd_hashset_get(&(m->exports),symbol))
    return fd_hashtable_get(&(m->bindings),symbol,FD_VOID);
  else return FD_VOID;
}

static void load_module(fd_u8char *name,fd_u8char *filename)
{
  if (filename == NULL) {
    fd_u8char *lname=fd_downcase_string(name,-1);
    lisp path=fd_getpath("MYFDPATH");
    DOLIST(dir,path) {
      fd_u8char *found=check_module(fd_strdata(dir),lname);
      if (found) {filename=found; break;}}
    if (filename == NULL) {
      lisp path=fd_getpath("%FDPATH");
      DOLIST(dir,path) {
	fd_u8char *found=check_module(fd_strdata(dir),lname);
	if (found) {filename=found; break;}}}
    free(lname);}
  if (filename == NULL)
    fd_raise_detailed_exception(fd_ModuleNotFound,name);
  else {
    fd_notify("Loading module %s from %s",name,filename);
    if (is_dllp(filename))
#if (FD_USING_DLLS)
      if (fd_load_dll(filename) < 0)
	fd_raise_detailed_exception(_("DLL error"),dlerror());
      else return;
#else
      fd_raise_detailed_exception(_("No DLL loading"),filename);
#endif
    else fd_process_file(filename,NULL,fd_enabled_env);}
}

static lisp make_upper_symbol(fd_u8char *s)
{
  fd_u8char *ustring=fd_upcase_string(s,-1);
  fd_lisp sym=fd_make_symbol(ustring);
  fd_xfree(ustring);
  return sym;
}

FRAMERD_EXPORT
/* fd_get_module:
     Arguments: a module name, a filename, and an int flag
     Returns: a pointer to a lispenv or NULL
  Returns the module with the given name, loading it from filename
if needed.  If filename is non-NULL, it is used to load the module's
definition.  The final flag argument determines whether the search
includes modules which have restricted access. */
fd_lispenv fd_get_module(fd_u8char *name,fd_u8char *filename,int all)
{
  lisp sym=make_upper_symbol(name);
  lisp val=FD_VOID;
  if (all) val=module_lookup(sym,fd_restricted_modules);
  if (FD_VOIDP(val)) val=module_lookup(sym,fd_global_modules);
  if (PRIM_TYPEP(val,env_type))
    return CPTR_DATA(val);
  else if (!(FD_VOIDP(val)))
    fd_raise_lisp_exception(fd_Type_Error,_("corrupted module table"),val);
  /* Now we declare that we are loading the module and when we're done,
     we try to look it up again. */
  if (loading_module(name)) {
    wait_for_module_to_load(name,MODULE_TIMEOUT);
    if (all) val=module_lookup(sym,fd_restricted_modules);
    if (FD_VOIDP(val)) val=module_lookup(sym,fd_global_modules);
    if (PRIM_TYPEP(val,env_type)) return CPTR_DATA(val);
    else if (!(FD_VOIDP(val)))
      fd_raise_lisp_exception(fd_Type_Error,_("corrupted module table"),val);}
  /* Now load the module and declare that you're done. */
  load_module(name,filename);
  done_loading_module(name);
  /* Go get the module which should have been loaded */
  if (all) val=module_lookup(sym,fd_restricted_modules);
  if (FD_VOIDP(val)) val=module_lookup(sym,fd_global_modules);
  /* If you still haven't found it, you've failed, so return NULL */
  if (FD_VOIDP(val)) return NULL;
  else if (PRIM_TYPEP(val,env_type)) return CPTR_DATA(val);
  else fd_type_error("Module is not an environment",val);
}

static lisp use_module_handler(fd_lisp expr,fd_lispenv env)
{
  lisp modules=fd_eval_in_env(fd_get_arg(expr,1,FD_VOID),env);
  lisp filename=fd_eval_in_env(fd_get_arg(expr,2,FD_FALSE),env);
  DO_CHOICES(module,modules) {
    fd_u8char *mname, *fname;
    fd_lispenv menv=NULL;
    if (STRINGP(module)) mname=FD_STRING_DATA(module);
    else if (SYMBOLP(module)) mname=FD_SYMBOL_NAME(module);
    else if (FD_PRIM_TYPEP(module,env_type)) menv=CPTR_DATA(module);
    else fd_type_error("can't be a module specifier",module);
    if (STRINGP(filename)) fname=FD_STRING_DATA(filename);
    else fname=NULL;
    if (menv == NULL) menv=fd_get_module(mname,fname,1);
    if (menv) fd_module_uses(env,menv);
    else fd_type_error("not a module",module);}
  END_DO_CHOICES;
  return FD_VOID;
} 

static lisp use_module_restricted_handler(fd_lisp expr,fd_lispenv env)
{
  lisp modules=fd_eval_in_env(fd_get_arg(expr,1,FD_VOID),env);
  lisp filename=fd_eval_in_env(fd_get_arg(expr,2,FD_FALSE),env);
  DO_CHOICES(module,modules) {
    fd_u8char *mname, *fname;
    fd_lispenv menv=NULL;
    if (STRINGP(module)) mname=FD_STRING_DATA(module);
    else if (SYMBOLP(module)) mname=FD_SYMBOL_NAME(module);
    else if (FD_PRIM_TYPEP(module,env_type)) menv=CPTR_DATA(module);
    else fd_type_error("can't be a module specifier",module);
    if (STRINGP(filename)) fname=FD_STRING_DATA(filename);
    else fname=NULL;
    if (menv == NULL) menv=fd_get_module(mname,fname,0);
    if (menv) fd_module_uses(env,menv);
    else fd_type_error("not a module",module);}
  END_DO_CHOICES;
  return FD_VOID;
}

FRAMERD_EXPORT
/* fd_load_library
     Arguments: a filename string, an encoding string, an environment
     Returns: a lisp pointer
 Loads a file into a particular environment with a particular encoding.
This searches for the file along FDMYPATH and FDPATH.
*/
lisp fd_load_library(char *filename,char *enc,fd_lispenv env)
{
  char *fname=fd_find_file(filename,fd_getpath("FDMYPATH"));
  if (fname == NULL)
    fname=fd_find_file(filename,fd_getpath("FDPATH"));
  if (fname) {
    lisp result=fd_process_file(fname,enc,env); free(fname); 
    return result;}
  else fd_raise_detailed_exception(fd_CantFindFile,filename);
}

static fd_lispenv get_load_env(fd_lispenv start_env)
{
  fd_lispenv env=start_env;
  while (env)
    if (env->module) return env;
    else env=env->parent;
  return env;
}

static char *interpret_encoding(lisp encoding)
{
  if (SYMBOLP(encoding)) return (SYMBOL_NAME(encoding));
  else if (STRINGP(encoding)) return (STRING_DATA(encoding));
  else fd_type_error(_("not a character encoding (string or symbol)"),
		     encoding);
}

static lisp lisp_load_file_handler(lisp expr,fd_lispenv env)
{
  char *encoding; fd_lispenv load_env;
  lisp filename=fd_eval_in_env(fd_get_arg(expr,1,FD_VOID),env);
  lisp enc_var=fd_eval_in_env(fd_get_arg(expr,2,FD_FALSE),env);
  lisp env_arg=fd_eval_in_env(fd_get_arg(expr,3,FD_FALSE),env);
  lisp result=FD_VOID;
  if (!(STRINGP(filename)))
    fd_type_error(_("filename must be string"),filename);
  if (FD_FALSEP(enc_var)) encoding=NULL;
  else encoding=interpret_encoding(enc_var);
  if (FD_FALSEP(env_arg)) load_env=get_load_env(env);
  else if (PRIM_TYPEP(env_arg,env_type)) 
    load_env=CPTR_DATA(env_arg);
  else fd_type_error(_("not an environment"),env_arg);
  result=fd_load_file(STRING_DATA(filename),encoding,load_env);
  decref(filename); decref(enc_var); decref(env_arg);
  return result;
}

static lisp lisp_traced_load_handler(lisp expr,fd_lispenv env)
{
  char *encoding; fd_lispenv load_env;
  lisp filename=fd_eval_in_env(fd_get_arg(expr,1,FD_VOID),env);
  lisp enc_var=fd_eval_in_env(fd_get_arg(expr,2,FD_FALSE),env);
  lisp env_arg=fd_eval_in_env(fd_get_arg(expr,3,FD_FALSE),env);
  lisp result=FD_VOID;
  if (!(STRINGP(filename)))
    fd_type_error(_("filename must be string"),filename);
  if (FD_FALSEP(enc_var)) encoding=NULL;
  else encoding=interpret_encoding(enc_var);
  if (FD_FALSEP(env_arg)) load_env=get_load_env(env);
  else if (PRIM_TYPEP(env_arg,env_type)) 
    load_env=CPTR_DATA(env_arg);
  else fd_type_error(_("not an environment"),env_arg);
  result=traced_process_file(STRING_DATA(filename),encoding,load_env);
  decref(filename); decref(enc_var); decref(env_arg);
  return result;
}

static lisp lisp_load_library_handler(lisp expr,fd_lispenv env)
{
  char *encoding; fd_lispenv load_env;
  lisp filename=fd_eval_in_env(fd_get_arg(expr,1,FD_VOID),env);
  lisp enc_var=fd_eval_in_env(fd_get_arg(expr,2,FD_FALSE),env);
  lisp env_arg=fd_eval_in_env(fd_get_arg(expr,3,FD_FALSE),env);
  lisp result=FD_VOID;
  if (!(STRINGP(filename)))
    fd_type_error(_("filename must be string"),filename);
  if (FD_FALSEP(enc_var)) encoding=NULL;
  else encoding=interpret_encoding(enc_var);
  if (FD_FALSEP(env_arg)) load_env=get_load_env(env);
  else if (PRIM_TYPEP(env_arg,env_type)) 
    load_env=CPTR_DATA(env_arg);
  else fd_type_error(_("not an environment"),env_arg);
  result=fd_load_library(STRING_DATA(filename),NULL,load_env);
  decref(filename); decref(enc_var); decref(env_arg);
  return result;
}

static lisp lisp_load_once_handler(lisp expr,fd_lispenv env)
{
  char *encoding; fd_lispenv load_env;
  lisp filename=fd_eval_in_env(fd_get_arg(expr,1,FD_VOID),env);
  lisp enc_var=fd_eval_in_env(fd_get_arg(expr,2,FD_FALSE),env);
  lisp env_arg=fd_eval_in_env(fd_get_arg(expr,3,FD_FALSE),env);
  lisp result=FD_VOID, loaded_files;
  if (!(STRINGP(filename)))
    fd_type_error(_("filename must be string"),filename);
  if (FD_FALSEP(enc_var)) encoding=NULL;
  else encoding=interpret_encoding(enc_var);
  if (FD_FALSEP(env_arg)) load_env=get_load_env(env);
  else if (PRIM_TYPEP(env_arg,env_type)) 
    load_env=CPTR_DATA(env_arg);
  else fd_type_error(_("not an environment"),env_arg);
  loaded_files=fd_symeval(fd_make_symbol("%FILES"),load_env);
  if ((FD_VOIDP(loaded_files)) ||
      (!(fd_choice_containsp(filename,loaded_files)))) {
    if (FD_VOIDP(loaded_files)) loaded_files=incref(filename);
    else {ADD_TO_CHOICE(loaded_files,incref(filename));}
    fd_set_value(fd_make_symbol("%FILES"),loaded_files,load_env);
    result=fd_load_file(STRING_DATA(filename),NULL,load_env);}
  decref(filename); decref(enc_var); decref(env_arg); decref(loaded_files);
  return result;
}

/** Loading 'components' (from the same directory as the current file) **/

FRAMERD_EXPORT char *fd_get_component_file(char *name)
{
  lisp root=fd_thread_symeval(current_file_symbol);
  if (STRINGP(root)) {
    char *dirname=fd_dirname(STRING_DATA(root));
    int dirlen=strlen(dirname), new_size=strlen(name)+dirlen+2;
    char *new=fd_xmalloc(sizeof(char)*new_size);
    strcpy(new,dirname);
    if ((*new) && (dirname[dirlen-1] != '/')) strcat(new,"/");
    strcat(new,name);
    decref(root); free(dirname);
    return new;}
  else return fd_strdup(name);
}

static lisp lisp_get_component(lisp filename)
{
  if (!(STRINGP(filename)))
    fd_type_error(_("filename must be string"),filename);
  else return fd_init_string(fd_get_component_file(STRING_DATA(filename)),-1);
}

/** Evaluating things just once **/

static fd_hashset evaluated_once;
#if FD_USING_THREADS
static fd_mutex eval_once_lock;
#endif

static lisp eval_once(lisp action,fd_lispenv env)
{
  lock_mutex(&eval_once_lock);
  if (fd_hashset_get(evaluated_once,action)) {
    unlock_mutex(&eval_once_lock); return FD_VOID;}
  else {
    lisp value;
    WITH_HANDLING {
      fd_hashset_add(evaluated_once,action);
      unlock_mutex(&eval_once_lock);
      value=fd_eval_in_env(action,env);}
    ON_EXCEPTION {
      fd_hashset_drop(evaluated_once,action);
      fd_reraise();}
    END_HANDLING;
    return value;}
  
}

static lisp eval_once_handler(lisp expr,lispenv env)
{
  lisp action=fd_get_arg(expr,1,FD_VOID);
  return eval_once(action,env);
}

/** Loading Components **/

static lisp lisp_load_component_handler(lisp expr,lispenv env)
{
  char *component_name, *encoding; fd_lispenv load_env;
  lisp filename=fd_eval_in_env(fd_get_arg(expr,1,FD_VOID),env);
  lisp enc_var=fd_eval_in_env(fd_get_arg(expr,2,FD_FALSE),env);
  lisp env_arg=fd_eval_in_env(fd_get_arg(expr,3,FD_FALSE),env);
  lisp result;
  if (!(STRINGP(filename)))
    fd_type_error(_("filename must be string"),filename);
  if (FD_FALSEP(enc_var)) encoding=NULL;
  else encoding=interpret_encoding(enc_var);
  if (FD_FALSEP(env_arg)) load_env=get_load_env(env);
  else if (PRIM_TYPEP(env_arg,env_type)) 
    load_env=CPTR_DATA(env_arg);
  else fd_type_error(_("not an environment"),env_arg);
  component_name=fd_get_component_file(STRING_DATA(filename));
  if (component_name == NULL) {
    fd_u8char *with_suffix=fd_malloc(STRING_LENGTH(filename)+10);
    strcpy(with_suffix,STRING_DATA(filename));
    strcat(with_suffix,".fdx");
    component_name=fd_get_component_file(with_suffix);
    fd_free(with_suffix,STRING_LENGTH(filename)+10);}
  if (component_name) 
    result=fd_load_file(component_name,NULL,load_env);
  else fd_raise_detailed_exception(fd_CantFindFile,component_name);
  decref(filename); decref(enc_var); decref(env_arg);
  free(component_name);
  return result;
}

/** Loading configurations **/

static lisp load_config_cproc(lisp filename_arg)
{
  char *filespec=NULL, *filename;
  /* Convert lisp arg to string */
  if (STRINGP(filename_arg))
    filespec=fd_strdup(STRING_DATA(filename_arg));
  else if (SYMBOLP(filename_arg))
    filespec=fd_string_getenv(SYMBOL_NAME(filename_arg));
  else fd_type_error(_("filename must be string"),filename_arg);
  /* Try to find a file which matches */
  filename=fd_find_file(filespec,fd_getpath("MYFDPATH"));
  if (filename == NULL)
    filename=fd_find_file(filespec,fd_getpath("FDPATH"));
  if (filename == NULL) { /* Try with .cfg suffix */
    char *buf=fd_malloc(strlen(filespec)+10);
    strcpy(buf,filespec); strcat(buf,".cfg");
    filename=fd_find_file(buf,fd_getpath("MYFDPATH"));
    if (filename == NULL)
      filename=fd_find_file(buf,fd_getpath("FDPATH"));
    fd_free(buf,strlen(filespec)+10);}
  if (filename) {
    fd_load_config(filename);
    fd_xfree(filename); fd_xfree(filespec);
    return FD_VOID;}
  else fd_raise_detailed_exception(fd_CantFindFile,filespec);
}

static lisp lisp_load_user_profile_cproc()
{
  fd_load_user_profile();
  return FD_VOID;
}

static lisp lisp_get_config_file_cproc()
{
  char *file=fd_get_config_file();
  if (file)
    return fd_make_string(file);
  else return FD_EMPTY_CHOICE;
}

/** Setting the current module */


#if 0
static void in_module_core(lisp expr,fd_lispenv env,int risky)
{
  lisp module_name=fd_get_arg(expr,1,FD_VOID), module;
  if (SYMBOLP(module_name))
    module=fd_symeval(module_name,env);
  else module=fd_eval_in_env(module_name,env);
  if ((SYMBOLP(module_name)) && (FD_VOIDP(module))) {
    fd_lispenv new_module=fd_make_module();
    module=fd_make_cptr(env_type,new_module);
    fd_bind_value(module_name,module,env);
    fd_bind_value(module_name,module,new_module);
    fd_module_uses(new_module,fd_global_env);
    fd_decref(module);
    set_cur_env(new_module);}
  else if (PRIM_TYPEP(module,env_type)) {
    fd_decref(module);
    set_cur_env(CPTR_DATA(module));}
  else if (SYMBOLP(module)) {
    lisp real_module=FD_VOID;
    if (risky)
      real_module=module_lookup(module,fd_restricted_modules);
    if (FD_VOIDP(real_module))
      real_module=module_lookup(module,fd_global_modules);
    if (PRIM_TYPEP(real_module,env_type))
      set_cur_env(CPTR_DATA(real_module));
    else if (FD_VOIDP(real_module)) {
      fd_lispenv fresh=fd_registered_module(SYMBOL_NAME(module),risky); 
      set_cur_env(fresh);}
    else fd_raise_lisp_exception
	   (fd_Type_Error,_("Corrupted module table"),module_name);}
  else {
    fd_decref(module);
    fd_type_error(_("not a module"),module_name);}
}

static lisp lisp_in_module_handler(lisp expr,fd_lispenv env)
{
  lisp arg2=fd_eval_in_env(fd_get_arg(expr,2,FD_TRUE),env);
  if (FD_TRUEP(arg2)) in_module_core(expr,env,1);
  else in_module_core(expr,env,0);
  fd_decref(arg2);
  return FD_VOID;
}

static lisp lisp_safe_in_module_handler(lisp expr,fd_lispenv env)
{
  lisp arg2=fd_eval_in_env(fd_get_arg(expr,2,FD_FALSE),env);
  if (FD_TRUEP(arg2))
    fd_raise_exception(_("Can't declare risky environment here"));
  else in_module_core(expr,env,0);
  fd_decref(arg2);
  return FD_VOID;
}
#endif

/** New IN-MODULE */

static fd_lisp get_unregistered_module
    (fd_lisp module_name,fd_lispenv env,int enabled)
{
  fd_lisp module;
  if (SYMBOLP(module_name))
    module=fd_symeval(module_name,env);
  else fd_type_error("not a module name",module_name);
  if (FD_VOIDP(module)) {
    fd_lispenv fresh=fd_make_module();
    if (enabled) fd_module_uses(fresh,fd_enabled_env);
    else fd_module_uses(fresh,fd_global_env);
    module=fd_make_cptr(env_type,fresh);
    fd_bind_value(module_name,module,env);
    fd_bind_value(module_name,module,fresh);
    return module;}
  else if (FD_VOIDP(module)) return module;
  else if (PRIM_TYPEP(module,env_type)) return module;
  else fd_raise_lisp_exception
	 (ModuleAlreadyBound,FD_SYMBOL_NAME(module_name),module);  
}

static fd_lisp get_registered_module(fd_lisp arg,int enabled)
{
  if (FD_PRIM_TYPEP(arg,env_type)) return fd_incref(arg);
  else if (FD_SYMBOLP(arg))
    if (enabled) {
      fd_lisp regmod=module_lookup(arg,fd_restricted_modules);
      if (FD_VOIDP(regmod)) regmod=module_lookup(arg,fd_global_modules);
      return regmod;}
    else {
      fd_lisp gmod=module_lookup(arg,fd_global_modules);
      return gmod;}
  else fd_type_error(fd_InvalidModule,arg);
}

static void initialize_module(fd_lispenv env,fd_lisp use_spec,int enabled)
{
  int safep=0;
  {DO_CHOICES(to_use,use_spec)
     if (FD_LISP_EQ(to_use,safe_symbol)) safep=1;
   END_DO_CHOICES;}
  if ((enabled == 0) || (safep)) 
    fd_module_uses(env,fd_global_env);
  else fd_module_uses(env,fd_enabled_env);
  {DO_CHOICES(to_use,use_spec) {
    fd_lispenv use_env=NULL;
    if (FD_LISP_EQ(to_use,safe_symbol)) {}
    else {
      if (FD_PRIM_TYPEP(to_use,env_type))
	use_env=FD_CPTR_DATA(to_use);
      else if (FD_SYMBOLP(to_use)) {
	fd_lisp module=get_registered_module(to_use,enabled);
	if (FD_PRIM_TYPEP(module,env_type)) use_env=FD_CPTR_DATA(module);
	else {
	  use_env=fd_get_module(FD_SYMBOL_NAME(to_use),NULL,enabled);}
	fd_decref(module);}
      if (use_env) fd_module_uses(env,use_env);}}
   END_DO_CHOICES;}
}

static fd_lisp lisp_neutered_in_module_handler(lisp expr,fd_lispenv env)
{
  fd_lisp module_name=fd_get_arg(expr,1,FD_VOID), module;
  fd_lisp module_uses=fd_eval_in_env(fd_get_arg(expr,2,FD_TRUE),env);
  if (SYMBOLP(module_name)) 
    module=get_unregistered_module(module_name,env,0);
  else module=fd_eval_in_env(expr,env);
  if (PRIM_TYPEP(module,env_type)) {
    initialize_module(FD_CPTR_DATA(module),module_uses,1);
    fd_decref(module_uses); set_cur_env(FD_CPTR_DATA(module));}
  else fd_type_error
	 ("Access restriction: IN-MODULE only works with unregistered modules",
	  module_name);
  return FD_VOID;
}
  
static fd_lisp lisp_in_module_handler(lisp expr,fd_lispenv env)
{
  fd_lisp module_name=fd_get_arg(expr,1,FD_VOID), module;
  fd_lisp module_uses=fd_eval_in_env(fd_get_arg(expr,2,FD_TRUE),env);
  if (FD_SYMBOLP(module_name)) 
    module=get_unregistered_module(module_name,env,1);
  else module=fd_eval_in_env(module_name,env);
  if (FD_PRIM_TYPEP(module,env_type)) {
    initialize_module(FD_CPTR_DATA(module),module_uses,1);
    fd_decref(module_uses); set_cur_env(FD_CPTR_DATA(module));}
  else if (FD_SYMBOLP(module)) {
    fd_lisp regmod=get_registered_module(module,1);
    fd_lispenv modenv;
    if (FD_PRIM_TYPEP(regmod,env_type)) modenv=FD_CPTR_DATA(regmod);
    else {
      modenv=fd_registered_module(FD_SYMBOL_NAME(module),1);
      regmod=fd_make_cptr(env_type,modenv);}
    initialize_module(modenv,module_uses,1);
    fd_decref(module_uses); set_cur_env(modenv);}
  else fd_type_error(fd_InvalidModule,module_name);
  return FD_VOID;
}

static fd_lisp lisp_in_safe_module_handler(lisp expr,fd_lispenv env)
{
  fd_lisp module_name=fd_get_arg(expr,1,FD_VOID), module;
  fd_lisp module_uses=fd_eval_in_env(fd_get_arg(expr,2,FD_TRUE),env);
  if (FD_SYMBOLP(module_name)) 
    module=get_unregistered_module(module_name,env,1);
  else module=fd_eval_in_env(module_name,env);
  if (FD_PRIM_TYPEP(module,env_type)) {
    initialize_module(FD_CPTR_DATA(module),module_uses,1);
    fd_decref(module_uses); set_cur_env(FD_CPTR_DATA(module));}
  else if (FD_SYMBOLP(module)) {
    fd_lisp regmod=get_registered_module(module,0);
    fd_lispenv modenv;
    if (FD_PRIM_TYPEP(regmod,env_type)) modenv=FD_CPTR_DATA(regmod);
    else {
      modenv=fd_registered_module(FD_SYMBOL_NAME(module),0);
      regmod=fd_make_cptr(env_type,modenv);}
    initialize_module(modenv,module_uses,1);
    fd_decref(module_uses); set_cur_env(modenv);}
  else fd_type_error(fd_InvalidModule,module_name);
  return FD_VOID;
}

/** WITHIN-MODULE **/

static lisp lisp_within_module_handler(lisp expr,fd_lispenv env)
{
  lisp module_spec=fd_get_arg(expr,1,FD_FALSE);
  lisp module_id=fd_eval_in_env(module_spec,env);
  lisp body=fd_get_body(expr,2), value=FD_VOID;
  fd_lispenv module=NULL;
  if (FD_SYMBOLP(module_id)) {
    fd_lisp v=module_lookup(module_id,fd_restricted_modules);
    if (!(FD_PRIM_TYPEP(v,env_type)))
      v=module_lookup(module_id,fd_global_modules);
    if (FD_PRIM_TYPEP(v,env_type)) module=CPTR_DATA(v);}
  else if (FD_PRIM_TYPEP(module_id,env_type))
    module=CPTR_DATA(module_id);
  else fd_type_error(_("not a module"),module_id);
  if (module == NULL)
    fd_raise_lisp_exception
      ("The module is not defined","WITHIN-MODULE",module_id);
  else {
    DOLIST(expr,body) {
      fd_decref(value); value=fd_eval_in_env(expr,module);}
    return value;}
}

static lisp lisp_safe_within_module_handler(lisp expr,fd_lispenv env)
{
  lisp module_spec=fd_get_arg(expr,1,FD_FALSE);
  lisp module_id=fd_eval_in_env(module_spec,env);
  lisp body=fd_get_body(expr,2), value=FD_VOID;
  fd_lispenv module=NULL;
  if (FD_SYMBOLP(module_id)) {
    fd_lisp v=module_lookup(module_id,fd_global_modules);
    if (FD_PRIM_TYPEP(v,env_type)) module=CPTR_DATA(v);}
  else if (FD_PRIM_TYPEP(module_id,env_type))
    module=CPTR_DATA(module_id);
  else fd_type_error(_("not a module"),module_id);
  if (module == NULL)
    fd_raise_lisp_exception
      ("The module is not defined","WITHIN-MODULE",module_id);
  else {
    DOLIST(expr,body) {
      fd_decref(value); value=fd_eval_in_env(expr,module);}
    return value;}
}

/** Initialization **/

void fd_initialize_load_c()
{
#if FD_USING_THREADS
  fd_init_mutex(&module_loading_lock);
  fd_init_mutex(&eval_once_lock);
  fd_new_tld_key(&current_file_key,NULL);
  fd_new_tld_key(&current_env_key,NULL);
#endif
  
  evaluated_once=fd_make_hashset(15);

  current_file_symbol=fd_make_symbol("*CURRENT-FILE*");
  safe_symbol=fd_make_symbol("SAFE");

  fd_add_restricted_special_form("LOAD",lisp_load_file_handler);
  fd_add_restricted_special_form("TRACED-LOAD",lisp_traced_load_handler);
  fd_add_restricted_special_form("LOAD-FILE",lisp_load_file_handler);
  fd_add_restricted_special_form("LOAD-LIBRARY",lisp_load_library_handler);
  fd_add_restricted_special_form("LOAD-ONCE",lisp_load_once_handler);
  fd_add_restricted_special_form("LOAD-COMPONENT",lisp_load_component_handler);
  fd_add_restricted_cproc("GET-COMPONENT",1,lisp_get_component);
  fd_add_restricted_special_form("EVAL-ONCE",eval_once_handler);

  fd_add_alias(fd_enabled_env,"TLOAD","TRACED-LOAD");

  fd_add_special_form
    (fd_global_env,"IN-MODULE",lisp_neutered_in_module_handler);
  fd_add_special_form
    (fd_global_env,"USE-MODULE",use_module_restricted_handler);
  fd_add_alias
    (fd_global_env,"USE-MODULE!","USE-MODULE");
  fd_add_restricted_special_form("IN-MODULE",lisp_in_module_handler);
  fd_add_restricted_special_form("IN-SAFE-MODULE",lisp_in_safe_module_handler);
  fd_add_restricted_special_form("USE-MODULE",use_module_handler);
  fd_add_alias(fd_enabled_env,"USE-MODULE!","USE-MODULE");

  fd_add_special_form
    (fd_global_env,"WITHIN-MODULE",lisp_safe_within_module_handler);
  fd_add_special_form
    (fd_enabled_env,"WITHIN-MODULE",lisp_within_module_handler);

  fd_add_restricted_cproc("LOAD-CONFIG",1,load_config_cproc);
  fd_add_restricted_cproc("LOAD-USER-PROFILE",0,lisp_load_user_profile_cproc);
  fd_add_restricted_cproc("GET-CONFIG-FILE",0,lisp_get_config_file_cproc);

  fd_add_restricted_cproc("SET-ENCODING!",1,lisp_set_encoding_cproc);

  fd_register_source_file("load",__DATE__,vcid);
}




/* File specific stuff */

/* The CVS log for this file
   $Log: load.c,v $
   Revision 1.25  2002/07/05 14:32:09  haase
   New in-module implementation

   Revision 1.24  2002/06/24 18:08:33  haase
   Fixed some source file registrations

   Revision 1.23  2002/06/15 20:42:31  haase
   Made IN-MODULE default to the same level of security as the current context

   Revision 1.22  2002/06/15 20:01:49  haase
   Fixes to risky arg to in-module

   Revision 1.21  2002/05/27 18:16:34  haase
   Added abstraction layer for thread-local data

   Revision 1.20  2002/05/01 21:46:31  haase
   Renamed mutex/condvar/rwlock types to have fd_ prefixes

   Revision 1.19  2002/04/27 17:47:18  haase
   Moved mutex/lock init and destroy into FramerD abstraction layer

   Revision 1.18  2002/04/22 11:57:53  haase
   Added FD_USING_THREAD conditional where it was needed

   Revision 1.17  2002/04/20 19:47:48  haase
   Renamed fd_hashset_zap to fd_hashset_drop

   Revision 1.16  2002/04/19 00:16:13  haase
   Fixed sense-inverting typo on module locking

   Revision 1.15  2002/04/17 17:50:36  haase
   Fixed some inconsistent returns

   Revision 1.14  2002/04/17 13:16:26  haase
   Decrement number of loading modules when finished with a module; renamed FDPATH to %FDPATH

   Revision 1.13  2002/04/17 12:25:36  haase
   Made locks for module loading to avoid double loading

   Revision 1.12  2002/04/17 11:46:11  haase
   Switched internal UTF-8 representation to real UTF8

   Revision 1.11  2002/04/10 17:10:18  haase
   Made error messags passed out through LOAD be more informative

   Revision 1.10  2002/04/02 21:39:32  haase
   Added log and emacs init entries to C source files

*/

/* Emacs local variables
;;;  Local variables: ***
;;;  compile-command: "cd ../..; make" ***
;;;  End: ***
*/
