########################################################################
#
#   class:  Shape
#
# purpose:  Draw a geometric shape on the canvas and allow the user
#           to drag or reshape it.
#         
#   usage:  To create a shape on a canvas:
#               gShape canvas
#           To set its options use "setShape", "setCoords", "setColor",
#           and "setRotation", or the convience method "setFullShape".
#
#           To have the shape use additional tags to which you can bind,
#           use "addTags".
#
#           Messages will be passed to the shape's owner (via a callback)
#           when the shape changes.  Become the shape's owner with the
#           "setOwner" method.  Messages currently passed are:
#                  shapeIsBeingModified (when user changes shape/coord/rot)
#                  shapeHasChanged      (when a series of changes ends)
#                  shapeIsSelected      (when a shape becomes selected)
#
########################################################################

class Shape {
   constructor { canvas } {}
   destructor {}

   public {
      method draw     {}
      method select   {}
      method deselect {}
      method clippedPolygon {}

      method beginModification  { {x 0} {y 0} }
      method finishModification {}

      method setClip      { x1 y1 x2 y2 }
      method setScale     { mx my       }
      method setFullShape { shp crds rot}
      method setCoords    { coords      }
      method setShape     { shape       }
      method setRotation  { rot         }
      method setColor     { clr         } { set itsColor $clr    }
      method setOwner     { owner       } { set itsOwner $owner  }

      method getShape     {             } { return $itsShape     }
      method getCoords    {             } { return $itsParams    }
      method getRotation  {             } { return $itsRotation  }

      method insertPt  { index      }
      method adjustPt  { index x  y }
      method rotatePt  {       x  y }
      method shift     {      dx dy }
      method drag      {       x  y }

      method getPolygon  { }
      method drawHandles { }

      method addTags    { tags }
      method removeTags { tags }

      method notifyOwner { args }

      method enterLeaveShape { mode }
   }


   protected {
      variable itsParams   {}
      variable itsRotation 0.0
      variable itsShape    "Circle"
      variable itsCanvas

      variable xScale 1.0
      variable yScale 1.0

      variable itsTag    ""
      variable allTags   ""
      variable itsOwner  ""
      variable itsColor  "black"
      variable itsIds    {}

      variable isSelected 0
      variable isBeingModified 0

      variable clipRect {}
      variable ignoreClip 0
      variable startX 0.0
      variable startY 0.0
      variable tmpIds {}

      method CircleToPoly  { x0 y0 dx dy }
      method EllipseToPoly { x0 y0 dx dy angle }
      method BoxToPoly     { x0 y0 dx dy angle }
      method PolyToPoly    { descr }
      method PointToPoly   { x0 y0 }
   }

   private {
      variable tmpAdjustPoint -1
      method checkPolygonPoint { index }
   }

}

########################################################################
#
#  gShape canvas
#
########################################################################

proc gShape { args } {
   return [uplevel #0 Shape #auto $args]
}

########################################################################
#
#
#
########################################################################

body Shape::constructor { canvas } {
   set itsCanvas $canvas

   set itsTag shp[namespace tail $this]
   $itsCanvas bind "$itsTag && DragAble" <Enter> \
         [code $this enterLeaveShape Enter]
   $itsCanvas bind "$itsTag && DragAble" <Leave> \
         [code $this enterLeaveShape Leave]

   $itsCanvas bind rgnHandle$itsTag <Enter> \
         "$itsCanvas configure -cursor sizing"
   $itsCanvas bind rgnHandle$itsTag <Leave> \
         "$itsCanvas configure -cursor \$powcursor"

   powBindBtn <<RGN_Drag>> "$itsCanvas bind \"$itsTag && DragAble\"" \
         [code $this beginModification %x %y] \
         [code $this drag %x %y] \
         [code $this finishModification]

   set allTags [list $itsTag shape DragAble]
}

body Shape::enterLeaveShape { mode } {
   if { $isBeingModified } return
   if { $mode=="Enter" } {
      $itsCanvas itemconfig $itsTag -width 2
      $itsCanvas configure -cursor fleur
   } else {
      $itsCanvas itemconfig $itsTag -width 1
      $itsCanvas configure -cursor $::powcursor
   }
}


body Shape::destructor {} {
   if { [winfo exists $itsCanvas] } {
      if { [llength $itsIds] } {
         $itsCanvas delete $itsTag
      }
      if { $isSelected } {
         $itsCanvas delete rgnHandle$itsTag
      }
   }
   notifyOwner shapeHasDied
}


body Shape::notifyOwner { args } {
   if { $itsOwner!="" } {
      eval $itsOwner $this $args
   }
}


body Shape::clippedPolygon { } {

   set coords [getPolygon]

   if {         ![llength $coords]          } { return ""              }
   if { $ignoreClip || ![llength $clipRect] } { return [list $coords]  }

   # Find bounding box of polygon

   set xMin [lindex $coords 0]
   set yMin [lindex $coords 1]
   set xMax $xMin
   set yMax $yMin
   foreach [list x y] [lrange $coords 2 end] {
      if { $x < $xMin } { set xMin $x } elseif { $x > $xMax } { set xMax $x }
      if { $y < $yMin } { set yMin $y } elseif { $y > $yMax } { set yMax $y }
   }

   #  Is polygon clipped by clipRect?

   foreach [list x1 y1 x2 y2] $clipRect {}
   if { $xMin > $x2 || $xMax < $x1 || $yMin > $y2 || $yMax < $y1 } {
      # poly is fully outside rectangle so is fully clipped
      return ""
   } elseif { $xMin >= $x1 && $xMax <= $x2 && $yMin >= $y1 && $yMax <= $y2 } {
      # poly is fully inside rectangle so no need to clip
      return [list $coords]
   } else {
      # poly overlaps rectangle bounds, so needs to be clipped
      set clipPoly [list $x1 $y1 $x1 $y2 $x2 $y2 $x2 $y1 $x1 $y1]
      return [powClipPolys $coords $clipPoly]
   }
}

body Shape::draw {} {

   set coords [clippedPolygon]
   if { $coords=="" } {
      $itsCanvas delete $itsTag
      set itsIds {}
      return
   }

   set newIds {}
   for { set i 0 } { $i < [llength $coords] } { incr i } {
      set coord [lindex $coords $i]
      set id [lindex $itsIds $i]

      if { $id == "" || [$itsCanvas find withtag $id]=="" } {
         lappend newIds [eval $itsCanvas create polygon \
               $coord -outline $itsColor -fill {{}} \
               -tags \$allTags ]
      } else {
         eval $itsCanvas coords $id $coord
         $itsCanvas itemconfigure $id -outline $itsColor
         lappend newIds $id
      }
   }
   while { $i<[llength $itsIds] } {
      set id [lindex $itsIds $i]
      $itsCanvas itemconfig $id -outline {}
      lappend tmpIds $id
      incr i
   }

   set itsIds $newIds
}

body Shape::addTags { tags } {
   foreach t $tags {
      if { [lsearch $allTags $t]==-1 } {
         lappend allTags $t
         if { [llength $itsIds] } {
            $itsCanvas addtag $t withtag $itsTag
         }
      }
   }
}

body Shape::removeTags { tags } {
   foreach t $tags {
      set idx [lsearch $allTags $t]
      if { $idx != -1 } {
         set allTags [lreplace $allTags $idx $idx]
         if { [llength $itsIds] } {
            $itsCanvas dtag $itsTag $t
         }
      }
   }
}

body Shape::deselect {} {
   $itsCanvas delete rgnHandle$itsTag
   set isSelected 0
}

body Shape::select {} {

   drawHandles

   $itsCanvas raise $itsTag
   $itsCanvas raise rgnHandle$itsTag
   if { ! $isSelected } {
      set isSelected 1
      notifyOwner shapeIsSelected
   }
}

body Shape::drawHandles { } {

   $itsCanvas delete rgnHandle$itsTag

   set coords [getCoords]
   set shape  [getShape]
   set rot    [getRotation]

   if { $shape == "Point" } return

   if { $shape=="Line" || $shape=="Polygon" } {
      set start 0
      set end   [expr [llength $coords]-1]
   } else {
      set start 2
      set end   3
   }

   set x0 [lindex $coords 0]
   set y0 [lindex $coords 1]
   if { !$ignoreClip && [llength $clipRect] } {
      foreach [list bx1 by1 bx2 by2] $clipRect {}
   } else {
      foreach [list bx1 by1 bx2 by2] [list -32000 -32000 32000 32000] {}
   }

# Create Move Point Handles

   set ptNum [expr $start/2]
   foreach {x y} [lrange $coords $start $end] {
      foreach {x y} [poly_rotate $x0 $y0 $rot [list $x $y] ] {}
      if { $x<$bx2 && $x>$bx1 && $y<$by2 && $y>$by1 } {
         set x1 [expr $x-2]
         set y1 [expr $y-2]
         set x2 [expr $x+2]
         set y2 [expr $y+2]
         set id [$itsCanvas create polygon $x1 $y1 $x1 $y2 $x2 $y2 $x2 $y1 \
               -outline green -fill {} \
               -tags "rgnHandle$itsTag rgnMovePt rgnHandle"]
         
         powBindBtn <<RGN_DragPt>> "$itsCanvas bind $id" \
               [code $this beginModification %x %y] \
               [code $this adjustPt $ptNum %x %y] \
               [code $this finishModification]
         
         powBindBtn <<RGN_InsertPt>> "$itsCanvas bind $id" \
              "[code $this beginModification %x %y]; [code $this insertPt $ptNum]" \
               [code $this adjustPt $ptNum %x %y] \
               [code $this finishModification]
      }
      incr ptNum
   }

# Create Rotate Region Handle... only for Box and Ellipse

   if { $shape=="Box" || $shape=="Ellipse" } {
      set dx [expr [lindex $coords 2]-$x0]
      if { $shape=="Ellipse" } {set dx [expr 1.41421356*$dx]}
      set x [expr $x0+$dx]
      set y [lindex $coords 1]
      foreach {x y} [poly_rotate $x0 $y0 $rot [list $x $y] ] {}
      if { $x<$bx2 && $x>$bx1 && $y<$by2 && $y>$by1 } {
         set x1 [expr $x-2]
         set y1 [expr $y-2]
         set x2 [expr $x+2]
         set y2 [expr $y+2]
         set id [$itsCanvas create polygon $x1 $y2 $x2 $y2 $x $y1 \
               -outline green -fill {} \
               -tags "rgnHandle$itsTag rgnRotate rgnHandle"]

         powBindBtn <<RGN_Rotate>> "$itsCanvas bind $id" \
               [code $this beginModification %x %y] \
               [code $this rotatePt %x %y] \
               [code $this finishModification]
      }
   }
}

body Shape::setClip { x1 y1 x2 y2 } {
   if { $x1 > $x2 } {
      set tmp $x1
      set x1  $x2
      set x2  $tmp
   }
   if { $y1 > $y2 } {
      set tmp $y1
      set y1  $y2
      set y2  $tmp
   }
   set clipRect [list $x1 $y1 $x2 $y2]
}


body Shape::setScale { mx my } {
   set xScale $mx
   set yScale $my
}


body Shape::setShape { shape } {
   set itsShape $shape
   if { $isBeingModified } {
      notifyOwner shapeIsBeingModified
   }
}

body Shape::setRotation { rot } {
   set itsRotation $rot
   if { $isBeingModified } {
      notifyOwner shapeIsBeingModified
   }
}

body Shape::setCoords { coords } {
   set nelem [llength $coords]
   if { [expr $nelem%2] } {
      error "Shape coordinates must contain an even number of elements"
   }
   set itsParams $coords
   if { $isBeingModified } {
      notifyOwner shapeIsBeingModified
   }
}

body Shape::setFullShape { shp crds rot } {
   setShape    $shp
   setRotation $rot
   setCoords   $crds
}


body Shape::insertPt { index } {
   #  Can only insert points into a Polygon.  Ignore all others
   if { [getShape] == "Polygon" } {
      set start [expr $index*2]
      set coords [getCoords]
      foreach {x y} [lrange $coords $start [expr $start+1] ] {}
      setCoords [linsert $coords $start $x $y]
   }
}

body Shape::adjustPt { index x y } {
   set tmpAdjustPoint $index

   set origX [$itsCanvas canvasx $x]
   set origY [$itsCanvas canvasy $y]

   set start [expr $index*2]
   set end   [expr $start+1]

   set coords [getCoords]
   set rot    [getRotation]

   if { abs($rot)>1e-10 } {
      set x0 [lindex $coords 0]
      set y0 [lindex $coords 1] 
      foreach {x y} [poly_rotate $x0 $y0 \
            [expr -$rot] [list $origX $origY]] {}
   } else {
      set x $origX
      set y $origY
   }

   switch [getShape] {
      "Point" {
         set coords [list $x $y $x $y]
      }
      "Box" {
         # Drawn/Sized to keep one corner fixed, so becomes complicated
         foreach {x0 y0 x2 y2} $coords {}
         set x1 [expr $x0-$x2+$x0]
         set y1 [expr $y0-$y2+$y0]
         if { abs($rot)>1e-10 } {
            foreach {x1 y1} [poly_rotate $x0 $y0 \
                  $rot [list $x1 $y1]] {}
         }
         set x0 [expr 0.5*($x1+$origX)]
         set y0 [expr 0.5*($y1+$origY)]

         if { abs($rot)>1e-10 } {
            foreach {x y} [poly_rotate $x0 $y0 \
                  [expr -$rot] [list $origX $origY]] {}
         }

         set coords [list $x0 $y0 $x $y]
      }
      "Circle" {
         # Need to set second point at 45 deg angle
         set x0 [lindex $coords 0]
         set y0 [lindex $coords 1]
         set dX [expr ($x0-$x)/$xScale]
         set dY [expr ($y0-$y)/$yScale]
         set dR [expr sqrt( 0.5*($dX*$dX + $dY*$dY) )]
         set x [expr $x0 + $dR*$xScale]
         set y [expr $y0 + $dR*$yScale]
         set coords [lreplace $coords 2 3 $x $y]
      }
      default {
         set coords [lreplace $coords $start $end $x $y]
      }
   }
   setCoords $coords
   draw
}

body Shape::checkPolygonPoint { index } {
   set coords [getCoords]
   set npts [llength $coords]

   #  Keep at least 2 points
   if { $npts <= 4 } return

   set start1 [expr $index*2]
   foreach [list x1 y1] [lrange $coords $start1 [expr $start1+1] ] {}

   set start0 [expr $start1-2]
   if {$start0<0} {set start0 [expr $npts-2]}
   foreach [list x0 y0] [lrange $coords $start0 [expr $start0+1] ] {}
   set dx [expr $x1-$x0]
   set dy [expr $y1-$y0]
   set r [expr $dx*$dx+$dy*$dy] 
   if { $r<16 } {

      setCoords [lreplace $coords $start1 [expr $start1+1] ]

   } else {

      set start0 [expr $start1+2]
      if {$start0>=$npts} {set start0 0}
      foreach [list x0 y0] [lrange $coords $start0 [expr $start0+1] ] {}
      set dx [expr $x1-$x0]
      set dy [expr $y1-$y0]
      set r [expr $dx*$dx+$dy*$dy] 
      if { $r<16 } {
         setCoords [lreplace $coords $start1 [expr $start1+1] ]
      }
   }
}

body Shape::beginModification { {x 0} {y 0} } {

   set x [$itsCanvas canvasx $x]
   set y [$itsCanvas canvasy $y]

   set startX $x
   set startY $y
   set ignoreClip 1
   set isBeingModified 1
}

body Shape::finishModification { } {
   if { $tmpAdjustPoint != -1 && [getShape]=="Polygon" } {
      #  If adjusted point is too close to an adjacent point, delete it
      checkPolygonPoint $tmpAdjustPoint
   }
   set tmpAdjustPoint -1
   set ignoreClip 0
   draw
   select
   foreach id $tmpIds {
      $itsCanvas delete $id
   }
   set tmpIds {}
   set isBeingModified 0

   notifyOwner shapeHasChanged
}


body Shape::drag { x y } {
   set x [$itsCanvas canvasx $x]
   set y [$itsCanvas canvasy $y]

   set dx [expr $x - $startX]
   set dy [expr $y - $startY]
   shift $dx $dy
   set startX $x
   set startY $y
}

body Shape::shift { dx dy } {
   set newParams {}
   foreach [list x y] [getCoords] {
      set x [expr $x + $dx]
      set y [expr $y + $dy]
      lappend newParams $x $y
   }
   setCoords $newParams
   draw
}

body Shape::rotatePt { x y } {
   set x [$itsCanvas canvasx $x]
   set y [$itsCanvas canvasy $y]

   set coords [getCoords]
   set dx [expr $x-[lindex $coords 0]]
   set dy [expr $y-[lindex $coords 1]]

   setRotation [expr -atan2($dy,$dx)*180.0/3.1415926535]
   draw
}


body Shape::getPolygon { } {

   set cnt 0
   set coords [getCoords]
   foreach {x y} $coords { 
      set x$cnt $x
      set y$cnt $y
      incr cnt
   }
   if { $cnt==0 } {
      return {}
   }

   if {$cnt>1} {
      set dx [expr $x1-$x0]
      set dy [expr $y1-$y0]
   }

   set rot [getRotation]

   switch [getShape] {
      Box     { set coords [BoxToPoly $x0 $y0 $dx $dy $rot] }
      Circle  { set coords [CircleToPoly $x0 $y0 $dx $dy] }
      Ellipse { set coords [EllipseToPoly $x0 $y0 [expr 1.41421356*$dx] \
                                                  [expr 1.41421356*$dy] $rot] }
      Polygon { set coords [PolyToPoly $coords] }
      Line    { set coords [list $x0 $y0 $x1 $y1 $x0 $y0] }
      Point   { set coords [PointToPoly $x0 $y0] }
   }
   return $coords
}

body Shape::CircleToPoly { x0 y0 dx dy } {
   global powPlotParam regionParam

   set xRad [expr $dx/$xScale]
   set yRad [expr $dy/$yScale]
   set radius [expr sqrt($xRad*$xRad + $yRad*$yRad)]
   set xRad [expr $radius*$xScale]
   set yRad [expr $radius*$yScale]
   set points ""
   foreach {x y} [circle] {
      set x [expr $xRad*$x+$x0]
      set y [expr $yRad*$y+$y0]
      lappend points $x $y
   }
   return $points
}

body Shape::EllipseToPoly { x0 y0 dx dy angle } {
    set points ""
    foreach {x y} [circle] {
	set x [expr $dx*$x+$x0]
	set y [expr $dy*$y+$y0]
	lappend points $x $y
    }
    if {[expr abs($angle)] < 1e-10} {return $points}
    return [poly_rotate $x0 $y0 $angle $points]
}

body Shape::BoxToPoly { x0 y0 dx dy angle } {
    set points ""
    foreach {x y} [square] {
	set x [expr $dx*$x+$x0]
	set y [expr $dy*$y+$y0]
	lappend points $x $y
    }
    if {[expr abs($angle)] < 1e-10} {return $points}
    return [poly_rotate $x0 $y0 $angle $points]
}

body Shape::PolyToPoly { descr } {
   lappend descr [lindex $descr 0] [lindex $descr 1]
   return $descr
}

body Shape::PointToPoly { x0 y0 } {
    global powPlotParam regionParam

    set halfxMag [expr 0.5*$xScale]
    set halfyMag [expr 0.5*$yScale]
    if {$halfxMag<1} {set halfxMag 1}
    if {$halfyMag<1} {set halfyMag 1}
    set x1 [expr $x0-$halfxMag]
    set y1 [expr $y0-$halfyMag]
    set x2 [expr $x0+$halfxMag]
    set y2 [expr $y0+$halfyMag]
    return "$x1 $y1 $x1 $y2 $x2 $y2 $x2 $y1 $x1 $y1"
}


#########
#
#   Object coordinates for primitive shapes
#

# generate a list of points for a circle. every 12 degrees should do
    
set circle_points ""
set cnvt [expr -3.1415926535 / 180.0]
for {set i 0} {$i <= 360} {incr i 12} {
    lappend circle_points [expr cos($i * $cnvt)]
    lappend circle_points [expr sin($i * $cnvt)]
}


# return coords of unit circle, radius 1 at the origin

proc circle {} {
    global circle_points
    return $circle_points
}

# return coords of square (bounding box of unit circle)

proc square {} {
    return "1 1 1 -1 -1 -1 -1 1 1 1"
}

# rotate a polygon around x0 and y0, return new coordinates

proc poly_rotate {x0 y0 angle coords} {
   set st [expr sin(-3.1415926535 * $angle / 180.0)]
   set ct [expr cos(-3.1415926535 * $angle / 180.0)]
   set result ""
   foreach {x y} $coords {
       lappend result [expr $ct*$x - $st*$y + $x0*(1.0 - $ct) + $y0*$st]
       lappend result [expr $st*$x + $ct*$y + $y0*(1.0 - $ct) - $x0*$st]
   }
   return $result
}

#
#
#
########################################################################
