/* GADMIN-DHCPD - An easy to use GTK+ frontend for ISC DHCPD.
 * Copyright (C) 2004 - 2009 Magnus Loef <magnus-swe@telia.com> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
*/



#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "widgets.h"
#include "gettext.h"
#include "allocate.h"
#include "show_info.h"
#include "get_ranges_from_current_scope.h"
#include "delete_scope.h"
#include "populate_scope_treeview.h"
#include "select_first_scope.h"
#include "set_num_ranges.h"
#include "populate_ranges.h"
#include "populate_scope_settings.h"
#include "failover_functions.h"
#include "reread_conf.h"
#include "handle_ssh.h"


extern char global_netmask[1024];
extern char global_subnet[1024];
extern char global_nic[1024];
extern char DHCPD_CONF_BUF[1024];



void apply_scope_settings(struct w *widgets)
{
    FILE *fp;
    long z=0, conf_size = 0;
    char *line, *new_conf;
    gchar *new_scope_begin, *failover_decl;
    gchar *nic, *subnet, *netmask;
    G_CONST_RETURN gchar *change_nic, *change_subnet, *change_netmask;
    GtkTextBuffer *textbuffer;
    GtkTextIter start, end;
    gchar *comments = NULL;
    gchar **comment_list = NULL;
    gchar *setting=NULL, *info;
    char *ranges;
    int once = 0, new_scope_def = 0;

    change_nic = gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[0]));
    change_subnet = gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[1]));
    change_netmask = gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[2]));

    subnet = g_strdup_printf("%s", global_subnet);
    netmask = g_strdup_printf("%s", global_netmask);
    nic = g_strdup_printf("%s", global_nic);

    /* Save any failover declaration so we can put it back later */
    failover_decl = get_failover(nic, subnet, netmask, "return-entire-declaration");

    g_free(nic);
    g_free(subnet);
    g_free(netmask);

    if( strlen(change_nic) < 2 )
    {
        info = g_strdup_printf(_("The name of the network card is too short, the scope was not changed.\n"));
        show_info(info);
        g_free(info);
	g_free(failover_decl);
	return;
    }
    if( strlen(change_subnet) < 7 )
    {
        info = g_strdup_printf(_("The network address is too short, the scope was not changed.\n"));
        show_info(info);
        g_free(info);
	g_free(failover_decl);
	return;
    }
    if( strlen(change_netmask) < 7 )
    {
        info = g_strdup_printf(_("The subnet mask is too short, the scope was not changed.\n"));
        show_info(info);
        g_free(info);
	g_free(failover_decl);
	return;
    }

    /* Only update the scope treeview if the scope defs differs */
    if( strcmp(change_nic, global_nic) || strcmp(change_subnet, global_subnet) 
    || strcmp(change_netmask, global_netmask) )
      new_scope_def = 1;


    /* Save the ranges so we can put them back (Includes pool { declaration) */
    ranges = get_ranges_from_current_scope();

    new_scope_begin = g_strdup_printf("subnet %s netmask %s {\n    interface %s;\n", 
								      change_subnet,
							              change_netmask,
								      change_nic);
    /* Delete the current scope */
    delete_scope(widgets);


    /* Append the changed scope */
    if((fp=fopen(DHCPD_CONF_BUF, "a"))==NULL)
    {
        printf("Could not open dhcpd.conf here: %s\n", DHCPD_CONF_BUF);
	g_free(new_scope_begin);
	
	if( failover_decl!=NULL )
	  g_free(failover_decl);
        return;
    }

    /* Add the beginning of the scope */    
    fputs(new_scope_begin, fp);

    if( strlen(ranges) > 0 )
      fputs(ranges, fp);

    free(ranges); /* Allocated in get_ranges.. */

    g_free(new_scope_begin);


    /* Default lease time */
    if( strlen(gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[3])))>0 )
    {
        setting = g_strdup_printf("    default-lease-time %s;\n", 
	gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[3])));
	fputs(setting, fp);
    }

    /* Max lease time */
    if( strlen(gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[4])))>0 )
    {
        setting = g_strdup_printf("    max-lease-time %s;\n", 
	gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[4])));
	fputs(setting, fp);
    }

    /* Domain name */
    if( strlen(gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[5])))>0 )
    {
        setting = g_strdup_printf("    option domain-name \"%s\";\n", 
	gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[5])));
	fputs(setting, fp);
    }

    /* Subnet mask */
    if( strlen(gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[6])))>0 )
    {
        setting = g_strdup_printf("    option subnet-mask %s;\n", 
	gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[6])));
	fputs(setting, fp);
    }

    /* Broadcast address */
    if( strlen(gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[7])))>0 )
    {
        setting = g_strdup_printf("    option broadcast-address %s;\n", 
	gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[7])));
	fputs(setting, fp);
    }

    /* Broadcast address */
    if( strlen(gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[8])))>0 )
    {
        setting = g_strdup_printf("    option routers %s;\n", 
	gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[8])));
	fputs(setting, fp);
    }

    /* Domain name servers */
    if( strlen(gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[9])))>0 )
    {
        setting = g_strdup_printf("    option domain-name-servers %s;\n", 
	gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[9])));
	fputs(setting, fp);
    }

    /* NIS domain */
    if( strlen(gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[10])))>0 )
    {
        setting = g_strdup_printf("    option nis-domain \"%s\";\n", 
	gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[10])));
	fputs(setting, fp);
    }

    /* Time offset */
    if( strlen(gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[11])))>0 )
    {
        setting = g_strdup_printf("    option time-offset %s;\n", 
	gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[11])));
	fputs(setting, fp);
    }

    /* NTP servers */
    if( strlen(gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[12])))>0 )
    {
        setting = g_strdup_printf("    option ntp-servers %s;\n", 
	gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[12])));
	fputs(setting, fp);
    }

    /* Netbios name servers */
    if( strlen(gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[13])))>0 )
    {
        setting = g_strdup_printf("    option netbios-name-servers %s;\n", 
	gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[13])));
	fputs(setting, fp);
    }

    /* Option T150 */
    if( strlen(gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[14])))>0 )
    {
        setting = g_strdup_printf("    option T150 \"%s\";\n", 
	gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[14])));
	fputs(setting, fp);
    }

    /* Option filename */
    if( strlen(gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[15])))>0 )
    {
        setting = g_strdup_printf("    filename \"%s\";\n", 
	gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[15])));
	fputs(setting, fp);
    }

    /* Option root-path */
    if( strlen(gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[16])))>0 )
    {
        setting = g_strdup_printf("    option root-path \"%s\";\n", 
	gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[16])));
	fputs(setting, fp);
    }

    /* Option next-server */
    if( strlen(gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[17])))>0 )
    {
        setting = g_strdup_printf("    next-server %s;\n", 
	gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[17])));
	fputs(setting, fp);
    }

    /* Option x-display-manager */
    if( strlen(gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[18])))>0 )
    {
        setting = g_strdup_printf("    option x-display-manager %s;\n", 
	gtk_entry_get_text(GTK_ENTRY(widgets->scope_set_entry[18])));
	fputs(setting, fp);
    }

    /* Insert comments from the textview if any */
    textbuffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(widgets->scope_comment_textview));
    gtk_text_buffer_get_bounds(textbuffer, &start, &end);

    comments = gtk_text_iter_get_text(&start, &end);    
    if( comments!=NULL && strlen(comments) > 0 )
    {    
	comment_list = g_strsplit(comments, "\n", 0);
	if( comment_list!=NULL )
	{
	    for(z=0; comment_list[z]!=NULL; z++)
	    {
		/* Dont add a new line after the last row of text */
		if( strlen(comment_list[z]) < 1 && comment_list[z+1] == NULL )
		  break;
			  
		fputs("    # ", fp);
		fputs(comment_list[z], fp);
		fputs(";\n", fp);
	    }	
	    g_strfreev(comment_list);
	}    
    }
    if( comments!=NULL )
      g_free(comments);

    /* Add end of the scope */							
    setting = g_strdup_printf("}\n");
    fputs(setting, fp);
    if( setting!=NULL )
      g_free(setting);

    fclose(fp);


    /* Change the globals to new nic, subnet and netmask if any */
    snprintf(global_nic, 1000, "%s", change_nic);
    snprintf(global_subnet, 1000, "%s", change_subnet);
    snprintf(global_netmask, 1000, "%s", change_netmask);

    
    /* Insert the saved failover declaration after any global values but
       before scope declarations or other failovers */
    if((fp=fopen(DHCPD_CONF_BUF, "r"))==NULL)
    {
        printf("Could not open dhcpd.conf here: %s\n", DHCPD_CONF_BUF);
	g_free(failover_decl);
        return;
    }
    fseek(fp, 0, SEEK_END);
    conf_size = ftell(fp);

    rewind(fp);
    
    line = allocate(conf_size+1);        
    new_conf = allocate(conf_size+4096+1); /* 4096 for the added failover decl */


    /* Put the failover back */
    while(fgets(line, conf_size, fp)!=NULL)    
    {
	if( (strstr(line, "failover peer") || strstr(line, "subnet")) 
	&& strstr(line, "{") && ! once )
	{
	    once = 1;

	    if( failover_decl!=NULL )
	      strcat(new_conf, failover_decl);

	    strcat(new_conf, line);
	}
	else
	  strcat(new_conf, line);
    }
    fclose(fp);
    free(line);

    if( failover_decl!=NULL )
      g_free(failover_decl);


    /* Write the new file */
    if((fp=fopen(DHCPD_CONF_BUF, "w+"))==NULL)
    {
        printf("Could not write dhcpd.conf here: %s\n", DHCPD_CONF_BUF);
	free(new_conf);
        return;
    }
    fputs(new_conf, fp);
    fclose(fp);    

    free(new_conf);


    /* Only update the scope settings if the scope defs differs */
    if( new_scope_def )
    {
	populate_scope_treeview(widgets);
	select_first_scope(widgets);

	set_num_ranges();
	populate_ranges(widgets, global_nic, global_subnet, global_netmask);

	populate_scope_settings(widgets);
    }

    reread_conf();
    
    
    /* Ask to update remote loadbalancer with this local
       dhcpd.conf, if a loadbalancer has been configured */
}
