/* GADMIN-DHCPD - An easy to use GTK+ frontend for ISC DHCPD.
 * Copyright (C) 2004 - 2009 Magnus Loef <magnus-swe@telia.com> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
*/



#include "../config.h"
#include <gtk/gtk.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "allocate.h"
#include "gettext.h"
#include "widgets.h"
#include "functions.h"
#include "commented.h"
#include "chars_are_digits.h"
#include "make_settings_entries.h"
#include "make_settings_combos.h"
#include "populate_scope_treeview.h"
#include "select_first_scope.h"
#include "set_num_ranges.h"
#include "populate_ranges.h"
#include "populate_scope_settings.h"
#include "populate_host_treeview.h"
#include "populate_host_settings.h"
#include "populate_leases.h"
#include "reread_conf.h"
#include "settings_window.h"

extern char global_netmask[1024];
extern char global_subnet[1024];
extern char global_nic[1024];

extern char LEASE_FILE_BUF[1024];
extern char DHCPD_CONF_BUF[1024];



void apply_settings_button_clicked(struct w *widgets)
{
    /* The apply button in the settings window */
    FILE *fp;
    long file_size;
    char *line, *new_conf;
    gint active_index = 0;
    gchar *cmd, *settings_file, *setting;

    G_CONST_RETURN gchar *conf_file = gtk_entry_get_text(GTK_ENTRY(widgets->settings_entry[0]));
    G_CONST_RETURN gchar *lease_file = gtk_entry_get_text(GTK_ENTRY(widgets->settings_entry[1]));
    G_CONST_RETURN gchar *default_lease_time = gtk_entry_get_text(GTK_ENTRY(widgets->settings_entry[2]));

    settings_file = g_strdup_printf("%s/settings.conf", SETTINGS_PATH);

    /* If the settings directory doesnt exist, create it */
    if( ! file_exists(SETTINGS_PATH) )
    {
	cmd = g_strdup_printf("mkdir -p %s", SETTINGS_PATH);
	if( ! run_command(cmd) )
	{
	    printf("Can not run command:\n%s\n", cmd);
	    g_free(cmd);
	    g_free(settings_file);
	    return;
	}
	g_free(cmd);
    }

    /* Save the paths */
    if((fp=fopen(settings_file, "w+"))==NULL)
    {
	printf("Can not open the settings file here:\n%s\n", settings_file);
	g_free(settings_file);
	return;
    }
    
    setting = g_strdup_printf("dhcpd_conf: %s\n", conf_file);
    fputs(setting, fp);
    g_free(setting);

    setting = g_strdup_printf("lease_file: %s\n", lease_file);
    fputs(setting, fp);
    g_free(setting);

    fclose(fp);
    g_free(settings_file);


    /* Make the paths current. */
    sprintf(DHCPD_CONF_BUF, "%s", conf_file);
    sprintf(LEASE_FILE_BUF, "%s", lease_file);


    if( ! file_exists((char *)conf_file) )
      printf("Creating a new dhcpd.conf here: %s\n", conf_file);


    /* Creates a default conf if its missing */
    add_missing_values((gchar *)conf_file);


    if((fp=fopen(DHCPD_CONF_BUF, "r"))==NULL)
    {
	printf("Cant open: %s\n", DHCPD_CONF_BUF);
	return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);
    
    line = allocate(file_size+1);
    new_conf = allocate(file_size+8192); /* For changes */

    /* DDNS update style */
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->settings_combo[0]));
    if( active_index <= 0 )
      strcat(new_conf, "ddns-update-style ad-hoc;\n");
    else
    if( active_index == 1 )
      strcat(new_conf, "ddns-update-style interim;\n");
    else
      strcat(new_conf, "ddns-update-style none;\n");

    /* DDNS updates */
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->settings_combo[1]));
    if( active_index <= 0 )
      strcat(new_conf, "ddns-updates on;\n");
    else
    if( active_index == 1 )
      strcat(new_conf, "ddns-updates off;\n");
    else
      strcat(new_conf, "ddns-updates off;\n");
      
    /* Client updates */    
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->settings_combo[2]));
    if( active_index <= 0 )
      strcat(new_conf, "allow client-updates;\n");
    else
    if( active_index == 1 )
      strcat(new_conf, "deny client-updates;\n");
    else
      strcat(new_conf, "ignore client-updates;\n");

    /* One lease per client */    
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->settings_combo[3]));
    if( active_index <= 0 )
      strcat(new_conf, "one-lease-per-client true;\n");
    else
      strcat(new_conf, "one-lease-per-client false;\n");

    /* BOOTP */    
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->settings_combo[4]));
    if( active_index <= 0 )
      strcat(new_conf, "allow bootp;\n");
    else
    if( active_index == 1 )
      strcat(new_conf, "deny bootp;\n");
    else
      strcat(new_conf, "ignore bootp;\n");

    /* Booting */    
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->settings_combo[5]));
    if( active_index <= 0 )
      strcat(new_conf, "allow booting;\n");
    else
    if( active_index == 1 )
      strcat(new_conf, "deny booting;\n");
    else
      strcat(new_conf, "ignore booting;\n");

    /* Unknown clients */    
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->settings_combo[6]));
    if( active_index <= 0 )
      strcat(new_conf, "allow unknown-clients;\n");
    else
    if( active_index == 1 )
      strcat(new_conf, "deny unknown-clients;\n");
    else
      strcat(new_conf, "ignore unknown-clients;\n");

    /* Authoritative */    
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->settings_combo[7]));
    if( active_index <= 0 )
      strcat(new_conf, "authoritative;\n");
    else
    if( active_index == 1 )
      strcat(new_conf, "not authoritative;\n");
    else
      strcat(new_conf, "not authoritative;\n");

    /* Option ip-forwarding */    
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->settings_combo[8]));
    if( active_index <= 0 )
      strcat(new_conf, "option ip-forwarding true;\n");
    else
    if( active_index == 1 )
      strcat(new_conf, "option ip-forwarding false;\n");
    else
      strcat(new_conf, "option ip-forwarding false;\n");

    /* Option mask-supplier */    
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->settings_combo[9]));
    if( active_index <= 0 )
      strcat(new_conf, "option mask-supplier true;\n");
    else
    if( active_index == 1 )
      strcat(new_conf, "option mask-supplier false;\n");
    else
      strcat(new_conf, "option mask-supplier false;\n");

    /* Default-lease-time */
    strcat(new_conf, "default-lease-time ");
    strcat(new_conf, default_lease_time);
    strcat(new_conf, ";\n");


    /* Get everything but the values to change */
    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
	/* Subnet, host or failover declaration reached. */
	if( ! commented(line) && strstr(line, "{") )
	{
	    strcat(new_conf, line);
	    break;
	}
    
	if( ! strstr(line, "ddns-update-style")
	&&  ! strstr(line, "ddns-updates")
	&&  ! strstr(line, "client-updates;")
	&&  ! strstr(line, "one-lease-per-client")
	&&  ! strstr(line, "bootp;")
	&&  ! strstr(line, "booting;")
	&&  ! strstr(line, "unknown-clients;")
	&&  ! strstr(line, "authoritative;")
	&&  ! strstr(line, "option ip-forwarding")
	&&  ! strstr(line, "option mask-supplier")
	&&  ! strstr(line, "default-lease-time ") )
	  strcat(new_conf, line);
    }
    /* Get everything after the global options */
    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
	strcat(new_conf, line);
    }
    fclose(fp);
    free(line);

    /* Write the new conf */
    if((fp=fopen(DHCPD_CONF_BUF, "w+"))==NULL)
    {
	printf("Cant write: %s\n", DHCPD_CONF_BUF);
	free(new_conf);
	return;
    }
    fputs(new_conf, fp);
    fclose(fp);

    free(new_conf);

    
    /* Reread entire gui */
    populate_scope_treeview(widgets);
    select_first_scope(widgets);

    set_num_ranges();
    populate_ranges(widgets, global_nic, global_subnet, global_netmask);

    populate_scope_settings(widgets);

    populate_host_treeview(widgets);
    populate_host_settings(widgets);

    populate_leases(widgets);
    
    gtk_widget_destroy(widgets->settings_window);

    reread_conf();
}


void cancel_settings_button_clicked(struct w *widgets)
{
    gtk_widget_destroy(widgets->settings_window);
}


void show_settings_window(struct w *widgets)
{
    FILE *fp;
    long file_size;
    int i=0, a=0, b=1;
    char *line, *tmp;
    GtkWidget *frame;
    GtkWidget *table;
    GtkWidget *settings_vbox;
    GtkTooltips *tooltips;
    gchar *info;
    gchar *utf8=NULL;

    widgets->settings_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_position(GTK_WINDOW (widgets->settings_window), GTK_WIN_POS_CENTER);
    gtk_widget_set_size_request(widgets->settings_window, -1, -1);

    /* Set window information */
    info = g_strdup_printf(_("GADMIN-DHCPD %s settings"), VERSION);
    gtk_window_set_title(GTK_WINDOW(widgets->settings_window), info);
    g_free(info);

    g_signal_connect(GTK_WINDOW(widgets->settings_window), "delete_event", 
		     G_CALLBACK (gtk_widget_destroy), NULL);

    settings_vbox = gtk_vbox_new(FALSE, 0);
    gtk_container_add (GTK_CONTAINER (widgets->settings_window), settings_vbox);


    tooltips = gtk_tooltips_new();
    frame = gtk_frame_new("Settings:");

    /* A table with 16 settings and 2 columns */
    table = gtk_table_new(16, 2, FALSE);

    gtk_box_pack_start(GTK_BOX(settings_vbox), frame, TRUE, TRUE, 1);
    gtk_container_add(GTK_CONTAINER(frame), table);

    /* Max length and input 350 chars */
    widgets->settings_entry[0] = make_entry_with_label(GTK_TABLE(table), _(" Path to dhcpd.conf: "),   0,1,a,b,350);
    gtk_tooltips_set_tip(tooltips, widgets->settings_entry[0], _("/path/to/dhcpd.conf"), NULL);
    a++; b++;

    widgets->settings_entry[1] = make_entry_with_label(GTK_TABLE(table), _(" Path to dhcpd.leases: "), 0,1,a,b,350);
    gtk_tooltips_set_tip(tooltips, widgets->settings_entry[1], _("/path/to/dhcpd.leases"), NULL);
    a++; b++;

    widgets->settings_entry[2] = make_entry_with_label(GTK_TABLE(table), _(" Default lease time: "),    0,1,a,b,350);
    gtk_tooltips_set_tip(tooltips, widgets->settings_entry[2], "604800", NULL);
    a++; b++;

    widgets->settings_combo[0] = make_combo_with_label(GTK_TABLE(table), _(" DDNS update style: "),    0,1,a,b,350);
    a++; b++;
    widgets->settings_combo[1] = make_combo_with_label(GTK_TABLE(table), _(" DDNS updates: "),         0,1,a,b,350);
    a++; b++;
    widgets->settings_combo[2] = make_combo_with_label(GTK_TABLE(table), _(" Client updates: "),       0,1,a,b,350);
    a++; b++;
    widgets->settings_combo[3] = make_combo_with_label(GTK_TABLE(table), _(" One lease per client: "), 0,1,a,b,350);
    a++; b++;
    widgets->settings_combo[4] = make_combo_with_label(GTK_TABLE(table), _(" BOOTP: "),                0,1,a,b,350);
    a++; b++;
    widgets->settings_combo[5] = make_combo_with_label(GTK_TABLE(table), _(" Booting: "),              0,1,a,b,350);
    a++; b++;
    widgets->settings_combo[6] = make_combo_with_label(GTK_TABLE(table), _(" Unknown clients: "),      0,1,a,b,350);
    a++; b++;
    widgets->settings_combo[7] = make_combo_with_label(GTK_TABLE(table), _(" Authoritative: "),        0,1,a,b,350);
    a++; b++;
    widgets->settings_combo[8] = make_combo_with_label(GTK_TABLE(table), _(" IP-forwarding: "),        0,1,a,b,350);
    a++; b++;
    widgets->settings_combo[9] = make_combo_with_label(GTK_TABLE(table), _(" Mask-supplier: "),        0,1,a,b,350);
    a++; b++;

    /* Set entry path texts */
    utf8 = g_locale_to_utf8(DHCPD_CONF_BUF, strlen(DHCPD_CONF_BUF), NULL, NULL, NULL);
    gtk_entry_set_text(GTK_ENTRY(widgets->settings_entry[0]), utf8);

    utf8 = g_locale_to_utf8(LEASE_FILE_BUF, strlen(LEASE_FILE_BUF), NULL, NULL, NULL);
    gtk_entry_set_text(GTK_ENTRY(widgets->settings_entry[1]), utf8);

    GtkWidget *settings_hbuttonbox = gtk_hbutton_box_new();
    gtk_box_pack_start(GTK_BOX(settings_vbox), settings_hbuttonbox, FALSE, FALSE, 0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(settings_hbuttonbox), GTK_BUTTONBOX_SPREAD);

    GtkWidget *apply_settings_button = gtk_button_new_from_stock(GTK_STOCK_APPLY);
    gtk_container_add(GTK_CONTAINER(settings_hbuttonbox), apply_settings_button);
    g_signal_connect_swapped(G_OBJECT(apply_settings_button), "clicked", 
                             G_CALLBACK(apply_settings_button_clicked), widgets);
    
    GtkWidget *cancel_settings_button = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
    gtk_container_add(GTK_CONTAINER(settings_hbuttonbox), cancel_settings_button);
    g_signal_connect_swapped(G_OBJECT(cancel_settings_button), "clicked", 
                             G_CALLBACK(cancel_settings_button_clicked), widgets);

    gtk_widget_show_all(widgets->settings_window);


    /* Set default combo options */
    tmp = allocate(1024);

    snprintf(tmp, 1000, "%s", _("Ad-hoc"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[0]), utf8);
    snprintf(tmp, 1000, "%s", _("Interim"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[0]), utf8);
    snprintf(tmp, 1000, "%s", _("None"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[0]), utf8);

    snprintf(tmp, 1000, "%s", _("Yes"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[1]), utf8);
    snprintf(tmp, 1000, "%s", _("No"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[1]), utf8);

    snprintf(tmp, 1000, "%s", _("Allow"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[2]), utf8);
    snprintf(tmp, 1000, "%s", _("Deny"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[2]), utf8);
    snprintf(tmp, 1000, "%s", _("Ignore"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[2]), utf8);

    snprintf(tmp, 1000, "%s", _("Yes"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[3]), utf8);
    snprintf(tmp, 1000, "%s", _("No"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[3]), utf8);

    snprintf(tmp, 1000, "%s", _("Allow"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[4]), utf8);
    snprintf(tmp, 1000, "%s", _("Deny"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[4]), utf8);
    snprintf(tmp, 1000, "%s", _("Ignore"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[4]), utf8);

    snprintf(tmp, 1000, "%s", _("Allow"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[5]), utf8);
    snprintf(tmp, 1000, "%s", _("Deny"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[5]), utf8);

    snprintf(tmp, 1000, "%s", _("Allow"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[6]), utf8);
    snprintf(tmp, 1000, "%s", _("Deny"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[6]), utf8);

    snprintf(tmp, 1000, "%s", _("Yes"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[7]), utf8);
    snprintf(tmp, 1000, "%s", _("No"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[7]), utf8);

    snprintf(tmp, 1000, "%s", _("Yes"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[8]), utf8);
    snprintf(tmp, 1000, "%s", _("No"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[8]), utf8);

    snprintf(tmp, 1000, "%s", _("Yes"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[9]), utf8);
    snprintf(tmp, 1000, "%s", _("No"));
    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->settings_combo[9]), utf8);

    /* Select the first of every combo option */
    for(i=0; i<10; i++)
       gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[i]), 0);

    if( utf8!=NULL )
      g_free(utf8);

    /* Insert values from dhcpd.conf if any */
    if((fp=fopen(DHCPD_CONF_BUF, "r"))==NULL)
    {
	free(tmp);    
	return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);
    line = allocate(file_size+1);

    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {    
	if( commented(line) )
	  continue;

	if( strstr(line, "{") )
	  break;

	if( strstr(line, "ddns-update-style") )
	{
	    /* Set selected. Cut 18 chars from the beginning of the line */
	    snprintf(tmp, 1000, "%s", &line[18]);
	    /* Cut ; and newline */
	    tmp[strlen(tmp)-2]='\0';
	    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
	    if( strcmp(tmp, "ad-hoc") == 0 )
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[0]), 0);
	    if( strcmp(tmp, "interim") == 0 )
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[0]), 1);
	    if( strcmp(tmp, "none") == 0 )
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[0]), 2);
	}
	else
	if( strstr(line, "ddns-updates") )
	{
	    /* Cut 21 chars from the beginning of the line */
	    snprintf(tmp, 1000, "%s", &line[13]);
	    tmp[strlen(tmp)-2]='\0';
	    if( strcmp(tmp, "on") == 0 )
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[1]), 0);
	    if( strcmp(tmp, "off") == 0 )
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[1]), 1);
	}
	else
	if( strstr(line, "client-updates;") )
	{
	    snprintf(tmp, 1000, "%s", line);
	    tmp[strlen(tmp)-17]='\0';
	    if( strcmp(tmp, "allow") == 0 )
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[2]), 0);
	    if( strcmp(tmp, "deny") == 0 )
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[2]), 1);
	    if( strcmp(tmp, "ignore") == 0 )
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[2]), 2);
	}
	else
	if( strstr(line, "one-lease-per-client") )
	{
	    /* Cut 21 chars from the beginning of the line */
	    snprintf(tmp, 1000, "%s", &line[21]);
	    tmp[strlen(tmp)-2]='\0';
	    if( strcmp(tmp, "true") == 0 )
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[3]), 0);
	    if( strcmp(tmp, "false") == 0 )
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[3]), 1);
	}
        else
	if( strstr(line, "bootp;") )
	{
	    snprintf(tmp, 1000, "%s", line);
	    tmp[strlen(tmp)-8]='\0';
	    if( strcmp(tmp, "allow") == 0 )
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[4]), 0);
	    if( strcmp(tmp, "deny") == 0 )
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[4]), 1);
	    if( strcmp(tmp, "ignore") == 0 )
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[4]), 2);
	}
        else
	if( strstr(line, "booting;") )
	{
	    snprintf(tmp, 1000, "%s", line);
	    tmp[strlen(tmp)-1]='\0';
	    if( strstr(tmp, "allow booting;") )
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[5]), 0);
	    else
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[5]), 1);
	}
	if( strstr(line, "unknown-clients;") )
	{
	    snprintf(tmp, 1000, "%s", line);
	    tmp[strlen(tmp)-1]='\0';
	    if( strstr(tmp, "allow") )
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[6]), 0);
	    else
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[6]), 1);
	}
	else
	if( strstr(line, "authoritative;") )
	{
	    snprintf(tmp, 1000, "%s", line);
	    tmp[strlen(tmp)-1]='\0';
	    if( ! strstr(tmp, "not ") )
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[7]), 0);
	    else
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[7]), 1);
	}
	else
	if( strstr(line, "option ip-forwarding") )
	{
	    snprintf(tmp, 1000, "%s", line);
	    tmp[strlen(tmp)-1]='\0';
	    if( strstr(tmp, "true") )
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[8]), 0);
	    else
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[8]), 1);
	}
	else
	if( strstr(line, "option mask-supplier") )
	{
	    snprintf(tmp, 1000, "%s", line);
	    tmp[strlen(tmp)-1]='\0';
	    if( strstr(tmp, "true") )
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[9]), 0);
	    else
	      gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->settings_combo[9]), 1);
	}
	else
	if( strstr(line, "default-lease-time ") )
	{
	    /* Isolate the digits. */
	    for(i=0; i<strlen(line)-1 && i<1000; i++)
	    {
	    	snprintf(tmp, 1000, "%s", &line[i]);
		tmp[strlen(tmp)-2]='\0'; /* Remove the last ";" char. */
		if( chars_are_digits(tmp) )
		{
		    utf8 = g_locale_to_utf8(tmp, strlen(tmp), NULL, NULL, NULL);
		    gtk_entry_set_text(GTK_ENTRY(widgets->settings_entry[2]), utf8);
		    break;
		}
	    }
	}
    }
    fclose(fp);
    free(line);
    free(tmp);

    if( utf8!=NULL )
      g_free(utf8);
}
