/**
 * @file person-list.c Person list
 *
 * Copyright (C) 2004-2006 Christian Hammond.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include "person-list.h"
#include "meta-account.h"

GHashTable *meta_people_table = NULL;
static GList *meta_people = NULL;
static GalagodMetaPerson *meta_me = NULL;
static size_t meta_person_count = 0;

static GalagodMetaPerson *
_find_existing_meta_person(GalagoPerson *person)
{
	GalagodMetaPerson *meta_person;
	GalagodMetaPerson *main_meta_person;
	const GList *l;

	main_meta_person = GALAGOD_META_PERSON(person);

	for (l = galago_person_get_accounts(person, FALSE);
		 l != NULL;
		 l = l->next)
	{
		GalagoAccount *account = (GalagoAccount *)l->data;

		meta_person = galagod_people_find_with_account(account);

		if (meta_person != NULL && main_meta_person != meta_person)
			return meta_person;
	}

	return NULL;
}

static void
_set_meta_persons(GalagoPerson *person, GalagodMetaPerson *meta_person)
{
	GalagodMetaPerson *meta_person2;
	const GList *l;
	GList *tmp_meta_people = NULL;

	meta_person2 = GALAGOD_META_PERSON(person);

	if (meta_person2 != NULL && meta_person2 != meta_person)
		tmp_meta_people = g_list_append(tmp_meta_people, meta_person2);

	for (l = galago_person_get_accounts(person, FALSE);
		 l != NULL;
		 l = l->next)
	{
		GalagoAccount *account = GALAGO_ACCOUNT(l->data);

		meta_person2 = galagod_people_find_with_account(account);

		if (meta_person != meta_person2 &&
			!g_list_find(tmp_meta_people, meta_person2))
		{
			tmp_meta_people = g_list_append(tmp_meta_people, meta_person2);
		}
	}

	for (l = tmp_meta_people; l != NULL; l = l->next)
	{
		meta_person2 = (GalagodMetaPerson *)l->data;

		meta_people = g_list_remove(meta_people, meta_person2);

		galagod_meta_person_merge(meta_person, meta_person2);
		meta_person_count--;
	}

	g_list_free(tmp_meta_people);
}

void
galagod_people_add(GalagoPerson *person)
{
	GalagodMetaPerson *meta_person;
	GalagoPerson *main_person;

	g_return_if_fail(person != NULL);

	if ((meta_person = _find_existing_meta_person(person)) != NULL)
	{
		_set_meta_persons(person, meta_person);
	}
	else
	{
		meta_person = galagod_meta_person_new();
	}

	if (galago_person_is_me(person) && GALAGOD_META_PERSON(person) != meta_me)
	{
		galagod_meta_person_destroy(meta_person);
		_set_meta_persons(person, meta_me);
		meta_person = meta_me;
	}

	galagod_meta_person_add_person(meta_person, person);

	main_person = galagod_meta_person_get_person(meta_person);

	if (galago_person_is_me(person) && !galago_person_is_me(main_person))
		galago_person_set_me(main_person);
}

void
galagod_people_remove(GalagoPerson *person)
{
	GalagodMetaPerson *meta_person;

	g_return_if_fail(person != NULL);

	meta_person = GALAGOD_META_PERSON(person);
	g_return_if_fail(meta_person != NULL);

	galagod_meta_person_remove_person(meta_person, person);

	if (galagod_meta_person_get_people(meta_person) == NULL &&
		meta_person != meta_me)
	{
		galagod_meta_person_destroy(meta_person);
	}
}

void
galagod_people_remove_with_coco(GalagodCoCo *coco)
{
	const GList *l;

	g_return_if_fail(coco != NULL);

	galago_context_push(galagod_coco_get_context(coco));

	for (l = galago_context_get_people(GALAGO_REMOTE); l != NULL; l = l->next)
		galagod_people_remove((GalagoPerson *)l->data);

	galago_context_pop();
}

void
galagod_people_add_meta_person(GalagodMetaPerson *meta_person)
{
	g_return_if_fail(meta_person != NULL);

	meta_people = g_list_append(meta_people, meta_person);
	meta_person_count++;
}

void
galagod_people_remove_meta_person(GalagodMetaPerson *meta_person)
{
	const char *id;

	g_return_if_fail(meta_person != NULL);

	id = galago_person_get_id(galagod_meta_person_get_person(meta_person));

	if (id != NULL)
		g_hash_table_remove(meta_people_table, id);

	meta_people = g_list_remove(meta_people, meta_person);

	meta_person_count--;
}

void
galagod_people_account_added(GalagoPerson *person)
{
	GalagodMetaPerson *meta_person;

	g_return_if_fail(person != NULL);

	if ((meta_person = _find_existing_meta_person(person)) != NULL)
		_set_meta_persons(person, meta_person);
}

GalagodMetaPerson *
galagod_people_get_meta_person(const char *id)
{
	g_return_val_if_fail(id  != NULL, NULL);
	g_return_val_if_fail(*id != '\0', NULL);

	return g_hash_table_lookup(meta_people_table, id);
}

const GList *
galagod_people_get_meta_people(void)
{
	return meta_people;
}

size_t
galagod_people_get_meta_person_count(void)
{
	return meta_person_count;
}

GalagodMetaPerson *
galagod_people_get_me(void)
{
	return meta_me;
}

GalagodMetaPerson *
galagod_people_find_with_account(const GalagoAccount *account)
{
	GalagoPerson *person;
	GalagodMetaAccount *meta_account;
	GalagoAccount *main_account;

	g_return_val_if_fail(account != NULL, NULL);

	meta_account = GALAGOD_META_ACCOUNT(account);
	g_return_val_if_fail(meta_account != NULL, NULL);

	main_account = galagod_meta_account_get_account(meta_account);
	person = galago_account_get_person(main_account);

	return GALAGOD_META_PERSON(person);
}

void
galagod_people_init(void)
{
	meta_people_table = g_hash_table_new_full(g_str_hash, g_str_equal,
											  g_free, NULL);

	/* Create our "Me" person. */
	meta_me = galagod_meta_person_new();
	galago_person_set_me(galagod_meta_person_get_person(meta_me));

	g_hash_table_insert(meta_people_table,
						g_strdup(galago_person_get_id(meta_me->person)),
						meta_me);
}

void
galagod_people_uninit(void)
{
	g_hash_table_destroy(meta_people_table);
	meta_people_table = NULL;

	g_list_free(meta_people);
	meta_people = NULL;

	galagod_meta_person_destroy(meta_me);
	meta_me = NULL;

	meta_person_count = 0;
}
