/* Copyright (C) 2000 Damir Zucic */

/*=============================================================================

				init_runtime.c

Purpose:
	Initialize some runtime data and allocate required memory.

Input:
	(1) Pointer to RuntimeS structure.

Output:
	(1) Runtime parameters initialized.
	(2) Return value.

Return value:
	(1) Positive on success.
	(2) Negative on failure.

Notes:
	(1) The buffers exposed_atom1IP and exposed_atom2IP will be
	    allocated only if docking is required. Otherwise, there
	    is no need to allocate these buffers. The allocation is
	    done in a function which creates docking window.

========includes:============================================================*/

#include <stdio.h>

#include <stdlib.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include "defines.h"
#include "typedefs.h"

/*======function prototypes:=================================================*/

void		ErrorMessage_ (char *, char *, char *,
			       char *, char *, char *, char *);

/*======initialize runtime:==================================================*/

int InitRuntime_ (RuntimeS *runtimeSP)
{
size_t		char_size, unsigned_char_size;
size_t		int_size, short_int_size, float_size;
size_t		elementsN;
int		i;
double		d;
int		width;

/* Initialize the scriptF (0 = interactive mode, 1 = script mode): */
runtimeSP->scriptF = 0;

/* Initialize the pauseF (0 = do not stop execution, 1 = pause): */
runtimeSP->pauseF = 0;

/* Default caught macromolecular complex: */
runtimeSP->default_complexI = 0;

/* Size of a single character: */
char_size = sizeof (char);

/* Size of unsigned character: */
unsigned_char_size = sizeof (unsigned char);

/* Size of integer: */
int_size = sizeof (int);

/* Size of short integer: */
short_int_size = sizeof (short int);

/* Size of float: */
float_size = sizeof (float);

/* Allocate memory for command strings (the history buffer): */
elementsN = MAXCOMMSTRINGS * COMMSTRINGSIZE;
runtimeSP->commandsP = (char *) calloc (elementsN, char_size);
if (runtimeSP->commandsP == NULL)
	{
	ErrorMessage_ ("garlic", "InitRuntime_", "",
		"Failed to allocate memory for command strings!\n",
		"", "", "");
	return -1;
	}

/* Initialize the curr_commandI and the highest_commandI: */
runtimeSP->curr_commandI  = 0;
runtimeSP->highest_commandI = 0;

/* Initialize the carriage (keyboard cursor) position: */
runtimeSP->carriage_position = 0;

/* Zero initialize the curr_commandA: */
runtimeSP->curr_commandA[0] = '\0';

/* Initialize the length of the current command and carriage position: */
runtimeSP->command_length = 0;
runtimeSP->carriage_position = 0;

/* Zero initialize the messageA: */
runtimeSP->messageA[0] = '\0';

/* Initialize the length of the message: */
runtimeSP->message_length = 0;

/* Allocate memory for titles: */
elementsN = MAXTITLES * TITLESTRINGSIZE;
runtimeSP->titlesP = (char *) calloc (elementsN, char_size);
if (runtimeSP->titlesP == NULL)
	{
	ErrorMessage_ ("garlic", "InitRuntime_", "",
		       "Failed to allocate memory for title strings!\n",
		       "", "", "");
	return -2;
	}

/* All titles are initially hidden: */
for (i = 0; i < MAXTITLES; i++) runtimeSP->title_hiddenF[i] = 1;

/* Initialize positions of titles: */
for (i = 0; i < MAXTITLES; i++)
	{
	runtimeSP->title_screen_x[i] = 10;
	runtimeSP->title_screen_y[i] = 10;
	}

/* Initialize the size of the sequence buffer: */
runtimeSP->sequence_buffer_size = MAXRESIDUES * (RESNAMESIZE - 1);

/* Initialize the number of residues in the sequence buffer: */
runtimeSP->residuesN = 0;

/* Allocate the sequence buffer: */
elementsN = runtimeSP->sequence_buffer_size;
runtimeSP->sequenceP = (char *) calloc (elementsN, char_size);
if (runtimeSP->sequenceP == NULL)
	{
	ErrorMessage_ ("garlic", "InitRuntime_", "",
		       "Failed to allocate the sequence buffer!\n",
		       "", "", "");
	return -3;
	}

/* Allocate memory for disulfide flags (see typedefs.h): */
elementsN = MAXRESIDUES;
runtimeSP->disulfideFP =
		(unsigned char *) calloc (elementsN, unsigned_char_size);
if (runtimeSP->disulfideFP == NULL)
	{
	ErrorMessage_ ("garlic", "InitRuntime_", "",
		       "Failed to allocate memory for disulfide flags!\n",
		       "", "", "");
	return -4;
	}

/* Allocate memory for serial numbers: */
elementsN = MAXRESIDUES;
runtimeSP->serialIP = (int *) calloc (elementsN, int_size);
if (runtimeSP->serialIP == NULL)
	{
	ErrorMessage_ ("garlic", "InitRuntime_", "",
		       "Failed to allocate memory",
		       " for residue serial numbers!\n",
		       "", "");
	return -5;
	}

/* Allocate memory for numeric residue codes: */
elementsN = MAXRESIDUES;
runtimeSP->residue_codeIP = (short int *) calloc (elementsN, short_int_size);
if (runtimeSP->residue_codeIP == NULL)
	{
	ErrorMessage_ ("garlic", "InitRuntime_", "",
		       "Failed to allocate memory",
		       " for numeric residue codes!\n",
		       "", "");
	return -6;
	}

/* Allocate memory for hydrophobicity values: */
elementsN = MAXRESIDUES;
runtimeSP->hydrophobicityP = (float *) calloc (elementsN, float_size);
if (runtimeSP->hydrophobicityP == NULL)
	{
	ErrorMessage_ ("garlic", "InitRuntime_", "",
		       "Failed to allocate memory",
		       " for hydrophobicity values!\n",
		       "", "");
	return -7;
	}

/* Allocate memory for averaged hydrophobicity: */
elementsN = MAXRESIDUES;
runtimeSP->averaged_hydrophobicityP = (float *) calloc (elementsN, float_size);
if (runtimeSP->averaged_hydrophobicityP == NULL)
	{
	ErrorMessage_ ("garlic", "InitRuntime_", "",
		       "Failed to allocate memory",
		       " for averaged hydrophobicity!\n",
		       "", "");
	return -8;
	}

/* Allocate memory for hydrophobic moment: */
elementsN = MAXRESIDUES;
runtimeSP->hydrophobic_momentP = (float *) calloc (elementsN, float_size);
if (runtimeSP->hydrophobic_momentP == NULL)
	{
	ErrorMessage_ ("garlic", "InitRuntime_", "",
		       "Failed to allocate memory",
		       " for hydrophobic moment!\n",
		       "", "");
	return -9;
	}

/* Initialize the number of residues in the reference sequence buffer: */
runtimeSP->reference_residuesN = 0;

/* Allocate the reference sequence buffer: */
elementsN = runtimeSP->sequence_buffer_size;
runtimeSP->reference_sequenceP = (char *) calloc (elementsN, char_size);
if (runtimeSP->reference_sequenceP == NULL)
	{
	ErrorMessage_ ("garlic", "InitRuntime_", "",
		       "Failed to allocate the reference sequence buffer!\n",
		       "", "", "");
	return -10;
	}

/* Allocate memory for reference serial numbers: */
elementsN = MAXRESIDUES;
runtimeSP->reference_serialIP = (int *) calloc (elementsN, int_size);
if (runtimeSP->reference_serialIP == NULL)
	{
	ErrorMessage_ ("garlic", "InitRuntime_", "",
		       "Failed to allocate memory",
		       " for reference residue serial numbers!\n",
		       "", "");
	return -11;
	}

/* Allocate memory for reference numeric residue codes: */
elementsN = MAXRESIDUES;
runtimeSP->reference_residue_codeIP =
			(short int *) calloc (elementsN, short_int_size);
if (runtimeSP->reference_residue_codeIP == NULL)
        {
        ErrorMessage_ ("garlic", "InitRuntime_", "",
                       "Failed to allocate memory",
                       " for reference numeric residue codes!\n",
                       "", "");
        return -12;
        }

/* Allocate memory for reference hydrophobicity values: */
elementsN = MAXRESIDUES;
runtimeSP->reference_hydrophobicityP =
			(float *) calloc (elementsN, float_size);
if (runtimeSP->reference_hydrophobicityP == NULL)
	{
	ErrorMessage_ ("garlic", "InitRuntime_", "",
		       "Failed to allocate memory",
		       " for reference hydrophobicity values!\n",
		       "", "");
	return -13;
	}

/* Initialize the zoom factor: */
runtimeSP->zoom_factor = 2;

/* Initialize  the sequence offsets.  Some  sequences */
/* are long,  so they  will not fit into main window. */
/* This may also happen if large zoom factor is used. */
runtimeSP->sequence_offset = 0;
runtimeSP->reference_offset = 0;

/* Initialize the segment width and the minimal */
/* score  (used when comparing  two sequences): */
runtimeSP->segment_width = SEGMENT_WIDTH;
runtimeSP->minimal_score = MINIMAL_SCORE;

/* Initialize the rama_selectionF: */
runtimeSP->rama_selectionF = 0;

/* Initialize the hydrophobicity scale index (default: Eisenberg scale, 2): */
runtimeSP->hydrophobicity_scaleI = 2;

/* Initialize the helix step angle, in degrees: */
runtimeSP->helix_step_angle = 100.0;

/* Initialize the arc (extent) angle, in degrees: */
runtimeSP->arc_angle = 10.0;

/* Initialize the sphere radius: */
runtimeSP->sphere_radius = 8.0;

/* Initialize the sliding window width: */
runtimeSP->sliding_window_width = 11;

/* Initialize flags related to the plotting of hydrophobicity functions: */
runtimeSP->averaged_hydrophobicityF = 0;
runtimeSP->hydrophobic_momentF      = 0;

/* Initialize range indices: */
runtimeSP->range_startI = 0;
runtimeSP->range_endI   = 0;

/* Initialize docking area width and docking cell width: */
runtimeSP->docking_area_width = DOCKING_AREA_WIDTH;
runtimeSP->docking_cell_width = DOCKING_CELL_WIDTH;

/* Initialize  the number of cells  across */
/* docking area; this number must be even. */
d = DOCKING_AREA_WIDTH / DOCKING_CELL_WIDTH;
width = (int) d;
if (width % 2) width++;
runtimeSP->docking_matrix_width = width;

/* Return positive value on success: */
return 1;
}

/*===========================================================================*/


